/* ======================================================================

	Functions to manage string buffers

	Filename:			BufType.c
	Last Edited:		March 7, 1997
	Authors:			Scott Manjourides, Laurence Lundblade, Bob Fronabarger
	Copyright:			1995, 1996 QUALCOMM Inc.
	Technical support:	<emsapi-info@qualcomm.com>
*/

#include <string.h>
#include "buftype.h"


static Boolean ValidBuf(BufTypeHandle bh);
static Boolean ResizeBuf_Buf(BufTypeHandle dst, unsigned long newSize);

//-------------------------------------------------------

Boolean ValidBuf(BufTypeHandle bh)
{
	if (!bh)
		return false;
	return ((**bh).bufHandle != nil);
}

//-------------------------------------------------------

BufTypeHandle MakeSize_Buf(unsigned long n)
{
	BufTypeHandle	bh;
	
	bh = (BufTypeHandle) NewHandleClear(sizeof(BufType));
	if (!bh)
		return nil;

	(**bh).bufHandle = NewHandleClear(n);
	if (!(**bh).bufHandle) {
		Delete_Buf(bh);
		bh = nil;
	}
	return bh;
}

//-------------------------------------------------------

BufTypeHandle Make_Buf(void)
{
	return (BufTypeHandle) NewHandleClear(sizeof(BufType));
}

//-------------------------------------------------------

Boolean IsEmpty_Buf(BufTypeHandle bh)
{
	if (!ValidBuf(bh))
		return true;
	return ((**bh).len == 0);
}

//-------------------------------------------------------

unsigned long BufLen_Buf(BufTypeHandle bh) /* Number of valid chars in buffer */
{
	return (IsEmpty_Buf(bh) ? 0 : (**bh).len);
}

//-------------------------------------------------------

Handle GetBuf_Buf(BufTypeHandle bh)
{
	return (bh ? (**bh).bufHandle : nil);
}

//-------------------------------------------------------

Ptr GetEnd_Buf(BufTypeHandle bh)
{
	Handle		h;
	char		*cp = nil;
	
	h = GetBuf_Buf(bh);
	if (h)
		cp = (*h + (**bh).len - 1);
	return cp;
}

//-------------------------------------------------------

unsigned long BufSize_Buf(BufTypeHandle bh)
{
	return (ValidBuf(bh) ? GetHandleSize((**bh).bufHandle) : 0);
}

//-------------------------------------------------------

Boolean SetLen_Buf(BufTypeHandle bh, unsigned long n)
{
	if (!ValidBuf(bh) || (n > GetHandleSize((**bh).bufHandle)))
		return false;

	(**bh).len = n;
	return true;
}

#pragma mark -
//-------------------------------------------------------

unsigned long BufNCat_Buf(BufTypeHandle dst, BufTypeHandle src, unsigned long n)
{
	unsigned long	oldSize, addSize, newSize;
	char			*dstStart;

	if ((IsEmpty_Buf(src)) || (n < 1) || !dst)
		return 0;

	oldSize = BufLen_Buf(dst);
	addSize = PosLen_Buf(src);
	if (n < addSize)
		addSize = n;

	newSize = oldSize + addSize;
	if (newSize > BufSize_Buf(dst)) {
		if (!ResizeBuf_Buf(dst, newSize))
			return 0;
	}
	if (oldSize == 0) {
		(**dst).pos = 0;
	}
	if (oldSize > 0)
		dstStart = GetEnd_Buf(dst) + 1;
	else
		dstStart = GetPos_Buf(dst);

	BlockMoveData(GetPos_Buf(src), dstStart, addSize);

	IncPos_Buf(src, addSize);
	(**dst).len = newSize;

	return addSize;
}

//-------------------------------------------------------

unsigned long BufIns_Buf(BufTypeHandle dst, BufTypeHandle src)
{
    unsigned long	oldLen, bufSize, addLen, newLen;

	if (IsEmpty_Buf(src))
		return 0;

	oldLen = PosLen_Buf(dst);
	bufSize = BufSize_Buf(dst);
	addLen = PosLen_Buf(src);
	newLen = oldLen + addLen;
	if (newLen > bufSize)
		addLen = bufSize - oldLen;

	if (addLen < 1)
		return 0; /* Buffer is already full */

	return BufNCat_Buf(dst, src, addLen);
}

//-------------------------------------------------------

unsigned long BufNIns_Buf(BufTypeHandle dst, BufTypeHandle src, unsigned long n)
{
    unsigned long	oldLen, bufSize, addLen, newLen;

	if ((IsEmpty_Buf(src)) || (n < 1))
		return 0;

	oldLen = PosLen_Buf(dst);
	bufSize = BufSize_Buf(dst);
	addLen = PosLen_Buf(src);
	newLen = oldLen + addLen;

	if (newLen > bufSize)
		addLen = bufSize - oldLen;

	if (n < addLen)
		addLen = n;

	if (addLen < 1)
		return (0); /* Buffer is already full */

	return (BufNCat_Buf(dst, src, addLen));
}

//-------------------------------------------------------

unsigned long StrCpy_Buf(BufTypeHandle dst, const StringPtr src)
{
	unsigned long	nLen;

	if ((!src) || (!dst))
		return (0);

	nLen = *src;
	if (nLen > 0) {
		if (nLen > BufSize_Buf(dst)) {
			if (!ResizeBuf_Buf(dst, nLen))
				return 0;
		}
		BlockMoveData(src+1, *((**dst).bufHandle), nLen);
		(**dst).pos = nLen;
	}
	(**dst).len = (**dst).pos = nLen;
	return nLen;
}

//-------------------------------------------------------

unsigned long StrCat_Buf(BufTypeHandle dst, const char *src)
{
	unsigned long	oldSize, addSize, newSize, nCopy;
	char			*cpDst, *cpSrc;	//, *cp;

	if (!src || !dst)
		return 0;

	oldSize = BufLen_Buf(dst);
	addSize = strlen(src);
	newSize = oldSize + addSize;
	if (newSize > BufSize_Buf(dst)) {
		if (!ResizeBuf_Buf(dst, newSize))
			return 0;
	}
	if (oldSize == 0)
		(**dst).pos = 0;

	cpSrc = (char*) src;
	if (oldSize > 0)
		cpDst = GetEnd_Buf(dst) + 1;
	else
		cpDst = GetPos_Buf(dst);

	nCopy = addSize;
	while (nCopy--)
		*cpDst++ = *cpSrc++;
	(**dst).len = newSize;
	return addSize;
}

//-------------------------------------------------------

unsigned long HandleCat_Buf(BufTypeHandle dst, const Handle src)
{
	unsigned long	oldSize, addSize, newSize;
	char			*dstStart;

	if ((!src) || (!dst))
		return 0;

	oldSize = BufLen_Buf(dst);
	addSize = GetHandleSize(src);
	newSize = oldSize + addSize;

	if (newSize > BufSize_Buf(dst)) {
		if (!ResizeBuf_Buf(dst, newSize))
			return 0;
	}
	if (oldSize == 0)
		(**dst).pos = 0;

	if (oldSize > 0)
		dstStart = GetEnd_Buf(dst) + 1;
	else
		dstStart = GetPos_Buf(dst);

    BlockMoveData(*src, dstStart, addSize);
	(**dst).len = (**dst).pos = newSize;
	return addSize;
}

#pragma mark -
//-------------------------------------------------------

unsigned long PosLen_Buf(BufTypeHandle bh)
{
	if (IsEmpty_Buf(bh))
		return 0;

	return ((**bh).len - (**bh).pos);
}

//-------------------------------------------------------

void IncPos_Buf(BufTypeHandle bh, unsigned long n)
{
	if (!ValidBuf(bh))
		return;

	if (PosLen_Buf(bh) >= n)
		(**bh).pos += n;
	else
		(**bh).pos = 0;
}

//-------------------------------------------------------

void ResetPos_Buf(BufTypeHandle bh)
{
	if (bh)
		(**bh).pos = 0;
}

//-------------------------------------------------------

Ptr GetPos_Buf(BufTypeHandle bh)
{
	Handle		h;
	char		*cp = nil;
	
	h = GetBuf_Buf(bh);
	if (h)
		cp = *h + (**bh).pos;
	return cp;
}

#pragma mark -
//-------------------------------------------------------

void Delete_Buf(BufTypeHandle bh)
{
	if (!bh)
		return;

	Free_Buf(bh);
	DisposeHandle((Handle) bh);
}

//-------------------------------------------------------

void Free_Buf(BufTypeHandle bh)
{
	if (!bh)
		return;

	DisposeHandle((**bh).bufHandle);
	Clear_Buf(bh);
}

//-------------------------------------------------------

void Clear_Buf(BufTypeHandle bh)
{
	if (!bh)
		return;

	(**bh).bufHandle = nil;
	EmptyBuf_Buf(bh);
}

//-------------------------------------------------------

void EmptyBuf_Buf(BufTypeHandle bh)
{
	if (!bh)
		return;

	(**bh).pos = 0;
	(**bh).len = 0;
}

//-------------------------------------------------------

static Boolean ResizeBuf_Buf(BufTypeHandle bh, unsigned long newSize)
{
	unsigned long	oldSize = GetHandleSize((**bh).bufHandle);

	if (oldSize == newSize)
		return (true);

	if ((**bh).bufHandle == nil)
		(**bh).bufHandle = NewHandleClear(newSize);
	else
		SetHandleSize((**bh).bufHandle, newSize);

	if(MemError() != noErr)
		return (false);

	if (newSize < oldSize) { /* Getting smaller */
		/* Check for valid length */
		if ((**bh).len > newSize)
			(**bh).len = newSize;
	}
	return (true);
}

//-------------------------------------------------------

// If doesn't continue match, returns zero
// otherwise returns number of chars of hSearchBuf that have been matched
unsigned long CompleteCount_Buf(BufTypeHandle buf, BufTypeHandle find,
									unsigned long nPreMatched)
{
	char			*pBuf, *pBufEnd, *pFind, *pFindEnd;
	unsigned long	nSkipCount = 0;

	if (IsEmpty_Buf(buf) || IsEmpty_Buf(find))
		return 0;

	if (nPreMatched >= PosLen_Buf(find))
		return 0;

    HLock((**buf).bufHandle);
    HLock((**find).bufHandle);
	pBuf = GetPos_Buf(buf);
	pBufEnd = GetEnd_Buf(buf);
	pFind = GetPos_Buf(find) + nPreMatched;
	pFindEnd = GetEnd_Buf(find);


	while ((pBuf <= pBufEnd) && (pFind <= pFindEnd) && ((*pBuf) == (*pFind)))
		pBuf++, pFind++, nSkipCount++;

    HUnlock((**buf).bufHandle);
    HUnlock((**find).bufHandle);

	if ((pBuf > pBufEnd) || (pFind > pFindEnd))
		return (nPreMatched + nSkipCount);

	return 0; /* match failed */
}

//-------------------------------------------------------

// Find match of hSearchBuf, either complete or end-spanning
// return number of chars to skip before match
// No match: return length of buf (skip the whole buffer)
unsigned long SkipCount_Buf(BufTypeHandle buf, BufTypeHandle find)
{
	char			*pBuf, *pBufEnd;
	char			*pFind, *pFindEnd;
	char			*s1 = nil, *s2 = nil;
	unsigned long	nSkipCount = 0;

	if (IsEmpty_Buf(buf))
		return 0;

	if (IsEmpty_Buf(find))
		return PosLen_Buf(buf);

    HLock((**buf).bufHandle);
    HLock((**find).bufHandle);
	pBuf = GetPos_Buf(buf);
	pBufEnd = GetEnd_Buf(buf);
	pFind = GetPos_Buf(find);
	pFindEnd = GetEnd_Buf(find);

	while (pBuf <= pBufEnd) {
		s1 = pBuf;
		s2 = pFind;

		while ((s1 <= pBufEnd) && (s2 <= pFindEnd) && ((*s1) == (*s2)))
			s1++, s2++;

		if ((s1 > pBufEnd) || (s2 > pFindEnd)) {
		    HUnlock((**buf).bufHandle);
    		HUnlock((**find).bufHandle);
			return (nSkipCount);
		}

		pBuf++;
		nSkipCount++;
	}
    HUnlock((**buf).bufHandle);
    HUnlock((**find).bufHandle);
	return (PosLen_Buf(buf));
}

//-------------------------------------------------------

Boolean Lock_Buf(BufTypeHandle bh)
{
	if (ValidBuf(bh)) {
		HLock((**bh).bufHandle);
		return true;
	}
	else
		return false;
}

//-------------------------------------------------------

void Unlock_Buf(BufTypeHandle bh)
{
	if (ValidBuf(bh))
		HUnlock((**bh).bufHandle);
}
