/*____________________________________________________________________________
	Copyright (C) 1999 Network Associates, Inc.
	All rights reserved.

	$Id: IPHandler.cp,v 1.23 1999/05/26 07:08:37 jason Exp $
____________________________________________________________________________*/

#include "pgpMem.h"
#include "pgpIPsecAH.h"
#include "pgpIPsecESP.h"
#include "pgpIPsecComp.h"
#include "pgpIPsecErrors.h"

#include "PolicyManager.h"
#include "IPHandler.h"


namespace {
	const PGPUInt8	kStandardIPHeaderLength		=	20;
	const PGPUInt8	kHeaderLengthOffset			=	0;
	const PGPUInt8	kPacketLengthOffset			=	2;
	const PGPUInt8	kPacketIDOffset				=	4;
	const PGPUInt8	kFlagsOffset				=	6;
	const PGPUInt8	kFragmentOffsetOffset		=	6;
	const PGPUInt8	kTimeToLiveOffset			=	8;
	const PGPUInt8	kProtocolOffset				=	9;
	const PGPUInt8	kChecksumOffset				=	10;
	const PGPUInt8	kSourceAddressOffset		=	12;
	const PGPUInt8	kDestinationAddressOffset	=	16;
	const PGPUInt8	kOptionsOffset				=	20;
	
	const PGPUInt16	kDontFragmentFlag			=	0x4000;
	const PGPUInt16	kMoreFragmentsFlag			=	0x2000;
	const PGPUInt16	kFragmentOffsetMask			=	0x1FFF;
	
	const PGPUInt8	kEndOfOptionListOption		=	0;
	const PGPUInt8	kNoOperationOption			=	1;
	const PGPUInt8	kSecurityOption				=	130;
	const PGPUInt8	kLooseSourceOption			=	131;
	const PGPUInt8	kStrictSourceOption			=	137;
	const PGPUInt8	kRecordRouteOption			=	7;
	const PGPUInt8	kStreamIdentifierOption		=	136;
	const PGPUInt8	kInternetTimestampOption	=	68;
	
	const PGPUInt8	kICMPHeaderLength			=	8;
	const PGPUInt8	kICMPExtraDataLength		=	8;
	const PGPUInt8	kICMPChecksumOffset			=	2;
	
	const PGPUInt8	kIPProtocolIGMP				=	2;
	const PGPUInt8	kIPProtocolTCP				=	6;
	const PGPUInt8	kIPProtocolUDP				=	17;
	
	const PGPUInt16	kUDPSourcePortOffset		=	0;
	const PGPUInt16	kUDPDestinationPortOffset	=	2;
	const PGPUInt16	kUDPPortIKE					=	500;
	const PGPUInt16	kUDPHeaderLength			=	8;
	
	const PGPUInt16	kMaxIPsecTransformGrowth	=	700;
	
	const PGPUInt32	kClassDMask					=	0xF0000000;
	const PGPUInt32	kClassEMask					=	0xF8000000;
	const PGPUInt32	kClassDPrefix				=	0xE0000000;
	const PGPUInt32 kClassEPrefix				=	0xF0000000;
}



void ICMPReplyToDontFragment(queue_t * inQ, mblk_t * inPacketBlock, PGPUInt16 inMaxSDU);


	inline PGPUInt8
GetIPHeaderHeaderLength(
	PGPByte *	inHeader)
{
	return (*(inHeader + kHeaderLengthOffset) & 0x0F) * 4;
}



	inline PGPUInt16
GetIPHeaderPacketLength(
	PGPByte *	inHeader)
{
	return *reinterpret_cast<PGPUInt16 *>(inHeader + kPacketLengthOffset);
}


	inline PGPUInt32
GetIPHeaderSourceAddress(
	PGPByte *	inHeader)
{
	return *reinterpret_cast<PGPUInt32 *>(inHeader + kSourceAddressOffset);
}


	inline PGPUInt32
GetIPHeaderDestinationAddress(
	PGPByte *	inHeader)
{
	return *reinterpret_cast<PGPUInt32 *>(inHeader + kDestinationAddressOffset);
}


	inline PGPUInt16
ComputeOnesComplementChecksum(
	PGPByte *	inData,
	PGPUInt16	inDataSize)
{
	PGPUInt32	total = 0;
	
	for (PGPUInt16 i = 0; i < inDataSize; i += sizeof(PGPUInt16)) {
		total += *reinterpret_cast<PGPUInt16 *>(inData + i);
	}
	
	return ~(total + (total >> 16));
}


	inline void
SetIPHeaderChecksum(
	PGPByte *	inHeader,
	PGPUInt8	inHeaderSize)
{
	PGPUInt32	total = 0;
	
	*reinterpret_cast<PGPUInt16 *>(inHeader + kChecksumOffset) = 0;	
	*reinterpret_cast<PGPUInt16 *>(inHeader + kChecksumOffset) = 
		ComputeOnesComplementChecksum(inHeader, inHeaderSize);
}



	inline PGPBoolean
CheckIPHeaderChecksum(
	PGPByte *	inHeader,
	PGPUInt16	inHeaderSize)
{
	PGPUInt32	total = 0;
	PGPByte *	end = inHeader + inHeaderSize;

	while (inHeader < end) {
		total += *reinterpret_cast<PGPUInt16 *>(inHeader);
		inHeader += sizeof(PGPUInt16);
	}
	total += total >> 16;

	return (static_cast<PGPUInt16>(total) == 0xFFFF);
}



	inline void
ClearIPHeaderDontFragment(
	PGPByte *	inHeader)
{
	*reinterpret_cast<PGPUInt16 *>(inHeader + kFlagsOffset) &= ~kDontFragmentFlag;
}



	inline PGPBoolean
IsIPHeaderDontFragmentSet(
	PGPByte *	inHeader)
{
	return (*reinterpret_cast<PGPUInt16 *>(inHeader + kFlagsOffset) & kDontFragmentFlag) != 0;
}



	inline PGPBoolean
IsIPHeaderAFragment(
	PGPByte *	inHeader)
{
	return (*reinterpret_cast<PGPUInt16 *>(inHeader + kFlagsOffset) & ~kDontFragmentFlag) != 0;
}


	inline PGPUInt16
GetIPHeaderOffset(
	PGPByte *	inHeader)
{
	return *reinterpret_cast<PGPUInt16 *>(inHeader + kFragmentOffsetOffset)
				& kFragmentOffsetMask;
}



	inline void
SetIPHeaderOffset(
	PGPByte *	inHeader,
	PGPUInt16	inOffset)
{
	// Note: This will write over the IP header flags
	*reinterpret_cast<PGPUInt16 *>(inHeader + kFragmentOffsetOffset) = inOffset;
}



	inline void
SetIPHeaderMoreFlag(
	PGPByte *	inHeader)
{
	*reinterpret_cast<PGPUInt16 *>(inHeader + kFlagsOffset) |= kMoreFragmentsFlag;
}



	inline PGPBoolean
IsIPHeaderMoreFlagSet(
	PGPByte *	inHeader)
{
	return (*reinterpret_cast<PGPUInt16 *>(inHeader + kFlagsOffset) & kMoreFragmentsFlag) != 0;
}

	inline void
SetIPHeaderPacketLength(
	PGPByte *	inHeader,
	PGPUInt16	inLength)
{
	*reinterpret_cast<PGPUInt16 *>(inHeader + kPacketLengthOffset) = inLength;
}



	inline void
ClearIPHeaderFragmentOptions(
	PGPByte *	inHeader,
	PGPUInt8	inHeaderSize)
{
	PGPByte *	end = inHeader + inHeaderSize;
	
	inHeader += kOptionsOffset;
	while (inHeader < end) {
		switch(*inHeader) {
			case kEndOfOptionListOption:
			{
				inHeader = end;
			}
			break;
			
			
			case kNoOperationOption:
			{
				inHeader++;
			}
			break;
			
			
			case kSecurityOption:
			case kLooseSourceOption:
			case kStrictSourceOption:
			case kStreamIdentifierOption:
			{
				inHeader += *(inHeader + 1);
			}
			break;
			
			
			case kRecordRouteOption:
			case kInternetTimestampOption:
			{
				PGPByte *	dest = inHeader;
				
				inHeader += *(inHeader + 1);
				pgpFillMemory(dest, *(dest + 1), kNoOperationOption);
			}
			break;
		}
	}
}



	void
ICMPReplyToDontFragment(
	queue_t *	inQ,
	mblk_t *	inPacketBlock,
	PGPUInt16	inMaxSDU)
{
	PGPUInt16	ipHeaderSize = GetIPHeaderHeaderLength(inPacketBlock->b_rptr);
	PGPUInt16	icmpReplySize = kStandardIPHeaderLength + kICMPHeaderLength
									+ ipHeaderSize + kICMPExtraDataLength;
	mblk_t *	icmpReply = allocb(icmpReplySize, 0);
	
	if (icmpReply != NULL) {
		PGPByte *	curr = icmpReply->b_wptr;
	
		icmpReply->b_wptr += icmpReplySize;
		*reinterpret_cast<PGPUInt32 *>(curr) = 0x45000000 + icmpReplySize;
		curr += sizeof(PGPUInt32);
		*reinterpret_cast<PGPUInt32 *>(curr) = 0;
		curr += sizeof(PGPUInt32);
		*reinterpret_cast<PGPUInt32 *>(curr) = 0xFF010000;
		curr += sizeof(PGPUInt32);
		*reinterpret_cast<PGPUInt32 *>(curr) = GetIPHeaderSourceAddress(inPacketBlock->b_rptr);
		curr += sizeof(PGPUInt32);
		*reinterpret_cast<PGPUInt32 *>(curr) = GetIPHeaderSourceAddress(inPacketBlock->b_rptr);
		curr += sizeof(PGPUInt32);
		*reinterpret_cast<PGPUInt32 *>(curr) = 0x03040000;
		curr += sizeof(PGPUInt32);
		*reinterpret_cast<PGPUInt32 *>(curr) = inMaxSDU;
		curr += sizeof(PGPUInt32);
		pgpCopyMemory(inPacketBlock->b_rptr, curr, ipHeaderSize + kICMPExtraDataLength);
		SetIPHeaderChecksum(icmpReply->b_rptr, kStandardIPHeaderLength);
		*reinterpret_cast<PGPUInt16 *>(icmpReply->b_rptr + kStandardIPHeaderLength
			+ kICMPChecksumOffset) = ComputeOnesComplementChecksum(icmpReply->b_rptr
										+ kStandardIPHeaderLength, icmpReplySize
										- kStandardIPHeaderLength);
DConLogNote("MaxSDU %hu\n", inMaxSDU);
DConDumpNote("ICMP ", icmpReply->b_rptr, icmpReply->b_wptr - icmpReply->b_rptr);
		qreply(inQ, icmpReply);
	}
}

	inline void
IPFragmentAndPutNext(
	queue_t *		inQ,
	const mblk_t *	inHeaderBlock,
	const mblk_t *	inPacketBlock)
{
	mblk_t *	newMessageHeader;
	mblk_t *	newIPHeader;
	mblk_t *	newDataBlock;
	PGPUInt32	maxSDU = GetPerStreamData(inQ)->maxSDU;
/*dopen("Fragments.txt");
DConDumpNote("Header", inHeaderBlock->b_rptr, inHeaderBlock->b_wptr - inHeaderBlock->b_rptr);
DConDumpNote("Packet", inPacketBlock->b_rptr, inPacketBlock->b_wptr - inPacketBlock->b_rptr);
*/
	if ((inPacketBlock->b_wptr - inPacketBlock->b_rptr) > maxSDU) {
		PGPByte *	ipHeader = reinterpret_cast<PGPByte *>(inPacketBlock->b_rptr);
		PGPUInt8	ipHeaderSize = GetIPHeaderHeaderLength(ipHeader);
		mblk_t *	dataBlock = dupb(const_cast<mblk_t *>(inPacketBlock));
		PGPUInt16	offset = GetIPHeaderOffset(ipHeader);
		PGPUInt32	nfb = (maxSDU - ipHeaderSize) >> 3;
		PGPBoolean	more = IsIPHeaderMoreFlagSet(ipHeader);

		DConLogNote("  Fragmenting packet\n");
		if (dataBlock != NULL) {
			dataBlock->b_rptr += ipHeaderSize;
			while (dataBlock->b_rptr < dataBlock->b_wptr) {
				newMessageHeader = dupb(const_cast<mblk_t *>(inHeaderBlock));
				if (newMessageHeader != NULL) {
					newIPHeader = newMessageHeader->b_cont = allocb(ipHeaderSize, 0);
					if (newIPHeader != NULL) {
						newDataBlock = newIPHeader->b_cont = dupb(dataBlock);
						if (newDataBlock != NULL) {
							pgpCopyMemory(ipHeader, newIPHeader->b_rptr, ipHeaderSize);
							newIPHeader->b_wptr += ipHeaderSize;
							SetIPHeaderOffset(newIPHeader->b_rptr, offset);
							if (newDataBlock->b_wptr > (newDataBlock->b_rptr + (nfb * 8))) {
								newDataBlock->b_wptr = newDataBlock->b_rptr + (nfb * 8);
								SetIPHeaderMoreFlag(newIPHeader->b_rptr);
							} else if (more) {
								SetIPHeaderMoreFlag(newIPHeader->b_rptr);
							}
							SetIPHeaderPacketLength(newIPHeader->b_rptr,
													ipHeaderSize + (newDataBlock->b_wptr
														- newDataBlock->b_rptr));
							if (offset != 0) {
								ClearIPHeaderFragmentOptions(newIPHeader->b_rptr, ipHeaderSize);
							}
							SetIPHeaderChecksum(newIPHeader->b_rptr, ipHeaderSize);
							offset += nfb;
							dataBlock->b_rptr += newDataBlock->b_wptr - newDataBlock->b_rptr;

/*
pullupmsg(newMessageHeader->b_cont, -1);
DConDumpNote("Header", newMessageHeader->b_rptr, newMessageHeader->b_wptr - newMessageHeader->b_rptr);
DConDumpNote("Packet", newMessageHeader->b_cont->b_rptr, newMessageHeader->b_cont->b_wptr - newMessageHeader->b_cont->b_rptr);
*/

							if (newMessageHeader->b_rptr == newMessageHeader->b_wptr) {
								// We special case a 0 length header as being no header
								putnext(inQ, newMessageHeader->b_cont);
								freeb(newMessageHeader);
							} else {
								putnext(inQ, newMessageHeader);
							}
							newMessageHeader = NULL;
						}
					}
				}
				
				if (newMessageHeader != NULL) {
					DConLogNote("    Header fragment: failed\n", offset * 8);
					freemsg(newMessageHeader);
					break;
				}
			}
		}
		freeb(dataBlock);
	} else {
		newMessageHeader = dupb(const_cast<mblk_t *>(inHeaderBlock));
		newDataBlock = dupb(const_cast<mblk_t *>(inPacketBlock));
		
		if ((newMessageHeader != NULL) && (newDataBlock != NULL)) {
			if (newMessageHeader->b_rptr == newMessageHeader->b_wptr) {
				putnext(inQ, newDataBlock);
				freeb(newMessageHeader);
			} else {
				newMessageHeader->b_cont = newDataBlock;
				putnext(inQ, newMessageHeader);
			}
		} else {
			if (newMessageHeader != NULL) {
				freeb(newMessageHeader);
			}
			if (newDataBlock != NULL) {
				freeb(newDataBlock);
			}
		}
	}
//dopen(NULL);
}



	inline PGPUInt16
ComputeIPHeaderDataLength(
	PGPByte *	inHeader)
{
	return GetIPHeaderPacketLength(inHeader) - GetIPHeaderHeaderLength(inHeader);
}


	inline PGPBoolean
IsIPHeaderSamePacket(
	PGPByte *	inHeader1,
	PGPByte *	inHeader2)
{
	return (*reinterpret_cast<PGPUInt16 *>(inHeader1 + kPacketIDOffset) ==
				*reinterpret_cast<PGPUInt16 *>(inHeader2 + kPacketIDOffset))
		&& (*reinterpret_cast<PGPUInt8 *>(inHeader1 + kProtocolOffset) ==
				*reinterpret_cast<PGPUInt8 *>(inHeader2 + kProtocolOffset))
		&& (*reinterpret_cast<PGPUInt32 *>(inHeader1 + kSourceAddressOffset) ==
				*reinterpret_cast<PGPUInt32 *>(inHeader2 + kSourceAddressOffset))
		&& (*reinterpret_cast<PGPUInt32 *>(inHeader1 + kDestinationAddressOffset) ==
				*reinterpret_cast<PGPUInt32 *>(inHeader2 + kDestinationAddressOffset));
}



	inline PGPUInt8
GetIPHeaderTTL(
	PGPByte *	inHeader)
{
	return *reinterpret_cast<PGPUInt8 *>(inHeader + kTimeToLiveOffset);
}



	inline PGPUInt8
GetIPHeaderProtocol(
	PGPByte *	inHeader)
{
	return *reinterpret_cast<PGPUInt8 *>(inHeader + kProtocolOffset);
}



/*	inline PGPInt8
IPHeaderComparePackets (
	PGPByte *	inHeader1,
	PGPByte *	inHeader2)
{
	PGPInt8		result = 0;
	PGPUInt16	header1Offset = GetIPHeaderOffset(inHeader1);
	PGPUInt16	header1Length = ComputeIPHeaderDataLength(inHeader1) >> 3;
	PGPUInt16	header2Offset = GetIPHeaderOffset(inHeader2);
	PGPUInt16	header2Length = ComputeIPHeaderDataLength(inHeader2) >> 3;
	
	if ((header1Offset + header1Length) <= header2Offset) {
		result = -1;
	} else if ((header2Offset + header2Length) <= header1Offset) {
		result = 1;
	}
	
	return result;
}
*/


	inline mblk_t *
IPInsertPacket(
	const mblk_t *		inPacket,
	SIPReassemblyList *	inReassemblyBlock)
{
	mblk_t *	result = NULL;
	mblk_t *	newPacket = dupb(const_cast<mblk_t *>(inPacket));

	DConLogNote("  IPInsertPacket Begin\n");
	if (newPacket != NULL) {
		mblk_t *	previous = NULL;
		mblk_t *	curr = inReassemblyBlock->packetMessage;
		PGPUInt16	packetOffset = GetIPHeaderOffset(inPacket->b_rptr);
		PGPUInt16	nextOffset = packetOffset + (ComputeIPHeaderDataLength(inPacket->b_rptr) >> 3);
		
		do {
			if (nextOffset <= GetIPHeaderOffset(curr->b_rptr)) {
				break;
			}
			previous = curr;
			curr = curr->b_cont;
		} while (curr != NULL);
		
		if (curr == NULL) {
//			DConLogNote("    Packet is after\n");
			PGPUInt16	previousNextOffset = GetIPHeaderOffset(previous->b_rptr) +
												(ComputeIPHeaderDataLength(previous->b_rptr) >> 3);
			if (previousNextOffset <= packetOffset) {
				if (previousNextOffset == packetOffset) {
					inReassemblyBlock->numRemaining--;
				}
				if (IsIPHeaderMoreFlagSet(inPacket->b_rptr)) {
					inReassemblyBlock->numRemaining++;
				}
				previous->b_cont = newPacket;
				newPacket = NULL;
			}
		} else if (previous == NULL) {
//			DConLogNote("    Packet is before\n");
			if (packetOffset == 0) {
				inReassemblyBlock->numRemaining--;
			}
			if (nextOffset != GetIPHeaderOffset(curr->b_rptr)) {
				inReassemblyBlock->numRemaining++;
			}
			newPacket->b_cont = curr;
			inReassemblyBlock->packetMessage = newPacket;
			newPacket = NULL;
		} else {
//			DConLogNote("    Packet is in the middle\n");
			PGPUInt16	previousNextOffset = GetIPHeaderOffset(previous->b_rptr) +
												(ComputeIPHeaderDataLength(previous->b_rptr) >> 3);
			
			if (previousNextOffset <= packetOffset) {
				if (previousNextOffset == packetOffset) {
					inReassemblyBlock->numRemaining--;
				}
				if (nextOffset != GetIPHeaderOffset(curr->b_rptr)) {
					inReassemblyBlock->numRemaining++;
				}
				previous->b_cont = newPacket;
				newPacket->b_cont = curr;
				newPacket = NULL;
			}
		}
/*if (previous == NULL) {
	DConLogNote("    Between Beginning\n");
} else {
	DConDumpNote("    Between ", previous->b_rptr, GetIPHeaderHeaderLength(previous->b_rptr));
}
if (curr == NULL) {
	DConLogNote("    And End\n");
} else {
	DConDumpNote("    And ", curr->b_rptr, GetIPHeaderHeaderLength(curr->b_rptr));
}
*/
		DConLogNote("  Number of packets remaining: %lu\n", inReassemblyBlock->numRemaining);
		
		if (inReassemblyBlock->numRemaining == 0) {
//			DConLogNote("  Packet reassembled\n");
			SetIPHeaderOffset(inReassemblyBlock->packetMessage->b_rptr, 0);
			curr = inReassemblyBlock->packetMessage->b_cont;
			do {
				if (curr->b_cont == NULL) {
					SetIPHeaderPacketLength(inReassemblyBlock->packetMessage->b_rptr,
						GetIPHeaderHeaderLength(inReassemblyBlock->packetMessage->b_rptr) +
							(GetIPHeaderOffset(curr->b_rptr) << 3)
							+ ComputeIPHeaderDataLength(curr->b_rptr));
				}
				curr->b_rptr += GetIPHeaderHeaderLength(curr->b_rptr);
				curr = curr->b_cont;
			} while (curr != NULL);
			SetIPHeaderChecksum(	inReassemblyBlock->packetMessage->b_rptr,
									GetIPHeaderHeaderLength(
										inReassemblyBlock->packetMessage->b_rptr));
			result = inReassemblyBlock->packetMessage;
//			DConDumpNote("    Reassembled packet header:", result->b_rptr, GetIPHeaderHeaderLength(result->b_rptr));
//pullupmsg(result, -1);
//DConDumpNote("\n\nReassembled Packet ", result->b_rptr, result->b_wptr - result->b_rptr);
		}
		
		if (newPacket != NULL) {
			freeb(newPacket);
		}
	}
	DConLogNote("  IPInsertPacket End\n");
	
	return result;
}



/*	inline mblk_t *
IPInsertPacket(
	const mblk_t *	inPacket,
	mblk_t **		inPacketList)
{
	mblk_t *	result = NULL;
	mblk_t **	previousBlock = inPacketList;
	mblk_t *	curr = *inPacketList;
	PGPBoolean	full = true;
	PGPBoolean	found = false;
	mblk_t *	newPacket = dupb(const_cast<mblk_t *>(inPacket));
	
	if (newPacket != NULL) {
		do {
			switch (IPHeaderComparePackets(newPacket->b_rptr, curr->b_rptr)) {
				case -1:
				{
					*previousBlock = newPacket;
					newPacket->b_cont = curr;
					found = true;
				}
				break;
				
				
				case 1:
				{
					if (curr->b_cont == NULL) {
						curr->b_cont = newPacket;
						found = true;
					}
				}
				break;
				
				
				case 0:
				{
					freeb(newPacket);
					found = true;
					full = false;
				}
				break;
			}
			previousBlock = &curr->b_cont;
			curr = curr->b_cont;
		} while (!found);
		
		curr = *inPacketList;
		while (full) {
			if (curr->b_cont == NULL) {
				if (IsIPHeaderMoreFlagSet(curr->b_rptr)) {
					full = false;
				}
				break;
			} else if ((GetIPHeaderOffset(curr->b_rptr) + 
			(ComputeIPHeaderDataLength(curr->b_rptr) >> 3))
			!= GetIPHeaderOffset(curr->b_cont->b_rptr)) {
				full = false;
			}
		}
		
		if (full) {
			SetIPHeaderPacketLength(	(*inPacketList)->b_rptr,
										GetIPHeaderOffset(curr->b_rptr) << 3
											+ ComputeIPHeaderDataLength(curr->b_rptr));
			SetIPHeaderOffset((*inPacketList)->b_rptr, 0);
			SetIPHeaderChecksum(	(*inPacketList)->b_rptr,
									GetIPHeaderHeaderLength((*inPacketList)->b_rptr));
			curr = (*inPacketList)->b_cont;
			while(curr != NULL) {
				curr->b_rptr += GetIPHeaderHeaderLength(curr->b_rptr);
				curr = curr->b_cont;
			}
			result = *inPacketList;
		}
	}
	
	return result;
}
*/

inline void
DumpReassembly(
	SIPReassemblyList *	inList)
{
	PGPUInt32	packetNumber = 0;
	DConLogNote("\n\n\n\nDumping List Begin\n");
	while (inList != NULL) {
		PGPUInt32	fragmentNumber = 0;
		mblk_t *	fragment = inList->packetMessage;
		DConLogNote("  NumRemaining: %lu\n", inList->numRemaining);
		DConLogNote("  Packet %lu\n", packetNumber);
		while (fragment != NULL) {
			DConLogNote("  Fragment %lu\n", fragmentNumber);
			DConDumpNote("", fragment->b_rptr, fragment->b_wptr - fragment->b_rptr);
			fragment = fragment->b_cont;
			fragmentNumber++;
		}
		DConLogNote("\n");
		inList = inList->next;
		packetNumber++;
	}
	DConLogNote("Dumping List End\n\n\n\n\n");
}
	inline void
IPReassemblePacket(
	SIPReassemblyList **	inList,
	const mblk_t *			inHeaderBlock,
	const mblk_t *			inPacketBlock,
	mblk_t **				outHeaderBlock,
	mblk_t **				outPacketMessage)
{
	SIPReassemblyList **	previousNext = inList;
	SIPReassemblyList *		curr = *inList;
	SIPReassemblyList *		temp;
	UInt32					clock = OTGetClockTimeInSecs();
	PGPBoolean				found = false;

	DConLogNote("  IPReassemblePacket Begin\n");
//DConDumpNote("Packet header", inPacketBlock->b_rptr, GetIPHeaderHeaderLength(inPacketBlock->b_rptr));

	while (curr != NULL) {
		if (IsIPHeaderSamePacket(inPacketBlock->b_rptr, curr->packetMessage->b_rptr)) {
			UInt32	newTTL = clock + GetIPHeaderTTL(inPacketBlock->b_rptr);
			
//			DConLogNote("    Found the Packet\n");
			found = true;
			*outPacketMessage = IPInsertPacket(inPacketBlock, curr);
			if (*outPacketMessage != NULL) {
				temp = curr;
				*outHeaderBlock = curr->header;
				curr = *previousNext = curr->next;
				OTFreeMem(temp);
			} else {
				if (newTTL > curr->expireTime) {
					curr->expireTime = newTTL;
				}
				previousNext = &curr->next;
				curr = curr->next;
			}
		} else if (clock > curr->expireTime) {
			DConDumpNote("  Expired packet ", curr->packetMessage->b_rptr, GetIPHeaderHeaderLength(curr->packetMessage->b_rptr));
			temp = curr;
			curr = *previousNext = curr->next;
			freeb(temp->header);
			freemsg(temp->packetMessage);
			OTFreeMem(temp);
		} else {
			previousNext = &curr->next;
			curr = curr->next;
		}
	}
	
	if (! found) {
		*previousNext = static_cast<SIPReassemblyList *>(OTAllocMem(sizeof(SIPReassemblyList)));

		DConLogNote("    Packet not found, adding it to the list\n");
		if (*previousNext != NULL) {
			(*previousNext)->header = dupb(const_cast<mblk_t *>(inHeaderBlock));
			(*previousNext)->packetMessage = dupb(const_cast<mblk_t *>(inPacketBlock));
			if (((*previousNext)->header == NULL)
			|| ((*previousNext)->packetMessage == NULL)) {
				if ((*previousNext)->header != NULL) {
					freeb((*previousNext)->header);
				}
				if ((*previousNext)->packetMessage != NULL) {
					freeb((*previousNext)->packetMessage);
				}
				OTFreeMem(*previousNext);
				*previousNext = NULL;
//				DConLogNote("    Allocation failed\n");
			} else {
				(*previousNext)->expireTime = clock + GetIPHeaderTTL(inPacketBlock->b_rptr);
				(*previousNext)->next = NULL;
				(*previousNext)->numRemaining = 0;
				if (GetIPHeaderOffset(inPacketBlock->b_rptr) != 0) {
					(*previousNext)->numRemaining++;
				}
				if (IsIPHeaderMoreFlagSet(inPacketBlock->b_rptr)) {
					(*previousNext)->numRemaining++;
				}
//				DConDumpNote("  New fragmented packet header ", inPacketBlock->b_rptr, GetIPHeaderHeaderLength(inPacketBlock->b_rptr));
				DConLogNote("  At least %lu remaining fragments.\n", (*previousNext)->numRemaining);
			}
		}
	}
//DumpReassembly(*inList);

	DConLogNote("  IPReassemblePacket End\n");
}


	inline void
IPTimeoutFragments(
	SIPReassemblyList **	inList)
{
	SIPReassemblyList **	previousNext = inList;
	SIPReassemblyList *		curr = *inList;
	SIPReassemblyList *		temp;
	UInt32					clock = OTGetClockTimeInSecs();

	while (curr != NULL) {
		if (clock > curr->expireTime) {
			DConDumpNote("  Expired packet ", curr->packetMessage->b_rptr, GetIPHeaderHeaderLength(curr->packetMessage->b_rptr));
			temp = curr;
			curr = *previousNext = curr->next;
			freeb(temp->header);
			freemsg(temp->packetMessage);
			OTFreeMem(temp);
		} else {
			previousNext = &curr->next;
			curr = curr->next;
		}
	}

}



	PGPIPsecBuffer *
IPMessageToIPsecBuffers(
	const mblk_t *	inMessage)
{
	PGPIPsecBuffer *	result = NULL;
	PGPUInt32			count = 0;
	const mblk_t *		curr = inMessage;
	
	do {
		count++;
		curr = curr->b_cont;
	} while (curr != NULL);
	result = reinterpret_cast<PGPIPsecBuffer *>(OTAllocMem(count * sizeof(PGPIPsecBuffer)));
	if (result != NULL) {
		curr = inMessage;
		count = 0;
		do {
			result[count].data = curr->b_rptr;
			result[count].dataSize = curr->b_wptr - curr->b_rptr;
			result[count].allocatedSize = result[count].dataSize;
			result[count].next = &result[count + 1];
			count++;
			curr = curr->b_cont;
		} while (curr != NULL);
		result[count - 1].next = NULL;
	}

	return result;
}



	void
IPGetSourceAddressAndSPI(
	PGPIPsecBuffer *	inBuffers,
	PGPUInt32 *			outAddress,
	PGPUInt32 *			outSPI)
{
	*outAddress = GetIPHeaderSourceAddress(inBuffers[0].data);
	pgpIPsecESPGetSPI(inBuffers, outSPI);
	if (*outSPI == 0) {
		pgpIPsecAHGetSPI(inBuffers, outSPI);
	}
}



	PGPError
IPApplySAToPacket(
	queue_t *			inQ,
	PGPBoolean			inIncoming,
	PGPBoolean			inTunnel,
	PGPikeSA *			inSA,
	PGPIPsecBuffer *	inBuffers,
	mblk_t **			outPacketMessage)
{
	PGPError				result = kPGPError_NoErr;
	PGPBoolean				useESP = false;
	PGPBoolean				useAH = false;
	PGPBoolean				useIPCOMP = false;
	PGPCipherAlgorithm		espCipher = kPGPCipherAlgorithm_None;
	PGPHashAlgorithm		espHash = kPGPHashAlgorithm_Invalid;
	PGPHashAlgorithm		ahHash = kPGPHashAlgorithm_Invalid;
	PGPCompressionAlgorithm	compAlg = kPGPCompAlgorithm_None;
	PGPByte *				espCipherKey;
	PGPByte *				espIV = NULL;
	PGPByte *				espHashKey;
	PGPByte *				ahHashKey;
	PGPUInt32				espSPI;
	PGPUInt32				ahSPI;
	
	DConLogNote("  IPApplySAToPacket Begin\n");
	for (PGPUInt16 i = 0; i < inSA->numTransforms; i++) {
		switch (inSA->transform[i].u.ipsec.protocol) {
			case kPGPike_PR_AH:
			{
				useAH = true;
				switch (inSA->transform[i].u.ipsec.u.ah.t.authAlg) {
					case kPGPike_AH_MD5:
					{
						ahHash = kPGPHashAlgorithm_MD5;
					}
					break;
					
					
					case kPGPike_AH_SHA:
					{
						ahHash = kPGPHashAlgorithm_SHA;
					}
					break;
					
					
					default:
					{
						result = kPGPError_BadHashNumber;
					}
					break;
				}
				if (inIncoming) {
					ahHashKey = inSA->transform[i].u.ipsec.u.ah.inAuthKey;
					ahSPI = *reinterpret_cast<PGPUInt32 *>(inSA->transform[i].u.ipsec.inSPI);
				} else {
					ahHashKey = inSA->transform[i].u.ipsec.u.ah.outAuthKey;
					ahSPI = *reinterpret_cast<PGPUInt32 *>(inSA->transform[i].u.ipsec.outSPI);
				}
			}
			break;
			
			
			case kPGPike_PR_ESP:
			{
				useESP = true;
				switch (inSA->transform[i].u.ipsec.u.esp.t.cipher) {
					case kPGPike_ET_3DES:
					{
						espCipher = kPGPCipherAlgorithm_3DES;
					}
					break;
					
					
					case kPGPike_ET_CAST:
					{
						espCipher = kPGPCipherAlgorithm_CAST5;
					}
					break;
					
					
					case kPGPike_ET_DES_IV64:
					{
						espCipher = kPGPCipherAlgorithm_3DES;
					}
					break;
					
					
					case kPGPike_ET_DES:
					{
						espCipher = kPGPCipherAlgorithm_3DES;
					}
					break;
					
					
					case kPGPike_ET_NULL:
					{
						espCipher = kPGPCipherAlgorithm_None;
					}
					break;
					
					
					default:
					{
						result = kPGPError_BadCipherNumber;
					}
					break;
				}
				switch (inSA->transform[i].u.ipsec.u.esp.t.authAttr) {
					case kPGPike_AA_HMAC_MD5:
					{
						espHash = kPGPHashAlgorithm_MD5;
					}
					break;
					
					
					case kPGPike_AA_HMAC_SHA:
					{
						espHash = kPGPHashAlgorithm_SHA;
					}
					break;
					
					
					case kPGPike_AA_None:
					{
						espHash = kPGPHashAlgorithm_Invalid;
					}
					break;
					
					
					default:
					{
						result = kPGPError_BadHashNumber;
					}
					break;
				}
				if (inIncoming) {
					espCipherKey = inSA->transform[i].u.ipsec.u.esp.inESPKey;
					espHashKey = inSA->transform[i].u.ipsec.u.esp.inAuthKey;
					espSPI = *reinterpret_cast<PGPUInt32 *>(inSA->transform[i].u.ipsec.inSPI);
				} else {
					espCipherKey = inSA->transform[i].u.ipsec.u.esp.outESPKey;
					espHashKey = inSA->transform[i].u.ipsec.u.esp.outAuthKey;
					espSPI = *reinterpret_cast<PGPUInt32 *>(inSA->transform[i].u.ipsec.outSPI);
					espIV = inSA->transform[i].u.ipsec.u.esp.explicitIV;
				}
			}
			break;
			
			
			case kPGPike_PR_IPCOMP:
			{
				useIPCOMP = true;
				switch (inSA->transform[i].u.ipsec.u.ipcomp.t.compAlg) {
					case kPGPike_IC_Deflate:
					{
						compAlg = kPGPCompAlgorithm_Deflate;
					}
					break;
					
					
					case kPGPike_IC_LZS:
					{
						compAlg = kPGPCompAlgorithm_LZS;
					}
					break;
					
					
					case kPGPike_IC_None:
					{
						compAlg = kPGPCompAlgorithm_None;
					}
					break;
					
					
					default:
					{
						result = kPGPError_BadParams;
					}
					break;
				}
			}
			break;
		}
		
		if (IsPGPError(result)) {
			break;
		}
	}
	
	if (IsntPGPError(result)) {
		PGPUInt16	outBufferSize = 0;
		PGPBoolean	tunnelIPCOMP = useIPCOMP && inTunnel;
		PGPBoolean	tunnelESP = (! tunnelIPCOMP) && inTunnel;
		PGPBoolean	tunnelAH = (! tunnelESP) && inTunnel;
		
		if ((! useESP) && (! useAH)) {
			useIPCOMP = false;
		}
		for (PGPIPsecBuffer * curr = inBuffers; curr != NULL; curr = curr->next) {
			outBufferSize += curr->dataSize;
		}
		if (useAH) {
			outBufferSize += kMaxIPsecTransformGrowth;
		}
		if (useESP) {
			outBufferSize += kMaxIPsecTransformGrowth;
		}
		if (useIPCOMP) {
			outBufferSize += kMaxIPsecTransformGrowth;
		}
		
		mblk_t *	tempBlock = allocb(outBufferSize, 0);
		mblk_t *	tempBlock2 = allocb(outBufferSize, 0);
		
		if ((tempBlock == NULL) || (tempBlock2 == NULL)) {
			if (tempBlock != NULL) {
				freeb(tempBlock);
			}
			if (tempBlock2 != NULL) {
				freeb(tempBlock2);
			}
			result = kPGPError_OutOfMemory;
		} else {
			PGPIPsecBuffer		tempBuffer = { tempBlock->b_wptr, 0, outBufferSize, NULL };
			PGPIPsecBuffer		tempBuffer2 = { tempBlock2->b_wptr, 0, outBufferSize, NULL };
			PGPIPsecBuffer *	input = inBuffers;
			PGPIPsecBuffer *	output = &tempBuffer;
			PGPIPsecBuffer *	extra = &tempBuffer2;
			PGPUInt32			gatewayIP = (inTunnel) ? inSA->ipAddress : 0;
			
			if (inIncoming) {
				PGPUInt32 sequenceWindow;
				PGPUInt32 upperSequence;
				PGPUInt32 lowerSequence;
				
				if (useAH) {
					sequenceWindow = reinterpret_cast<PGPnetIKESAUserData *>(inSA->userData)->
										sequenceWindow;
					upperSequence = reinterpret_cast<PGPnetIKESAUserData *>(inSA->userData)->
										sequenceWindowUpper;
					lowerSequence = reinterpret_cast<PGPnetIKESAUserData *>(inSA->userData)->
										sequenceWindowLower;
					result = pgpVerifyIPsecAuthentication(GetPerStreamData(inQ)->readContext,
								0, input, tunnelAH, ahHash, ahHashKey, &sequenceWindow,
								&upperSequence, &lowerSequence, output);
					input = output;
					output = extra;
					output->dataSize = 0;
					extra = input;
					DConLogNote("  pgpVerifyIPsecAuthentication = %ld\n", result);
				}
				if (useESP && IsntPGPError(result)) {
					sequenceWindow = reinterpret_cast<PGPnetIKESAUserData *>(inSA->userData)->
										sequenceWindow;
					upperSequence = reinterpret_cast<PGPnetIKESAUserData *>(inSA->userData)->
										sequenceWindowUpper;
					lowerSequence = reinterpret_cast<PGPnetIKESAUserData *>(inSA->userData)->
										sequenceWindowLower;
					result = pgpIPsecDecrypt(GetPerStreamData(inQ)->readContext, 0, input,
								tunnelESP, espCipher, espCipherKey, espHash, espHashKey,
								&sequenceWindow, &upperSequence, &lowerSequence,
								output);
					input = output;
					output = extra;
					output->dataSize = 0;
					extra = input;
					DConLogNote("  pgpIPsecDecrypt = %ld\n", result);
				}
				if (IsntPGPError(result) && (useESP || useAH)) {
					reinterpret_cast<PGPnetIKESAUserData *>(inSA->userData)->
										sequenceWindow = sequenceWindow;
					reinterpret_cast<PGPnetIKESAUserData *>(inSA->userData)->
										sequenceWindowUpper = upperSequence;
					reinterpret_cast<PGPnetIKESAUserData *>(inSA->userData)->
										sequenceWindowLower = lowerSequence;
				}
				if (useIPCOMP && IsntPGPError(result)) {
					result = pgpIPsecDecompress(GetPerStreamData(inQ)->readContext, 0, input,
								tunnelIPCOMP, compAlg, output);
					input = output;
					DConLogNote("  pgpIPsecDecompress = %ld\n", result);
				}
			} else {
				if (useIPCOMP) {
					PGPBoolean	usedIPCOMP;
					
					result = pgpIPsecCompress(GetPerStreamData(inQ)->writeContext, 0, input,
								tunnelIPCOMP, gatewayIP, compAlg, output, &usedIPCOMP);
					if (tunnelIPCOMP && (! usedIPCOMP)) {
						if (useESP) {
							tunnelESP = true;
						} else if (useAH) {
							tunnelAH = true;
						}
					}
					input = output;
					output = extra;
					output->dataSize = 0;
					extra = input;
					DConLogNote("  pgpIPsecCompress = %ld\n", result);
				}
				if (useESP && IsntPGPError(result)) {
					result = pgpIPsecEncrypt(GetPerStreamData(inQ)->writeContext, 0, input,
								tunnelESP, gatewayIP, espCipher, espCipherKey, espIV,
								espHash, espHashKey, espSPI,
								reinterpret_cast<PGPnetIKESAUserData *>(inSA->userData)->
									packetsSent + 1, output);
					input = output;
					output = extra;
					output->dataSize = 0;
					extra = input;
					DConLogNote("  pgpIPsecEncrypt = %ld\n", result);
				}
				if (useAH && IsntPGPError(result)) {
					result = pgpApplyIPsecAuthentication(GetPerStreamData(inQ)->writeContext,
								0, input, tunnelAH, gatewayIP, ahHash, ahHashKey, ahSPI,
								reinterpret_cast<PGPnetIKESAUserData *>(inSA->userData)->
									packetsSent + 1, output);
					input = output;
					DConLogNote("  pgpApplyIPsecAuthentication = %ld\n", result);
				}
			}
			
			if (input->data == tempBlock2->b_wptr) {
				freeb(tempBlock);
				tempBlock = tempBlock2;
			} else {
				freeb(tempBlock2);
			}
			if (IsPGPError(result)) {
				freeb(tempBlock);
			} else {
				tempBlock->b_wptr += GetIPHeaderPacketLength(tempBlock->b_rptr);
				*outPacketMessage = tempBlock;
			}
		}
	}
	
	DConLogNote("  IPApplySAToPacket End = %ld\n", result);
	return result;
}



	inline PGPBoolean
IsAddressClassD(
	PGPUInt32	inAddress)
{
	return (inAddress & kClassDMask) == kClassDPrefix;
}



	inline PGPBoolean
IsAddressClassE(
	PGPUInt32	inAddress)
{
	return (inAddress & kClassEMask) == kClassEPrefix;
}



	inline PGPBoolean
IPTransformPacket(
	PGPBoolean		inIncoming,
	queue_t *		inQ,
	const mblk_t *	inPacketMessage,
	SGlobals *		inGlobals,
	mblk_t **		outPacketBlock)
{
	PGPBoolean			result;
	PGPUInt32			destAddress = GetIPHeaderDestinationAddress(inPacketMessage->b_rptr);
	PGPUInt32			address = (inIncoming) ? GetIPHeaderSourceAddress(
							inPacketMessage->b_rptr) : destAddress;
	EPMStatus			status;
	
	DConLogNote("  IPTransformPacket Begin\n");
	if ((destAddress == inGlobals->broadcastAddress)
	|| IsAddressClassD(destAddress) || IsAddressClassE(destAddress)) {
		status = pmStatus_PassThrough;
	} else {
		status = pmStatus_DropPacket;
		switch (GetIPHeaderProtocol(inPacketMessage->b_rptr)) {
			case kIPProtocolIGMP:
			{
				DConLogNote("  Protocol: IGMP\n");
				status = pmStatus_PassThrough;
			}
			break;
			
			case kIPProtocolUDP:
			{
				PGPUInt16	headerLength = GetIPHeaderHeaderLength(inPacketMessage->b_rptr);
				
				DConLogNote("  Protocol: UDP\n");
				if (pullupmsg(const_cast<mblk_t *>(inPacketMessage),
				headerLength + kUDPHeaderLength)) {
					if ((*reinterpret_cast<PGPUInt16 *>(inPacketMessage->b_rptr + headerLength +
					kUDPSourcePortOffset) == kUDPPortIKE) || (*reinterpret_cast<PGPUInt16 *>(
					inPacketMessage->b_rptr + headerLength + kUDPDestinationPortOffset))
					== kUDPPortIKE) {
						if (pullupmsg(const_cast<mblk_t *>(inPacketMessage), -1)) {
							*outPacketBlock = dupb(const_cast<mblk_t *>(inPacketMessage));
							if (*outPacketBlock != NULL) {
								status = pmStatus_PassThrough;
								ClearIPHeaderDontFragment((*outPacketBlock)->b_rptr);
								SetIPHeaderChecksum((*outPacketBlock)->b_rptr,
									GetIPHeaderHeaderLength((*outPacketBlock)->b_rptr));
							}
						}
					}	
	//				DConLogNote("  Source port: %hu, Destination port: %hu\n", sourcePort,
	//					destPort);
				} else {
					DConLogNote("  Failed to pull up udp header\n");
				}
			}
			break;

	/*
			default:
			{
				PGPUInt16	headerLength = GetIPHeaderHeaderLength(inPacketMessage->b_rptr);
				
				DConLogNote("  Protocol: %d\n", static_cast<int>(GetIPHeaderProtocol(
					inPacketMessage->b_rptr)));
				if (pullupmsg(const_cast<mblk_t *>(inPacketMessage), headerLength + 8)) {
					PGPUInt16 	sourcePort = *reinterpret_cast<PGPUInt16 *>(
												inPacketMessage->b_rptr + headerLength);
					PGPUInt16 	destPort = *reinterpret_cast<PGPUInt16 *>(
												inPacketMessage->b_rptr + headerLength + 2);
								
					DConLogNote("  Header length: %hu, Source port: %hu, Destination port: %hu\n",
						headerLength, sourcePort, destPort);
					DConDumpNote("", inPacketMessage->b_rptr, headerLength + 8);
				} else {
					DConLogNote("  Failed to pull up tcp header\n");
				}
			}
			break;
			
	*/		
		}
	}

	if (status == pmStatus_DropPacket) {
		status = PMCheckPacketPolicy(address, inIncoming, inPacketMessage,
					inGlobals->allowUnconfigured, inGlobals->requireSecure,
					inGlobals->saList, inGlobals->hostEntryList,
					inGlobals->pendingSAList);
	}
	switch (status) {
		case pmStatus_PassThrough:
		{
			DConLogNote("  Status = pass through\n");
			result = true;
		}
		break;
		
		
		case pmStatus_ApplySA:
		{
			PGPUInt32			spi;
			PGPIPsecBuffer *	buffers = IPMessageToIPsecBuffers(inPacketMessage);
			
			DConLogNote("  Status = apply SA\n");
			result = false;
			ClearIPHeaderDontFragment(inPacketMessage->b_rptr);
			SetIPHeaderChecksum(	inPacketMessage->b_rptr,
									GetIPHeaderHeaderLength(
										inPacketMessage->b_rptr));
			if (buffers != NULL) {
				pgpIPsecESPGetSPI(buffers, &spi);
				if (spi == 0) {
					pgpIPsecAHGetSPI(buffers, &spi);
				}
				DConLogNote("  Transforming packet\n");
				status = PMDoTransform(inQ, buffers, address, inIncoming, spi,
							inGlobals->saList, inGlobals->hostEntryList, outPacketBlock);
				if (status == pmStatus_ApplySA) {
					DConLogNote("  SA Applied\n");
					result = true;
				} else {
					const mblk_t *	theMessage = inPacketMessage;
					
					DConLogNote("  Unable to apply SA\n");
					while (theMessage != NULL) {
						DConDumpNote("Packet beginning ", theMessage->b_rptr,
							theMessage->b_wptr - theMessage->b_rptr);
						DConLogNote("\n");
						theMessage = theMessage->b_cont;
					}
				}
				OTFreeMem(buffers);
			} else {
				DConLogNote("  Unable to allocate buffers\n");
			}
		}
		break;
		
		
		case pmStatus_DropPacket:
		{
			DConLogNote("  Status = drop packet\n");
			result = false;
		}
		break;
	}
	
	DConLogNote("  IPTransformPacket End = %d\n", static_cast<int>(result));

	return result;
}



	PGPBoolean
IPProcessAndPutNext(
	PGPBoolean				inIncoming,
	queue_t *				inQ,
	const mblk_t *			inHeaderBlock,
	const mblk_t *			inPacketMessage,
	SGlobals *				inGlobals,
	SIPReassemblyList **	inReassemblyList)
{
	PGPBoolean	result = false;
	
	DConLogNote("  IPDecodeAndPutNext Begin %d\n", msgdsize(inPacketMessage));	
//DConDumpNote("", inPacketMessage->b_rptr, inPacketMessage->b_wptr - inPacketMessage->b_rptr);
	if (pullupmsg(const_cast<mblk_t *>(inPacketMessage), kStandardIPHeaderLength)
	&& pullupmsg(const_cast<mblk_t *>(inPacketMessage), GetIPHeaderHeaderLength(
	inPacketMessage->b_rptr))) {
		PGPByte *	ipHeader = reinterpret_cast<PGPByte *>(inPacketMessage->b_rptr);
		PGPUInt8	ipHeaderSize = GetIPHeaderHeaderLength(ipHeader);
		mblk_t *	header = NULL;
		mblk_t *	packetMessage = NULL;
		PGPBoolean	reassembled;

		if ((ipHeaderSize >= 20) && CheckIPHeaderChecksum(ipHeader, ipHeaderSize)) {
//dopen("Packets.txt");
//pullupmsg(const_cast<mblk_t *>(inPacketMessage), -1);
//DConDumpNote(inIncoming ? "Incoming " : "Outgoing ", inPacketMessage->b_rptr, inPacketMessage->b_wptr - inPacketMessage->b_rptr);

			if (IsIPHeaderAFragment(ipHeader)) {
				DConLogNote("  IP packet is a fragment\n");
				if (pullupmsg(const_cast<mblk_t *>(inPacketMessage), -1)) {
					IPReassemblePacket(inReassemblyList,
						inHeaderBlock, inPacketMessage, &header, &packetMessage);
					reassembled = true;
				} else {
					DConLogNote("  Unable to pull up the packet\n");
				}
			} else {
				IPTimeoutFragments(inReassemblyList);
				header = const_cast<mblk_t *>(inHeaderBlock);
				packetMessage = const_cast<mblk_t *>(inPacketMessage);
				reassembled = false;
			}
			
			if (header != NULL) {
				mblk_t *	outPacketBlock = NULL;
				
				result = IPTransformPacket(inIncoming, inQ, packetMessage, inGlobals,
							&outPacketBlock);
				if (reassembled) {
					if (outPacketBlock != NULL) {
						IPFragmentAndPutNext(inQ, header, outPacketBlock);
						freeb(outPacketBlock);
					} else if (result) {
						if (pullupmsg(packetMessage, -1)) {
							IPFragmentAndPutNext(inQ, header, packetMessage);
						} else {
							DConLogNote("  Unable to pull up packet\n");
						}
					}
					result = false;
					freeb(header);
					freemsg(packetMessage);
				} else if (outPacketBlock != NULL) {
					IPFragmentAndPutNext(inQ, header, outPacketBlock);
					freeb(outPacketBlock);
					result = false;
				}
			}
//dopen(NULL);
		} else {
			DConLogNote(" Invalid ip header\n");
		}
	} else {
		DConLogNote("  Unable to pull up ip header\n");
	}
		
	DConLogNote("  IPDecodeAndPutNext End = %d\n", static_cast<int>(result));
//DConDumpNote("", inPacketMessage->b_rptr, inPacketMessage->b_wptr - inPacketMessage->b_rptr);
	return result;
}



/*
	PGPBoolean
IPDecodeAndPutNext(
	queue_t *		inQ,
	const mblk_t *	inHeaderBlock,
	const mblk_t *	inPacketMessage,
	SGlobals *		inGlobals)
{
	PGPBoolean	result = false;
	
	DConLogNote("  IPDecodeAndPutNext Begin %d\n", msgdsize(inPacketMessage));	
//DConDumpNote("", inPacketMessage->b_rptr, inPacketMessage->b_wptr - inPacketMessage->b_rptr);
	if (pullupmsg(const_cast<mblk_t *>(inPacketMessage), kStandardIPHeaderLength)
	&& pullupmsg(const_cast<mblk_t *>(inPacketMessage), GetIPHeaderHeaderLength(
	inPacketMessage->b_rptr))) {
		PGPByte *	ipHeader = reinterpret_cast<PGPByte *>(inPacketMessage->b_rptr);
		PGPUInt8	ipHeaderSize = GetIPHeaderHeaderLength(ipHeader);
		mblk_t *	header = NULL;
		mblk_t *	packetMessage = NULL;
		PGPBoolean	reassembled;
	
		if ((ipHeaderSize >= 20) && CheckIPHeaderChecksum(ipHeader, ipHeaderSize)) {
			if (IsIPHeaderAFragment(ipHeader)) {
				DConLogNote("  IP packet is a fragment\n");
				if (pullupmsg(const_cast<mblk_t *>(inPacketMessage), -1)) {
					IPReassemblePacket(&GetPerStreamData(inQ)->reassemblyList,
						inHeaderBlock, inPacketMessage, &header, &packetMessage);
					reassembled = true;
				} else {
					DConLogNote("  Unable to pull up the packet\n");
				}
			} else {
				IPTimeoutFragments(&GetPerStreamData(inQ)->reassemblyList);
				header = const_cast<mblk_t *>(inHeaderBlock);
				packetMessage = const_cast<mblk_t *>(inPacketMessage);
				reassembled = false;
			}
			
			if (header != NULL) {
				mblk_t *	outPacketBlock = NULL;
				
				result = IPTransformPacket(true, inQ, packetMessage, inGlobals,
							&outPacketBlock);
				if (reassembled) {
					if (outPacketBlock != NULL) {
						IPFragmentAndPutNext(inQ, header, outPacketBlock);
						freeb(outPacketBlock);
					} else if (result) {
						if (pullupmsg(packetMessage, -1)) {
							IPFragmentAndPutNext(inQ, header, packetMessage);
						} else {
							DConLogNote("  Unable to pull up packet\n");
						}
					}
					result = false;
					freeb(header);
					freemsg(packetMessage);
				} else if (outPacketBlock != NULL) {
					IPFragmentAndPutNext(inQ, header, outPacketBlock);
					freeb(outPacketBlock);
					result = false;
				}
			}
		} else {
			DConLogNote(" Invalid ip header\n");
		}
	} else {
		DConLogNote("  Unable to pull up ip header\n");
	}
		
	DConLogNote("  IPDecodeAndPutNext End = %d\n", static_cast<int>(result));
//DConDumpNote("", inPacketMessage->b_rptr, inPacketMessage->b_wptr - inPacketMessage->b_rptr);
	return result;
}



	PGPBoolean
IPEncodeAndPutNext(
	queue_t *			inQ,
	const mblk_t *		inHeaderBlock,
	const mblk_t *		inPacketMessage,
	SGlobals *			inGlobals)
{
	PGPBoolean	result = true;

	DConLogNote("  IPEncodeAndPutNext Begin %d\n", msgdsize(inPacketMessage));	
//DConDumpNote("", inPacketMessage->b_rptr, inPacketMessage->b_wptr - inPacketMessage->b_rptr);
	if (pullupmsg(const_cast<mblk_t *>(inPacketMessage), kStandardIPHeaderLength)
	&& pullupmsg(const_cast<mblk_t *>(inPacketMessage), GetIPHeaderHeaderLength(
	inPacketMessage->b_rptr))) {
		mblk_t *	outPacketBlock = NULL;
//(void)inGlobals;
//pullupmsg(const_cast<mblk_t *>(inPacketMessage), -1);
//outPacketBlock = dupb(const_cast<mblk_t *>(inPacketMessage));
		
		result = IPTransformPacket(false, inQ, inPacketMessage, inGlobals, &outPacketBlock);
		if (outPacketBlock != NULL) {
			IPFragmentAndPutNext(inQ, inHeaderBlock, outPacketBlock);
			freeb(outPacketBlock);
			result = false;
		}
	} else {
		DConLogNote("  Unable to pull up ip header\n");
		result = false;
	}
	
	DConLogNote("  IPEncodeAndPutNext End = %d\n", static_cast<int>(result));
//DConDumpNote("", inPacketMessage->b_rptr, inPacketMessage->b_wptr - inPacketMessage->b_rptr);
	return result;
}
*/