/*____________________________________________________________________________
	Copyright (C) 1994-1998 Network Associates Inc. and affiliated companies.
	All rights reserved.
	
	$Id: PGPDiskUtils.c,v 1.33 1999/03/15 17:43:29 heller Exp $
____________________________________________________________________________*/

#include <Finder.h>
#include <Sound.h>
#include <Timer.h>
#include <FSM.h>
#include <Icons.h>
#include <Math64.h>
#include <stdarg.h>
#include <stdio.h>
#include <LowMem.h>
#include "CipherProc.h"
#include "fp.h"

#include <PP_Messages.h>
#include <UModalDialogs.h>
#include "CPGPStDialogHandler.h"

#include "MacErrors.h"
#include "MacEvents.h"
#include "MacFiles.h"
#include "UInt64.h"
#include "MacResources.h"
#include "MacStrings.h"
#include "MacSecureMemory.h"
#include "MacDriverUtils.h"
#include "pgpMacMemory.h"
#include "Beta.h"

#include "PGPDiskEncryptDecrypt.h"
#include "PGPDiskFile.h"
#include "PGPDiskGestalt.h"
#include "PGPDiskPreferences.h"
#include "PGPDiskResources.h"
#include "PGPDiskUtils.h"

#include "CWarningAlert.h"
#include "AddDriveStruct.h"
#include "CTuner.h"
#include "MacDriverUtils.h"
#include "pflPrefs.h"
#include "pflPrefTypes.h"
#include "pgpAdminPrefs.h"
#include "pgpClientErrors.h"
#include "pgpErrors.h"
#include "pgpUtilities.h"
#include "PGPOpenPrefs.h"

#define MIN(a,b)		( (a) <= (b) ? (a) : (b) )


short			gApplicationResFile;


	OSStatus
CheckForLocalPGPDiskVolume(short vRefNum)
{
	OSStatus		err;
	HParamBlockRec	pb;
	Str255			volumeName;
	
	pgpClearMemory( &pb, sizeof( pb ) );
	
	pb.volumeParam.ioNamePtr = volumeName;
	pb.volumeParam.ioVRefNum = vRefNum;
	
	err = PBHGetVInfoSync( &pb );
	if( IsntErr( err ) )
	{	
		if( pb.volumeParam.ioVFSID != 0 )
		{
			GetVolParmsInfoBuffer	volParms;
			
			pgpClearMemory( &pb, sizeof( pb ) );
			pgpClearMemory( &volParms, sizeof( volParms ) );
		
			pb.volumeParam.ioNamePtr 	= volumeName;
			pb.volumeParam.ioVRefNum 	= vRefNum;
			pb.ioParam.ioBuffer 		= (Ptr) &volParms;
			pb.ioParam.ioReqCount 		= sizeof( volParms );
			
			err = PBHGetVolParmsSync( &pb );
			if( IsntErr( err ) )
			{
				if( volParms.vMServerAdr != 0 )
				{
					err = kPGPDiskRemoteVolumeTypeError;
				}
				else
				{
					err = kPGPDiskForeignVolumeTypeError;
				}
			}
		}
		else
		{
			long			result;
			const OSType	kMAEGestaltSelector	= 'cith';
			
			/*
				From (Byron Han):
				check the fsID of the disk you are saving the PGPdisk on.
				if it is zero (for standard MacOS HFS disk), you need to
				check for MAE. to check for MAE, see if gestalt selector
				'cith' is implemented. if it is, the fsID (returned from
				PBHGetVInfo) may be a lie!  you need to check if the fsID
				is 0x0017 in the VCB.  0x0017 is the file system ID used for
				MAE's boot disk.  If you detect that this is the case, you
				cannot mount a disk image on this type of volume.
			*/
			if ( Gestalt( kMAEGestaltSelector, &result ) == noErr )
			{
				if ( GetVCBForVolume( vRefNum )->vcbFSID == 0x0017 )
				{
					err = kPGPDiskForeignVolumeTypeError;
				}
			}
			
			if( IsntErr( err ) )
			{
				const VCB	*vcb;
				
				/* Check for HFS/HFS Plus. */
				
				vcb	= GetVCBForVolume( vRefNum );
				if( IsntNull( vcb ) )
				{
					if( vcb->vcbSigWord != kHFSVolumeSignature &&
						vcb->vcbSigWord != kHFSPlusVolumeSignature )
					{
						err = kPGPDiskForeignVolumeTypeError;
					}
				}
			}
		}
	}
	
	return( err );
}

	Boolean
IsPGPDiskFileType(OSType fileCreator, OSType fileType)
{
	return( fileCreator == kPGPDiskFileCreator && 
			fileType == kPGPDiskFileType );
}

/*___________________________________________________________________________
	Checks to see if the the passed in file spec is a mounted PGP Disk file.
___________________________________________________________________________*/

	static Boolean
IsPGPDiskFileInUseByDriver(const FSSpec *fileSpec, short *driveNumber)
{
	OSStatus		err = noErr;
	ushort			numDrives;
	const ushort	kMaxDrives	= 200;
	short			driveNumbers[ kMaxDrives ];
	Boolean			inUse;
	
	AssertSpecIsValid( fileSpec, "IsPGPDiskFileMounted" );
	pgpAssertAddrValid( driveNumber, short );

	*driveNumber 	= 0;
	inUse			= FALSE;
	
	GetAllPGPDrives( driveNumbers, kMaxDrives, &numDrives );

	for( ushort driveIndex = 0; driveIndex < numDrives; ++driveIndex )
	{
		PGPDriveInfoStruct	info;
		
		err = GetPGPDriveInfo( driveNumbers[driveIndex], &info );
		if( IsntErr( err ) )
		{
			FSSpec	checkFileSpec;
			
			err = GetSpecFromRefNum( info.fileRefNum, &checkFileSpec );
			if( IsntErr( err ) )
			{
				if( FSpEqual( fileSpec, &checkFileSpec ) )
				{
					inUse 			= TRUE;
					*driveNumber	= driveNumbers[driveIndex];

					break;
				}
			}
		}
		else
		{
			pgpAssert( "IsPGPDiskFileMounted: Error getting drive info" );
		}
	}
	
	return( inUse );
}

	OSStatus
GetPGPDiskFileInfo(
	const FSSpec 		*fileSpec,
	PGPDiskFileInfo		*fileInfo)
{
	OSStatus	err;
	CInfoPBRec	cpb;
	
	AssertSpecIsValid( fileSpec, "GetPGPDiskFileInfo" );
	pgpAssertAddrValid( fileInfo, PGPDiskFileInfo );
	
	pgpClearMemory( fileInfo, sizeof( *fileInfo ) );
	
	err = FSpGetCatInfo( fileSpec, &cpb );
	if( IsntErr( err ) )
	{
		if( cpbIsFile( &cpb ) )
		{
			err = CheckForLocalPGPDiskVolume( fileSpec->vRefNum );
			if( IsntErr( err ) )
			{
				if( IsPGPDiskFileType( cpbFileCreator( &cpb ),
						cpbFileType( &cpb ) ) )
				{
					HParamBlockRec	pb;
					Str255			volumeName;
					
					if( ( cpbAttributes( &cpb ) &
								kFileAttributeFileOpenMask ) != 0 )
					{
						fileInfo->fileIsOpen = TRUE;
						fileInfo->fileIsInUseByDriver =
								IsPGPDiskFileInUseByDriver( fileSpec,
									&fileInfo->fileDriveNumber );
						
						if( fileInfo->fileIsInUseByDriver )
						{
							fileInfo->fileIsMounted =
								IsntNull( GetVCBForDrive(
									fileInfo->fileDriveNumber ) );
						}
					}
					
					if( ( cpbAttributes( &cpb ) &
							kFileAttributeFileLockedMask ) != 0 )
					{
						fileInfo->fileIsLocked = TRUE;
					}
					
					pgpClearMemory( &pb, sizeof( pb ) );
					
					pb.volumeParam.ioNamePtr = volumeName;
					pb.volumeParam.ioVRefNum = fileSpec->vRefNum;
					
					err = PBHGetVInfoSync( &pb );
					if( IsntErr( err ) )
					{	
						if( ( pb.volumeParam.ioVAtrb &
								kVolumeAttributeLockMask ) != 0 )
						{
							fileInfo->fileIsOnLockedVolume = TRUE;
						}
					}
				}
				else
				{
					err = kPGPDiskUnknownFileKindError;
				}
			}
		}
		else
		{
			pgpDebugMsg( "GetPGPDiskFileInfo: Spec is not a file" );
			err = notAFileErr;
		}
	}
	
	return( err );
}

/*___________________________________________________________________________
	Change a passphrase, providing that it matches
___________________________________________________________________________*/

	OSStatus
ChangePassphrase(
	PGPContextRef	context,
 	const FSSpec 	*fileSpec,
 	const uchar 	*oldPassphrase,
 	const uchar 	*newPassphrase)
{
	OSStatus		err;
	PGPDiskFileRef	diskFileRef;
	
	AssertSpecIsValid( fileSpec, "ChangePassphrase" );
	pgpAssertAddrValid( oldPassphrase, uchar );
	pgpAssertAddrValid( newPassphrase, uchar );
	
	err = OpenPGPDiskFile( context, fileSpec, &diskFileRef );
	if( IsntErr( err ) )
	{
		PGPDiskKeyRef	keyRef;
		
		err = FindPassphrasePGPDiskKey( diskFileRef, oldPassphrase, &keyRef );
		if( IsntErr( err ) )
		{
			err = ChangePGPDiskKeyPassphrase( keyRef, oldPassphrase,
							newPassphrase );
			if( IsntErr( err ) )
			{
				err = SavePGPDiskFile( diskFileRef );
			}
			
			DisposePGPDiskKey( keyRef );
		}
	
		ClosePGPDiskFile( diskFileRef );
	}
	
	return( err );
}

/*___________________________________________________________________________
	Verify a passphrase is valid
___________________________________________________________________________*/
	OSStatus
VerifyPassphrase(
	PGPContextRef		context,
	const FSSpec		*fileSpec,
	ConstStr255Param	passphrase )
{
	OSStatus		err;
	PGPDiskFileRef	diskFileRef;
	
	err = OpenPGPDiskFile( context, fileSpec, &diskFileRef );
	if( IsntErr( err ) )
	{
		PGPDiskKeyRef	keyRef;
		
		err = FindPassphrasePGPDiskKey( diskFileRef, passphrase, &keyRef );
		if( IsntErr( err ) )
		{
			// Finding it is sufficient to verify the passphrase
			DisposePGPDiskKey( keyRef );
		}
	
		ClosePGPDiskFile( diskFileRef );
	}
	
	return( err );
}



/*___________________________________________________________________________
	Verify that a passphrase is the master passphrase
___________________________________________________________________________*/

	OSStatus
VerifyMasterPassphrase(
	PGPContextRef		context,
	const FSSpec		*fileSpec,
	ConstStr255Param	passphrase )
{
	OSStatus		err;
	PGPDiskFileRef	diskFileRef;
	
	err = OpenPGPDiskFile( context, fileSpec, &diskFileRef );
	if( IsntErr( err ) )
	{
		PGPDiskKeyRef	keyRef;
		
		err = GetMasterPGPDiskKey( diskFileRef, &keyRef );
		if( IsntErr( err ) )
		{
			err = VerifyPGPDiskKeyPassphrase( keyRef, passphrase );
			
			DisposePGPDiskKey( keyRef );
		}
	
		ClosePGPDiskFile( diskFileRef );
	}
	
	return( err );
}

/*___________________________________________________________________________
	Copy the custom icons to the drive if not already there.
___________________________________________________________________________*/
	static OSStatus
CopyCustomVolumeIconToDrive(short driveNumber)
{
	OSStatus		err;
	HParamBlockRec	pb;
	Str255			volumeName;
	
	pgpClearMemory( &pb, sizeof( pb ) );

	// Call PBHGetVInfo to turn our drive number into a vRefNum.
	
	pb.volumeParam.ioVRefNum	= driveNumber;
	pb.volumeParam.ioNamePtr	= volumeName;
	
	err = PBHGetVInfoSync( &pb );
	AssertNoErr( err, nil );
	
	if( IsntErr( err ) )
	{
		FSSpec		iconFileSpec;
		short		iconFileRef;
		short		saveResFile;
		CInfoPBRec	cpb;
		
		saveResFile = CurResFile();
		
		iconFileSpec.vRefNum	= pb.volumeParam.ioVRefNum;
		iconFileSpec.parID		= fsRtDirID;
		
		CopyPString( "\pIcon\r", iconFileSpec.name );
		
		err	= FSpGetCatInfo( &iconFileSpec, &cpb );
		if( IsntErr( err ) )
		{
			// The custom icon file may be empty. Blow it away if the length
			// of the resource fork iz zero. Mac OS 8.1 appears to leave empty
			// files instead of deleting them
			
			if( cpbResForkSize( &cpb ) == 0 )
				(void) FSpDelete( &iconFileSpec );
				
			err = FSpGetCatInfo( &iconFileSpec, &cpb );
		}
		
		if ( IsErr( err ) )
		{
			// icon file does not yet exist, so create one
			
			// The system creates custom icon files with no type and creator.
			// So shall we
			FSpCreateResFile( &iconFileSpec, 0, 0, smSystemScript );
			
			iconFileRef = FSpOpenResFile( &iconFileSpec, fsRdWrPerm );
			if( iconFileRef > 0 )
			{
				UseResFile( iconFileRef );
				
				err = CopyResource( gApplicationResFile, kLarge1BitMask,
									kPGPDiskVolumeIconResID,
									iconFileRef, kLarge1BitMask,
									kCustomIconResource );
				AssertNoErr( err, nil );
				
				if( IsntErr( err ) )
				{
					// The black and white icon must succeed. The rest are
					// optional
					
					(void) CopyResource( gApplicationResFile, kLarge4BitData,
									kPGPDiskVolumeIconResID, iconFileRef,
									kLarge4BitData, kCustomIconResource );
					(void) CopyResource( gApplicationResFile, kLarge8BitData,
									kPGPDiskVolumeIconResID, iconFileRef,
									kLarge8BitData, kCustomIconResource );
					(void) CopyResource( gApplicationResFile, kSmall1BitMask,
									kPGPDiskVolumeIconResID, iconFileRef,
									kSmall1BitMask, kCustomIconResource );
					(void) CopyResource( gApplicationResFile, kSmall4BitData,
									kPGPDiskVolumeIconResID, iconFileRef,
									kSmall4BitData, kCustomIconResource );
					(void) CopyResource( gApplicationResFile, kSmall8BitData,
									kPGPDiskVolumeIconResID, iconFileRef,
									kSmall8BitData, kCustomIconResource );
					(void) CopyResource( gApplicationResFile, 'icns',
									kPGPDiskVolumeIconResID, iconFileRef,
									'icns', kCustomIconResource );
				}
		
				CloseResFile( iconFileRef );
				UseResFile( saveResFile );
			}
			else
			{
				err = ForceResError( fnfErr );
			}
		}
		
		if( IsntErr( err ) )
		{
			// Make the custom icon file invisible
	
			err = FSpGetCatInfo( &iconFileSpec, &cpb );
			AssertNoErr( err, nil );
			
			if( IsntErr( err ) )
			{
				cpb.hFileInfo.ioFlFndrInfo.fdFlags |= kIsInvisible;
				
				err = FSpSetCatInfo( &iconFileSpec, &cpb );
				AssertNoErr( err, nil );
			}
		}
		
		if( IsntErr( err ) )
		{
			CInfoPBRec	cpb;
			FSSpec		volumeSpec;
			
			// Set the custom icon bit of the root directoy catalog entry.
			
			volumeSpec.vRefNum	= iconFileSpec.vRefNum;
			volumeSpec.parID	= fsRtParID;
			
			CopyPString( volumeName, volumeSpec.name );
			
			err = FSpGetCatInfo( &volumeSpec, &cpb );
			AssertNoErr( err, nil );
			
			if( IsntErr( err ) )
			{
				cpb.dirInfo.ioDrUsrWds.frFlags |= kHasCustomIcon;
				
				err = FSpSetCatInfo( &volumeSpec, &cpb );
				AssertNoErr( err, nil );
			}
		}
		
		if( IsErr( err ) )
		{
			pgpDebugMsg( "CopyCustomVolumeIconToDrive: Error!" );
			
			(void) FSpDelete( &iconFileSpec );
		}
	}
	
	return( err );
}



	static ulong
DetermineGoodWipeSize(
	short		driveNumber,
	short		driverRefNum )
{
	const UInt32		kMaxWipeSize	= 1024UL * 1024UL;
	const UInt32		kMinWipeSize	= 4UL * 1024UL;
	const UInt32		kDefaultSize	= 64UL * 1024UL;
	void *			buffer;
	UInt32			goodSize	= kDefaultSize;
	UInt32		bufferSize;
	OSStatus		err	= noErr;
		
	buffer	= pgpNewPtrMost( goodSize, kMinWipeSize,
				kMacMemory_UseCurrentHeap | kMacMemory_PreferTempMem,
				&bufferSize);
	pgpAssert( IsntNull( buffer ) );
	if ( IsntNull( buffer ) )
	{
		ParamBlockRec	pb;
		UnsignedWide	startMS;
		UnsignedWide	stopMS;
	
		pb.ioParam.ioCompletion		= nil;
		pb.ioParam.ioVRefNum		= driveNumber;
		pb.ioParam.ioRefNum			= driverRefNum;
		pb.ioParam.ioBuffer			= (Ptr)buffer;
		pb.ioParam.ioReqCount		= bufferSize;
		pb.ioParam.ioPosMode		= fsFromStart;
		pb.ioParam.ioPosOffset		= 0;
		
		Microseconds( &startMS );
		err	= PBWriteSync( &pb );
		Microseconds( &stopMS );
		
		if ( IsntErr( err ) )
		{
			pgpAssert( UnsignedWideToUInt64(stopMS) >=
				UnsignedWideToUInt64(startMS) );
			UInt32	elapsedMilli	= (stopMS - startMS).lo / 1000;
			
			if ( elapsedMilli != 0 )
			{
				float	elapsedSeconds	= (float)elapsedMilli / (float)1000;
				UInt32	bytesPerSecond	= pb.ioParam.ioActCount /
								elapsedSeconds;
				
				// we'd like to be able to cancel after a delay of no more
				// than a second
				goodSize	= bytesPerSecond;
				
				if ( goodSize > kMaxWipeSize )
					goodSize	= kMaxWipeSize;
					
				goodSize	= ( goodSize / kDiskBlockSize ) * kDiskBlockSize;
				
				if ( goodSize < kMinWipeSize )
					goodSize	= kMinWipeSize;
			}
		}
		
		pgpFreeMac( buffer );
	}
	
	return( goodSize );
}
	
	
	static OSStatus
WipeDiskUsingDrive(
	short						driveNumber,
	short						driverRefNum,
	UInt32						numBlocksToWipe,
	MountDiskProgressProcPtr	progressProc,
	void						*progressUserValue)
{
	OSStatus		err	= noErr;
	void *			buffer;
	UInt32			bufferSize;
	const UInt32	kMinWipeSize		= kPGPDiskBlockSize;
	
	bufferSize = DetermineGoodWipeSize( driveNumber, driverRefNum );
	
	// We have a memory problem here. If we ask for a lot of memory and it
	// comes out of the application heap, then we may trigger the
	// GrowZoneProc for the app and a low memory warning to the user.
	// To alleviate this, we explicitly try for Temp Mem first
	// and then the App heap, leaving a sizeable amount free.

	buffer = pgpNewPtrMost( bufferSize, kMinWipeSize, kMacMemory_UseTempMem,
					&bufferSize);
	if( IsNull( buffer ) )
	{
		#define	kMinAppFreeSpace	(64 * 1024L)
		
		bufferSize = MaxBlock();
		if( bufferSize >= 2 * kMinAppFreeSpace )
		{
			bufferSize -= kMinAppFreeSpace;
		}
		else if( bufferSize >= kMinAppFreeSpace )
		{
			bufferSize = bufferSize / 2;
		}
		else
		{
			bufferSize = 2 * kMinWipeSize;
		}

		buffer = pgpNewPtrMost( bufferSize, kMinWipeSize,
						kMacMemory_UseCurrentHeap, &bufferSize);
	}

	pgpAssert( IsntNull( buffer ) );
		
	if ( IsntNull( buffer ) )
	{
		UInt32	remainingBlocks;
		UInt32	bufferSizeInBlocks;
		UInt32	percentComplete 	= 0;
		UInt32	blocksWrittenSoFar	= 0;
		
		// make sure buffer size is an integral number of blocks
		bufferSizeInBlocks	= bufferSize / kPGPDiskBlockSize;
		bufferSize			= bufferSizeInBlocks * kPGPDiskBlockSize;
		pgpAssert( bufferSize != 0 &&
						( bufferSize % kPGPDiskBlockSize ) == 0 );

		remainingBlocks	= numBlocksToWipe;
		
		
		while( remainingBlocks != 0 && IsntErr( err ) )
		{
			volatile ParamBlockRec	pb;	// IMPORTANT: must be volatile
			UInt32					blocksThisWrite;
			
			blocksThisWrite	= MIN( remainingBlocks, bufferSizeInBlocks );
		
			pb.ioParam.ioCompletion		= nil;
			pb.ioParam.ioVRefNum		= driveNumber;
			pb.ioParam.ioRefNum			= driverRefNum;
			pb.ioParam.ioBuffer			= (Ptr)buffer;
			pb.ioParam.ioReqCount		= blocksThisWrite * kPGPDiskBlockSize;
			pb.ioParam.ioPosMode		= fsFromStart;
			pb.ioParam.ioPosOffset		= blocksWrittenSoFar *
												kPGPDiskBlockSize;
			
			(void)PBWriteAsync( (ParamBlockRec *) &pb );
			
			// yield time while/after write completes
			do
			{
				err = (*progressProc)( percentComplete, progressUserValue );
				if( IsErr( err ) )
				{
					while ( pb.ioParam.ioResult == 1 )
					{
						// wait till done
					}

					break;
				}
				
			} while ( ( err = pb.ioParam.ioResult ) == 1 );
			
			remainingBlocks		-= blocksThisWrite;
			blocksWrittenSoFar	+= blocksThisWrite;
		
			if( IsntErr( err ) )
			{
				percentComplete = (double) blocksWrittenSoFar * 100.0 /
										(double) numBlocksToWipe;
			
				err = (*progressProc)( percentComplete, progressUserValue );
			}
		}
		
		pgpFreeMac( buffer );
	}
	else
	{
		err = memFullErr;
	}
	
	return( err );
}


	static void
GetMediaAndPhysicalIcons(
	BWIcon *	mediaIcon,
	BWIcon *	physicalIcon )
	{
	Handle	iconHandle	= nil;
	short	physicalID	= kPGPDiskVolumeIconResID;
	short	mediaID		= physicalID;
	short	saveResFile	= CurResFile();
	
	UseResFile(gApplicationResFile);
	
	iconHandle	= Get1Resource('ICN#', physicalID);
	if( IsntNull( iconHandle ) )
		{
		pgpAssert( HomeResFile( iconHandle ) == gApplicationResFile );
		*mediaIcon	= *(BWIcon *)*iconHandle;
		ReleaseResource( iconHandle );
		}
	else
		{
		pgpDebugPStr( "\pGetMediaAndPhysicalIcons: can't get icon");
		}
	
	// get the icon for the physical media
	iconHandle	= Get1Resource('ICN#', mediaID);
	if( IsntNull( iconHandle ) )
		{
		pgpAssert( HomeResFile( iconHandle ) == gApplicationResFile );
		*physicalIcon	= *(BWIcon *)*iconHandle;
		ReleaseResource( iconHandle );
		}
	else
		{
		pgpDebugPStr( "\pGetMediaAndPhysicalIcons: can't get icon");
		}
	
	UseResFile( saveResFile );
	}

// Removes the header and trailer blocks from the extent list.

	static OSErr
PGPDiskFileExtentsToPGPDiskDataExtents(
	DiskExtent 	*extentsList,
	PGPUInt32	numFileExtents,
	PGPUInt32	firstDataBlock,
	PGPUInt32	numDataBlocks,
	PGPUInt32 	*numDataExtents)
{
	OSErr		err	= noErr;
	DiskExtent	*curExtent;
	DiskExtent	*lastExtent;
	UInt32		remainingHeaderBlocks;
	UInt32		remainingDataBlocks;

	pgpAssertAddrValid( extentsList, DiskExtent );
	pgpAssertAddrValid( numDataExtents, UInt32 );
	pgpAssert( firstDataBlock != 0 );
	
	curExtent 				= extentsList;
	lastExtent				= curExtent + numFileExtents - 1;
	remainingHeaderBlocks	= firstDataBlock;
	remainingDataBlocks		= numDataBlocks;
	
	pgpAssert( remainingHeaderBlocks < remainingDataBlocks );
	
	while( remainingDataBlocks != 0 && curExtent <= lastExtent )
	{
		if( remainingHeaderBlocks != 0 )
		{
			if( remainingHeaderBlocks >= curExtent->numBlocks )
			{
				// Entire extent is header data.
				remainingHeaderBlocks 	-= curExtent->numBlocks;
				curExtent->numBlocks 	= 0;
			}
			else
			{
				// Extent is partially header data.
				curExtent->numBlocks 		-= remainingHeaderBlocks;
				curExtent->diskBlockIndex 	+= remainingHeaderBlocks;
				remainingHeaderBlocks		= 0;
			}
		}

		// Check to see if this is the last data block extent.
		if( curExtent->numBlocks > remainingDataBlocks )
		{
			curExtent->numBlocks 	= remainingDataBlocks;
			remainingDataBlocks		= 0;
		}
		else
		{
			remainingDataBlocks	-= curExtent->numBlocks;
		}
		
		++curExtent;
	}
		
	// At this point, the extent list may have zero-length entries.
	// Shift the list data to remove these entries.

	*numDataExtents = numFileExtents;
	
	for( UInt32 index = 0; index < numFileExtents; ++index )
	{
		if( extentsList[index].numBlocks != 0 )
		{
			if( index > 0 )
			{
				UInt32	bytesToMove;
				
				// This is the first non-empty block which is not the first
				// block. Shift the entries from this point.
				
				bytesToMove = ( *numDataExtents - index ) *
									sizeof( DiskExtent );
				
				BlockMoveData( &extentsList[index], &extentsList[0],
							bytesToMove );
				*numDataExtents -= index;
			}
			
			break;
		}
	}

	return( err );
}


	static CComboError
AddInfoForFile(
	PGPDiskFileRef				diskFileRef,
	PGPDiskKeyRef				keyRef,
	short						fileRefNum,
	AddDriveStruct *			info,
	Boolean						mountReadOnly,
	ConstStr255Param			passphrase)
{
	CComboError					err;
	const PGPDiskFileHeader		*fileHeader;
	
	AssertFileRefNumIsValid( fileRefNum, "AddInfoForFile" );
	pgpAssertAddrValid( info, AddDriveStruct );
	pgpAssertAddrValid( passphrase, uchar );
	
	fileHeader = PeekPGPDiskFileHeader( diskFileRef );
	
	info->fileRefNum		= fileRefNum;
	info->numBlocks			= fileHeader->numDataBlocks;
	info->mountReadOnly		= mountReadOnly;
	
	err.err = GetDiskExtentsForFork( fileRefNum, (DiskExtent **) &info->extents,
					&info->numExtents );
	if( err.IsntError() )
	{
		err.err = PGPDiskFileExtentsToPGPDiskDataExtents(
						(DiskExtent *) info->extents,
						info->numExtents, fileHeader->numHeaderBlocks,
						fileHeader->numDataBlocks, &info->numExtents );
		if( err.IsntError() )
		{
			info->salt = fileHeader->salt;

			err = DecryptPGPDiskKey( keyRef, passphrase,
						&info->decryptionKey );
		}
	}
	
	return( err );
}
	
/*___________________________________________________________________________
	On some machines, breaking up PBs into smaller ones just gives us slower
	I/O due to the smaller requests we make, with payback.  On modern machines
	with DMA, we get ample time back (at least with SCSI).
	
	By testing a particular driver, we can optimize our behavior for the
	particular drive, driver and machine in use.
	
	*Important note*:
		This used to be done inside a sync control call inside the driver and
	the driver would wait for the IO to complete.  This caused a problem where
	the sync call got queued and then executed with interrupts off.  So we must
	make this determination here, in the app, and wait for IO to complete at
	 a "nice" time.
___________________________________________________________________________*/
	static Boolean
ShouldBreakUpRequestsToDriver(
	short	driveNumber,
	short	driverRefNum )
{
	pgpAssert( driverRefNum != 0 );
	pgpAssert( driveNumber > 0 );

	Boolean			shouldBreakUp	= TRUE;
	LogicalAddress	buffer;
	UInt32			bufferSize;
	UInt32			numBlocks;
	const UInt32		kDefaultIOBufferSize	= 128UL * 1024UL;
	const UInt32		kMinIOBufferSize		= 8UL * 1024UL;
	
	buffer = pgpNewPtrMost( kDefaultIOBufferSize, kMinIOBufferSize,
					kMacMemory_PreferTempMem | kMacMemory_UseApplicationHeap,
					&bufferSize );
	pgpDebugMsgIf( IsNull( buffer ),
				"ShouldBreakUpPBs: Unable to allocate buffer" );
	
	if ( IsntNull( buffer ) )
	{
		volatile ParamBlockRec	pb;
	
		numBlocks = bufferSize / kPGPDiskBlockSize;

		Str255	driverName;
		GetDriverNameUsingRefNum( driverRefNum, driverName );
		if ( PStringsAreEqual( driverName, "\p.Sony" ) )
		{
			// reduce the test size so it doesn't take too long
			const UInt32	kMaxFloppyBlocks	= 16;
			
			numBlocks	= MIN( numBlocks, kMaxFloppyBlocks );
		}

		pb.ioParam.ioCompletion		= nil;
		pb.ioParam.ioVRefNum		= driveNumber;
		pb.ioParam.ioRefNum			= driverRefNum;
		pb.ioParam.ioBuffer			= (Ptr)buffer;
		pb.ioParam.ioReqCount		= bufferSize;
		pb.ioParam.ioPosMode		= fsFromStart;
		pb.ioParam.ioPosOffset		= 0;
		
		(void)PBReadAsync( (ParamBlockRec *) &pb );
		
		// see how high we can count while we wait for I/O
		UInt32	counter	= 0;
		while ( pb.ioParam.ioResult == kAsyncIOInProgress )
		{
			++counter;
		}
		
		// unclear where the cutoff should be...do we get dribs and drabs of
		// time back on some machines that we should ignore?
		shouldBreakUp	= counter > 200;
		
		pgpFreeMac( buffer );
		buffer	= nil;
	}
	
	return( shouldBreakUp );
}


	static OSErr
InstallNewPGPDiskDriver(
	ConstStr255Param	name,
	short				refNum)
{
	THz		saveZone		= GetZone();
	Handle	driverHandle	= nil;
	OSErr	err				= noErr;
	
	SetZone( SystemZone() );
	
	driverHandle = Get1NamedResource( kPGPDiskDriverType, kPGPDiskDriverName);
	err = ResError();
	if( IsNull( driverHandle ) )
	{
		err	= resNotFound;
		pgpDebugPStr("\pGet1NamedResource failed!");
	}
	else
	{
		HLock( driverHandle );
		DetachResource( driverHandle );
		MacLeaks_IgnoreItem( driverHandle );

		SetZone( ApplicationZone() );
		
		SetDriverName( (DRVRHeader *) *driverHandle, name );
		
		err = MyDRVRInstall( *driverHandle, refNum);
		if ( IsntErr( err ))
		{
			short	dummy	= refNum;
			
			err = OpenDriver( name, &dummy);
			AssertNoErr( err, "InstallNewPGPDiskDriver" );
			pgpAssert( dummy == refNum );
			if( IsntErr( err ) )
			{
				err = SendPreferencesToDriver( refNum,
							&gPGPDiskPrefs.driverPrefs );
			}
		}
		
		if ( IsErr( err ) )
		{
			short			unitNum	= -1 * ( refNum + 1 );
			AuxDCEHandle	*unitTable	= (AuxDCEHandle *)LMGetUTableBase();
			AuxDCEHandle	dce;
		
			pgpDebugPStr( "\perror installing driver" );
			dce	= unitTable[ unitNum ];
			unitTable[ unitNum ] = nil;
			if ( IsntNull( dce ) )
			{
				DisposeHandle( (Handle)dce );
			}
			
			DisposeHandle( driverHandle );
		}
	}
	
	SetZone( saveZone );
	
	AssertNoErr( err, "GetInstallAndOpenDriver" );
	return( err );
}



	void
GetAllPGPDiskDrivers(
	short		driverRefNums[],
	ushort		maxRefNums,
	ushort *	numDrivers)
{
	OSStatus	err	= noErr;
	ushort		numDCtlEntries;
	
	*numDrivers	= 0;
	
	numDCtlEntries	= GetNumDCtlEntries();
	for( ushort dCtlIndex = 0;
			dCtlIndex < numDCtlEntries && *numDrivers < maxRefNums;
				++dCtlIndex )
	{
		DCtlHandle		dCtl;
		
		dCtl	= GetIndDCtlEntry( dCtlIndex );
		if ( IsntNull( dCtl ) )
		{
			pgpAssert( IsntNull( *dCtl ) );
			
			if ( IsPGPDiskDriver( (**dCtl).dCtlRefNum ) )
			{
				driverRefNums[ *numDrivers ]	= (**dCtl).dCtlRefNum;
				*numDrivers	+= 1;
			}
		}
	}
}

	void
GetAllPGPDrives(
	short		driveNumbers[],
	ushort		maxDrives,
	ushort *	numDriveNumbers)
{
	OSStatus		err	= nsDrvErr;
	const DrvQEl	*driveQEl = nil;
	
	*numDriveNumbers	= 0;
	
	driveQEl = (const DrvQEl *)(GetDrvQHdr()->qHead);
	while( IsntNull( driveQEl ) && *numDriveNumbers < maxDrives)
	{
		if( IsPGPDiskDriver( driveQEl->dQRefNum ) )
		{
			driveNumbers[ *numDriveNumbers ]	= driveQEl->dQDrive;
			*numDriveNumbers	+= 1;
		}
			
		driveQEl	= (const DrvQEl *)(driveQEl->qLink);
	}
}
	
	
	
/*___________________________________________________________________________
	Determine if there is a PGPDisk driver which already controls a PGPDisk
	on a base volume.
___________________________________________________________________________*/
	static Boolean
FindPGPDiskDriverWhichUsesDriver(
	short		baseDriverRefNum,
	short *		driverRefNum )
{
	pgpAssert( baseDriverRefNum < 0 );
	
	OSStatus		err	= nsDrvErr;
	ushort			numDrives;
	const ushort	kMaxDrives	= 200;
	short			driveNumbers[ kMaxDrives ];
	Boolean			foundDriver	= false;
	
	*driverRefNum	= 0;

	GetAllPGPDrives( driveNumbers, kMaxDrives, &numDrives );

	for( ushort driveIndex = 0; driveIndex < numDrives; ++driveIndex )
	{
		PGPDriveInfoStruct	info;
		
		err	= GetPGPDriveInfo( driveNumbers[ driveIndex ], &info);
		if ( IsntErr( err ) )
		{
			short	tempDriverRefNum;
			
			GetDriverRefNumForDrive( info.baseDriveNumberOut,
						&tempDriverRefNum );
			if ( tempDriverRefNum == baseDriverRefNum )
			{
				// a match--same base drive driver as one we're looking for
				GetDriverRefNumForDrive( driveNumbers[ driveIndex ],
							driverRefNum );
				foundDriver		= TRUE;
				break;
			}
		}
	}
	
	return( foundDriver );
}
	
	

/*___________________________________________________________________________
	Find a PGPDisk driver which isn't currently handling any drives
___________________________________________________________________________*/
	static Boolean
FindDriverWithNoDrives( short *	driverRefNum)
{
	const ushort	kMaxDrivers	= 200;
	short			driverRefNums[ kMaxDrivers ];
	ushort			numDrivers;
	OSStatus		err	= noErr;
	Boolean			foundDriver	= false;
	
	*driverRefNum	= 0;
	
	GetAllPGPDiskDrivers( driverRefNums, kMaxDrivers, &numDrivers );
	for( ushort driverIndex = 0; driverIndex < numDrivers; ++driverIndex )
	{
		PGPDriverInfoStruct	info;
		
		err	= GetPGPDiskDriverInfo( driverRefNums[ driverIndex ], &info );
		if ( IsntErr( err ) && info.numDrivesOut == 0 )
		{
			*driverRefNum	= driverRefNums[ driverIndex ];
			
			foundDriver	= TRUE;
			break;
		}
	}
	
	return( foundDriver );
}


/*__________________________________________________________________________
	Return TRUE if the driver is for a hardware device such as SCSI, IDE,
	RAM disk, etc.  Rather than have something algorithmic, just
	"know about" common drivers.
	
	We want to know this so we can reuse the same driver for multiple
	PGPDisks without fear of our driver calling something which calls our
	driver.  If a driver is missing from the list, it's not a big deal;
	it just means we may install another driver in some cases.
___________________________________________________________________________*/
	static Boolean
IsHardwareDriver( short driverRefNum )
{
	Boolean		isHardwareDriver	= false;
	OSStatus	err	= noErr;
	Str255		driverName;
	
	err	= GetDriverNameUsingRefNum( driverRefNum, driverName );
	if ( IsntErr( err ) )
	{
		static ConstStr255Param	sKnownHardwareDrivers[] =
		{
			// abbreviated to the non-changing prefix of the driver name
			"\p.ASYC",					// Apple Drive Setup
			"\p.PTAD!",					// FWB Hard Disk Toolkit
			"\p.#CharisMacSCSIBus",		// APS
			"\p.SC/IOMEGA",				// zip drive and Jaz drive
			"\p.EDisk",					// Apple RAM Disk
			"\p.Sony",					// floppy drive
			"\p.LaCie",					// SilverLining
			"\p.CONLEY SCSI Driver",	// Apple RAID
			"\p.CONLEY PCI",			// Conley raid 1.5
			"\p.Conley PCI Driver",		// Conley raid 2.0
			"\p.ATADisk",				// Apple IDE driver
		};
		static const ushort	kNumDriverNames	=
				sizeof( sKnownHardwareDrivers ) /
					sizeof(sKnownHardwareDrivers[ 0 ]);
		
		for( ushort nameIndex = 0; nameIndex < kNumDriverNames; ++nameIndex )
		{
			ConstStr255Param	prefix	= sKnownHardwareDrivers[ nameIndex ];
			
			if ( pgpMemoryEqual( &driverName[ 1 ], &prefix[ 1 ], prefix[ 0 ] ) )
			{
				isHardwareDriver	= TRUE;
				break;
			}
		}
	}
	
	return( isHardwareDriver );
}
	
	
	
/*___________________________________________________________________________
	Find a PGPDisk driver which uses a base device whose driver is a hardware
	driver.
___________________________________________________________________________*/
	static Boolean
FindPGPDiskDriverWhichUsesHardwareDriver(
	short *	pgpDriverRefNumPtr)
{
	Boolean			foundDriver	= false;
	OSStatus		err	= nsDrvErr;
	ushort			numDrives;
	const ushort	kMaxDrives	= 200;
	short			driveNumbers[ kMaxDrives ];
	
	*pgpDriverRefNumPtr	= 0;

	GetAllPGPDrives( driveNumbers, kMaxDrives, &numDrives );

	for( ushort driveIndex = 0; driveIndex < numDrives; ++driveIndex )
	{
		PGPDriveInfoStruct	info;
		const short			pgpDriveNumber	= driveNumbers[ driveIndex ];
		
		err	= GetPGPDriveInfo( pgpDriveNumber, &info);
		if ( IsntErr( err ) )
		{
			short	tempDriverRefNum;
			
			GetDriverRefNumForDrive( info.baseDriveNumberOut,
					&tempDriverRefNum );
			if ( IsHardwareDriver( tempDriverRefNum ) )
			{
				GetDriverRefNumForDrive( pgpDriveNumber, pgpDriverRefNumPtr );
				foundDriver		= TRUE;
				break;
			}
		}
	}
	
	return( foundDriver );
}



/*___________________________________________________________________________
	Is there already a PGPDisk driver installed which can handle a PGPDisk
	file on the volume baseVRefNum.
___________________________________________________________________________*/

	static Boolean
HaveAppropriateDriver(
	short		baseVRefNum,
	short *		driverRefNumPtr )
	{
	pgpAssert( baseVRefNum < 0 );
	
	Boolean		haveDriver	= false;
	OSStatus	err	= noErr;
	short		baseDriveNumber;
	short		baseDriverRefNum;
	
	*driverRefNumPtr	= 0;
	
	// first, see if we can find a driver which controls an existing PGP disk on
	// the same baseVRefNum
	GetVolumeDriveNumberAndRefNum( baseVRefNum, &baseDriveNumber,
				&baseDriverRefNum );
	if ( FindPGPDiskDriverWhichUsesDriver( baseDriverRefNum, driverRefNumPtr) )
		{
		pgpAssert( *driverRefNumPtr < 0 );
		haveDriver	= TRUE;
		}
	else if ( FindDriverWithNoDrives( driverRefNumPtr ) )
		{
		// a driver could be installed which is not currently handling any
		// drives
		pgpAssert( *driverRefNumPtr < 0 );
		haveDriver	= TRUE;
		}
	else if ( VolumeIsPGPDisk( baseVRefNum ) )
		{
		// we must create a new driver because the underlying drive is
		// already a PGPDisk and there is not yet a driver loaded which is
		// handling this situation
		haveDriver	= false;
		}
	else
		{
		// volume is not a PGPDisk.  It could still be a shrink-wrap volume
		// or some other volume which could reside on a PGPDisk. To prevent
		// this potential problem install a new driver unless we can reuse
		// a PGPDiskDriver which is based on a hardware driver.
		if ( IsHardwareDriver( baseDriverRefNum ) )
			{
			haveDriver	= FindPGPDiskDriverWhichUsesHardwareDriver(
								driverRefNumPtr );
			}
		}
	
	return( haveDriver );
	}
	


/*___________________________________________________________________________
	Return a driver ref num which is open and available for adding drives.
___________________________________________________________________________*/
	static OSStatus
SmartInstallPGPDiskDriver(
	short		baseVRefNum,
	short *		driverRefNum)
{
	OSStatus	err	= noErr;
	
	*driverRefNum	= 0;
	
	if ( HaveAppropriateDriver( baseVRefNum, driverRefNum ) )
	{
		err	= noErr;
	}
	else
	{
		Str255	newName;
		// assume default driver name
		GetUniquePGPDiskDriverName( newName );
		
		err	= GetUnusedDriverRefNum( driverRefNum );
		if ( IsntErr( err ) )
		{
			err	= InstallNewPGPDiskDriver( newName, *driverRefNum );
		}
	}

	return( err );
}


	
	static OSStatus
InstallPGPGestalt(void)
{
	OSStatus	err = noErr;
	long		dummyResult;
	
	// Install our gestalt selector and globals if they are not already
	// present
	
	if( IsErr( Gestalt( gestaltPGPDisk, &dummyResult ) ) )
	{
		Handle	gestaltCodeHandle;
		THz		saveZone;
			
		saveZone = GetZone();
		SetZone( SystemZone() );
		
		gestaltCodeHandle = Get1Resource( kGestaltProcResType,
					kGestaltProcResID );
		if( IsntNull( gestaltCodeHandle ) )
		{
			DetachResource( gestaltCodeHandle );
			MacLeaks_ForgetHandle( gestaltCodeHandle,
						kMacLeaks_DisposeHandleDeallocType );
			
			HLock( gestaltCodeHandle );
			
			err = CallGestaltInstallProc(
						(InstallGestaltUPP) *gestaltCodeHandle );
			if( IsntErr( err ) )
			{
				err = Gestalt( gestaltPGPDisk, &dummyResult );
			}
		}
		else
		{
			err = ForceResError( resNotFound );
		}
		
		SetZone( saveZone );
	}

	return( err );
}

	static OSStatus
DoMountFailedDialog(ConstStr255Param diskFileName)
{
	Str255		msg;
	MessageT	result;
	OSStatus	err = noErr;
	
	GetIndString( msg, kErrorStringListResID,
			kQueryAllowUnmountedVolumeStrIndex );
	PrintPString( msg, msg, diskFileName );
	 
	SysBeep( 1 );
	
	result = CWarningAlert::Display( kWACautionAlertType, kWACancelOKStyle,
					msg );
	if( result == msg_OK )
	{
		GetIndString( msg, kErrorStringListResID,
				kConfirmAllowUnmountedVolumeStrIndex );
		PrintPString( msg, msg, diskFileName );
	 
		SysBeep( 1 );
		
		result = CWarningAlert::Display( kWACautionAlertType,
						kWACancelOKStyle, msg );
	}
	
	if( result == msg_OK )
	{
		err = noErr;
	}
	else
	{
		err = userCanceledErr;
	}
	
	return( err );
}

	CComboError
MountPGPDisk(
	PGPContextRef				context,
	const FSSpec *				fileSpec,
	Boolean 					firstTime,
	Boolean						mountReadOnly,
	ConstStr255Param			passphrase,
	Boolean						allowRecoveryUI,
	ConstStr255Param			newVolumeName,
	MountDiskProgressProcPtr	progressProc,
	void						*progressUserValue)
{
	short				driveNum = 0;
	short				driverRefNum = 0;
	short				fileRefNum	= 0;
	CComboError			err;
	PGPDiskFileInfo		fileInfo;
	
	AssertSpecIsValid( fileSpec, "MountPGPDisk" );
	pgpAssertAddrValid( passphrase, uchar );
	
	(void)FlushVol( nil, fileSpec->vRefNum );
	
	// Validate mount request
	err.err = GetPGPDiskFileInfo( fileSpec, &fileInfo );
	if( err.IsntError() )
	{
		if( fileInfo.fileIsOpen )
		{
			if( fileInfo.fileIsInUseByDriver )
			{
				if( fileInfo.fileIsMounted )
				{
					err.err = kPGPDiskFileInUseByDriverError;
				}
				else
				{
					// This is the window case where the volume had been
					// unmounted, but the drive queue element has not been
					// deleted yet. If the user is attempting to open the file
					// again, then just try to mount the disk.
					
					ParamBlockRec	pb;
					
					pgpClearMemory( &pb, sizeof( pb ) );
					
					pb.volumeParam.ioVRefNum = fileInfo.fileDriveNumber;
					
					err.err = PBMountVol( &pb );
					if( err.IsError() && allowRecoveryUI )
					{
						err.err = DoMountFailedDialog( fileSpec->name );
					}
					
					goto Exit;
				}
			}
			else
			{
				err.err = kPGPDiskFileAlreadyOpenError;
			}
		}
		
		if( fileInfo.fileIsOnLockedVolume || fileInfo.fileIsLocked )
		{
			mountReadOnly = TRUE;
		}
	}

	// Validate the headers	and passphrase	
	if ( err.IsntError() )
	{
		PGPDiskFileRef	diskFileRef;
		AddDriveStruct	addInfo;
		CMemoryWiper	addInfoWiper( &addInfo, sizeof( addInfo ) );
		
		pgpClearMemory( &addInfo, sizeof( addInfo ) );
		
		err.err = OpenPGPDiskFile( context, fileSpec, &diskFileRef );
		if( err.IsntError() )
		{
			PGPDiskKeyRef	keyRef;
			
			err = FindPGPDiskKey( diskFileRef, passphrase, &keyRef );
			if( err.IsntError() )
			{
				if( ! mountReadOnly )
				{
					err.err = GetPGPDiskKeyBoolean( keyRef,
									kPGPDiskKeyProperty_MountReadOnly,
									&mountReadOnly );
				}		

				if ( err.IsntError() )
				{
					volatile HParamBlockRec	pb;
					
					pgpClearMemory( (void *) &pb, sizeof( pb ) );
					
					pb.ioParam.ioVRefNum	= fileSpec->vRefNum;
					pb.fileParam.ioDirID	= fileSpec->parID;
					pb.ioParam.ioNamePtr	= (uchar *)fileSpec->name;
					pb.ioParam.ioPermssn	= fsRdPerm;
					
					// open async so it's not tracked to this application;
					// it must remain open
					(void)PBHOpenDFAsync( (HParamBlockRec *) &pb );
					
					while ( pb.ioParam.ioResult == 1  )
					{
						// wait till done
					}

					err.err 		= pb.ioParam.ioResult;
					fileRefNum 	= pb.ioParam.ioRefNum;	
				}
					
			#if BETA
				if ( BetaExpired() )
				{
					mountReadOnly = TRUE;
				}
			#endif

				if ( err.IsntError() )
				{
					GetMediaAndPhysicalIcons( &addInfo.mediaIcon,
							&addInfo.physicalIcon );
					
					err = AddInfoForFile( diskFileRef, keyRef, fileRefNum,
								&addInfo, mountReadOnly, passphrase );
				}
				
				DisposePGPDiskKey( keyRef );
			}
			
			ClosePGPDiskFile( diskFileRef );
		}
			
		if ( err.IsntError() )
		{
			err.err = InstallPGPGestalt();
		}

		if ( err.IsntError() )
		{
			err.err = SmartInstallPGPDiskDriver( fileSpec->vRefNum,
							&driverRefNum);
		}
				
		if ( err.IsntError() )
		{
			ParamBlockRec	pb;
			Boolean			fileIsOnPGPDisk;
			
			// if this driver sits on top of another PGPDisk,
			// then reduce its memory requirements
			fileIsOnPGPDisk = VolumeIsPGPDisk( fileSpec->vRefNum );
			addInfo.minimizeWriteBufferSize	= fileIsOnPGPDisk;
			
			if ( fileIsOnPGPDisk )
			{
				addInfo.shouldBreakUpRequests	= false;
			}
			else
			{
				short			baseDriveNumber;
				short			baseDriverRefNum;
				
				GetVolumeDriveNumberAndRefNum( fileSpec->vRefNum,
						&baseDriveNumber, &baseDriverRefNum );
				addInfo.shouldBreakUpRequests =
						ShouldBreakUpRequestsToDriver( baseDriveNumber,
							baseDriverRefNum );
			}
			
			(void) PGPGetClientPrefsFileFSSpec( &addInfo.prefsSpec );
			
			err.err = PGPDiskControl( &pb, kAddDriveCC, (long)&addInfo,
							driverRefNum);
			if( err.IsntError() )
			{
				driveNum = addInfo.resultDriveNumber;
			}
		}
		
		if ( err.IsntError() )
		{
			if(firstTime)
			{
				err.err = WipeDiskUsingDrive( driveNum, driverRefNum,
								addInfo.numBlocks, progressProc,
								progressUserValue );
				if ( err.IsntError() )
				{
					err.err = DIZero( driveNum, newVolumeName );
					AssertNoErr( err.err, "MountPGPDisk: DIZero" );
				}
			}
			else
			{
				ParamBlockRec	pb;
				
				// Mount the disk.  This is where the OS takes over.
				pb.volumeParam.ioCompletion	= nil;
				pb.volumeParam.ioVRefNum	= driveNum;
				err.err	= PBMountVol( &pb );
				AssertNoErr( err.err, "error from PBMountVol" );
			
				if( err.IsError() && allowRecoveryUI )
				{
					err.err = DoMountFailedDialog( fileSpec->name );
				}
			}
				
			if ( err.IsntError() )
			{
				(void)CopyCustomVolumeIconToDrive( driveNum );
			}
		}
	
		FreeDiskExtentList( (DiskExtent *) addInfo.extents );
	}
	
Exit:
		
	// Remove all traces of an error if we encountered a problem
	if( err.IsError() )
	{
		if( driveNum > 0 )
		{
			(void)PGPDriverRemoveDrive( driverRefNum, driveNum );
		}

		if( fileRefNum > 0 )
		{
			FSClose( fileRefNum );
		}
		
		if ( firstTime )
		{
			FSpDelete( fileSpec );
		}
	}

	pgpAssertMsg( err.IsntError() || err.IsCancelError(), "MountPGPDisk" );
	
	FlushVol( nil, fileSpec->vRefNum );
	
	return err;
}


/*___________________________________________________________________________
	Remove one of the alternate passphrases.  The master passphrase cannot be
	removed.
___________________________________________________________________________*/

	OSStatus
RemovePassphrase(
	PGPContextRef		context,
	const FSSpec *		fileSpec,
	ConstStr255Param	passphrase)
{
	OSStatus		err;
	PGPDiskFileRef	diskFileRef;
	
	AssertSpecIsValid( fileSpec, "RemovePassphrase" );
	pgpAssertAddrValid( passphrase, uchar );
	
	err = OpenPGPDiskFile( context, fileSpec, &diskFileRef );
	if( IsntErr( err ) )
	{
		PGPDiskKeyRef	keyRef;
		
		err = FindPassphrasePGPDiskKey( diskFileRef, passphrase, &keyRef );
		if( IsntErr( err ) )
		{
			Boolean	isMasterKey;
			
			err = GetPGPDiskKeyBoolean( keyRef,
						kPGPDiskKeyProperty_IsMasterKey, &isMasterKey );
			if( IsntErr( err ) )
			{
				if( ! isMasterKey )
				{
					err = RemovePGPDiskKey( keyRef );
					if( IsntErr( err ) )
					{
						err = SavePGPDiskFile( diskFileRef );
					}
				}
				else
				{
					err = kPGPDiskMasterPassphraseError;
				}
			}
			
			DisposePGPDiskKey( keyRef );
		}
	
		ClosePGPDiskFile( diskFileRef );
	}
	
	return( err );
}

/*___________________________________________________________________________
	Remove all of the alternate passphrases.  The master passphrase cannot be
	removed.
___________________________________________________________________________*/
	OSStatus
RemoveAlternatePassphrases(
	PGPContextRef	context,
	const FSSpec 	*fileSpec)
{
	OSStatus		err;
	PGPDiskFileRef	diskFileRef;
	
	err = OpenPGPDiskFile( context, fileSpec, &diskFileRef );
	if( IsntErr( err ) )
	{
		while( IsntErr( err ) )
		{
			PGPDiskKeyRef	keyRef;
			
			// Always fetch index 1 (first non-master passphrase) as we're
			// deleting entries.
			
			err = GetIndPassphrasePGPDiskKey( diskFileRef, 1, &keyRef );
			if( IsntErr( err ) )
			{
				err = RemovePGPDiskKey( keyRef );
				
				DisposePGPDiskKey( keyRef );
			}
		}
		
		if( err == kEndOfKeyIterationError )
		{
			err = SavePGPDiskFile( diskFileRef );
		}
		
		ClosePGPDiskFile( diskFileRef );
	}
	
	return( err );
}

	OSStatus
AddPassphrase(
	PGPContextRef		context,
	const FSSpec *		fileSpec,
	ConstStr255Param	masterPassphrase,
	ConstStr255Param	newPassphrase,
	Boolean				readOnly )
{
	OSStatus		err;
	PGPDiskFileRef	diskFileRef;
	
	err = OpenPGPDiskFile( context, fileSpec, &diskFileRef );
	if( IsntErr( err ) )
	{
		err = AddPassphrasePGPDiskKey(diskFileRef, masterPassphrase,
					newPassphrase, readOnly );
		if( IsntErr( err ) )
		{
			err = SavePGPDiskFile( diskFileRef );
		}
		
		ClosePGPDiskFile( diskFileRef );
	}
	
	return( err );
}


	static OSErr
FSWipeBytesUsingDriver(
	short	fileRef,
	UInt32	startOffset,
	UInt32	numBytes,
	uchar	wipeChar )
{
	OSStatus	err	= noErr;
	
	err	= SetFPos( fileRef, fsFromStart, startOffset );
	if ( IsntErr( err ) )
	{
		void *		buffer	= nil;
		UInt32		bufferSize;
		uchar		emergencyBuffer[ 512 ];
		Boolean		disposeBuffer	= false;
		UInt32		remaining;
		UInt32		writeOffset;
		
		buffer	= pgpNewPtrMost( numBytes, MIN( 512, numBytes ),
					kMacMemory_PreferTempMem | kMacMemory_UseApplicationHeap,
					&bufferSize );
		if ( IsntNull( buffer ) )
		{
			disposeBuffer	= TRUE;
		}
		else
		{
			buffer		= &emergencyBuffer[ 0 ];
			bufferSize	= sizeof( emergencyBuffer );
		}
		
		
		pgpFillMemory( buffer, bufferSize, wipeChar );
		
		writeOffset	= startOffset;
		remaining	= numBytes;
		while( remaining != 0 )
		{
			long	count;
			
			count	= MIN( remaining, bufferSize );
			err	= FSWriteFileUsingDriver( fileRef, writeOffset, count, buffer,
						nil );
			if ( IsErr( err ) )
				break;
				
			remaining	-= count;
			writeOffset	+= count;
		}
		
		if ( disposeBuffer )
		{
			pgpFreeMac( buffer );
		}
	}
	
	return( err );
}


	OSStatus
VerifyPGPDiskFileIsModifiable(const FSSpec *fileSpec)
{
	PGPDiskFileInfo		fileInfo;
	OSStatus			err;
	
	err = GetPGPDiskFileInfo( fileSpec, &fileInfo );
	if( IsntErr( err ) )
	{
		if( fileInfo.fileIsOpen )
		{
			if( fileInfo.fileIsInUseByDriver )
			{
				 err = kPGPDiskFileInUseByDriverError;
			}
			else
			{
				err = kPGPDiskFileAlreadyOpenError;
			}
		}
		else if( fileInfo.fileIsLocked )
		{
			err = kPGPDiskFileLockedError;
		}
		else if( fileInfo.fileIsOnLockedVolume )
		{
			err = kPGPDiskFileOnLockedVolumeError;
		}
	}
	
	return( err );
}

//	Gets the error string for the error err assuming a phrasing such as
// 	Couldn't do X because Y, where Y is the error string.

	void
GetErrorString(OSStatus err, StringPtr errorStr)
{
	pgpAssertAddrValid( errorStr, uchar );
	pgpAssert( err != noErr && err != userCanceledErr );
	
	errorStr[0] = 0;
	
	if( err > 0 )
	{
		short	strIndex = 0;
		
		// Application specific errors
	
		switch( err )
		{
			case kPGPDiskInvalidFileHeaderError:
			case kPGPDiskUnknownFileKindError:
			case kPGPDiskUnknownAlgorithmError:
				strIndex = kNotAValidPGPDiskFileStrIndex;
				break;

			case kPGPDiskNewerFileVersionError:
				strIndex = kPGPDiskFileCreatedByNewerVersionStrIndex;
				break;

			case kPGPDiskRemoteVolumeTypeError:
			case kPGPDiskForeignVolumeTypeError:
				strIndex = kVolumeRemoteOrForeignStrIndex;
				break;
			
			case kPGPDiskIncorrectPassphraseError:
				strIndex = kIncorrectPassphraseStrIndex;
				break;

			case kPGPDiskMasterPassphraseError:
				strIndex = kMasterPassphraseStrIndex;
				break;
			
			case kPGPDiskFileInUseByDriverError:
				strIndex = kPGPDiskFileMountedStrIndex;
				break;
				
			case kPGPDiskFileAlreadyOpenError:
				strIndex = kPGPDiskFileBusyStrIndex;
				break;
			
			case kPGPDiskFileLockedError:
				strIndex = kPGPDiskFileLockedStrIndex;
				break;
				
			case kPGPDiskFileOnLockedVolumeError:
				strIndex = kPGPDiskFileIsOnLockedVolumeStrIndex;
				break;
			
			case kCastSBoxesInvalidError:
				strIndex = kCastSBoxesDamagedStrIndex;
				break;
				
			case kPGPDiskUnknownFileVersionError:	
				strIndex = kVersionObsoleteStrIndex;
				break;
			
			case kNoMorePassphrasesAvailableError:
				strIndex = kPGPDiskMaxPassphrasesAddedStrIndex;
				break;
			
			case kPGPdiskADKNotFoundError:
				strIndex = kPGPdiskADKNotFoundStrIndex;
				break;
		
			case kPGPDiskIncorrectPassphraseNoKeyringsError:
				strIndex = kIncorrectPassphraseNoKeyringStrIndex;
				break;
				
			case kPGPDiskIncorrectPassphraseKeyNotFoundError:
				strIndex = kIncorrectPassphraseKeyNotFoundStrIndex;
				break;
		
			default:
				pgpDebugMsg( "GetErrorString: Unhandled application error" );
				break;
		}
		
		if( strIndex != 0 )
		{
			GetIndString( errorStr, kErrorStringListResID, strIndex );
			pgpAssert( errorStr[0] != 0 );
		}
	}

	if( errorStr[0] == 0 )
	{
		Str255	osErrStr;
		
		GetIndString( errorStr, kErrorStringListResID,
					kAnErrorOccurredStrIndex );
		GetOSErrorString( err, osErrStr );
		PrintPString( errorStr, errorStr, osErrStr );
	}
}

	void
GetComboErrorString(
	CComboError	err,
	StringPtr 	errorStr)
{
	pgpAssertAddrValid( errorStr, uchar );
	pgpAssert( err.IsError() && ! err.IsCancelError() );
	
	if( err.HavePGPError() )
	{
		char	cErrorStr[256];
		Str255	pErrorStr;
		
		GetIndString( errorStr, kErrorStringListResID,
					kAnErrorOccurredStrIndex );
		PGPGetClientErrorString( err.pgpErr, sizeof( cErrorStr ), cErrorStr );
		CToPString( cErrorStr, pErrorStr );
		PrintPString( errorStr, errorStr, pErrorStr );
	}
	else
	{
		GetErrorString( err.err, errorStr );
	}
}

#if PGP_BUSINESS_SECURITY	/* [ */

	PGPError
GetADKInfo(
	PGPContextRef			context,
	PGPBoolean				*haveADK,
	PGPKeyID				*keyID,
	PGPPublicKeyAlgorithm	*algorithm)
{
	PGPError		err = kPGPError_NoErr;
	PGPPrefRef		adminPrefsRef;
	
	*haveADK 	= FALSE;
	*algorithm	= kPGPPublicKeyAlgorithm_Invalid;
	
	pgpClearMemory( keyID, sizeof( *keyID ) );
	
	err = PGPOpenAdminPrefs( PGPGetContextMemoryMgr( context ),
						&adminPrefsRef );
	if( IsntPGPError( err ) )
	{
		err = PGPGetPrefBoolean( adminPrefsRef,
							kPGPPrefUsePGPdiskADK, haveADK );
		if( IsntPGPError( err ) && *haveADK )
		{
			PGPUInt32	tempAlg;
			
			err = PGPGetPrefNumber( adminPrefsRef,
								kPGPPrefPGPdiskADKPublicKeyAlgorithm,
								&tempAlg );
			if( IsntPGPError( err ) )
			{
				PGPSize	dataLength;
				void	*data;
				
				*algorithm = (PGPPublicKeyAlgorithm) tempAlg;
				
				err = PGPGetPrefData( adminPrefsRef,
									kPGPPrefPGPdiskADKKeyID,
									&dataLength, &data );
				if( IsntPGPError( err ) )
				{
					err = PGPImportKeyID( data, keyID );

					PGPFreeData( data );
				}
			}				
		}
		
		PGPClosePrefFile( adminPrefsRef );
	}

	return( err );
}

#endif	/* ] PGP_BUSINESS_SECURITY */













