(in-package "VIDEO")

(defclass video-disk-player ()
  ((tty :initform "/dev/ttyb" :type string :accessor tty)
   (disk :initform nil :accessor disk)
   (stream :initform nil)))

(defun make-video-disk-player (&rest args)
  (apply #'make-instance 'video-disk-player :allow-other-keys t args))

(setf *current-player* (make-video-disk-player))

(defun check-reply (player)
  t)

(defun timecode-to-frame (tc)
  (let (h m s f)
    (setq f (mod tc 100)
	  tc (truncate tc 100)
	  s (mod tc 100)
	  tc (truncate tc 100)
	  m (mod tc 100)
	  tc (truncate tc 100)
	  h (mod tc 100))
    (+ f (* s 30) (* m 30 60) (* h 30 60 60))))

(defun frame-to-timecode-str (f)
  (let (hour min sec frame)
    (setf frame (mod f 30)
	  f (truncate f 30)
	  sec (mod f 60)
	  f (truncate f 60)
	  min (mod f 60)
	  f (truncate f 60)
	  hour f)
    (format nil "~d~2,'0d~2,'0d~2,'0d" hour min sec frame)))

(defun check-player (player)
  (if (not (typep player 'video-disk-player))
      (error "Attempt to use invalid player ~s~%" player))
  (if (and *real-player-p* (null (slot-value player 'stream)))
      (error "Attempt to used unconnected player ~s~%" player)))

(defconstant *stx* (character 2))
(defconstant *etx* (character 3))

(defun output-to-player (player str &rest args)
  (check-player player)
  (let ((stream (slot-value player 'stream)))
    (setq *last-arg* (apply #'format nil str args))
    (if *real-player-p*
	(format (slot-value player 'stream) "~a~a~a" *stx* *last-arg* *etx*)
      (format t "~a" *last-arg*))
    (if *real-player-p*
	(force-output stream)
      (format t "~%"))))

(defun input-from-player (player)
  (check-player player)
  (if *real-player-p*
      (let ((str (make-array 256 :fill-pointer t))
	    (stream (slot-value player 'stream))
	    (ch nil)
	    (start nil))
	(setf (fill-pointer str) 0)
	(loop
	 (setq ch (read-char stream))
	 (cond ((eq ch *stx*) (setq start t))
	       ((eq ch *etx*) (return))
	       (start (vector-push ch str))
	       (t (warn "Character read from player without stx~%"))))
	(coerce str 'string))
    "R"))

(defun connect-player (&optional (player *current-player*))
  (when (slot-value player 'stream)
	(warn "connect-player: Player ~s already connected~%" player)
	(return-from connect-player (slot-value player 'stream)))
  (if *real-player-p*
      (setf (slot-value player 'stream)
	    (open (tty player) :direction :io :if-exists :append)))
  (if (null (slot-value player 'stream))
      (progn
	(setf *real-player-p* nil)
	(warn "connect-player: Cannot connect on tty ~s~%" (tty player)))
    ;; set up the (open) tty...
    (let* ((tty (tty player))
	   (str "stty 9600 raw -parity -cstopb -echo clocal")
	   (cmd (format nil "~a > ~a" str tty)))
      (#+allegro excl:run-shell-command
		 #+lucid run-program cmd :wait t)))
  (slot-value player 'stream))

(defun disconnect-player (&optional (player *current-player*))
  (when (slot-value player 'stream)
	(stop-disc)
	(close (slot-value player 'stream))
	(setf (slot-value player 'stream) nil)))

(defun mode (&optional (player *current-player*))
  :pause)

(defun door-open (&optional (player *current-player*))
  "Eject the video tape"
  (output-to-player player "A@A")
  )

(defun stop-disc (&optional (player *current-player*))
  "Stop the video disc"
  t)

(defun start-disc (&optional (player *current-player*))
  "Calibrate the zero point on the tape"
  (output-to-player player "A@V"))

(defun search-to-frame (frame &optional (player *current-player*))
  "Goto frame passed"
  (case (mode player)
	((:eject :no-tape)
	 (cerror "Insert a tape and retry" "No tape in player")
	 (search-to-frame  frame player))
	((:still :pause :play :multi-speed)
	 (output-to-player player "A@T~a" (frame-to-timecode-str frame))
	 )
	(:park
	 (search-to-frame  frame player))))

(defun play (&key (player *current-player*)
	     (end nil) (speed 60) &aux (forward t))
  "Play to the specified address"
  (setq speed (max -255 (min speed 255)))
  (if (< speed 0)
      (setq speed (- speed)
	    forward nil))
  (case (mode player)
	((:eject :no-tape)
	 (cerror "Insert a tape and retry" "No tape in player")
	 (play :player player :end end :speed speed))
	((:still :pause)
	 (if (and (= speed 60) forward)
	     (if (numberp end)
		 (output-to-player player "A@U~a" (frame-to-timecode-str end))
		 (output-to-player player "A@J"))
	     (error "Command not supported")
	     )
	 )
	((:play :multi-speed)
	 (still player)
	 (play :player player :end end :speed speed))
	))

(defun frame-number (&optional (player *current-player*))
;  :unknown
  (output-to-player player "ZI")
  (read-from-string (input-from-player player)))

(defun still (&optional (player *current-player*))
  "Pause at picture displayed"
  (output-to-player player "A@F")
  )

(defun pause (&optional (player *current-player*))
  "Pause the disc"
  (output-to-player player "A@@")
  )

(defun rewind (&optional (player *current-player*))
  "Rewind the disc"
  (output-to-player player "A@B")
  )

(defun fast-forward (&optional (player *current-player*))
  "Forward the disc"
  (output-to-player player "A@C")
  )

(defun eject (&optional (player *current-player*))
  "Eject the disc"
  (output-to-player player "A@A")
  )

(defun step-forward (&optional (player *current-player*))
  "Step forward one frame"
  (output-to-player player "A@L")
  )

(defun step-backward (&optional (player *current-player*))
  "Step forward one frame"
  (output-to-player player "A@M")
  )

(defun play-range (start end video ch1 ch2 &optional (player *current-player*))
  "Play the specified address range"
  (case (mode player)
	((:eject :no-tape)
	 (cerror "Insert a tape and retry" "No tape in player")
	 (play-range start end video ch1 ch2 player))
	(t
	 (output-to-player player "A@Q~a ~a" (frame-to-timecode-str start)
			   (frame-to-timecode-str end)))))

