(in-package "VIDEO")

(defclass video-disk ()
  ((disk-title :initform "" :type string :accessor disk-title)
   (num-chapters :initform 1 :type integer :accessor num-chapters)
   (num-frames :initform most-positive-fixnum :accessor num-frames)))

(defclass indexed-video-disk (video-disk)
  ((disk-index :initform (make-hash-table :test #'equal) :accessor disk-index)))

(defmethod load-disk ((disk video-disk) &optional (player *current-player*))
  (setf (disk player) disk))

(defun make-indexed-video-disk (&rest keys)
  (apply #'make-instance 'indexed-video-disk :allow-other-keys t keys))

(defmethod initialize-instance :after ((disk video-disk)
				       &key
				       (file nil)
				       &allow-other-keys)
  (if file
      (load-index disk file)))

(defun load-index (disk path &aux (index (disk-index disk)))
  (with-open-file (in path :direction :input)
		  (do ((kv-pair (read in nil nil #+allegro t #+lucid nil)
				(read in nil nil #+allegro t #+lucid nil)))
		      ((null kv-pair) t)
		      (setf (gethash (car kv-pair) index) (cadr kv-pair)))))

(defun save-index (disk path)
  (with-open-file (out path :direction :output :if-does-not-exist :create
		       :if-exists :overwrite)
		  (maphash #'(lambda (k v) (format out "(~s ~s)~%" k v))
			   (disk-index disk))))

(defun clear-index (disk)
  (clrhash (disk-index disk)))

(defun enter-index (disk key val)
  (setf (gethash key (disk-index disk)) val))

(defun lookup-index (disk key)
  (gethash key (disk-index disk)))

(defun delete-index (disk key)
  (remhash key (disk-index disk)))

(defun play-index (disk key)
  (let ((params (lookup-index disk key)))
       (if params (apply #'play-range params))))
