;;;
;;; Copyright (c) 1990 Regents of the University of California
;;; 
;;; $Author: bsmith $
;;; $Source: /pic2/picasso/new/widgets/text/RCS/text-widget.cl,v $
;;; $Revision: 1.1 $
;;; $Date: 1991/08/04 19:09:31 $
;;;

(in-package "PT")

;;;
;;; text-widget class - a simple text editor.
;;;

;; Description of slots.
;;
;; load-db, save-db, append-db, insert-db, search-db, replace-db -
;;   these slots hold the dialog boxes for the load, save, append to file,
;;   insert into text-widget, search, and search and replace operations.
;; editable - can the text buffer of the text-widget be modified?
;; scroll-right-at - the number of columns to the right of left-of-screen
;;   at which the text-widget will automatically scroll
;;   right by the number of columns specified in the
;;   horizontal-scroll-step slot.
;; horizontal-scroll-step - the number of columns to scroll at a time in the
;;   horizontal (left and right) direction.
;; vertical-scroll-step - the number of rows to scroll at a time in the 
;;   vertical (up and down) direction.
;; tab-step - how far apart from each other the tab stops are.
;; insert-mode - If t, characters to the right of the cursor will be pushed
;;   over and preserved.  If nil, they will be overwritten when
;;   there is new keyboard input.
;;

(defclass text-widget (widget text-buffer-gadget)
  ((name
    :initform "A Text-Widget")
   (mf-selectable-widget :initarg :mf-selectable-widget :initform t)
   (event-mask
    :initform '(:key-press :button-press :expose-region :button-1-motion 
			   :double-click))
   (load-db :initform nil :type widget :accessor load-db)
   (save-db :initform nil :type widget :accessor save-db)
   (append-db :initform nil :type widget :accessor append-db)
   (insert-db :initform nil :type widget :accessor insert-db)
   (search-db :initform nil :type widget :accessor search-db)
   (replace-db :initform nil :type widget :accessor replace-db)
   (confirm-db :initform nil :type widget :accessor confirm-db)
   (inform-db :initform nil :type widget :accessor inform-db)
   (editable :initarg :editable :initform t  :type t  :accessor editable)
   (changed :initarg :changed :initform nil  :type t  :accessor changed)
   (scroll-right-at 
    :initarg :scroll-right-at 
    :initform nil  
    :type integer  
    :accessor scroll-right-at)
   (horizontal-scroll-step
    :initarg :horizontal-scroll-step 
    :initform nil  
    :type integer  
    :accessor horizontal-scroll-step)
   (vertical-scroll-step
    :initarg :vertical-scroll-step 
    :initform nil  
    :type integer  
    :accessor vertical-scroll-step)
   (tab-step
    :initarg :tab-step 
    :initform 8 
    :type integer 
    :accessor tab-step)
   (insert-mode
    :initarg :insert-mode 
    :initform nil  
    :type t  
    :accessor insert-mode)))

;;;
;;; class event mapping for text-widget
;;;

;; These are the key bindings:
;; up, down, left, right arrow keys - move in the direction of the arrow by
;;   one column or one row.
;; meta-arrow keys - scroll in the direction of the arrow by either
;;   vertical-scroll-step or horizontal-scroll-step.
;; shift-right and shift-left arrows - move by words.
;; control-meta-up-arrow - go to beginning of buffer.
;; control-meta-down-arrow - go to end of buffer.
;; control-left-arrow - go to beginning of line.
;; control-right-arrow - go to end of line.
;; control-l - recenter screen around cursor position, leaving cursor in
;;   the middle row of the screen.
;; alt-l - load file
;; alt-s - save file
;; alt-a - append to file
;; alt-i - insert into widget
;; control-s - search
;; control-r - search and replace
;; alt-c - copy marked text into cut-buffer
;; alt-x - cut marked text into cut-buffer
;; alt-p - paste cut-buffer into text.
;; control-d - delete character under cursor.
;; control-k - delete to end of line
;; backspace, delete - delete backward.
;; control-i - toggle replace mode.
;; left button click at row, column - place cursor at row, column.
;; click-drag-release left mouse button - mark text.
;; right button click - place mark at row, column
;; tab - go to next tab stop
;; shift-tab - go back to previous tab stop.

(defun make-text-widget (&rest keys)
  (apply #'make-instance 'text-widget :allow-other-keys t keys))

(defmethod new-instance ((self text-widget)
			 &key
			 (key-dummy nil)
			 &allow-other-keys)
  (declare (ignore key-dummy)) 
  (setf (cursor self) 
	(make-cursor :name "text" :file "text.cursor" 
		     :mask-file "text_mask.cursor"))
  (call-next-method))

;;;
;;; text-widget event operations (callback-functions)
;;;

;; save, plus auxiliary functions
(defhandler save ((text-widget text-widget) &rest args
		  &default (:key-press :meta #\s))
  (declare (ignore args))
  (when (or (not (editable text-widget))
	    (not (changed text-widget)))
	(call-dialog (find-po-named (list* "picasso" "notifier" "dialog"))
		     :msg '("No changes need" "to be saved."))
	(return-from text-widget-save nil))
  (let ((filename (call
		   (find-po-named (list* "picasso" "save-file" "dialog"))
		   )))
    (when (and filename
	       (> (length filename) 0)
	       (or (not (probe-file filename))
		   (call-dialog
		    (find-po-named (list* "picasso" "confirmer" "dialog"))
		    :msg  (list (concatenate 'string
					     filename
					     " exists.")
				"Overwrite?"))))
	  (save-file text-widget filename)
	  (setf (changed text-widget) nil))))

;; append, plus auxiliary functions
;; NOT IMPLEMENTED YET!
(defhandler append ((text-widget text-widget) &rest args
		    &default (:key-press :meta #\a))
  (declare (ignore args text-widget)))

;; insert, plus auxiliary functions
;; NOT IMPLEMENTED YET!
(defhandler insert ((text-widget text-widget) &rest args
		    &default (:key-press :meta #\i))
  (declare (ignore args text-widget)))

;; load, plus auxiliary functions
(defhandler load ((text-widget text-widget) &rest args
		  &default (:key-press :meta #\l))
  (declare (ignore args))
  (when (or (not (changed text-widget))
	    (call (find-po-named (list* "picasso" "confirmer" "dialog"))
		  :msg
		  '("Buffer was modified"
		    "but not saved."
		    "Continue?")))
	(let ((filename (call
			 (find-po-named
			  (list* "picasso" "open-file" "dialog")))))
	  (when filename
		(if (probe-file filename)
		    (load-file text-widget filename)
		  (call-dialog
		   (find-po-named (list* "picasso" "notifier" "dialog"))
		   :msg '(filename "does not exist.")))))))

;; search, plus auxiliary functions
(defhandler search ((text-widget text-widget) &rest args
		    &default (:key-press :control #\s))
  (declare (ignore args))
  (call
   (find-po-named (list* "picasso" "search" "dialog"))
   :tw text-widget))

;; replace, plus aux. funcs.
(defhandler replace ((text-widget text-widget) &rest args
		     &default (:key-press :control #\r))
  (declare (ignore args))
  (when (not (editable text-widget))
	(call
	 (find-po-named '("picasso" "notifier" . "dialog"))
	 :msg '("This text-widget" "is read-only."))
	(return-from text-widget-replace nil))
  (call
   (find-po-named (list* "picasso" "replace" "dialog"))
   :tw text-widget))

;;
;; character handler
;;
;; (arguably the most important part of text-widgets.)
;; Probably a place to explore for speed-ups.

(defhandler char-handler ((text-widget text-widget) &key display code state 
			  &allow-other-keys
			  &default :key-press)
  "Handle key presses.  The number of rows in the
  text-widget can never change as a result of a call to this method, since
  newline characters (the return key) are mapped to another function."
  (when (not (editable text-widget))
	(return-from text-widget-char-handler nil))
  (let* ((column (column text-widget))
	 (columns (columns text-widget))
	 (overwrite (not (insert-mode text-widget)))
	 (left-of-screen (left-of-screen text-widget))
	 (scroll-right-at (scroll-right-at text-widget))
	 (sra (if scroll-right-at scroll-right-at (1- columns)))
	 (mark-row (mark-row text-widget))
	 (mark-column (mark-column text-widget))
	 (str (xlib:keycode->character display code state)))
    (when (or (keywordp str) (> state 1))
	  (return-from text-widget-char-handler))
    (setq str (string str))
    (when (and (and mark-row mark-column)
	       (> (length str) 0))
	  (delete-mark text-widget))
    
    (when (> (length str) 0)
	  (when (>= column (+ left-of-screen sra))
		(let ((hss (horizontal-scroll-step text-widget)))
		  (tbg-scroll-right text-widget 
				    (if hss hss 
				      (round (* 2 (/ columns 3)))))))
	  (put text-widget :string str :overwrite overwrite)
	  (setf (changed text-widget) t))))

;; more callback functions

(defhandler copy-mark ((text-widget text-widget) &rest args
		       &default (:key-press :meta #\c))
  (declare (ignore args))
  (when (and (mark-row text-widget)
	     (mark-column text-widget))
	(copy-mark text-widget)
	(unmark text-widget)))

(defhandler cut-mark ((text-widget text-widget) &rest args
		      &default (:key-press :meta #\x))
  (declare (ignore args))
  (when (and (mark-row text-widget)
	     (mark-column text-widget))
	(copy-mark text-widget)
	(delete-mark text-widget)))

(defhandler paste-cut-buffer ((text-widget text-widget) &rest args
			      &default ((:key-press :meta #\p)
					(:button-press :detail :middle-button)))
  (declare (ignore args))
  (let ((d (res (display text-widget)))
	(overwrite (not (insert-mode text-widget)))
	(row (row text-widget)))
    (put text-widget :string (copy-seq (xlib:cut-buffer d)) :overwrite overwrite)
    (tbg-repaint text-widget :start-row row)))

(defhandler toggle-replace-mode ((text-widget text-widget) &rest args
				 &default (:key-press :control "i"))
  (declare (ignore args))
  (cond ((insert-mode text-widget)
	 (setf (insert-mode text-widget) nil)
	 (setf (cursor-mode text-widget) :overwrite))
	(t
	 (setf (insert-mode text-widget) t)
	 (setf (cursor-mode text-widget) :insert)))
  (tbg-repaint text-widget 
	       :start-row (row text-widget) 
	       :end-row (row text-widget)))

(defhandler backspace ((text-widget text-widget) &rest args
		       &default ((:key-press :detail #\Delete)
				 (:key-press :detail #\backspace)))
  (declare (ignore args))
  (when (not (editable text-widget))
	(return-from text-widget-backspace nil))
  
  (let* ((row (row text-widget))
	 (old-row row)
	 (column (column text-widget))
	 (mark-row (mark-row text-widget))
	 (mark-column (mark-column text-widget)))
    (setf (changed text-widget) t)
    (cond ((and (and mark-row mark-column))
	   (text-widget-cut-mark text-widget nil))
	  (t
	   (multiple-value-setq
	    (row column)
	    (delete-backward (buffer text-widget) row column))
	   (setf (row text-widget) row)
	   (when (= column  (left-of-screen text-widget))
		 (let ((hss (horizontal-scroll-step text-widget)))
		      (tbg-scroll-left 
		       text-widget 
		       (if hss hss (round (* 2 (/ (columns text-widget) 3)))))))
	   (setf (column text-widget) column)
	   (if (/= row old-row)
	       (tbg-repaint text-widget :start-row row)
	     (tbg-repaint text-widget :start-row old-row :end-row old-row))))))

(defhandler delete-char ((text-widget text-widget) &rest args
			 &default (:key-press :control #\d))
  (declare (ignore args))
  (when (not (editable text-widget))
	(return-from text-widget-delete-char nil))
  (let* ((row (row text-widget))
	 (columns (buf-columns (buffer text-widget) row))
	 (column (column text-widget)))
    ;; remember that (delete-char buffer) never changes row and column.
    (setf (changed text-widget) t)
    (cond ((= column columns)
	   (delete-char (buffer text-widget) row column)
	   (tbg-repaint text-widget :start-row row))
	  (t
	   (delete-char (buffer text-widget) row column)
	   (tbg-repaint text-widget :start-row row :end-row row)))))

(defhandler forward-char ((text-widget text-widget) &rest args
			  &default (:key-press :control "f"))
  "Move the point one character forward, wrapping to the
next line if necessary.  Do nothing if at the end of the buffer."
  (declare (ignore args))
  (let* ((row (row text-widget))
	 (column (column text-widget))
	 (buffer (buffer text-widget)))
    (cond ((= column (buf-columns buffer row))
	   ;; end of a line.  
	   (when (not (= row (1- (rows buffer))))
		 (setf (row text-widget) (1+ row))
		 (setf (column text-widget) 0)))
	  (t  ; the "normal" case
	   (setf (column text-widget) (1+ column))))))

(defhandler backward-char ((text-widget text-widget) &rest args
			   &default (:key-press :control "b"))
  "Move the point back one character, wrapping to the
previous line, if necessary.  Do nothing if at the beginning
of the buffer."
  (declare (ignore args))
  (let* ((row (row text-widget))
	 (column (column text-widget))
	 (buffer (buffer text-widget)))
    (cond ((= column 0)
	     ;; beginning of a line
	   (when (not (= row 0))
		 (setf (row text-widget) (1- row))
		 (setf (column text-widget) (buf-columns buffer (1- row)))))
	  (t  ; the "normal" case
	   (setf (column text-widget) (1- column))))))

(defhandler beginning-of-line ((text-widget text-widget) &rest args
			       &default (:key-press :control "a"))
  (declare (ignore args))
  (setf (column text-widget) 0))

(defhandler end-of-line ((text-widget text-widget) &rest args
			 &default (:key-press :control "e"))
  (declare (ignore args))
  (setf (column text-widget) (buf-columns (buffer text-widget) (row text-widget))))

(defhandler delete-line ((text-widget text-widget) &rest args
			 &default (:key-press :control "u"))
  (declare (ignore args))
  (setf (column text-widget) 0)
  (text-widget-kill-line text-widget))

(defhandler kill-line ((text-widget text-widget) &rest args
		       &default (:key-press :control "k"))
  (declare (ignore args))
  (when (not (editable text-widget))
	(return-from text-widget-kill-line nil))
  (let ((row (row text-widget))
	(d (res (display text-widget)))
	(column (column text-widget))
	(buffer (buffer text-widget)))
    (when (cursor-moved text-widget)
	  (setf (xlib:cut-buffer d) "")
	  (setf (cursor-moved text-widget) nil))
    (add-bytes (res (display text-widget))
	       (if (< column (buf-columns buffer row))
		   (value buffer :row row :column column)
		   (string #\newline)))
    (multiple-value-setq (row column)
			 (kill-line buffer row column))
    (setf (changed text-widget) t)
    ;; kill-line will never change the row and column, so don't call these
    ;; setf methods.  It's necessary not to call them because they would
    ;; set cursor-moved to t inappropriately.
    ;;    (setf (row text-widget) row)
    ;;    (setf (column text-widget) column)
    (tbg-repaint text-widget :start-row row)))

(defhandler recenter ((text-widget text-widget) &rest args
		      &default (:key-press :control "l"))
  (declare (ignore args))
  (let ((buffer-rows (rows (buffer text-widget)))
	(widget-rows (rows text-widget)))
    (setf (top-of-screen text-widget)
	  (max 0
	       (min (- buffer-rows widget-rows)
		    (- (row text-widget)
		       (truncate (rows text-widget) 2)))))
    (tbg-repaint text-widget)))

(defhandler newline ((text-widget text-widget) &rest args
		     &default (:key-press :detail #\Return))
  (declare (ignore args))
  (when (not (editable text-widget))
	(return-from text-widget-newline nil))
  (let ((overwrite (not (insert-mode text-widget))))
    (put text-widget :string (string '#\Newline)
       :overwrite overwrite)
    (setf (changed text-widget) t)
    (when (not overwrite)
	  (tbg-repaint text-widget :start-row (1- (row text-widget))))))

(defhandler next-line ((text-widget text-widget) &rest args
		       &default (:key-press :control "n"))
  (declare (ignore args))
  (let ((row (row text-widget))
	(column (column text-widget))
	(new-row 0))
    (setq new-row (min (1+ row) (1- (rows (buffer text-widget)))))
    (setf (row text-widget) new-row)
    (setf (column text-widget)
	  (min column (buf-columns (buffer text-widget) new-row)))))

(defhandler previous-line ((text-widget text-widget) &rest args
			   &default (:key-press :control "p"))
  (declare (ignore args))
  (let ((row (row text-widget))
	(column (column text-widget))
	(new-row 0))
    (declare (type integer row column new-row))
    (setq new-row (max (1- row) 0))
    (setf (row text-widget) new-row)
    (setf (column text-widget)
	  (min column 
	       (buf-columns (buffer text-widget) new-row)))))

(defhandler next-word ((text-widget text-widget) &rest args
		       &default (:key-press :meta "f"))
  "move cursor to next word on the line.  If end of line, go to beginning of next line"
  (declare (ignore args))
  (let* ((column (column text-widget))
	 (row (row text-widget))
	 (buffer (buffer text-widget))
	 (curr-line (value buffer :row row))
	 (length (length curr-line))
	 (end-of-word (next-non-alphanumeric-pos curr-line column)))
    ;; the next word starts at the first alphanumeric character followed
    ;; by a non-alphanumeric character.
    (cond ((= column length) ;end of line case
	   (text-widget-forward-char text-widget nil))
	  (t ;"normal" case
	   (setf (column text-widget)
		 (next-alphanumeric-pos curr-line end-of-word))))))

(defhandler previous-word ((text-widget text-widget) &rest args
			   &default (:key-press :meta "b"))
  "move cursor to previous word on the line.  If beginning of line go
to end of previous line."
  (declare (ignore args))
  (let* ((column (column text-widget))
	 (row (row text-widget))
	 (buffer (buffer text-widget))
	 (curr-line (value buffer :row row))
	 (beginning-of-word
	  (previous-non-alphanumeric-pos
	   curr-line 
	   (previous-alphanumeric-pos
	    curr-line
	    (previous-non-alphanumeric-pos curr-line column)))))
    ;; the previous word starts at the first alphanumeric character after
    ;;  a non-alphanumeric character (going from right to left).
    (cond ((= column 0) ;beginning of line case
	   (text-widget-backward-char text-widget nil))
	  (t ;"normal" case
	   (setf (column text-widget) beginning-of-word)))))
	     
(defhandler scroll-down ((text-widget text-widget) &rest args
			 &default (:key-press :control "v"))
  "Scroll the text down one full screenful."
  (declare (ignore args))
  (let* ((vss (vertical-scroll-step text-widget))
	 (n (if vss vss (rows text-widget))))
    (tbg-scroll-down text-widget n)))

(defhandler scroll-up ((text-widget text-widget) &rest args
		       &default (:key-press :meta "v"))
  "Scroll the text up one full page."
  (declare (ignore args))
  (let* ((vss (vertical-scroll-step text-widget))
	 (n (if vss vss (rows text-widget))))
    (tbg-scroll-up text-widget n)))

(defhandler scroll-left ((text-widget text-widget) &rest args
			 &default ((:key-press :control "<")
				   (:key-press (:control :shift) "<")))
  "Scroll the text horizontally to the left.  Will not scroll farther
left than column 0"
  (declare (ignore args))
  (let* ((hss (horizontal-scroll-step text-widget))
	 (n (if hss hss (columns text-widget))))
    (tbg-scroll-left text-widget n)))

(defhandler scroll-right ((text-widget text-widget) &rest args
			  &default ((:key-press :control ">")
				    (:key-press (:control :shift) ">")))
  "Scroll the text horizontally to the right one screenful (minus 2 cols)."
  (declare (ignore args))
  (let* ((hss (horizontal-scroll-step text-widget))
	 (n (if hss hss (columns text-widget))))
    (tbg-scroll-right text-widget n)))

(defhandler beginning-of-buffer ((text-widget text-widget) &rest args
				 &default ((:key-press :meta "<")
					   (:key-press (:shift :meta) "<")))
  "Move the point to the beginning of the buffer."
  (declare (ignore args))
  (setf (row text-widget) 0)
  (setf (column text-widget) 0))

(defhandler end-of-buffer ((text-widget text-widget) &rest args
			   &default ((:key-press :meta ">")
				     (:key-press (:shift :meta) ">")))
  "Move the point to the end of the buffer."
  (declare (ignore args))
  (setf (row text-widget) (rows (buffer text-widget)))
  (setf (column text-widget) (buf-columns (buffer text-widget) (row text-widget))))

(defhandler tab ((text-widget text-widget) &rest args
		 &default (:key-press :detail #\Tab))
  (declare (ignore args))
  (when (and (not (editable text-widget))
	     (insert-mode text-widget))
	(return-from text-widget-tab nil))
  (let* ((column (column text-widget))
	 (row (row text-widget))
	 (columns (columns text-widget))
	 (insert-mode (insert-mode text-widget))
	 (tab-step (tab-step text-widget))
	 (target-column (* tab-step (1+ (truncate column tab-step))))
	 (delta (- target-column column)))
    (cond (insert-mode
	   (put text-widget :string (make-string delta :initial-element '#\Space))
	   (setf (changed text-widget) t)
	   (tbg-repaint text-widget :start-row row :end-row row))
	  (t ;overwrite
	   (setf (column text-widget)
		 (min target-column columns))))))

(defhandler back-tab ((text-widget text-widget) &rest args
		      &default (:key-press :shift #\Tab))
  (declare (ignore args))
  (let* ((column (column text-widget))
	 (tab-step (tab-step text-widget))
	 (target-column (* tab-step (truncate (1- column) tab-step))))
    (setf (column text-widget)
	  (max target-column 0))))

;; marking text
(defhandler position-mark ((text-widget text-widget) &key x y &allow-other-keys
			   &default ((:button-press :detail :right-button)
				     (:button-release :detail :left-button)
				     (:pointer-motion :detail :left-button)))
  (let* ((font (font text-widget))
	 (buffer (buffer text-widget))
	 (fw (width font))
	 (fh (height font))
	 (top-pad (top-pad text-widget))
	 (mark-row (mark-row text-widget))
	 (mark-column (mark-column text-widget))
    
	 ;; the new row is either where the mouse was clicked, or the last
	 ;; line of the buffer, whichever is smaller.
	 (new-mark-row (min (+ (top-of-screen text-widget)
			       (truncate (- y top-pad) fh))
			    (1- (rows buffer))
			    (+ (top-of-screen text-widget)
			       (rows text-widget))))
    
	 ;; the new col is either where the mouse was clicked, or the
	 ;; last column of the new row, whichever is smaller.
	 (new-mark-column (min (+ (left-of-screen text-widget)
				  (truncate x fw))
			       (+ (left-of-screen text-widget)
				  (columns text-widget))
			       (buf-columns buffer new-mark-row))))

    (when (or (null mark-row)
	      (null mark-column)
	      (/= new-mark-row mark-row) 
	      (/= new-mark-column mark-column))
	  (mark text-widget 
		:mark-row new-mark-row 
		:mark-column new-mark-column))))

;; moving the cursor
(defhandler position-cursor ((text-widget text-widget) &key x y 
			     &allow-other-keys
			     &default (:button-press :detail :left-button))
  (let* ((font (font text-widget))
	 (fw (width font))
	 (fh (height font))
	 (top-pad (top-pad text-widget))
	 (row (row text-widget))
	 (column (column text-widget)))
    ;; the new row is either where the mouse was clicked, or the last
    ;; line of the buffer, whichever is smaller.
    (setq row (min (+ (top-of-screen text-widget)
		      (truncate (- y top-pad) fh))
		   (1- (rows (buffer text-widget)))
		   (+ (top-of-screen text-widget)
		      (rows text-widget))))
    
    ;; the new col is either where the mouse was clicked, or the
    ;; last column of the new row, whichever is smaller.
    (setq column (min (+ (left-of-screen text-widget)
			 (truncate x fw))
		      (+ (left-of-screen text-widget)
			 (columns text-widget))
		      (buf-columns (buffer text-widget) row)))

    (setf (row text-widget) row)
    (setf (column text-widget) column)))
    
(defhandler mark-word ((text-widget text-widget) &rest args
		       &default :double-click)
  (declare (ignore args))
  (let* ((row (row text-widget))
	 (column (column text-widget))
	 (buffer (buffer text-widget))
	 (curr-line (value buffer :row row))
	 (ch (if (>= column (length curr-line)) (character " ")
	       (aref curr-line column)))
	 (new-column (if (alphanumericp ch)
			 (next-alphanumeric-pos
			  curr-line
			  (previous-non-alphanumeric-pos curr-line column))
		       column))
	 (new-mark-column (if (alphanumericp ch)
			      (previous-alphanumeric-pos
			       curr-line
			       (next-non-alphanumeric-pos curr-line
							  new-column))
			    column)))
    (setf (changed text-widget) t)

    (setf (column text-widget) new-column)
    (mark text-widget :mark-row row :mark-column new-mark-column)))

(defhandler mark-line ((text-widget text-widget) &rest args
		       &default :triple-click)
  (declare (ignore args))
  (let ((row (row text-widget))
	(buffer (buffer text-widget)))
    (setf (column text-widget) 0)
    (mark text-widget :mark-row row :mark-column (buf-columns buffer row))))

;;;
;;;	Activate/Deactivate methods restore/hide cursor
;;;


(defmethod activate ((self text-widget))
  (call-next-method)
  (unless (current self)
	  (setf (current self) t)
	  (let ((row (row self))
		(col (column self)))
	       (tbg-repaint self :start-row row :end-row row
			:start-column col :end-column col))))


(defmethod deactivate ((self text-widget))
  (call-next-method)
  (when (current self)
	  (setf (current self) nil)
	  (let ((row (row self))
		(col (column self)))
	       (tbg-repaint self :start-row row :end-row row
			:start-column col :end-column col))))
