;;;
;;; Copyright (c) 1990 Regents of the University of California
;;; 
;;; $Author: smoot $
;;; $Source: /pic2/picasso/widgets/table/RCS/matrix-field-methods.cl,v $
;;; $Revision: 1.5 $
;;; $Date: 1991/11/15 21:58:33 $
;;;

(in-package "PT")

(defun make-matrix-field (&rest keys)
  (apply #'make-instance 'matrix-field :allow-other-keys t keys))

;;;
;;;	Data accessors
;;;

(defun mref (mf row col &aux data)
  (if (not data)
      (setq data (data mf)))
  (aref data row col))

(defun mref-setf (mf row col val &optional data)
  (if (not (matrix-field-p mf))
      (error "mref:  bad-matrix:  ~S" mf))
  (if (null data) (setq data (data mf)))
  (setf (aref data row col) val)
  (if (and (typep val 'dtext) (attached-p mf))
      (setf (slot-value val 'gc) 
	    (make-shared-gc mf (list :font (font val) :paint (paint val)))))
  (mf-sync-data mf row col))

;;;
;;;	Function to manually dispatch an event
;;;

(defun mf-find-window (self &rest args &key x y &allow-other-keys
			    &aux icp irp ft tot row col cur md)
  (declare (ignore args))
  (setq icp (inter-col-pad self)
	irp (inter-row-pad self)
	md (gm-data self)
	ft (field-table self))
  ;;	Find column in which event occured (if any)
  (setq tot icp)
  (setq col
	(do ((c (md-col-index md) (1+ c)))
	    ((>= tot x) (when (and (>= (setq c (1- c)) 0)
				   (< x (- tot icp)))
			      c))
	    (setq cur (aref ft 0 c))
	    (setq tot 
		  (if (listp cur)
		      (+ tot (getf cur :width) icp)
		      (+ tot (width cur) (* 2 (border-width cur)) icp)))))
  ;;	Find row in which event occured (if any)
  (setq tot irp)
  (setq row
	(do ((r (md-row-index md) (1+ r)))
	    ((>= tot y) (when (and (>= (setq r (1- r)) 0)
				   (< y (- tot irp)))
			      r))
	    (setq cur (aref ft r 0))
	    (setq tot 
		  (if (listp cur)
		      (+ tot (getf cur :height) irp)
		      (+ tot (height cur) (* 2 (border-width cur)) irp)))))
  ;;	Return appropriate entry in field-table or nil if none
  (when (and row col)
	(aref ft row col)))

(defun mf-dispatch-event (win args &aux func click-time double-time)
  (setq click-time *button-event-time*
	double-time *button-double-time*)
  (setq *button-event-time* 0
	*button-double-time* 0)
  ;; get event-mappings
  (setq func (find-entry (class-event-map win) (descriptor args)))
  (setq *button-event-time* click-time
	*button-double-time* double-time)
  (when (function-p func)
	(apply func (cons win args))))

(defhandler advance-nomad ((self matrix-field) &rest args
			   &default (:key-press :detail #\Return))
  (declare (ignore args))
  (execute 'return-func self))

(defhandler dispatch ((self matrix-field) &rest args &aux cf 
		      &default :key-press)
  (if (x-window-p (setq cf (current-field self)))
      (mf-dispatch-event cf args)))

(defhandler focus ((self matrix-field) &key display &allow-other-keys &aux cf
		   &default :enter-window)
  (if (x-window-p (setq cf (current-field self)))
	(xlib:set-input-focus display (res cf) :pointer-root)))

(defhandler unfocus ((self matrix-field) &key display &allow-other-keys &aux cf 
		     &default :enter-window)
  (if (x-window-p (setq cf (current-field self)))
	(xlib:set-input-focus display :pointer-root :pointer-root)))

;;	Selections field to be current, all others uncurrent
(defhandler select-unique ((self matrix-field) &rest args &key child 
			   &allow-other-keys &aux gs
			   &default (:button-press :detail :left-button))
  ;;	Find subwindow in which event occurred (if any)
  (cond ((null child)
	 (setq child (apply #'mf-find-window (cons self args))) 
	 (when (or (null child) 
		   (and (atom child) 
			(eq (third (geom-spec child)) :unselectable))
		   (and (listp child)
			(getf child :unselectable)))
	       (return-from matrix-field-select-unique))))
  
  ;;	Make window current
  (setq gs 
	(if (listp child)
	    (getf child :geom-spec)
	    (geom-spec child)))
  
  (case (selection self)
	(:entry 
	 (let ((ri (slot-value self 'row-index))
	       (ci (slot-value self 'col-index)))
	      (if (and (< (+ (car gs) ri) (data-rows self)) 
		       (< (+ (cadr gs) ci) (data-cols self)))
		  (setf (current-indices self)
			(list (list (+ (car gs) ri) 
				    (+ (cadr gs) ci)))))))
	(:row
	 (let ((ri (slot-value self 'row-index)))
	      (if (< (+ (car gs) ri) (data-rows self))
		  (setf (current-indices self) 
			(list (+ (car gs) ri))))))
	((:column :col)
	 (let ((ci (slot-value self 'col-index)))
	      (if (< (+ (cadr gs) ci) (data-cols self))
		  (setf (current-indices self)
			(list (+ (cadr gs) ci))))))))

;;	If field is current, make uncurrent, otherwise add to current list.
(defhandler select-multiple ((self matrix-field) &rest args &key child 
			     &allow-other-keys &aux gs il 
			     &default (:button-press :detail :right-button))
  ;;	Find subwindow in which event occurred (if any)
  (when (null child) 
	(setq child (apply #'mf-find-window (cons self args))) 
	(when (or (null child) 
		  (and (atom child) 
		       (eq (third (geom-spec child)) :unselectable))
		  (and (listp child)
		       (getf child :unselectable)))
	      (return-from matrix-field-select-multiple)))
  
  ;;	Make window current
  (setq gs 
	(if (listp child)
	    (getf child :geom-spec)
	    (geom-spec child)))
  (setq il (list (+ (car gs) (slot-value self 'row-index)) 
		 (+ (cadr gs) (slot-value self 'col-index))))
  (case (selection self)
	(:entry
	 (if (and (< (car il) (data-rows self)) 
		  (< (cadr il) (data-cols self)))
	     (if (member il (slot-value self 'current-indices) :test #'equal)
		 (delete-current il self)
		 (add-current il self))))
	(:row
	 (if (< (car il) (data-rows self))
	     (if (member (car il) (slot-value self 'current-indices) 
			 :test #'equal)
		 (delete-current (car il) self)
		 (add-current (car il) self))))
	((:column :col)
	 (if (< (cadr il) (data-cols self))
	     (if (member (cadr il) (slot-value self 'current-indices) :test #'equal)
		 (delete-current (cadr il) self)
		 (add-current (cadr il) self))))))

(defhandler select-drag ((self matrix-field) &rest args &key child 
			     &allow-other-keys &aux gs il 
			     &default (:pointer-motion :detail :right-button))
  ;;	Find subwindow in which event occurred (if any)
  (when (null child) 
	(setq child (apply #'mf-find-window (cons self args))) 
	(when (or (null child) 
		  (and (atom child) 
		       (eq (third (geom-spec child)) :unselectable))
		  (and (listp child)
		       (getf child :unselectable)))
	      (return-from matrix-field-select-drag)))
  
  ;;	Make window current
  (setq gs 
	(if (listp child)
	    (getf child :geom-spec)
	    (geom-spec child)))
  (setq il (list (+ (car gs) (slot-value self 'row-index)) 
		 (+ (cadr gs) (slot-value self 'col-index))))
  (case (selection self)
	(:entry
	 (if (and (< (car il) (data-rows self)) 
		  (< (cadr il) (data-cols self)))
	     (unless (member il (slot-value self 'current-indices) 
			     :test #'equal)
		     (add-current il self))))
	(:row
	 (if (< (car il) (data-rows self))
	     (unless (member (car il) (slot-value self 'current-indices) 
			 :test #'equal) 
		     (add-current (car il) self))))
	((:column :col)
	 (if (< (cadr il) (data-cols self))
	     (unless (member (cadr il) (slot-value self 'current-indices) 
			     :test #'equal) 
		     (add-current (cadr il) self))))))

(defhandler unselect-drag ((self matrix-field) &rest args &key child 
			     &allow-other-keys &aux gs il 
			     &default (:pointer-motion :detail :middle-button))
  ;;	Find subwindow in which event occurred (if any)
  (when (null child) 
	(setq child (apply #'mf-find-window (cons self args))) 
	(when (or (null child) 
		  (and (atom child) 
		       (eq (third (geom-spec child)) :unselectable))
		  (and (listp child)
		       (getf child :unselectable)))
	      (return-from matrix-field-unselect-drag)))
  
  ;;	Make window current
  (setq gs 
	(if (listp child)
	    (getf child :geom-spec)
	    (geom-spec child)))
  (setq il (list (+ (car gs) (slot-value self 'row-index)) 
		 (+ (cadr gs) (slot-value self 'col-index))))
  (case (selection self)
	(:entry
	 (if (and (< (car il) (data-rows self)) 
		  (< (cadr il) (data-cols self)))
	     (if (member il (slot-value self 'current-indices) :test #'equal)
		 (delete-current il self))))
	(:row
	 (if (< (car il) (data-rows self))
	     (if (member (car il) (slot-value self 'current-indices) 
			 :test #'equal)
		 (delete-current (car il) self))))
	((:column :col)
	 (if (< (cadr il) (data-cols self))
	     (if (member (cadr il) (slot-value self 'current-indices) 
			 :test #'equal)
		 (delete-current (cadr il) self))))))

;;	If field is a current, activate field
(defhandler select-input ((self matrix-field) &rest args &key child 
			  &allow-other-keys 
			  &default (:button-press :detail :middle-button))
  ;;	Find subwindow in which event occurred (if any)

  (declare (ignore args))
  (when (member child (current-fields self))
	(mf-deactivate (current-field self) self)
	(mf-activate child self)
	(setf (slot-value self 'current-field) child)))

;;	Messages from geometry manager
(defun mf-resize-update (self visible-rows visible-cols &aux nr nc sup 
			      uni-rows uni-cols (flag nil))
  (when (flag self) (return-from mf-resize-update))
  (setq nr (data-rows self))
  (setq nc (data-cols self))
  (when (and (setq uni-rows (uniform-rows self)) 
	     (< nr (+ (row-index self) visible-rows)))
	(setf (row-index self) (max 0 (- nr visible-rows)))
	(setq flag t)
	(mf-uni-propagate self nil nil))
  (when (and (setq uni-cols (uniform-cols self))
	     (< nc (+ (col-index self) visible-cols)))
	(setf (col-index self) (max 0 (- nc visible-cols 1)))
	(unless flag
		(mf-uni-propagate self nil nil)))
  (setq sup (parent self))
  (if (and (typep sup 'table-field) (eq self (matrix-field sup)))
	(ta-resize-update sup visible-rows visible-cols))
  (if (or uni-rows uni-cols) 
      (update-indices self)
;;      (setf (current-indices self) (slot-value self 'current-indices))
)
  
  ;;	Update inverted fields
  (update-inverts self))

;;;
;;;	Clear synthetic gadgets
;;;

(defun mf-clear (self &aux nw (gs nil)) 
  (setq nw (nomad-widget self))
  (when (consp nw)
	(setq gs (getf nw :geom-spec)))
  (cond ((uni-row-types self)
	 (let* ((ft (field-table self))
		(ef nil)
		(vci (visible-col-index self))
		(mc (+ (visible-cols self) vci)) 
		(nc (car gs)))
	       (do ((c vci (1+ c)))
		   ((>= c mc))
		   (setq ef (aref ft 0 c)) 
		   (cond ((eq c nc)
			  (clear-region self 
					(getf nw :x) (getf nw :y) 
					(getf nw :width) (height self)))
			 ((listp ef)
			  (clear-region self 
					(getf ef :x) (getf ef :y) 
					(getf ef :width) (height self)))))))
	((uni-col-types self)
	 (let* ((ft (field-table self))
		(ef nil)
		(vri (visible-row-index self))
		(mr (+ (visible-rows self) vri))
		(nr (cadr gs)))
	       (do ((r vri (1+ r)))
		   ((>= r mr))
		   (setq ef (aref ft r 0)) 
		   (cond ((eq r nr)
			  (clear-region self 
					(getf nw :x) (getf nw :y) 
					(width self) (getf nw :height)))
			 ((listp ef)
			  (clear-region self 
					(getf ef :x) (getf ef :y) 
					(width self) (getf ef :height))))))))
  (repaint-inverts self :clear t))

;;;
;;;	Repaint synthetic gadgets
;;;

(defun mf-put-entry (ef self &aux gs r c)
  (clear-region self 
		(getf ef :x) (getf ef :y) 
		(getf ef :width) (getf ef :height)) 
  (setq gs (getf ef :geom-spec))
  (setq r
	(if (uniform-rows self)
	    (+ (row-index self) (car gs))
	    (car gs)))
  (setq c
	(if (uniform-cols self)
	    (+ (col-index self) (cadr gs))
	    (cadr gs)))
  (apply #'put
	 `(,(aref (data self) r c)
	    ,@ef
	    :window ,self 
	    :gc ,(gc-res self))))

(defun mf-put (self) 
  ;;	Go through and draw those values which are not associated with "hard"
  ;;	widgets.  These values are represented by attribute lists (as opposed
  ;;	to widgets or gadgets) in the field table.
  (cond ((uni-row-types self)
	 (let* ((ft (field-table self))
		(da (data self))
		(ri (row-index self))
		(ci (col-index self))
		(md (gm-data self))
		(vci (md-col-index md))
		(vri (md-row-index md))
		(ef nil)
		(mc (+ (visible-cols self) vci))
		(mr (+ (visible-rows self) ri))
		(nw (nomad-widget self))
		(nr nil)
		(gc (gc-res self))
		(nc nil))
	       (setq nw (when (consp nw) (getf nw :geom-spec)))
	       (setq nr (car nw) nc (cadr nw))
	       (if (uniform-rows self)
		   (do ((c vci (1+ c)))
		       ((>= c mc))
		       (setq ef (aref ft 0 c)
			     nr (car nw)) 
		       (cond ((eq nc c)
			      (setq ef (aref ft nr nc))
			      (clear-region self 
					    (x-offset ef) (y-offset ef) 
					    (width ef) (height self))
			      (setq nr (+ nr ri))
			      (do ((r ri (1+ r)))
				  ((>= r nr))
				  (apply #'put
					 `(,(aref da r (+ c ci))
					    ,@(aref ft (- r ri) c)
					    :window ,self
					    :gc ,gc)))
			      (do ((r (1+ nr) (1+ r)))
				  ((>= r mr))
				  (apply #'put
					 `(,(aref da r (+ c ci))
					    ,@(aref ft (- r ri) c)
					    :window ,self 
					    :gc ,gc))))
			     ((listp ef) 
			      (clear-region self 
					    (getf ef :x) (getf ef :y) 
					    (getf ef :width) (height self))
			      (do ((r ri (1+ r)))
				  ((>= r mr))
				  (apply #'put
					 `(,(aref da r (+ c ci))
					    ,@(aref ft (- r ri) c)
					    :window ,self 
					    :gc ,gc))))))
		   (progn 
		    (setq mr (+ vri (visible-rows self)))
		    (do ((c vci (1+ c)))
			((>= c mc))
			(setq ef (aref ft 0 c)) 
			(cond ((eq nc c) 
			       (setq ef (aref ft nr nc))
			       (clear-region self 
					    (x-offset ef) (y-offset ef) 
					    (width ef) (height self))
			       (do ((r vri (1+ r)))
				   ((>= r nr))
				   (apply #'put
					  `(,(aref da r c)
					     ,@(aref ft r c)
					     :window ,self 
					     :gc ,gc)))
			       (do ((r (1+ nr) (1+ r)))
				   ((>= r mr))
				   (apply #'put
					  `(,(aref da r c)
					     ,@(aref ft r c)
					     :window ,self 
					     :gc ,gc))))
			      ((listp ef) 
			       (clear-region self 
					     (getf ef :x) (getf ef :y) 
					     (getf ef :width) (height self))
			       (do ((r vri (1+ r)))
				   ((>= r mr))
				   (apply #'put
					  `(,(aref da r c)
					     ,@(aref ft r c)
					     :window ,self 
					     :gc ,gc))))))))))
	((uni-col-types self)
	 (let* ((ft (field-table self))
		(da (data self))
		(md (gm-data self))
		(ri (row-index self))
		(ci (col-index self))
		(vci (md-row-index md))
		(vri (md-row-index md))
		(ef nil)
		(mc (+ (visible-cols self) ci))
		(mr (+ (visible-rows self) vri))
		(nw (nomad-widget self))
		(nr nil)
		(gc (gc-res self))
		(nc nil))
	       (setq nw (when (consp nw) (getf nw :geom-spec)))
	       (setq nr (car nw) nc (cadr nw))
	       (if (uniform-cols self)
		   (do ((r vri (1+ r)))
		       ((>= r mr))
		       (setq ef (aref ft r 0)) 
		       (cond ((eq nr r) 
			      (setq ef (aref ft nr nc))
			      (clear-region self 
					    (x-offset ef) (y-offset ef) 
					    (width self) (height ef))
			      (setq nc (+ nc ci))
			      (do ((c ci (1+ c)))
				  ((>= c nc))
				  (apply #'put
					 `(,(aref da (+ r ri) c)
					    ,@(aref ft r (- c ci))
					    :window ,self 
					    :gc ,gc)))
			      (do ((c (1+ nc) (1+ c)))
				  ((>= c mc))
				  (apply #'put
					 `(,(aref da (+ r ri) c)
					    ,@(aref ft r (- c ci))
					    :window ,self 
					    :gc ,gc))))
			     ((listp ef)
			      (clear-region self 
					    (getf ef :x) (getf ef :y) 
					    (getf ef :width) (height self))
			      (do ((c ci (1+ c)))
				  ((>= c mc))
				  (apply #'put
					 `(,(aref da (+ r ri) c)
					    ,@(aref ft r (- c ci))
					    :window ,self 
					    :gc ,gc))))))
		   (progn 
		    (setq mc (+ vci (visible-cols self)))
		    (do ((r vri (1+ r)))
			((>= r mr))
			(setq ef (aref ft r 0)) 
			(cond ((eq nr r) 
			       (setq ef (aref ft nr nc)) 
			       (clear-region self 
					     (x-offset ef) (y-offset ef) 
					     (width self) (height ef))
			       (do ((c vci (1+ c)))
				   ((>= c nc))
				   (apply #'put
					  `(,(aref da r c)
					     ,@(aref ft r c)
					     :window ,self 
					     :gc ,gc)))
			       (do ((c (1+ nc) (1+ c)))
				   ((>= c mc))
				   (apply #'put
					  `(,(aref da r c)
					     ,@(aref ft r c)
					     :window ,self 
					     :gc ,gc))))
			      ((listp ef)
			       (clear-region self 
					     (getf ef :x) (getf ef :y) 
					     (getf ef :width) (height self))
			       (do ((c vci (1+ c)))
				   ((>= c mc))
				   (apply #'put
					  `(,(aref da r c)
					     ,@(aref ft r c)
					     :window ,self 
					     :gc ,gc))))))))))))

(defun mf-dir-put (self dir num &aux r c end x w y h hs ws max) 
  ;;	Go through and draw those values which are not associated with "hard"
  ;;	widgets.  These values are represented by attribute lists (as opposed
  ;;	to widgets or gadgets) in the field table.
  (let* ((ft (field-table self))
	 (da (data self))
	 (ri (row-index self))
	 (ci (col-index self))
	 (md (gm-data self))
	 (vci (md-col-index md))
	 (vri (md-row-index md))
	 (ef nil)
	 (vr (visible-rows self))
	 (vc (visible-cols self))
	 (mc (+ vc vci))
	 (mr (+ vr ri))
	 (nw (nomad-widget self))
	 (nr nil)
	 (temp nil)
	 (rpad (inter-row-pad self))
	 (cpad (inter-col-pad self))
	 (nc nil)
	 (gc (gc-res self))
	 (res (res self)))
	(setq nw (when (consp nw) (getf nw :geom-spec)))
	(setq nr (car nw) nc (cadr nw)) 
	(case dir
	      ((:up :down) 
	       (when (>= num vr) 
		     (mf-put self)
		     (if (grid-lines self)
			 (mf-draw-grid self))
		     (return-from mf-dir-put))
	       (setq r (1- mr))
	       (setq ef (aref ft (1- vr) 0))
	       (setq hs (if (consp ef)
			    (+ (getf ef :y) (getf ef :height) rpad)
			    (+ (y-offset ef) (height ef) rpad)))
	       (setq max (width self))
	       (do ((c vci (1+ c)))
		   ((>= c mc))
		   (setq ef (aref ft 0 (- c ci))) 
		   (unless (or (consp ef) (eq nc c))
			   (continue))
		   ;;	Find range to move
		   (setq end 
			 (do ((end (1+ c)))
			     ((>= end mc) end)
			     (setq temp (aref ft 0 (- end ci)))
			     (unless (or (eq nc end) (consp temp))
				     (return end))
			     (setq end (1+ end))))
		   (when (= c (1+ end)) (continue))
		   (setq ef (aref ft 0 c))
		   (setq x 
			 (if (consp ef)
			     (- (getf ef :x) rpad)
			     (- (x-offset ef) rpad)))
		   (setq temp (aref ft 0 (1- end)))
		   (setq w 
			 (if (consp temp)
			     (+ (getf temp :x) (getf temp :width) 
				rpad (- x))
			     (+ (x-offset temp) (width temp) rpad (- x)))) 
		   (if (eq dir :up)
		       (progn
			(setq ef (aref ft num 0))
			(setq h
			      (if (consp ef)
				  (- (getf ef :y) rpad)
				  (- (y-offset ef) rpad)))
			(setq y (- hs h))
			;;	Move screen down
			(xlib:copy-area res gc x 0 w y res x h) 
			;;	Clear and fill in first [num] lines 
			(xlib:clear-area res :x x :y 0 :width w :height h
					 :exposures-p nil)
			(if (grid-lines self) 
			    (progn
			     (setq temp (round (/ rpad 2)))
			     (setq y (truncate (/ rpad 2)))
			     (do ((r 0 (1+ r)) 
				  (synth nil)
				  (gc-gray (gc-gray self)))
				 ((>= r num))
				 (do ((cur c (1+ cur)))
				     ((>= cur end)) 
				     (setq synth (aref ft r cur))
				     (incf y h)
				     (apply #'put
					    `(,(aref da (+ ri r) cur)
					      ,@synth
					      :window ,self 
					      :gc ,gc)))
				 (xlib:draw-line res gc-gray 0 y max y))
			     (xlib:clear-area res :x 0 :y (- (height self) temp)
					      :width max :height 1 
					      :exposures-p nil))
			    (do ((r 0 (1+ r)))
				((>= r num))
				(do ((cur c (1+ cur)))
				    ((>= cur end))
				    (apply #'put
					   `(,(aref da (+ ri r) cur)
					      ,@(aref ft r cur)
					      :window ,self 
					      :gc ,gc))))))
		       (progn
			(setq ef (aref ft num 0))
			(setq y (if (consp ef) 
				    (- (getf ef :y) rpad)
				    (- (y-offset ef) rpad)))
			(setq h (- hs y))
			;;	Move screen up
			(xlib:copy-area res gc x y w h res x 0)
			;;	Clear and fill in last [num] lines
			(xlib:clear-area res :x x :y h :width w :height y
					 :exposures-p nil)
			(if (grid-lines self)
			    (progn
			     (setq temp (round (/ rpad 2)))
			     (psetq y h h y)
			     (setq y (- (height self) temp))
			     (do ((r (- vr num) (1+ r))
				  (synth nil)
				  (gc-gray (gc-gray self)))
				 ((>= r vr))
				 (do ((cur c (1+ cur)))
				     ((>= cur end)) 
				     (setq synth (aref ft r cur)) 
				     (decf y h)
				     (decf y rpad)
				     (apply #'put
					    `(,(aref da (+ ri r) cur)
					      ,@synth
					      :window ,self 
					      :gc ,gc)))
				 (xlib:draw-line res gc-gray 0 (1- y) 
						 max (1- y)))
			     (xlib:clear-area res :x 0 :y (1- temp) :width max 
					      :height 1 :exposures-p nil))
			    (do ((r (- vr num) (1+ r)))
				((>= r vr))
				(do ((cur c (1+ cur)))
				    ((>= cur end)) 
				    (apply #'put
					   `(,(aref da (+ ri r) cur)
					      ,@(aref ft r cur)
					      :window ,self 
					      :gc ,gc)))))))
		   (setq c end)))
	      (t 
	       (when (>= num vc) 
		     (mf-put self)
		     (if (grid-lines self)
			 (mf-draw-grid self))
		     (return-from mf-dir-put))
	       (setq mc (+ vc ci) 
		     mr (+ vr vri)
		     max (height self))
	       (setq c (1- mc))
	       (setq ef (aref ft 0 (1- vc)))
	       (setq ws (if (consp ef)
			    (+ (getf ef :x) (getf ef :width) cpad)
			    (+ (x-offset ef) (width ef) cpad)))
	       (do ((r vri (1+ r)))
		   ((>= r mr))
		   (setq ef (aref ft (- r ri) 0)) 
		   (unless (or (consp ef) (eq nr r))
			   (continue))
		   ;;	Find range to move
		   (setq end
			 (do ((end (1+ r)))
			     ((>= end mr) end)
			     (setq temp (aref ft (- end ri) 0))
			     (unless (or (eq nr end) (consp temp))
				     (return end))
			     (setq end (1+ end))))
		   (when (= r (1+ end)) (continue))
		   (setq ef (aref ft r 0))
		   (setq y 
			 (if (consp ef)
			     (- (getf ef :y) cpad)
			     (- (y-offset ef) cpad)))
		   (setq temp (aref ft (1- end) 0))
		   (setq h 
			 (if (consp temp)
			     (+ (getf temp :y) (getf temp :height) 
				cpad (- y))
			     (+ (y-offset temp) (height temp) cpad (- y)))) 
		   (if (eq dir :left)
		       (progn
			(setq ef (aref ft 0 num))
			(setq w
			      (if (consp ef)
				  (- (getf ef :x) cpad)
				  (- (x-offset ef) cpad)))
			(setq x (- ws w))
			;;	Move screen down
			(xlib:copy-area res gc 0 y x h res w y)
			;;	Clear and fill in first [num] lines 
			(xlib:clear-area res :x 0 :y y :width w :height h
					 :exposures-p nil)
			(if (grid-lines self)
			    (progn
			     (setq temp (round (/ cpad 2)))
			     (do ((c 0 (1+ c))
				  (synth nil)
				  (gc-gray (gc-gray self)))
				 ((>= c num))
				 (do ((cur r (1+ cur)))
				     ((>= cur end))
				     (setq synth (aref ft cur c))
				     (setq x (- (getf synth :x) temp))
				     (apply #'put
					    `(,(aref da cur (+ ci c))
					       ,@synth
					       :window ,self 
					       :gc ,gc))) 
				 (xlib:draw-line res gc-gray x 0 x max)))
			    (do ((c 0 (1+ c)))
				((>= c num))
				(do ((cur r (1+ cur)))
				    ((>= cur end))
				    (apply #'put
					   `(,(aref da cur (+ ci c))
					      ,@(aref ft cur c)
					      :window ,self 
					      :gc ,gc))))))
		       (progn
			(setq ef (aref ft 0 num))
			(setq x (if (consp ef) 
				    (- (getf ef :x) cpad)
				    (- (x-offset ef) cpad)))
			(setq w (- ws x))
			;;	Move screen up
			(xlib:copy-area res gc x y w h res 0 y)
			;;	Clear and fill in last [num] lines
			(xlib:clear-area res :x x :y h :width w :height h
					 :exposures-p nil)
			(if (grid-lines self)
			    (progn
			     (setq temp (round (/ cpad 2)))
			     (do ((c (- vc num) (1+ c))
				  (synth nil)
				  (gc-gray (gc-gray self)))
				 ((>= c vc))
				 (do ((cur r (1+ cur)))
				     ((>= cur end)) 
				     (setq synth (aref ft cur c))
				     (setq x (- (getf synth :x) temp))
				     (apply #'put
					    `(,(aref da cur (+ ci c))
					       ,@synth
					       :window ,self 
					       :gc ,gc)))
				 (xlib:draw-line res gc-gray x 0 x max))) 
			    (do ((c (- vc num) (1+ c)))
				((>= c vc))
				(do ((cur r (1+ cur)))
				    ((>= cur end)) 
				    (apply #'put
					   `(,(aref da cur (+ ci c))
					      ,@(aref ft cur c)
					      :window ,self 
					      :gc ,gc)))))))
		   (setq r end))))))


(defmethod do-repaint-region ((self matrix-field) x y w h
			      &key (clear t) &allow-other-keys)
  ;;	Go through and draw those values which are not associated with "hard"
  ;;	widgets.  These values are represented by attribute lists (as opposed
  ;;	to widgets or gadgets) in the field table.
  ;;   But only do it for things in the appropriate region
  ;;   In this new version the number of min/max col/rows vars is confuzing and messy.
  ;;   sorry.  It'll p'bly stay that way until at least summer 92....
  ;;   unless someone yells at me enough... (-smoot)
  (declare (integer x y w h))
  (if clear (clear-region self x y w h))
  (if (typep (parent self) 'table-field)
      (border-repaint (border-type (parent self)) (parent self))
    (border-repaint (border-type self) self))
  (when (grid-lines self) (mf-draw-grid self))
  (cond ((uni-row-types self)
	 (let* ((ft (field-table self))
		(da (data self))
		(ri (row-index self))
		(md (gm-data self))
		(vci (md-col-index md))
		(vri (md-row-index md))
		(ef nil)
		(nw (nomad-widget self))
		(nr nil)
		(gc (gc-res self))
		(min-c vci)
		(max-c (+ (visible-cols self) vci))
		(min-r vri)
		(max-r (+ (visible-rows self) ri))
		(nc nil))
	   (setq nw (if (consp nw) (getf nw :geom-spec) 
		      (if (not (null nw)) (geom-spec nw))))
	   (setq nr (car nw) nc (cadr nw))
 	   (do* ((c min-c (1+ c)))
 		((>= c max-c))
		(setq ef (aref ft ri c))
		(let ((ef-x (if (consp ef) (getf ef :x) (x-offset ef)))
		      (ef-w (if (consp ef) (getf ef :width) (width ef))))
		  (if (< (+ ef-x ef-w) x) (setq min-c c))
		  (if (> ef-x (+ x w)) (setq max-c c))))
	   (do* ((r min-r (1+ r)))
		((>= r max-r))
		(setq ef (aref ft r min-c))
		(let ((ef-y (if (consp ef) (getf ef :y) (y-offset ef)))
		      (ef-h (if (consp ef) (getf ef :height) (height ef))))
		  (if (< (+ ef-y ef-h) y) (setq min-r r))
		  (if (> ef-y (+ y h)) (setq max-r r))))
	   (if (uniform-rows self)
	       (do ((c min-c (1+ c)))
		   ((>= c max-c))
		   (setq ef (aref ft ri c))
		   (cond ((eq nc c)
			  (setq ef (aref ft ri c))
			  (setq nr (+ nr ri))
			  (if (< nr min-r) (setq nr max-r))
			  (do ((r min-r (1+ r)))
			      ((>= r nr))
			      (apply #'put
				     `(,(aref da r c )
				       ,@(aref ft r  c)
				       :window ,self
				       :gc ,gc)))
			  (do ((r (1+ nr) (1+ r)))
			      ((>= r max-r))
			      (apply #'put
				     `(,(aref da r c )
				       ,@(aref ft r c)
				       :window ,self 
				       :gc ,gc))))
			 ((listp ef)
			  (do ((r min-r (1+ r)))
			      ((>= r max-r))
			      (apply #'put
				     `(,(aref da r c )
				       ,@(aref ft r  c)
				       :window ,self
				       :gc ,gc))))))
	     (progn
	       (setq max-r (+ vri (visible-rows self)))
	       (if (< nr min-r) (setq nr max-r))
	       (do ((c min-c (1+ c)))
		   ((>= c max-c))
		   (setq ef (aref ft 0 c)) 
		   (cond ((eq nc c) 
			  (setq ef (aref ft nr nc))
			  (do ((r min-r (1+ r)))
			      ((>= r nr))
			      (apply #'put
				     `(,(aref da r c)
				       ,@(aref ft r c)
				       :window ,self 
				       :gc ,gc)))
			  (do ((r (1+ nr) (1+ r)))
			      ((>= r max-r))
			      (apply #'put
				     `(,(aref da r c)
				       ,@(aref ft r c)
				       :window ,self 
				       :gc ,gc))))
			 ((listp ef) 
			  (do ((r min-r (1+ r)))
			      ((>= r max-r))
			      (apply #'put
				     `(,(aref da r c)
				       ,@(aref ft r c)
				       :window ,self 
				       :gc ,gc))))))))))
	((uni-col-types self)
	 (let* ((ft (field-table self))
		(da (data self))
		(md (gm-data self))
		(ri (row-index self))
		(ci (col-index self))
		(vci (md-row-index md))
		(vri (md-row-index md))
		(ef nil)
		(nw (nomad-widget self))
		(nr nil)
		(gc (gc-res self))
		(min-c vci)
		(max-c (+ (visible-cols self) ci))
		(min-r vri)
		(max-r (+ (visible-rows self) vri))
		(nc nil))
	   (setq nw (if (consp nw) (getf nw :geom-spec) (geom-spec nw)))
	   (setq nr (car nw) nc (cadr nw))
	   (do* ((r min-r (1+ r)))
		((>= r max-r))
		(setq ef (aref ft r min-c))
		(let ((ef-y (if (consp ef) (getf ef :y) (y-offset ef)))
		      (ef-h (if (consp ef) (getf ef :height) (height ef))))
		  (if (< (+ ef-y ef-h) y) (setq min-r r))
		  (if (> ef-y (+ y h)) (setq max-r r))))
	   ;;; note this should be done more efficiently,
           ;;; using uniformity.  but is harder to debug,
	   ;;; I ripped out my version which did it.
 	   (do* ((c min-c (1+ c)))
 		((>= c max-c))
		(setq ef (aref ft ri c))
		(let ((ef-x (if (consp ef) (getf ef :x) (x-offset ef)))
		      (ef-w (if (consp ef) (getf ef :width) (width ef))))
		  (if (< (+ ef-x ef-w) x) (setq min-c c))
		  (if (> ef-x (+ x w)) (setq max-c c))))
	   (if (< nc min-c) (setq nc max-c))
	   (if (uniform-cols self)
	       (do ((r min-r (1+ r)))
		   ((>= r max-r))
		   (setq ef (aref ft r 0)) 
		   (cond ((eq nr r) 
			  (setq ef (aref ft nr nc))
			  (setq nc (+ nc ci))
			  (do ((c min-c (1+ c)))
			      ((>= c nc))
			      (apply #'put
				     `(,(aref da r  c)
				       ,@(aref ft r c )
				       :window ,self 
				       :gc ,gc)))
			  (do ((c (1+ nc) (1+ c)))
			      ((>= c max-c))
			      (apply #'put
				     `(,(aref da  r  c)
				       ,@(aref ft r c )
				       :window ,self 
				       :gc ,gc))))
			 ((listp ef)
			  (do ((c min-c (1+ c)))
			      ((>= c max-c))
			      (apply #'put
				     `(,(aref da  r c)
				       ,@(aref ft r c )
				       :window ,self 
				       :gc ,gc))))))
	     (progn 
	       (do ((r min-r (1+ r)))
		   ((>= r max-r))
		   (setq ef (aref ft r 0)) 
		   (cond ((eq nr r) 
			  (setq ef (aref ft nr nc)) 
			  (do ((c min-c (1+ c)))
			      ((>= c nc))
			      (apply #'put
				     `(,(aref da r c)
				       ,@(aref ft r c)
				       :window ,self 
				       :gc ,gc)))
			  (do ((c (1+ nc) (1+ c)))
			      ((>= c max-c))
			      (apply #'put
				     `(,(aref da r c)
				       ,@(aref ft r c)
				       :window ,self 
				       :gc ,gc))))
			 ((listp ef)
			  (do ((c min-c (1+ c)))
			      ((>= c max-c))
			      (apply #'put
				     `(,(aref da r c)
				       ,@(aref ft r c)
				       :window ,self 
				       :gc ,gc))))))))))))


;;;
;;;	Draw grid lines
;;;

(defun mf-draw-grid (self &aux temp ft bw)
  (setq ft (field-table self))
  
  ;;	Horizontal
  (setq temp (visible-rows self))
  (if (> temp 1)
      (if (uniform-rows self)
	  (let* ((irp (inter-row-pad self))
		 (y (truncate (/ irp 2)))
		 (w (width self))
		 (res (res self))
		 (gc (gc-gray self))
		 (obj (aref ft 0 0))
		 (off (if (listp obj)
			  (+ (getf obj :height) irp) 
			  (progn
			   (setq bw (border-width obj))
			   (if (listp bw)
			       (setq bw (cadr bw)))
			   (+ (height obj) bw bw irp)))))
		(dotimes (r (1- temp))
			 (incf y off)
			 (xlib:draw-line res gc 0 y w y)))
	  (let* ((irp (truncate (/ (inter-row-pad self) 2)))
		 (y 0)
		 (w (width self))
		 (res (res self))
		 (gc (gc-gray self)))
		(dotimes (r (1- temp))
			 (setq y (aref ft (1+ r) 0)) 
			 (setq y 
			       (if (listp y)
				   (getf y :y)
				   (progn
				    (setq bw (border-width y))
				    (if (listp bw)
					(setq bw (cadr bw)))
				    (- (y-offset y) bw))))
			 (decf y irp)
			 (xlib:draw-line res gc 0 y w y)))))
  (setq temp (visible-cols self))
  (if (> temp 1)
      (if (uniform-cols self)
	  (let* ((icp (inter-col-pad self))
		 (x (truncate (/ icp 2)))
		 (h (height self))
		 (res (res self))
		 (gc (gc-gray self)) 
		 (obj (aref ft 0 0))
		 (off (+ icp (if (listp obj) 
				 (getf obj :width) 
				 (width obj)))))
		(dotimes (c (1- temp))
			 (incf x off)
			 (xlib:draw-line res gc x 0 x h)))
	  (let* ((icp (truncate (/ (inter-col-pad self) 2)))
		 (x 0)
		 (h (height self))
		 (res (res self))
		 (gc (gc-gray self)))
		(dotimes (c (1- temp))
			 (setq x (aref ft 0 (1+ c)))
			 (setq x 
			       (if (listp x)
				   (getf x :x)
				   (progn
				    (setq bw (border-width x))
				    (if (listp bw)
					(setq bw (car bw)))
				    (- (x-offset x) bw))))
			 (decf x icp)
			 (xlib:draw-line res gc x 0 x h))))))

;;;
;;;	Do-attach method attaches nomad-widget
;;;

(defmethod do-attach ((self matrix-field) &aux font disp)
  ;;	Attach self
  (setq font (font self)
	disp (display self))
  (call-next-method)

  ;;	Attach fonts
  (dolist (synth (synths self))
	  (setf (getf synth :gc)
		(make-shared-gc self (getf synth :gc-info)))))

(defmethod do-detach ((self matrix-field) &aux font)
  ;;	Detach fonts
  (setq font (font self))
  (dolist (f (fonts self))
	  (when (font-p f) (font-detach f)))
  
  ;;	Detach self
  (call-next-method))

;;;
;;;	Draw matrix-field
;;;

;;	Draw methods goes through and draws synthetic gadgets
(defmethod do-repaint ((self matrix-field)
		       &key 
		       &allow-other-keys)
  (call-next-method)

  ;;	Update synthetic fields
  (mf-put self)
  ;;	Draw lines
  (if (grid-lines self)
      (mf-draw-grid self))
  ;;	Update inverted fields
  (repaint-inverts self)
  (if (typep (parent self) 'table-field)
      (border-repaint (border-type (parent self)) (parent self))
    (border-repaint (border-type self) self))
  )


;;	Used to redraw widgets
(defun mf-repaint (self)
  ;;	Repaint widgets & gadgets
  (mapc #'repaint (children self))

  ;;	Update synthetic fields
  (mf-put self)
  ;;	Draw lines
  (if (grid-lines self)
      (mf-draw-grid self))
  ;;	Update inverted fields
  (repaint-inverts self))

;;	Used to directedly redraw widgets
(defun mf-dir-repaint (self dir num)
  ;;	Repaint widgets & gadgets
  (mapc #'repaint (children self))

  ;;	Update synthetic fields
  (mf-dir-put self dir num)
  ;;	Update inverted fields
  (repaint-inverts self))

(defun mf-gad-repaint (self)
  (dolist (ch (children self))
	  (unless (opaque-window-p ch)
		  (repaint ch))))

(defun data-array (self)
  (data self))

;;;
;;;	Do propogation (from data to fields)
;;;

(defun mf-propagate (mf &aux data ri ci gs)
  (unless (auto-prop mf)
	  (mf-repaint mf)
	  (return-from mf-propagate))
  (setq data (data mf)
	ri (slot-value mf 'row-index)
	ci (slot-value mf 'col-index))
  (mapc #'(lambda (ef) 
		  (setq gs (geom-spec ef)) 
		  (setf (value ef) 
			(aref data (+ ri (car gs)) (+ ci (cadr gs)))))
	(children mf))
  (mf-put mf))

(defun mf-uni-propagate (mf dir n &aux data ri ci gs)
  (unless (auto-prop mf)
	  (if dir
	      (mf-dir-repaint mf dir n)
	      (mf-repaint mf))
	  (return-from mf-uni-propagate))
  (setq data (data mf)
	ri (slot-value mf 'row-index)
	ci (slot-value mf 'col-index))
  (mapc #'(lambda (ef) 
		  (setq gs (geom-spec ef)) 
		  (setf (value ef) 
			(aref data (+ ri (car gs)) (+ ci (cadr gs)))))
	(exposed-of (children mf)))
  (mf-put mf)
  (mf-draw-grid mf))

;;	Update the field corresponding to indices into the data
(defun mf-sync-data (mf row col &aux field r c)
  (setq r (- row (row-index mf)) 
	c (- col (col-index mf)))
  (when (and (<= 0 r (1+ (rows mf))) 
	     (<= 0 c (1+ (cols mf))))
	(setq field (aref (field-table mf) r c))
	(cond ((atom field)
	       (setf (value field) (aref (data mf) row col))
	       (if (repaint-flag mf)
		   (repaint field)))
	      ((repaint-flag mf)
	       (mf-put-entry field mf)))))

;;	Pass in a field to be updated
(defun mf-sync-field (mf field &aux gs)
  (cond ((atom field)
	 (setq gs (geom-spec field))
	 (setf (value field) (aref (data mf) (+ (row-index mf) (car gs))
				(+ (col-index mf) (cadr gs))))
	 (if (repaint-flag mf)
	     (repaint field)))
	((repaint-flag mf)
	 (mf-put-entry field mf))))

;;	Pass in the index into the data for the row to be updated
(defun mf-sync-row (mf num &optional (ft nil) (da nil) (ri nil) (ci nil) rn
		       &aux ef)
  (unless ri (setq ri (row-index mf)))
  (when (or (< num ri) (>= num (+ ri (rows mf))))
	(return-from mf-sync-row))
  (unless ft (setq ft (field-table mf)))
  (unless da (setq da (data mf)))
  (unless ci (setq ci (col-index mf)))
  (setq rn (- num ri))
  (dotimes (c (cols mf))
	   (unless (listp (setq ef (aref ft rn (- c ci))))
		   (setf (value ef) 
			 (aref da num c)))))

;;	Pass in the index into the data for the column to be updated
(defun mf-sync-col (mf num &optional (ft nil) (da nil) (ri nil) (ci nil) cn
		       &aux ef)
  (unless ci (setq ci (col-index mf)))
  (when (or (< num ci) (>= num (+ ci (cols mf))))
	(return-from mf-sync-col))
  (unless ft (setq ft (field-table mf)))
  (unless da (setq da (data mf)))
  (unless ri (setq ri (row-index mf)))
  (setq cn (- num (if ci ci (col-index mf))))
  (dotimes (r (rows mf))
	   (unless (listp (setq ef (aref ft (- r ri) cn)))
		   (setf (value (aref ft (- r ri) cn))
			 (aref da r num)))))

;;;
;;;	Do backwards propogation (from field to data)
;;;

(defun mf-propagate-field (mf ef &aux gs r c l)
  (setq gs (geom-spec ef) 
	r (+ (car gs) (slot-value mf 'row-index))
	c (+ (cadr gs) (slot-value mf 'col-index)))
  (setf (aref (data mf) r c) (value ef))
  (unless (member (setq l (list r c)) (slot-value mf 'changed-indices) 
		  :test #'equal)
	  (setf (slot-value mf 'changed-indices) 
		(push (list r c) (slot-value mf 'changed-indices)))))

;;;
;;;	Functions to toggle event-mask
;;;

;;;	Toggle event-mask so that car is :expose and cadr is normal mask
(defun mf-rec-init-events (ef func)
  (if (typep ef 'widget) 
      (register-callback ef func :button-press))
  (if (typep ef 'collection-gadget)
      (dolist (ch (children ef))
	      (mf-rec-init-events ch func))))

;;;	Toggle event-mask so that cadr is :expose and car is normal mask
(defun mf-activate (ef mf &aux em x y)
  (when (listp ef)
	(when (getf ef :editable)
	      (let* ((gs (getf ef :geom-spec)) 
		     (nw (nomad-widget mf))
		     (item (aref (data mf) (+ (car gs) (row-index mf))
				 (+ (cadr gs) (col-index mf)))))
		    (setf (aref (field-table mf) (car gs) (cadr gs)) nw) 
		    (setf (value nw)
			  (if (typep item 'dtext)
			      (value item)
			      item))
		    (setf (slot-value mf 'nomad-widget) ef
			  (slot-value mf 'current-field) nw)
		    (setf (repaint-flag mf) nil)
		    (setq x (getf ef :x)
			  y (getf ef :y))
		    (when (free-nomad mf)
			  (multiple-value-setq 
			   (x y)
			   (xlib:translate-coordinates 
			    (res mf) x y (res (root-window)))))
		    (reshape nw x y 
			     (getf ef :width) (getf ef :height))
		    (expose nw)
		    (flush-display (display nw))
		    (setf (return-func nw)
			  (or (getf ef :return-func)
			      (return-func mf)))
		    (setf (repaint-flag mf) t)))
	(return-from mf-activate))
  (when (typep ef 'opaque-window)
	(setq em (event-mask ef))
	(when (and (not (member :current em)) 
		   (or (listp (car em)) (listp (cadr em))))
	      (setf (event-mask ef)
		    (append (car (last em)) (list (butlast em))))))
  (when (typep ef 'collection-gadget)
	(dolist (ch (children ef))
		(mf-activate ch mf))))

(defun mf-deactivate (ef mf &aux em nw gs oldval r c item)
  (when (or (listp ef) (concealed-p ef))
	(return-from mf-deactivate))
  (when (consp (setq nw (nomad-widget mf)))
;;	(setf 
;;	 (getf nw :x) (x-offset ef) 
;;	 (getf nw :y) (y-offset ef)
;;	      (getf nw :width) (width ef)
;;	      (getf nw :height) (height ef))
	(setq gs (getf nw :geom-spec))
	(setf (aref (field-table mf) (car gs) (cadr gs))
	      nw)
	(setq oldval (aref (data mf) 
			   (setq r (+ (car gs) (row-index mf)))
			   (setq c (+ (cadr gs) (col-index mf)))))
	(setq item (aref (data mf) r c))
	(if (typep item 'dtext)
	    (setf (value item) (value ef))
	    (setf (aref (data mf) r c) (value ef)))
	(if (not (equal (value ef) 
			(if (typep oldval 'dtext) 
			    (value oldval) 
			    oldval)))
		(setf (slot-value mf 'changed-indices)
		      (push (list r c) (slot-value mf 'changed-indices))))
	(setf (slot-value mf 'nomad-widget) ef)
	(mf-put-entry nw mf) 
	(setf (repaint-flag mf) nil)
	(conceal ef) 
	(setf (repaint-flag mf) t)
	(return-from mf-deactivate))
  (when (typep ef 'opaque-window)
	(setq em (event-mask ef))
	(when (member :current em)
	      (setf (event-mask ef)
		    (append (car (last em)) (list (butlast em))))))
  (when (typep ef 'collection-gadget)
	(dolist (ch (children ef))
		(mf-quick-deactivate ch)))
  ;;	Propogate back to matrix
  (setf (aref (data mf)
	      (+ (slot-value mf 'row-index) (car (geom-spec ef)))
	      (+ (slot-value mf 'col-index) (cadr (geom-spec ef))))
	(value ef)))

(defun mf-quick-deactivate (ef &aux em)
  (when (typep ef 'opaque-window)
	(setq em (event-mask ef))
	(when (member :current em)
	      (setf (event-mask ef)
		    (append (car (last em)) (list (butlast em))))))
  (when (typep ef 'collection-gadget)
	(dolist (ch (children ef))
		(mf-quick-deactivate ch))))

;;;
;;;	Create a new matrix-field instance
;;;

(defmethod new-instance ((self matrix-field)
			 &key
			 ;;	Slots. . .
			 (width		nil)
			 (height	nil)
			 (base-width	nil)
			 (base-height	nil)
			 (base-size 	nil)
			 (row-index 	0)
			 (col-index 	0)
			 (data-rows	nil)
			 (data-cols	nil)
			 (field-table	nil)
			 (data		nil)
			 (value		nil)
			 (row-titles	nil)
			 (col-titles	nil)
			 (inter-col-pad 3)
			 (inter-row-pad 3)
			 (row-title-selectable	nil)
			 (col-title-selectable	nil)
			 ;;	Creation options. . .
			 (size		nil)
			 (region	nil)
			 (rows		nil)	
			 (cols		nil)
			 (data-array-size nil)
			 (row-elements	nil)
			 (col-elements	nil)
			 (col-widths	nil)
			 (row-heights	nil)
			 (initial-rows	nil)
			 (initial-cols	nil)
			 (font		nil)
			 (editable	nil)
			 (editable-row-titles	nil)
			 (editable-col-titles	nil)
			 (self-adjusting nil)
			 (unique-selection nil)
			 (return-func nil rfp)
			 (title-select-func nil)
			 (row-title-select-func nil)
			 (reverts-p	nil)
			 (just		:center)
			 (horiz-just	:center)
			 (vert-just	:center)
			 ;;	Titles
			 (titles	nil)
			 (title-font	nil)
			 (title-height	nil)
			 (title-width	nil)
			 (row-title-width nil)
			 (row-title-elements nil)
			 (row-title-font nil)
			 (col-title-font nil)
			 (col-title-height nil)
			 (col-title-elements nil)
			 (default-titles t)

			 &allow-other-keys 
			 &aux rh cw x y r c x-init y-init l temp ef cur val
			      (uni-rows nil) (uni-cols nil) md unselectable
			      nomad-widget flag fonts)

  (call-next-method)

  ;;	Enforce defaults
  (if base-size
      (setq base-width (car base-size)
	    base-height (cadr base-size)))
  (when value (setq data value))
  (setq font (slot-value self 'font))
  (unless row-title-font (setq row-title-font "8x13bold"))
  (when col-titles
	(when title-font (setq col-title-font title-font))
	(unless col-title-font (setq col-title-font "8x13bold"))
	(when (and title-height (null col-title-height))
	      (setq col-title-height title-height))
	(unless col-title-height 
		(unless (get-font col-title-font)
			(make-font :name col-title-font :attach-p t))
		(setq col-title-height 
		      (+ 15 (font-height col-title-font)))))
  (when row-titles
	(when title-font (setq col-title-font title-font))
	(unless row-title-font (setq row-title-font "8x13bold"))
	(when (and title-width (null row-title-width))
	      (setq row-title-width title-width))
	(unless row-title-width 
		(unless (get-font row-title-font)
			(make-font :name row-title-font :attach-p t))
		(setq row-title-width
		      (apply #'max
			     (mapcar
			      #'(lambda (title)
					(cond ((stringp title)
					       (+ (text-width 
						   title :font row-title-font) 
						  15))
					      ((image-p title) 
					       (+ (width title) 15))
					      (t 15)))
			      row-titles)))))

  (cond ((not rfp)
	 (setq return-func
	       `(mf-advance-current ',self))
	 (setf (slot-value self 'return-func) return-func))
	((null return-func) nil)
	(t
	 (setq return-func 
	       `(let ((self ',self))
		     ,return-func))))

  ;;	Calculate how many rows/cols to comprise matrix
  (setq rows 
	(cond (field-table (array-dimension field-table 0))
	      (rows)
	      (row-titles (length row-titles))
	      ((consp row-heights) (length row-heights))
	      (row-elements (length row-elements))
	      (data (min 10 (num-elements data)))
	      (initial-rows)
		 ;;	Default to 3 if absolutely necessary
	      (t (warn "Unable to calculate how many rows to comprise matrix")
		 3)))
  (setq cols
	(cond (field-table (num-cols field-table))
	      (cols)
	      (titles (length titles))
	      (col-titles (length col-titles))
	      ((consp col-widths) (length col-widths))
	      (col-elements (length col-elements))
	      (data (min 10 (num-cols data)))
	      (initial-cols)
		 ;;	Default to 3 if absolutely necessary
	      (t (warn "Unable to calculate how many cols to comprise matrix")
		 3)))

  (cond (region
	  (setq width (third region)
		height (fourth region)))
	(size
	  (setq width (car size)
		height (cadr size))))

  (when titles
	(setq col-titles titles))

  (unless data-rows 
	  (setf (slot-value self 'data-rows)
		(setq data-rows (num-elements data))))
  (unless data-cols 
	  (setf (slot-value self 'data-cols)
		(setq data-cols (num-cols data))))

  (if just
      (setq horiz-just just))

  ;;	Make data into array if necessary
  (when (null data)
	(setq data (make-array (list rows cols)))
	(setf (slot-value self 'data) data))
  (when (and data (listp data))
	(if data-array-size
	    (setq r (car data-array-size)
		  c (cadr data-array-size))
	    (setq r (max (num-elements data) rows)
		  c (max (num-cols data) cols)))
	(setq data (make-array (list r c)
			       :initial-contents (fill-2d-list data r c)))
	(setf (slot-value self 'data) data))

  ;;	Better way of determining initial-rows/cols?
  (unless initial-rows
	  (setq initial-rows (min rows 10)))
  (unless initial-cols 
	  (setq initial-cols (min cols 10)))

  ;;	Make col-widths & row-heights lists of appropriate length
  (when (and col-elements (null col-widths))
	(setq cw (if (and col-widths (atom col-widths)) 
		     col-widths 
		     50))
	(setq col-widths nil)
	(dolist (cur col-elements)
		(unless (keywordp (car cur))
			(setq cur (cdr cur)))
		(push (if (setq l (getf cur :base-width))
			  (progn
			   (remf cur :base-width)
			   l)
			  cw)
		      col-widths))
	(setq col-widths (nreverse col-widths))
	(when (> (length col-widths) 1)
	      (setf (uni-col-types self) nil))
	(setq cols (max cols data-cols)))
  (cond ((consp col-widths) (setq cols (max data-cols (length col-widths))))
	((null col-widths) (setq col-widths 100)))
  (cond ((atom col-widths)
	 (setq col-widths (list col-widths)) 
	 (setq uni-cols t) 
	 (setf (uniform-cols self) t)) 
	((= (length col-widths) 1)
	 (setq uni-cols t) 
	 (setf (uniform-cols self) t)))
  (when	(< (length col-widths) cols)
	(setq l (make-list (- cols (length col-widths))
			   :initial-element (car (last col-widths))))
	(setq col-widths (append col-widths l)))

  (when (and row-elements (null row-heights))
	(setq rh (if (atom row-heights) row-heights (car row-heights)))
	(setq row-heights nil)
	(dolist (cur row-elements)
		(unless (keywordp (car cur))
			(setq cur (cdr cur)))
		(push (if (setq l (getf cur :base-height))
			  (progn
			   (remf cur :base-height)
			   l)
			  rh)
		      row-heights))
	(setq row-heights (nreverse row-heights))
	(when (> (length row-heights) 1)
	      (setf (uni-row-types self) nil))
	(setq rows (max rows data-rows)))
  (cond ((consp row-heights) (setq rows (max data-rows (length row-heights))))
	((null row-heights) (setq row-heights 40)))
  (cond ((atom row-heights)
	 (setq row-heights (list row-heights))
	 (setq uni-rows t)
	 (setf (uniform-rows self) t)) 
	((= (length row-heights) 1)
	 (setq uni-rows t)
	 (setf (uniform-rows self) t))) 
  (when	(< (length row-heights) rows)
	(setq l (make-list (- rows (length row-heights))
			   :initial-element (car (last row-heights))))
	(setq row-heights (append row-heights l)))

  ;;	Set-up geometry-manager
  (gm-matrix-init self :rows rows :cols cols 
		  :inter-row-pad inter-row-pad :inter-col-pad inter-col-pad
		  :conform self-adjusting)

  ;;	Get field-table 
  (setq field-table (md-children (gm-data self)))

  ;;	Set-up revert table
  (when reverts-p
	(setf (revert-table self) (copy-matrix-detail (gm-data self))
	      (revert-data self) 
	      (if (typep (setq data (data self)) 'portal)
		  (copy-1d-array (slot-value data 'array)) 
		  (copy-2d-array data))))

  ;;	Set-up initial offsets
  (setq x-init (if (zerop col-index) 
		   inter-col-pad
		   (let ((res 0)
			 (cw col-widths))
			(dotimes (col-index res)
				 (setq res (- res (car cw)))
				 (setq cw (if (cdr cw) (cadr cw) cw))))))
  (setq y-init (if (zerop row-index)
		   inter-row-pad
		   (let ((res 0)
			 (rh row-heights))
			(dotimes (row-index res)
				 (setq res (- res (car rh)))
				 (setq rh (if (cdr rh) (cadr rh) rh))))))

  ;;	Create default titles
  (unless (or col-titles (not default-titles) 
	      (not (typep data 'portal)))
	  (setq col-titles (mapcar #'(lambda (x) (string (clos::slotd-name x)))
			       (clos::class-slots (clos::class-of 
						  (element data 0))))))

  ;;	Coerce row-titles into a list of appropriate length and then 
  ;;	make a new matrix for the row-titles
  (when row-titles
	(when (atom row-titles)
	      (setq row-titles (list row-titles)))
	(setq temp (- (max rows data-rows) 
		      (length row-titles)))
	(do ((i 0 (1+ i)))
	    ((>= i temp))
	    (setq row-titles (append row-titles '(nil))))
	(setq row-titles (mapcar #'list row-titles))
	(setf (slot-value self 'row-titles)
	      (setq row-titles
		    (make-matrix-field :col-index 0 :row-index row-index 
				       :select-func (or
						     row-title-select-func
						     title-select-func)
				       :data row-titles :rows rows :cols 1
				       :border-width 0
				       :row-elements row-title-elements
				       :col-widths row-title-width
				       :row-heights (if uni-rows 
							(car row-heights)
							row-heights)
				       :default-titles nil
				       :selection (if row-title-selectable
						   :row nil)
				       :font row-title-font
				       :editable editable-row-titles
				       :initial-rows initial-rows))))
  ;;	Do same for col-titles. . .
  (when col-titles
	(when (atom col-titles)
	      (setq col-titles (list col-titles)))
	(setq temp (- (max cols data-cols) 
		      (length col-titles)))
	(do ((i 0 (1+ i)))
	    ((>= i temp))
	    (append col-titles '(nil)))
	(setq col-titles (list col-titles))
	(setf (slot-value self 'col-titles) 
	      (setq col-titles
		    (make-matrix-field :row-index 0 :col-index col-index
				       :select-func (or
						     row-title-select-func
						     title-select-func)
				       :border-width 0
				       :data col-titles :cols cols :rows 1
				       :col-elements col-title-elements
				       :row-heights col-title-height
				       :col-widths (if uni-cols (car col-widths)
						       col-widths)
				       :default-titles nil
				       :selection (if col-title-selectable
						   :column nil)
				       :font col-title-font
				       :editable editable-col-titles
				       :initial-cols initial-cols))))

  ;;	Create field-table entries
  (setq x x-init 
	y y-init
	c 0
	r 0)
  ;;	If row/col-elements are specified, use those, else make generic widget
  (setq md (gm-data self))
  (cond (col-elements 
	 (defun mf-notify-matrix (field &rest args)
		(mf-propagate-field self field)
		(mf-dispatch-event field args))
	     (dolist (cw col-widths)
		     (setq cur (car col-elements))
		     (unless (keywordp (car cur))
			     (setq unselectable (getf (cdr cur) :unselectable))
			     (remf (cdr cur) :unselectable))
		     (cond ((keywordp (car cur)) 
			    (if (getf cur :editable)
				(setq nomad-widget (list 0 c)))
			    (unless (getf cur :font)
				    (setf (getf cur :font) (font self)))
			    (if (member :return-func cur)
				(setq return-func
				      (getf cur :return-func)))
			    (dolist (rh row-heights)
				    (push
				     (setf (aref field-table r c)
					   (list :exposed nil
						 :x x
						 :y y
						 :width cw 
						 :height rh
						 :gc nil
						 :geom-spec
						 (list r c)
						 :editable (getf cur :editable)
						 :return-func return-func
						 :horiz-just horiz-just
						 :vert-just vert-just
						 :gc-info cur))
				     (synths self))
				    (setq y (+ y rh inter-row-pad)
					  r (1+ r))))
			   (t 
			    (dolist (rh row-heights)
				    (setq val (aref data r c)) 
				    (setq ef 
					  (eval
					   (append `(,(car cur) 
						      :base-width ,cw
						      :base-height ,rh 
						      :geom-spec '(,r 
								   ,c) 
						      :value ',val
						      :x-offset ,x
						      :y-offset ,y
						      :parent ',self)
						   `,(cdr cur)
						   `(:base-width 
						     ,cw 
						     :base-height ,rh
						     :geom-spec '(,r ,c)
						     :value ',val
						     :x-offset ,x
						     :y-offset ,y))))
				    (when unselectable
					  (setf (geom-spec ef)
						(nconc (geom-spec ef)
						       (list :unselectable))))
				    (setq y (+ y rh inter-row-pad)
					  r (1+ r))
				    
				    ;;	Recursively toggle event-processing 
				    (if (x-window-p ef)
					(if (mf-selectable-widget ef) 
					    (setf (event-mask ef) 
						  (remove :button-pressed 
							  (event-mask ef)))
					    (progn
					     (setq flag 'cols)
					     (mf-rec-init-events
					      ef #'mf-notify-matrix)))))))
		     (setq x (+ x c inter-col-pad)
			   c (1+ c)
			   y y-init
			   r 0)
		     (setq col-elements (if (cdr col-elements) 
					    (cdr col-elements) 
					    col-elements)))
	 ;;	Do automatic propogation
	 (setf (auto-prop self) t))

	(row-elements 
	 (defun mf-notify-matrix (field &rest args)
		(mf-propagate-field self field)
		(mf-dispatch-event field args))
	     (dolist (rh row-heights)
		     (setq cur (car row-elements))
		     (unless (keywordp (car cur))
			     (setq unselectable (getf (cdr cur) :unselectable))
			     (remf (cdr cur) :unselectable))
		     (cond ((keywordp (car cur)) 
			    (if (getf cur :editable)
				(setq nomad-widget (list r 0)))
			    (unless (getf cur :font)
				    (setf (getf cur :font) (font self)))
			    (if (member :return-func cur)
				(setq return-func
				      (getf cur :return-func)))
			    (dolist (cw col-widths)
				    (push
				     (setf (aref field-table r c)
					   (append (list :exposed nil
							 :x x
							 :y y
							 :width cw 
							 :height rh
							 :gc nil 
							 :editable 
							 (getf cur :editable) 
							 :horiz-just horiz-just 
							 :vert-just vert-just
							 :geom-spec
							 :return-func 
							 return-func
							 (list r c)
							 :gc-info cur)))
				     (synths self))
				    (setq x (+ x cw inter-col-pad)
					  c (1+ c))))
			   (t 
			    ;;	See if it's a gadget 
			    (setq temp (string-downcase (car cur))) 
			    (setq temp (subseq temp (- (length temp) 6)))
			    (dolist (cw col-widths) 
				    (setq val (aref data r c))
				    (setq ef 
					  (eval
					   (append 
					    `(,(car cur) 
					       :base-width ,cw
					       :base-height ,rh 
					       :geom-spec '(,r ,c) 
					       :value ',val
					       :x-offset ,x
					       :y-offset ,y
					       :parent ',self)
					    `,(cdr cur)
					    `(:base-width 
					      ,cw 
					      :base-height ,rh
					      :geom-spec '(,r ,c)
					      :value ',val
					      :x-offset ,x
					      :y-offset ,y))))
				    (when unselectable
					  (setf (geom-spec ef) 
						(nconc (geom-spec ef)
						       (list :unselectable))))
				    (setq x (+ x cw inter-col-pad)
					  c (1+ c)))
			    
			    ;;	Recursively toggle event-processing
			    (if (x-window-p ef)
				(if (mf-selectable-widget ef) 
				    (setf (event-mask ef) 
					  (remove :button-pressed 
						  (event-mask ef)))
				    (progn
				     (setq flag 'rows)
				     (mf-rec-init-events
				      ef #'mf-notify-matrix))))))
		     (setq y (+ y r inter-row-pad)
			   r (1+ r)
			   x x-init
			   c 0)
		     (setq row-elements (if (cdr row-elements)
					    (cdr row-elements)
					    row-elements)))
	 ;;	Do automatic propogation
	 (setf (auto-prop self) t))

	(t
	 (if editable 
	     (setq nomad-widget (list 0 0)))
	 (dotimes (r rows)
		  (dotimes (c cols) 
			   (setf (aref field-table r c)
				      (list :exposed nil 
					    :x 0
					    :y 0
					    :width 0 
					    :height 0
					    :editable editable 
					    :return-func return-func
					    :horiz-just horiz-just 
					    :vert-just vert-just
					    :geom-spec (list r c)))))))

  (when fonts (setf (fonts self) fonts))

  ;;	Set data-rows/cols to be equal to rows/cols if flag
  (case flag
	('rows (setf (data-rows self) (num-elements data)))
	('cols (setf (data-cols self) (num-cols data))))

  ;;	Set-up nomad-widget
  (when nomad-widget
	(setq temp nomad-widget)
	(setq nomad-widget
	      (apply #'aref (cons field-table nomad-widget))) 
	(setf (slot-value self 'nomad-widget)
	      (apply (nomad-creator self)
		     (list :parent self :status :concealed
			   :border-width 0 :geom-spec temp)))
	(setf (aref field-table (car temp) (cadr temp))
	      nomad-widget))

  ;; 	Set current row/col stuff in gm-data for matrix and titles
  (setf (md-row-current md) (make-array rows :initial-contents row-heights)
	(md-col-current md) (make-array cols :initial-contents col-widths)) 
  (when row-titles
	(setf (md-row-current (gm-data row-titles)) (md-row-current md)))
  (when col-titles
	(setf (md-col-current (gm-data col-titles)) (md-col-current md)))

  ;;	Set height/width to accomodate initial-rows/cols
  (if (not base-width)
      (setf (base-width self) (apply #'min col-widths)))
  (if (not base-height)
      (setf (base-height self) (apply #'min row-heights)))

  (setf (height self)
	(if height
	    height
	    (apply #'+ (cons (* inter-row-pad (1+ rows))
			     (butlast (nthcdr row-index row-heights) 
				      (- rows initial-rows))))))
  (setf (width self)
	(if width
	    width
	    (apply #'+ (cons (* inter-col-pad (1+ cols)) 
			     (butlast (nthcdr col-index col-widths) 
				      (- cols initial-cols))))))

  ;;	Cache scrolling functions
  (if uni-cols
      (setf (left-func self) #'mf-uni-scroll-left
	    (right-func self) #'mf-uni-scroll-right)
      (setf (left-func self) #'mf-var-scroll-left
	    (right-func self) #'mf-var-scroll-right))
  (if uni-rows
      (setf (up-func self) #'mf-uni-scroll-up
	    (down-func self) #'mf-uni-scroll-down)
      (setf (up-func self) #'mf-var-scroll-up
	    (down-func self) #'mf-var-scroll-down))

  ;;	If unique-selection, alter event-map
  (if unique-selection
      (register-callback self #'matrix-field-select-unique :button-press))

  ;;	Return instance
  self)
