;;;
;;; Copyright (c) 1990 Regents of the University of California
;;; 
;;; $Author: bsmith $
;;; $Source: /pic2/picasso/new/widgets/table/RCS/browse-widget.cl,v $
;;; $Revision: 1.1 $
;;; $Date: 1991/08/04 19:09:11 $
;;;

(in-package "PT")

;;;
;;;  A browse-widget is a complex widget that allows the user to browse
;;;  through a list of objects that have a hierarchial arrangemnet.  Consider,
;;;  for example, a list of objects that have department, course and section
;;;  slots.  Assume there are readers for all these slots, named dept, course 
;;;  and section, respectively.  If a browse-widget is created with:
;;;  
;;;  	(make-browse-widget :data <list> 
;;;  			    :sort-keys '(("Department".#'dept)
;;;  					 ("Course".#'course)
;;;  					 ("Section".#'section)))
;;;  
;;;  the resulting widget will consist of 3 tables, the first of which displays
;;;  a department, with the second and third blank.  When the user selects a
;;;  department, the second table fills in with a list of all the course names
;;;  within that department.  If the user selects a course, the third table 
;;;  lists all the sections of that course.  In this way, a user may browse a
;;;  hierarcial data structure.
;;;
;;;  In terms of creating one of these puppies, the data should be a list
;;;  of objects, and the sort keys sould be a list of cons cells; the car of
;;;  each cell is the column title, and the cdr is the accessor function.
;;;
;;;  The "selection" of a browse widget is the maximal list of selected
;;;  items.  Thus, if the "CS" department is selected, all courses in this
;;;  department are selected.  The current-selection is the list of
;;;  individual items that have been fully specified.  This is equivalent
;;;  to the items selected in the last column.
;;;

;;; ================================================================
;;;
;;; bw-sync class -- class object to hold synchronization code for
;;; browse-widgets.
;;;
(defclass bw-sync (pmc)
  ((rest-list :initform nil :type list :accessor rest-list)
   (key-func :initform nil :type t :accessor key-func)
   (table :initform nil :type t :accessor table)
   (browser :initform nil :type t :accessor browser)
   (next :initform nil :type t :accessor next)))

;;;
;;; Synchronize the rest of the tables that depend on this one.
;;;
(defmethod (setf rest-list) (rl (self bw-sync))
  (unless (equalp rl (rest-list self))
	  (setf (slot-value self 'rest-list) rl)
	  (let* ((kf (key-func self))
		 (tbl (table self))
		 (mf (matrix-field tbl))
		 (next (next self))
		 (str "")
		 (val nil))
		;; Extract out the unique items of the key into val
		(dolist (obj rl)
			(setq str (funcall kf obj))
			(if (not (member str val :test #'equalp))
			    (push str val)))
		(setq val (sort val #'string<))
		;; Set the data of the table to unique items and sync slider
		(setf (value tbl) val
		      (current-indices mf) nil
		      (row-index mf) 0)
		(if next (setf (rest-list next) nil)))))

(defun restrict-to (rl kf kv)
  (if kv
      (remove-if-not  #'(lambda (x) (member (funcall kf x) kv :test #'string=))
		      rl)
      nil))

;;; ================================================================
;;;
;;; browse-widget class
;;;
(defclass browse-widget (collection-widget)
  ((name :initform "A Browser")
   (gm :initform 'anchor-gm)
   (data :initarg :data :initform nil :type list :accessor data)
   (selection :initarg :selection :initform nil :type list :accessor selection)
   (current-selection :initarg :current-selection :initform nil :type list :accessor current-selection)
   (sort-keys :initarg :sort-keys :initform nil :type t :accessor sort-keys)
   (event-mask 
    :initform '(:exposure :button-press))
   (sync-array :initform nil :type array :accessor sync-array)))

(defun make-browse-widget (&rest keys)
  (apply #'make-instance 'browse-widget :allow-other-keys t keys))

(defmethod (setf data) (value (self browse-widget))
  (unless (equalp value (data self))
	  (setf (slot-value self 'data) value)
	  (let ((syncer (aref (sync-array self) 0)))
	       (when syncer
		     (setf (rest-list syncer) value)
		     (setf (selection self) nil)
		     (setf (current-selection self) nil)))))

;;;
;;; Called when the curr-index of a table for a bw-sync has changed.
;;;
(defun curr-idx-changed (self)
  (let* ((new-rl nil)
	 (rl (rest-list self))
	 (browser (browser self))
	 (kf (key-func self))
	 (kv nil)
	 (tbl (table self))
	 (mf (matrix-field tbl))
	 (curr (current-indices mf))
	 (val (value tbl))
	 (next (next self)))
	;; Get the key-val if there is a selection...
	(dolist (i curr)
		(push (aref val i 0) kv))
	;; Propagate to next table
	(setq new-rl (restrict-to rl kf kv))
	(if next 
	    (progn
	     (setf (rest-list next) new-rl)
	     (setf (current-selection browser) nil))
	    (setf (current-selection browser) new-rl))
	(setf (selection browser) new-rl)))

;;;
;;;	New-instance method
;;;
(defmethod new-instance ((self browse-widget)
			 &key 
			 (title-font (make-font))
			 (col-widths nil)
			 (font (make-font))
			 (sort-keys nil)
			 (data nil)
			 &allow-other-keys)
  
  (call-next-method)

  ;; Compute/adjust column widths
  (if col-widths
      (let ((sum (apply #'+ col-widths)))
	   (setq col-widths (mapcar #'(lambda (x) (/ x sum)) col-widths)))
      (let ((sum (apply #'+ (mapcar #'(lambda (x) (length (car x)))
				    sort-keys))))
	   (setq col-widths (mapcar #'(lambda (x) (/ (length (car x)) sum))
				    sort-keys))))

  (let* ((num-tables (length sort-keys))
	 (array-dims (list num-tables))
	 (x 0)
	 (w 0)
	 (sync-array (make-array array-dims))
	 (prev-syncer nil)
	 (syncer nil)
	 (tbl nil)
	 (tg nil)
	 (cw nil)
	 (key nil))
	(setf (sync-array self) sync-array)
	(dotimes (i num-tables)
		 (setf w (car col-widths)
		       col-widths (cdr col-widths)
		       key (elt sort-keys i)
		       cw (make-collection-gadget :gm 'packed-gm
						  :parent self
						  :geom-spec 
						  (list 
						   x 0 w 1 
						   :arrow '(:horiz :vert)))
		       tg (make-text-gadget :parent cw
					    :value (car key)
					    :self-adjusting t
					    :geom-spec :top
					    :font title-font)
		       syncer (make-instance 'bw-sync)
		       (aref sync-array i) syncer
		       tbl (make-list-box
			    :parent cw
			    :grid-lines t
			    :just :left
			    :geom-spec '(:bottom 0)
			    :font font)
		       (slot-value syncer 'rest-list) nil
		       (browser syncer) self
		       (key-func syncer) (cdr key)
		       (select-func (matrix-field tbl)) `(bw-select ',syncer)
		       (table syncer) tbl)
		 (if prev-syncer (setf (next prev-syncer) syncer))
		 (setq prev-syncer syncer)
		 (incf x w)))
  (setf (slot-value self 'data) nil)
  (setf (data self) data)
  self)

(defun bw-select (syncer)
  (curr-idx-changed syncer))
