;;;
;;; Copyright (c) 1991 Regents of the University of California
;;; 
;;; $Author: bsmith $
;;; $Source: /pic2/picasso/new/widgets/sound/RCS/sound-widget.cl,v $
;;; $Revision: 1.1 $
;;; $Date $
;;;

(in-package "PT")

;;;
;;;	A sound-widget is a widget for displaying a sparcstation
;;;	audio file in a two dimensional graph.  The x coordinate
;;;	is given in seconds of sound data.
;;;
;;;	Sample usage:
;;;	(setq audio-data (read-audio "/usr/demo/SOUND/sounds/sample.au"))
;;;	(make-sound-widget :base-size '(200 200) 
;;;			   :domain (cons 0 50)
;;;			   :x-label "Time" :y-label "Audio" 
;;;			   :value audio-data 
;;;			   :color "green")
;;;

(defclass sound-widget (widget)
  ((color			; To draw data in
    :initarg :color                  
    :initform "black"
    :accessor color)
   (start-time
    :initarg :start-time
    :initform 0
    :reader start-time)
   (pps			;; Number of pixels per second on x axis
    :initarg :pps
    :initform 100
    :reader pps)
   (selection
    :initarg :selection
    :initform nil
    :reader selection)
   ;---------------------------------------------------------------
   ; Internal use only
   ; Graphic contexts used by sound-widgets
   (gc-spec :initform '((gc-res "default")
			(highlight-gc (:foreground "red"))))
   (highlight-gc :initform nil :type vector :reader highlight-gc)

   ; Overide some defaults
   (name :initform "A Sound-Widget")
   (background :initform "gray50")
   (event-mask                  ; The event mask for this widget.
    :initform '(:exposure :button-motion :button-press :button-release)
    :type list)
   (cursor :initform (default-cursor))
   (border-width :initform 0)))

(defun make-sound-widget (&rest keys)
  (apply #'make-instance 'sound-widget :allow-other-keys t keys))

(defmethod (setf selection) (val (self sound-widget))
  (setf (slot-value self 'selection) val)
  (repaint self))

(defmethod (setf start-time) (val (self sound-widget))
  (setf (slot-value self 'start-time) val)
  (repaint self))

(defmethod (setf pps) (val (self sound-widget))
  (setf (slot-value self 'pps) val)
  (repaint self))

(defmethod do-attach ((self sound-widget))
  (call-next-method)
  (if (stringp (color self))
      (setf (slot-value self 'color) (make-color :name (color self))))
  (attach (color self))
  (setf (xlib:gcontext-foreground (gc-res self)) (pixel (color self))))

(defmethod do-detach ((self sound-widget))
  (call-next-method)
  (when (color-p (color self))
	(detach (color self))
	(setf (slot-value self 'color) (name (color self)))))

;;;
;;;	Convert sound data to point data
;;;

(defmethod do-repaint ((self sound-widget)
		       &key 
		       &allow-other-keys)
  (let* ((h (height self))
	 (w (width self))
	 (sound (value self))
	 (len (sound-length sound))
	 (data (sound-data sound))
	 (sample-rate (sound-sample-rate sound))
	 (bytes-per-unit (sound-bytes-per-unit sound))
	 (channels (sound-channels sound))
	 (samples-per-unit (sound-samples-per-unit sound))
	 (C (/ (* channels bytes-per-unit sample-rate) samples-per-unit))
	 (res (res self))
	 (gc (gc-res self))
	 (hgc  (highlight-gc self))
	 x1 y1 y2 start end inc
	 (start-time (start-time self))
	 (pps (pps self))
	 (selection (selection self))
	 (highlight-start (if selection 
			      (round (* pps (- (car selection) start-time)))
			      1))
	 (highlight-end (if selection
			    (round (* pps (- (cdr selection) start-time)))
			    0))
	 )
	(setq start (round (* start-time C)))
	(setq end (+ start (* (/ w pps) C)))
	(setq end (min end len))
	(setq inc (round (* (/ 1 pps) C)))
	;; Clear the background.
	(clear-region self 0 0 w h)
	(setq x1 0)
	(setq y1 (ash (* h (+ 32767 (sound-to-int (aref data start)))) -16))
	(do* ((i (+ inc start) (+ i inc))
	      (x2 1 (1+ x2)))
	     ((>= i end))
	     (setq y2 
		   (ash (* h (+ 32767 (sound-to-int (aref data i)))) -16))
	     (if (<= highlight-start x1 highlight-end)
		 (xlib:draw-line res hgc x1 y1 x2 y2)
		 (xlib:draw-line res gc x1 y1 x2 y2))
	     (setq x1 x2 y1 y2))))

(defun find-sound-time (self x)
  (let* ((start-time (start-time self))
	 (pps (pps self)))
	(float (+ start-time (/ x pps)))))

(defun find-sound-index (self time)
  (let* ((sound (value self))
	 (sample-rate (sound-sample-rate sound)))
	(round (* time sample-rate))))

(defhandler select ((self sound-widget) &rest args &key x y &allow-other-keys
		    &default (:button-press :detail :left-button))
  (declare (ignore args))
  (let* ((x1 x)
	 (x2 (get-region self x y :cursor (default-cursor))))
	(setf (selection self)
	      (cons (find-sound-time self x1) (find-sound-time self x2))))
  nil)

(defhandler play ((self sound-widget) &rest args
		  &default (:button-press :detail :right-button))
  (declare (ignore args))
  (let ((selection (selection self)))
       (play-sound (value self) 
		   :start (find-sound-index self (car selection))
		   :end (find-sound-index self (cdr selection))))
  nil)

(defhandler drop-on-floor ((self sound-widget) &rest args
			   &default ((:button-release)
				     (:pointer-motion)))
  (declare (ignore self args))
  nil)

(defmethod zoom-extent ((self sound-widget))
  (let* ((w (width self))
	 (sound (value self))
	 (len (sound-length sound))
	 (sample-rate (sound-sample-rate sound)))
	(setf (pps self) (round w (/ len sample-rate)))))

#|
;; some sample code I assume
(setq sd (read-sound "/usr/demo/SOUND/sounds/sample.au"))
(setq sd (read-sound "/usr/demo/SOUND/sounds/splat.au"))
(setq sw (make-sound-widget :value sd :parent (root-window)
			    :base-size '(200 100)))
(zoom-extent sw)
(attach sw)
|#
