(in-package clos)
(export '(dbclass-cache-size
          object-cache-size
          *dbmode-list*
          *default-mode*
          *current-mode*
          dbclass
          direct-update
          deferred-update
          local-copy
          touch
          dbobject
          make-instance
          dbobject-p
          store-dbobject
          store-dbobject-to-db
          fetch-dbobject
          mfetch-dbobject
          reinitialize-dbobject)  
        'clos)


(in-package clos)

(defvar *dbclass-cache-size* 1024)

(defvar *dbclass-cache* (make-hash-table :size *dbclass-cache-size*
                                         :test #'equal))

(defvar *object-cache-size* 8192)

(defvar *cache-full* (/ *object-cache-size* 2))

(defvar *cache-count* 0
  "the number of entries in the hashtable")

(defvar *object-cache* (make-hash-table :size *object-cache-size*
                                          :test #'equal))

;;;
;;; UPDATING-MODES
;;; Define the updating modes for instances of shared classes (or dbobjects).
;;;     direct-update: all updates to the object are flushed to DB immdiately
;;;     deferred-update: updates remain in the cache until store-dbobject
;;;     local-copy: should be used only in the cache, but can be stored
;;;                 by store-dbobject.
;;;

(defvar *dbmode-list* '(direct-update local-copy deferred-update))
(defvar *default-mode* 'deferred-update)

;;;
;;; The *current-mode* is useful for composite objects.
;;; When a composite object is created, if any of its slots
;;; is another dbobject, then it should be created with
;;; the same mode instead of the *default-mode*.
;;; However, since the arguments to a function is evaluated
;;; before the function, the user may have to set this
;;; variable to get the correct behavior.
;;; For example, in (make-instance 'foo :a (make-instance 'bar)),
;;; the instance of bar is created first.
;;;
;;; When a composite object is fetched from the database,
;;; we are able to set this variable before fetching the slots.
;;;
(defvar *current-mode* *default-mode*)

(defvar *max-objid* 2147483648 ) ;; 2^31 breaks

;;;
;;; DBCLASS CREATION MODES
;;; When a dbclass is defined, if a class with the same name exists in the
;;; database, creation mode decides the behavior of DEFDBCLASS.
;;; Four creation modes are supported:
;;;     remove-create: if definition different, remove the one in DB, and
;;;                    create a new one.
;;;     rename-create: if definition different, rename the one in DB to
;;;                    class_old, and create a new one.
;;;     always-create: always remove the old one from DB and create a new one.
;;;     never-recreate: do not change the one in DB.
;;;

(defvar *legal-creation-modes*
  '(always-create rename-create remove-create never-recreate from-database))

(defvar *default-creation-mode* 'remove-create)







(defmacro setf-dbclass-cache-size (size)
  `(setf *dbclass-cache-size* ,size))

(defsetf dbclass-cache-size setf-dbclass-cache-size)

(defmacro put-into-dbclass-cache (class relid)
  `(setf (gethash ,relid *dbclass-cache*) ,class))

(defmacro cached-dbclass (relid &optional no-error-p)
  `(or (gethash ,relid *dbclass-cache*)
       (if ,no-error-p
           nil
         (error "The dbclass with relid ~S is not in cache." ,relid))))

(defmacro setf-object-cache-size (size)
  `(progn
     (setf *object-cache-size* ,size)
     (setq *cache-full* (/ *object-cache-size* 2))))

(defsetf object-cache-size setf-object-cache-size)

;;;
;;; GET-DBOBJECT
;;; Returns the real dbobject given a handle:
;;; fetches it from the database if it is not already in the cache.
;;;

(defmacro get-dbobject (handle &optional no-error-p)
  "Returns the real dbobject given its objid"
  `(or (handle-instance ,handle)
       (handle-instance (fetch-dbobject ,handle nil ,no-error-p))))

(defmacro illegal-mode (dbmode)
  "Return T if dbmode is not one of the legal update modes"
  `(or (null ,dbmode)
       (null (member ,dbmode *dbmode-list*))))

;;;
;;; Check if a handle with the objid already exists.
;;; If it does exist, it must be in the cache.
;;;

(defmacro handle-exist-p (objid)
  "Return T if objid (a list) is in *object-cache*, NIL otherwise"
  `(gethash ,objid *object-cache*))

(defmacro get-handle (objid)
  `(gethash ,objid *object-cache*))

(defmacro record-modification (handle)
  `(progn
     (handle-set-modified ,handle t)
     (if (eq (handle-mode ,handle) 'direct-update)
         (store-dbobject ,handle))))

(defmacro sohid (self)
  ;; Must access sohid via "front door" so it can be trapped to objid
  `(slot-value ,self 'sohid))


;;; macros used for cling and not libpq 

(defmacro cling-slot-update-list (slots)
    `(mapcar #'(lambda (slot-value) (cons '= slot-value)) ,slots))

(defmacro cling-slot-name-list (slots)
    `(mapcar #'first ,slots))

(defmacro cling-slot-value-list (slots)
    `(mapcar #'second ,slots))

;;;
;;; DEFDBCLASS
;;; The definition of a dbclass is in the same form of a private class
;;; except there are two more options.
;;;     :creation-mode  -- could be any of the legal mode defined above.
;;;     :local-slots    -- a list of slots not to be stored in database.
;;; The :metaclass option can only be dbclass or defaulted.
;;;
;;; The options are processed so the :creation-mode and :local-slots
;;; are taken out. The arguments are them passed to the macro DEFCLASS
;;; to define the class in memory.
;;;
;;; Then the function ADD-DBCLASS is called to add the class
;;; definition to the database.  The :local-slots are stored as
;;; the first element in the options list.
;;; It is also used to set the slot soh-local-slots in the dbclass object.
;;;
(defmacro defdbclass (name includes slots &rest options)
  (let ((local-slots nil)
        (creation-mode *default-creation-mode*))
    (dolist (option options)
            (if (not (listp option))
                (error "~S is not a legal defdbclass option." option)
              (cond ((eq (car option) ':metaclass)
                     (unless (eq (cadr option) 'dbclass)
                             (error "The value of the :metaclass option~%~
                                     must be 'DBCLASS."))
                     (setq options (remove option options)))
                    ((eq (car option) ':local-slots)
                     (unless (listp option)
                             (error "The value of the :local-slots option~%~
                                     must be a list."))
                     (setq local-slots (cadr option))
                     (setq options (remove option options)))
                    ((eq (car option) ':creation-mode)
                     (if (illegal-creation-mode (cadr option))
                         (error "The value of the :creation-mode option~%~
                                 is illegal."))
                     (setq creation-mode (cadr option))
                     (setq options (remove option options))))))
    (setq options `((:metaclass dbclass)
                    ,@options))
    `(progn
       (defclass ,name ,includes ,slots ,@options)
       (add-dbclass ',name ',includes ',slots ',options
                    ',creation-mode ',local-slots))))









(defclass dbclass (standard-class)
  ()
  (:metaclass standard-class))

(defclass dbobject (object)
  ()
  (:metaclass dbclass))



(defgeneric all-std-class-readers-miss-1 (self wrapper slot-name))

(defgeneric check-super-metaclass-compatibility (self super))

(defgeneric dbclass-no-of-handle-slots (self))

(defgeneric dbclass-relid (self))
(defgeneric (setf dbclass-relid) (value self))

(defgeneric dbobject-p (self))

(defgeneric delete-dbobject (self &optional no-error-p))

(defgeneric do-db-delete (self))

(defgeneric fetch-dbobject (self &rest the-rest))

(defgeneric lookup-pv-miss-1 (self slots pv))

(defgeneric make-reader-method-function (self slotd))

(defgeneric make-writer-method-function (self slotd))

(defgeneric mfetch-dbobject (self &rest the-rest))

(defgeneric reinitialize-dbobject (self))

(defgeneric soh-local-slots (self))
(defgeneric (setf soh-local-slots) (value self))

(defgeneric store-dbobject (self &optional store-components-p))
