;;;
;;; Shared Object Hierarchy
;;;
;;; Copyright (c) 1986 Regents of the University of California
;;; 
;;; Permission to use, copy, modify, and distribute this software and its
;;; documentation for any purpose and without fee is hereby granted,
;;; provided that the above copyright notice appear in all copies and
;;; that both that copyright notice and this permission notice appear in
;;; supporting documentation, and that the name of the University of
;;; California not be used in advertising or publicity pertaining to
;;; distribution of the software without specific, written prior
;;; permission.  The University of California makes no representations
;;; about the suitability of this software for any purpose.  It is
;;; provided "as is" without express or implied warranty.
;;; 
;;; $Author: bsmith $
;;; $Source: /pic2/picasso/new/widgets/soh/RCS/defdbclass.cl,v $
;;; $Revision: 1.1 $
;;; $Date: 1991/08/04 19:08:49 $
;;;

(in-package clos)

(use-package 'libpq 'cling)


(defun illegal-creation-mode (creation-mode)
  (or (null creation-mode)
      (null (member creation-mode *legal-creation-modes*))))

	      
#|
;;;
;;; When we bring in the definition of a dbclass from the database, we need to
;;; call add-named-class to make an in-core definition.  However, we cannot use
;;; the method add-named-class defined for dbclass, because it will redefine
;;; this dbclass in the database.  Therefore, we store the function called by
;;; the method add-named-class defind for class in %add-named-class, and call
;;; it when we need to make an in-core definition for a dbclass brought from
;;; the database.  This is a HACK.
;;; 

(defun find-add-named-class-function ()
  "Return the function called by the method (add-named-class (proto class))"
  (let* ((generic-function (fboundp 'add-named-class))
       (method (car (slot-value generic-function 'methods))))
    (slot-value method 'function)))

(defvar *loaded* nil)

(eval-when (compile load eval)
	   (when (null *loaded*)
		 (setf (symbol-function '%add-named-class)
		       (find-add-named-class-function))
		 (unintern 'add-named-class)
		 (setq *loaded* t)))
		 
(defmethod add-named-class ((proto class) name superclasses slots options)
  (%add-named-class proto name superclasses slots options))

(defvar %add-named-class nil)
(setf %add-named-class (find-add-named-class-function))
|#

#|
;;;
;;; ADD-NAMED-CLASS for DBCLASS.  
;;; Besides creating the class in memory, it also stores the 
;;; definition of a dbclass into the database.
;;;
(defmethod add-named-class ((proto dbclass) name superclasses slots options)
  ;; We know where creation-mode is, because we put it there.
  (write name) (write superclasses) (write slots) (write options)
  (let* ((creation-mode (cadr (car options)))
	 (options (cdr options))
	 ;; First call the generic method add-named-class,
	 ;; but we can't pass the :soh-local-slots option to it.
	 (class (call-next-method proto name superclasses slots 
				  (cdr options))))
    ;; debugging
    (write class)
    ;; If the class definition is fetched from the database (in the 
    ;; function fetch-dbclass-1, 'from-database is assigned to classes
    ;; fetched from the database) or there is an error when defining 
    ;; the class in memory, then we don't define it in database.
    (setf (soh-local-slots class) (cadr (car options)))
    (if (and (not (equal creation-mode 'from-database)) class)
	(add-class-to-database class superclasses slots options 
			       creation-mode))))
|#

(defun add-dbclass (name superclasses slots options creation-mode local-slots)
  (let ((class (find-class name))
	(options `((:local-slots ,local-slots) ,@options)))
    (when class
	  (setf (soh-local-slots class) local-slots)
	  (if (not (equal creation-mode 'from-database))
	      (add-class-to-database class superclasses slots options 
				 creation-mode)))))
    
;;;
;;; ADD-CLASS-TO-DATABASE 
;;; It first checks to see if a class with the same name already exists in the
;;; database.  If not, it stores the definition in the database. 
;;; If so, it behaves according to the creation mode.
;;;
(defun add-class-to-database (class superclasses slots options creation-mode)
  "Store the definition of \fBclass\fP into the database"
  (let* ((class-name (class-name class))
	 (old-definition (rest (definition-in-database class-name))))
    (if (null old-definition)
	(add-class-to-database-1 class superclasses slots options)
      (cond ((eq creation-mode 'remove-create)
	     (when (not (same-dbclass old-definition 
				      superclasses slots options))
		   (remove-class-from-database class-name)
		   (add-class-to-database-1 class superclasses slots options)))
	    ((eq creation-mode 'rename-create)
	     (when (not (same-dbclass old-definition 
				      superclasses slots options))
		   (rename-class-in-database class-name)
		   (add-class-to-database-1 class superclasses slots options)))
	    ((eq creation-mode 'always-create)
	     (remove-class-from-database class-name)
	     (add-class-to-database-1 class superclasses slots options))
	    ((eq creation-mode 'never-recreate)
	     (when (not (same-dbclass old-definition 
				      superclasses slots options))
		   (error "Definition for class ~A changed, but the mode is~%~
                           NEVER-RECREATE." 
			  class-name)))
	    (t (error "This should never happen."))))
    (cache-relid class)))

;;;
;;; Actually add the definition of a class to the database.
;;; 1) append the class definition to the relation DBCLASS
;;; 2) append the direct superclasses to the relation SUPERCLASSES
;;; 3) create an instance relation with the same name as the class
;;; 4) create a single tuple relation to store the class variables
;;;
;;; need to consider the order of queries to DB to make them cosistent.
;;; or use beginxact ... endxact facility.
;;;

(defun add-class-to-database-1 (class superclasses slots options)
  (add-class-definition-to-database class superclasses slots options)
  (add-superclasses-to-database class superclasses)
  (create-instance-relation class)
  (create-class-variable-relation class)
  (cache-relid class))

(defun add-class-definition-to-database (class superclasses slots options)
  "Append the class definition of \fBclass\fP to the relation DBCLASS"
  (do-db-add-class 
   (class-name-to-db (class-name class))
   (system:getenv "USER")
   (of-value-to-db superclasses)
   (of-value-to-db slots)
   (of-value-to-db options)))

#+libpq
(defun do-db-add-class (name owner superclasses slots options)
  (let ((query 
	 (format nil 
		 "append dbclass (name = ~S, owner = text[~S], superclasses = text[~S], slots = text[~S], options = text[~S])"
		 name owner superclasses slots options)))
    (pqexec query)))

#+cling
(defun do-db-add-class (name owner superclasses slots options)
  (db-insert 'dbclass 
	     '(name owner superclasses slots options)
	     (list name owner superclasses slots options)))

(defun add-superclasses-to-database (class superclasses)
  "Append the direct superclasses (if any) to the relation SUPERCLASSES"
  (unless (null superclasses)
	  (let ((dbname (class-name-to-db (class-name class))))
	    (dolist (s superclasses)
		    (do-db-add-super dbname (class-name-to-db s))))))

#+libpq
(defun do-db-add-super (class superclass)
  (let ((query (format nil "append superclasses (class = ~S, superclass = ~S)"
		       class superclass)))
    (pqexec query)))

#+cling
(defun do-db-add-super (class superclass)
  (db-insert 'superclasses 
	     '(class superclass)
	     (list class superclass)))

(defun create-instance-relation (class)
  "Create an instance relation for \fBclass\fP named (class-name \fBclass\fP)"
  (let ((instance-slots (class-instance-slots class))
	(local-slots (soh-local-slots class))
	(slots nil))
    (when instance-slots
	  (dolist (c instance-slots)
		  (unless (member (slotd-name c) local-slots)
			  (setq slots
				(extend-instance-slots
				 slots
				 (symbol-to-db (slotd-name c))
				 ;; convert OBJFADS type to database type,
				 ;; each type must have a class
				 (of-type-to-db
				  ;;(find-class (slotd-type c))
				  (slotd-type c))))))
	  (do-db-create-instance (symbol-to-db (class-name class)) slots))))

#+libpq
(defun extend-instance-slots (slots name value)
  (format nil "~A~A = ~A, "
	  (if slots slots "")
	  name
	  value))

#+cling
(defun extend-instance-slots (slots name value)
  (cons (list name value) slots))

#+libpq
(defun do-db-create-instance (name slots)
  (let ((query (format nil "create ~A (~A)" 
		       name 
		       (subseq slots 0 (- (length slots) 2)))))
    (pqexec query)))

#+cling
(defun do-db-create-instance (name slots)
  (db-create-table name slots))

(defun create-class-variable-relation (class)
  ;; the relation is named class-name_cvars
  (let ((class-slots (class-class-slots class))
	 (var-name (symbol-to-db (class-var-name (class-name class))))
	 (slots nil))
    (when class-slots
	  (dolist (c class-slots)
		  (setq slots (extend-instance-slots
			       slots
			       (symbol-to-db (slotd-name c))
			       (of-type-to-db 
				;;(find-class (slotd-type c))
				(slotd-type c)))))
	  (do-db-create-instance var-name slots)
	  ;; append the only tuple of this relation
	  (do-db-add-locals 
	   var-name
	   (symbol-to-db 'soh-local-slots)
	   (of-value-to-db 
	    (slot-value (class-prototype class) 'soh-local-slots))))))

#+libpq
(defun do-db-add-locals (name col value)
  (let ((query (format nil "append ~A (~A = ~A)" name col value)))
    (pqexec query)))

#+cling
(defun do-db-add-locals (name col value)
  (db-insert name (list col) (list value)))

(defun class-var-name (class-name)
  (read-from-string (format nil "~A_cvars" class-name)))

#+libpq
(defun cache-relid (class)
  "Fetch relid of the instance relation for \fBclass\fP into *dbclass-cache*"
  (let* ((class-name (class-name class)) 
	 (relid 0)
	 (query (format nil "retrieve (pg_relation.oid) where pg_relation.relname = ~S"
			(class-name-to-db class-name))))
    (pqexec query)
    (setq relid (read-from-string (pqgetvalue (pqparray "blank") 0 0)))
    (put-into-dbclass-cache class relid)
    (setf (dbclass-relid class) relid)))

#+cling
(defun cache-relid (class)
  "Fetch relid of the instance relation for \fBclass\fP into *dbclass-cache*"
  (let ((relid (class-name-to-db (class-name class))))
    (put-into-dbclass-cache class relid)
    (setf (dbclass-relid class) relid)))

;;;
;;; Retrieve the definition of a class from the database.
;;;

#+libpq
(defun definition-in-database (class-name)
 (let ((query (format nil "retrieve (dbclass.all) where dbclass.name = ~S"
                      (class-name-to-db class-name))))
   (definition-in-database-1 query)))

#+libpq
(defun definition-in-database-1 (query)
  (pqexec query)
  ;; if the dbclass does not exist, return nil
  (if (zerop (pqntuples (pqparray "blank")))
      nil
    (let* ((parray (pqparray "blank")))
      (list (portal-value parray "name")
	    (portal-value parray "superclasses")
	    (portal-value parray "slots")
	    (portal-value parray "options")))))


#+cling
(defun definition-in-database (class-name)
  (let ((result (db-select '(d.name d.superclasses d.slots d.options)
			   :from '((dbclass d))
			   :where `(= d.name ,(class-name-to-db class-name)))))
    (if result 
	(mapcar #'read-from-string (first result)) 
      nil)))


#+libpq
(defun definition-in-database-relid (relid)
  (let ((query (format nil "retrieve (dbclass.all) where dbclass.name = pg_relation.relname and pg_relation.oid = oid[~S]" 
		       (prin1-to-string relid))))
    (definition-in-database-1 query)))

(defun portal-value (parray field-name)
  "Get a field value from the portal \fBparray\fP"
  (read-from-string (pqgetvalue parray 0 (pqfnumber parray 0 field-name))))

;;;
;;; Check if the two class definitions are the same.
;;; Defns should not include name, as it may be dbified.
;;;
(defun same-dbclass (old-definition &rest new-definition)
  (equal old-definition new-definition))

;;;
;;; REMOVE-CLASS-FROM-DATABASE
;;; Remove the definition of a dbclass from the database.
;;;
#+libpq
(defun remove-class-from-database (class-name)
  "Remove the definition of \fBclass\fP from the database"
  (let* ((class-name (class-name-to-db class-name))
	 (delete-dbclass (format nil "delete dbclass where dbclass.name = ~S"
				 class-name))
	 (delete-super (format nil "delete superclasses where superclasses.class = ~S"
			       class-name))
	 (destroy-instance (format nil "destroy ~A"
				   class-name)))
    ;; need to add destroy class instance slot relation
    (pqexec delete-dbclass)
    (pqexec delete-super)
    (pqexec destroy-instance)))

#+cling
(defun remove-class-from-database (class-name)
  "Remove the definition of \fBclass\fP from the database"
  (let* ((class-db-name (class-name-to-db class-name)))
    (db-delete 'dbclass :where `(= dbclass.name ,class-db-name))
    (db-delete 'superclasses :where `(= superclasses.class ,class-db-name))
    (db-drop class-db-name)))

;;;
;;; RENAME-CLASS-IN-DATABASE
;;;
(defun rename-class-in-database (class-name)
  (error "Renaming class in database is not implemented yet."))

    
;;;
;;; FETCH-DBCLASS
;;; Get the definition of a dbclass from the database and define it in core.
;;; 

(defmethod fetch-dbclass (class-name dont-fetch-super-p 
				     &optional no-error-p)
  "Define the dbclass \fBclass-name\fP from the definition in the database"
  (if (stringp class-name)
      (setq class-name (read-from-string class-name)))
  ;; If the class is already defined in memory, return it.
  (or (find-class class-name nil)
      (fetch-dbclass-1 (definition-in-database class-name) dont-fetch-super-p)
      (if no-error-p
	  nil
	(error "Dbclass ~A is not stored in the database." class-name))))

(defmethod fetch-dbclass ((relid integer) dont-fetch-super-p
			  &optional no-error-p)
  (or (cached-dbclass relid)
      (fetch-dbclass-1 (definition-in-database relid) dont-fetch-super-p)
      (if no-error-p
	  nil
	(error "Dbclass with relid ~A is not in the database" relid))))

(defun fetch-dbclass-1 (class-definition &optional dont-fetch-super-p)
  (if (null class-definition)
      nil
    (let ((dbclass-proto (class-prototype (find-class 'dbclass)))
	  (class-name (symbol-from-db (car class-definition)))
	  (superclasses (cadr class-definition))
	  (slots (caddr class-definition))
	  (options (cadddr class-definition)))
      (if (null dont-fetch-super-p)
	  ;; If any of the superclasses is not already in core, bring it in
	  (dolist (s superclasses)
		  (if (null (find-class s nil))
		      (fetch-dbclass s dont-fetch-super-p))))
      ;; Actually define the dbclass in core
      (setq options `((:creation-mode from-database) ,@options))
      (eval `(defdbclass ,class-name ,superclasses ,slots ,@options))
      ;; put this dbclass in the dbclass/relid cache
      (cache-relid (find-class class-name))
      t)))

;;;
;;; Exports symbols
;;;
(export '(remove-create
	  rename-create
	  always-create
	  never-create
	  *default-creation-mode*
	  defdbclass
	  remove-class-from-database
	  fetch-dbclass)
	(find-package 'pcl))



