;;;
;;; Shared Object Hierarchy
;;;
;;; Copyright (c) 1986 Regents of the University of California
;;; 
;;; Permission to use, copy, modify, and distribute this software and its
;;; documentation for any purpose and without fee is hereby granted,
;;; provided that the above copyright notice appear in all copies and
;;; that both that copyright notice and this permission notice appear in
;;; supporting documentation, and that the name of the University of
;;; California not be used in advertising or publicity pertaining to
;;; distribution of the software without specific, written prior
;;; permission.  The University of California makes no representations
;;; about the suitability of this software for any purpose.  It is
;;; provided "as is" without express or implied warranty.
;;; 
;;; $Author: bsmith $
;;; $Source: /pic2/picasso/new/widgets/soh/RCS/dbclass.cl,v $
;;; $Revision: 1.1 $
;;; $Date: 1991/08/04 19:08:49 $
;;;

(in-package clos)

;;;
;;; DBCLASS
;;;	Define a meta-class "dbclass" which is the metaclass of all
;;; 	the shared classes.
;;;

(defclass dbclass (standard-class)
  ((no-of-handle-slots
    :initform 6
    :type fixnum
    :reader dbclass-no-of-handle-slots
    :allocation :class)
   (relid
    :initform 0
    :type integer
    :accessor dbclass-relid
    :allocation :instance)
   (soh-local-slots
    :initform nil
    :type cons
    :accessor soh-local-slots
    :allocation :instance))
  (:metaclass standard-class))

;;;
;;; HANDLE
;;; 	Each instance of a shared class is represented by a handle.
;;; 	The handle is an instance of the iwmc-class (see pcl/low.cl).
;;; 	Its class-wrapper points to the class-wrapper of the shared
;;; 	class.  A handle has 6 static slots:
;;;	objid: 		the unique id, a list of (relid oid).
;;;	instance:	the real instance.
;;;	mode:		the update mode.
;;;	indb:		whether this instance is in the database.
;;;	mofified:	whether this instance is modified.
;;;	pinned:		whether this instance is pinned (remain in the cache).
;;;
;;;	When an instance of a shared class is created, a handle
;;;	representing the instance is created and returned.
;;;	Each instance has one and only one handle at any time.
;;;
;;;	The class of an instance is determined by its class-wrapper.
;;;	Since the handle and the instance shares the same class-wrapper,
;;;	the handle is treated just like the instance.  Therefore we
;;;	have no problem with method invocation.
;;;	

;;;
;;; ALLOCATE-INSTANCE
;;; The handle for an instance is allocated here.
;;; This method is called in two occasions: when an instance is first
;;; created and when an instance fetched from the database is allocated.
;;; When an instance is first created, objid is NIL.  
;;; But an instance fetched from the database already has an objid.  
;;;
;;; I should have used call-next-method, but that's slow.
;;; So the code of allocate-instance for standard-class is 
;;; replicated in this method.  (see pcl/slots.cl)
;;;

(defmethod allocate-instance ((class dbclass) &rest initargs)
  "Allocate an instance of a shared class"
  (let* ((sohid (cadr (member ':sohid (car initargs))))
	 (objid (if sohid (list (symbol-to-db (class-name class)) sohid) nil))
	 (class-wrapper (class-wrapper class))
         (instance (%allocate-instance--class
                     (class-no-of-instance-slots class)))
	 (handle (allocate-handle class objid)))
    (setf (iwmc-class-class-wrapper instance) class-wrapper)
    (handle-set-instance handle instance)
    ;; The update-mode must be set to 'local-copy so even when an 
    ;; instance is made with the 'direct-update mode, put-slot-using-class
    ;; won't go to database when each slot is initilized.
    ;; The correct-mode is set in initialize-instance.
    (handle-set-mode handle 'local-copy)
    handle))

;;;
;;; This looks better, but slower.
;;;
#|
(defmethod allocate-instance ((class dbclass) &rest initargs)
  (let* ((objid (cadr (member ':objid initargs)))
	 (instance (call-next-method))
	 (handle (allocate-handle class objid)))
    (handle-set-instance handle instance)
    (handle-set-mode handle 'local-copy)
    handle))
|#

;;;
;;; ALLOCATE-HANDLE
;;; Just as allocate-instance, this function is also called in two 
;;; occasions. When an instance fetched is first created,
;;; a handle is allocated just like an iwmc-class instance.  
;;; When an instance fetched from the database is allocated, 
;;; we have to check if a handle for this instance already exists
;;; in the object cache.  A handle could exist because when 
;;; a composite object is fetched, objects referred by it might not 
;;; be allocated immediately (delayed fetch). Only the handle is created.
;;; If a handle indeed exists, we must not create another one.
;;;
(defun allocate-handle (class objid)
  (let ((handle nil)
	(class-wrapper (class-wrapper class)))
    (if (and (not (null objid)) (handle-exist-p objid))
	(setq handle (get-handle objid))
      (progn 
	(setq handle (%allocate-instance--class
		      (dbclass-no-of-handle-slots class)))
	(setf (iwmc-class-class-wrapper handle) class-wrapper)
	(handle-set-id handle objid)))
    handle))

;;;
;;; MAKE-HANDLE
;;; This function is called only when delayed fetch is desired.
;;; The handle is stored in the object cache.  
;;;
(defun make-handle (class objid)
  (let ((handle nil))
    (if (handle-exist-p objid)
	(setq handle (get-handle objid))
      (progn 
	(setq handle (allocate-handle class objid))
	(handle-set-instance handle nil)
	;; Instance is indb, but not modified or pinned.
	(handle-set-indb handle t)
	(handle-set-modified handle nil)
	(handle-set-pinned handle nil)
	(put-into-object-cache handle)))
    handle))

;;;
;;; SLOT-VALUE
;;; When we access the slots of an instance, the object passed
;;; to slot-value is the handle instead of the real instance.
;;; So we have to get the real instance.
;;; (pcl/slots.cl)
;;;

(defmethod slot-value-using-class ((class dbclass) handle slot-name)
  (cond ((eq slot-name 'objid) (handle-id handle))
	((eq slot-name 'sohid) (cadr (handle-id handle)))
	(t
	 (let ((object (get-dbobject handle)))
	   #+debug (format t "Slot ~S from object ~S (instance ~S)~%" 
			   slot-name object (handle-instance object))
	   (bind-wrapper-and-static-slots--std object (slot-value-using-class-1))))))

(defmethod (setf slot-value-using-class) (nv (class dbclass) handle slot-name)
  (let ((object (get-dbobject handle)))
    (bind-wrapper-and-static-slots--std 
     object (setf-slot-value-using-class-1))
    (record-modification handle)
    nv))

(defmethod slot-boundp-using-class ((class dbclass) handle slot-name)
  (let ((object (get-dbobject handle)))
    (bind-wrapper-and-static-slots--std
     object (slot-boundp-using-class-1))))

;;;
;;; Cleaner but slower implementation.
;;;
#|
(defmethod slot-value-using-class ((class dbclass) handle slot-name)
  (cond ((eq slot-name 'objid) (handle-id handle))
	((eq slot-name 'sohid) (cadr (handle-id handle)))
	(t
	 (call-next-method class (get-dbobject handle) slot-name))))

(defmethod (setf slot-value-using-class) (nv (class dbclass) handle slot-name)
  (call-next-method nv class (get-dbobject handle) slot-name)
  (record-cache-flag handle %mode))

(defmethod slot-boundp-using-class ((class dbclass) handle slot-name)
  (class-next-method class (get-dbobject handle) slot-name))
|#

;;;
;;; ACCESSORS
;;; Automatically generated accessor methods should do the correct thing.
;;; We need to make sure PCL uses the right code for getting at these slots.
;;; (pcl/defclass.cl)
;;;

(defmethod make-reader-method-function ((class dbclass) slotd)
  (let* ((slot-name (slotd-name slotd))
	 (sxhash (sxhash slot-name)))
    #'(lambda (handle)
	(let* ((object (get-dbobject handle))
	       (wrapper (iwmc-class-class-wrapper object))
	       (slots (iwmc-class-static-slots object)))
	  (get-slot-value-1 object wrapper slots sxhash slot-name)))))

(defmethod make-writer-method-function ((class dbclass) slotd)
  (let* ((slot-name (slotd-name slotd))
	 (sxhash (sxhash slot-name)))
    #'(lambda (nv handle)
	(let* ((object (get-dbobject handle))
	       (wrapper (iwmc-class-class-wrapper object))
	       (slots (iwmc-class-static-slots object)))
	  (set-slot-value-1 nv object wrapper slots sxhash slot-name)
	  (record-modification handle)))))
	
;;;
;;; Cleaner but slower implementation.
;;;
#|
(defmethod make-reader-method-function ((class dbclass) slotd)
  (let ((slot-name (slotd-name slotd)))
    #'(lambda (object)
	(slot-value object slot-name))))

(defmethod make-writer-method-function ((class dbclass) slotd)
  (let ((slot-name (slotd-name slotd)))
    #'(lambda (object new-value)
	(setf (slot-value object slot-name) new-value))))
|#

;;; 
;;; OPTIMIZED-SLOT-VALUE
;;; This part is suggested by Gregor.  I still have to understand it.
;;;
;;; We need to make sure that any optimized calls to slot-value we inherit
;;; get deoptimized appropriately.  We also need to have optimized accessor
;;; methods get deoptimized appropriately.  De-optimization means having
;;; any calls to these trap out to calling slot-value-using-class or
;;; put-slot-using-class.
;;;
;;; This code takes advantage of undocumented PCL functionality.
;;; There will be a more elegant mechanism for doing this in future 
;;; versions of PCL, as well as in the CLOS spec.
;;; (see pcl/vector.cl).
;;;

;;;
;;; This method arranges for all optimized accesses to any slot of a
;;; dbclass to trap through slot-value-using-class.
;;;
(defmethod lookup-pv-miss-1 ((class dbclass) slots pv)
  (dolist (slot slots)
    (push nil pv))
  pv)

;;;
;;; This method arranges for all inherited reader and writer methods to
;;; trap through slot-value-using-class.
;;;
(defmethod all-std-class-readers-miss-1 ((class dbclass) wrapper slot-name)
  (declare (ignore wrapper slot-name))
  ())

;;;
;;; Given this, we would allow inheritance from different meta-classes.
;;;

(defmethod check-super-metaclass-compatibility ((class dbclass)
						(new-super standard-class))
  t)

(defmethod check-super-metaclass-compatibility ((class standard-class)
						(new-super dbclass))
  t)

;;;
;;; It doesn't seem worth doing slot-value optimization because of the time
;;; required to go to the database.  We have to disable the optimization
;;; method we would otherwise inherit.
;;;

;;; optimize-slot-value is not a generic function in current CLOS

;;; (defmethod optimize-slot-value ((class dbclass) form) form)
;;; (defmethod optimize-set-slot-value ((class dbclass) form) form)

