;;;
;;; Copyright (c) 1982 Regents of the University of California
;;; Copyright (c) 1987 Regents of the University of California
;;;
;;; Permission to use, copy, modify, and distribute this software and its
;;; documentation for any purpose and without fee is hereby granted,
;;; provided that the above copyright notice appear in all copies and
;;; that both that copyright notice and this permission notice appear in
;;; supporting documentation, and that the name of the University of
;;; California not be used in advertising or publicity pertaining to
;;; distribution of the software without specific, written prior
;;; permission.  The University of California makes no representations
;;; about the suitability of this software for any purpose.  It is
;;; provided "as is" without express or implied warranty.
;;; 

(in-package "SLING")

;;;
;;; Utility functions:
;;;

;;;
;;;; state-check
;;;
;;; This function checks the state of cling before a quel command takes
;;; place.  An error is returned if
;;; there is a is no database open.  If a retrieve is in progress, it is
;;; killed to prevent errors.
;;;

(defun state-check ()
   (when (and (not (nested-command-mode)) retrieve-in-progress)
         (ii-break)
	 (setf retrieve-in-progress NIL))
   (when (Not current-db-name) (error "No database has been opened")))

;;;
;;;; LP & RP
;;;
;;; These keep the code cleaner.  Parentheses in strings confuse the editor.
;;;
(defun write-out-lp ()
  (write-out "(" ;)
	     ))

(defun write-out-rp ()
  (write-out ;(
	       ")"))

;;;
;;;; STRING
;;;
;;; Given a string'able argument, put it out with quotes around it.
;;; The user should take care that the string is suitable for ingres.
;;; Thus, the following convertions must be made from C print style to
;;; lisp print style.
;;; Each backslash not escaping a quote must be doubled.  Each backslash
;;; escaping a quote must be tripled.  Of course, you don't have to build
;;; a string using the read syntax -- you can just construct one.
;;; Examples:
;;;  "hello" -> "hello"
;;;  "hello\n" -> "hello\\n"
;;;  "\"hello\"" -> "\\\"hello\\\""
;;;  "\\\"hello\"\n" -> "\\\\\\\"hello\\\"\\n"
;;;
(defun write-out-string (str)
  (write-out "'")
  (write-out (string str))
  (write-out "'"))

;;;
;;;; NUMBER
;;;
;;; Handle doubles specially by changing 'd' to 'e'
;;;
(defun write-out-number (value)
  (if (typep value 'double-float)
      (write-out (nsubstitute #\e #\d (prin1-to-string value)
			      :test 'char-equal))
      (write-out (prin1-to-string value))))

;;;
;;;; FILE-NAME
;;;
;;; Used by the copy command.  Takes a string as input.
;;; Try to be helpful by prepending the current directory to the name
;;; if the name isn't a full pathname [since ingres will bitch and moan
;;; if it doesn't get a full pathname].
;;;
(defun write-out-file-name (name)
  (write-out-string (full-path-name name)))

;;;
;;;; NAME
;;;
;;; This takes a symbol or a string and prints it out, without quotes,
;;; to a maximum of 12 characters.  This keeps ingres from complaining
;;; about names being too long.  Names include relations, attribute,
;;; variable, indexes, and locations.
;;; If the item could be a column name it should be sent to write-out-column.
;;;
(defun write-out-name (name &aux (s (string name)))
  (if (> (length s) max-ingres-name-length)
      (write-out (subseq s 0 max-ingres-name-length))
      (write-out s)))

;;;
;;;; COLUMN
;;;
;;; The argument must be either a symbol, string, or a dotted pair.
;;; If there is no dot, then the argument is sent to write-out-name.
;;; Otherwise, each half is printed, up to 12 characters, with a dot
;;; separating them.
;;; ### Should use a subroutine that breaks the argument up into strings.
;;;
(defun write-out-column (name &aux dot-posn)
  (if (consp name)	; structural version
      (progn
       (write-out-name (first name))
       (write-out ".")
       (write-out-name (rest name)))
      (progn
       (setf name (string name))	; textual version
       (if (setf dot-posn (position #\. name))	; there's a dot
	   (progn
	    (write-out (subseq name 0 (min dot-posn max-ingres-name-length)))
	    (write-out (subseq name dot-posn
			       (+ dot-posn (min (- (length name) dot-posn)
						(1+ max-ingres-name-length))))))
	   (write-out-name name)))))

;;;
;;;; LOCATION
;;;
;;; The argument is structured the same as a column, ie, either a
;;; symbol, string, or dotted pair.
;;; However, in printing the result, a colon is used to separate the names.
;;;
(defun write-out-location (location &aux dot-posn)
  (if (consp location)	; structural version
      (progn
       (write-out-name (first location))
       (write-out ":")
       (write-out-name (second location)))
      (progn
       (setf location (string location))
       (if (setf dot-posn (position #\. location))
	   (progn
	    (write-out (subseq location 0
			       (min dot-posn max-ingres-name-length)))
	    (write-out ":")
	    (write-out (subseq location (1+ dot-posn)
			       (+ dot-posn (min (- (length location)
						   dot-posn)
						(1+ max-ingres-name-length))))))
	   (write-out-name location)))))

;;;
;;;; SEQUENCE
;;;
;;; Use the function argument to print a comma separated version of the list.
;;;
(defun write-out-sequence (list function)
  (do ((l list (rest l))
       )
      ((null l)
       )
      (funcall function (first l))
      (if (rest l) (write-out ", "))))

;;;
;;;; LIST
;;;
;;; Surround the comma separated list by parentheses.
;;; Same arguments as to write-out-sequence.
;;;
(defun write-out-list (list function)
  (write-out-lp)
  (write-out-sequence list function)
  (write-out-rp))

;;;
;;;; ITEM
;;;
;;; This differs from an expression in that strings are printed without
;;; quotes.  Lists aren't supported either.  Used for savepoints,
;;; permit elements, format and other '=' lists, or anything that isn't
;;; an expression and doesn't need quotes around strings.
;;;
(defun write-out-item (item)
  (typecase
   item
   (number (write-out-number item))
   (atom (write-out-name item))
   (t (error "A list is not valid here: ~s" item))))

;;;
;;;; EXPRESSION
;;;
;;; Given an expression or a qualification, print it out.
;;; The atomic elements are strings and numbers and column names, which
;;; are either symbols or dotted pairs.
;;; Otherwise, we have a function, which is either unary, infix, or
;;; functional.  Functions may be aggregate, in which case :by and :where
;;; introduce new material - each takes one argument.
;;; Function names are treated as items, and thus could be strings.
;;; All other strings get quotes around them.
;;; Binary operators must be symbols, however.
;;;
(defun write-out-expression (expression)
  (typecase
   expression
   (string (write-out-string expression))
   (number (write-out-number expression))
   (atom (write-out-column expression))
   (list 
    (if (not (listp (rest expression)))
	(write-out-column expression)	; is a dotted pair column name
	(case (length expression)
	      ;; put parenthses around something
	      (1 (write-out-lp)
		 (write-out-expression (first expression))
		 (write-out-rp))
	      ;; function notation, one argument, works for 'not', too.
	      (2 (if (eq (first expression) '-)  ; check for unary minus
		     (progn
		      (write-out-lp)
		      (write-out "-")
		      (write-out-expression (second expression))
		      (write-out-rp))
		     (progn
		      (write-out-item (first expression))
		      (write-out-lp)
		      (write-out-expression (second expression))
		      (write-out-rp))))
	      ;; function, two args, either infix or functional
	      (3 (if (member (first expression) *binary-ops*)
		     (progn
		      (write-out-lp)
		      (write-out-expression (second expression))
		      (write-out " ")
		      (write-out-item (first expression))
		      (write-out " ")
		      (write-out-expression (third expression))
		      (write-out-rp))
		     (progn
		      (write-out-item (first expression))
		      (write-out-list (rest expression)
				      'write-out-expression))))
	      ;; function, lots of args, check for aggregate function
	      (t (cond
		  ((member (first expression) *binary-ops*)
		   (write-out-lp)
		   (do ((operator (first expression))
			(l (rest expression) (rest l))
			)
		       ((null l)
			)
		       (write-out-expression (first l))
		       (when (rest l)
			     (write-out " ") ; needed for 'and' and 'or'
			     (write-out-expression operator)
			     (write-out " "))
		       )
		   (write-out-rp))
		  ((or (member :by expression)
		       (member :where expression))
		   (write-out-aggregate expression))
		  (t
		   (write-out-item (first expresssion))
		   (write-out-list (rest expression)
				   'write-out-expression)))))))))


;;;
;;;; AGGREGATE
;;;
;;; This method of printing a function call is invoked when the
;;; :by or :where keywords are detected.
;;; The call looks like '(function argument [:by var|(vars)][:where qual])
;;;
(defun write-out-aggregate (aggregate)
  (write-out-name (pop aggregate))
  (write-out-lp)
  (write-out-expression (pop aggregate))
  (when (eq :by (first aggregate))
	(write-out " by ")
	(write-out-sequence (listify (second aggregate)) 'write-out-column)
	(setf aggregate (nthcdr 2 aggregate)))
  (when (eq :where (first aggregate))
	(write-out-qual (second aggregate)))
  (write-out-rp))

;;;
;;;; TARGET-LIST
;;;
;;; A target list is a list of expressions, with a few exceptions at the
;;; top level.  An element may be a magic function (:'type') that takes
;;; one argument that is a top-level element.
;;; Otherwise, a top level element must be a column name or the '=' function.
;;; 
(defun write-out-target-list (list)
  (if list
      (write-out-list list 'write-out-target-element)
      (error "Target list cannot be nil")))

;;;
;;;; TARGET-SEQUENCE
;;;
;;; Same as above, no parentheses surrounding it.
;;;
(defun write-out-target-sequence (list)
  (if list
	  (write-out-sequence list 'write-out-target-element)
	  (error "Target sequence cannot be nil")))

(defun write-out-target-element (element)
  (cond ((consp element)
	 (cond ((not (listp (rest element)))
		(write-out-column element))
	       ((member (first element) '(:integer :float :string :varchar))
		(write-out-target-element (second element)))
	       ((eq (first element) '=)
		(write-out-name (second element))
		(write-out "=")
		(write-out-expression (third element)))
	       (t
		(error "Invalid target-list element: ~s" element))))
	((atom element)
	 (write-out-column element))
	(t
	 (error "Invalid targaet-list element: ~s" element))))

;;;
;;;; QUAL
;;;
;;; The second argument is used for integrities, where "where" is not used.
;;; The qualification is generally optional.
;;;
(defun write-out-qual (qual &optional (intro " where "))
  (unless (null qual)
	  (write-out intro)
	  (write-out-expression qual)))

;;;
;;;; GROUP-BY
;;;
;;;
(defun write-out-group-by (cols)
  (unless (null cols)
	  (write-out " group by ")
	  (write-out-target-sequence cols)))


;;;
;;;; FROM-LIST
;;;
;;;
(defun write-out-from-list (tables)
  (unless (null tables)
	  (write-out " from ")
	  (if (listp tables)
	      (write-out-sequence tables 'write-out-corr-element)
	      (write-out-name tables))))

(defun write-out-corr-element (element)
  (cond ((consp element)
	 (write-out-name (first element))
	 (write-out " ")
	 (write-out-name (second element)))
	(t (write-out-name element))))

;;;
;;;; COLUMN-FORMAT
;;;
;;;
(defun write-out-column-format (element)
  (if (atom element)
      (write-out-name element)
      (progn
       (write-out-name (first element))
       (write-out " ")
       (if (and (listp (second element)) (eql (car (second element)) 'char))
	   (progn
	    (write-out "varchar")
	    (write-out-lp)
	    (write-out-number (cadr (second element)))
	    (write-out-rp))
	   (write-out-name (second element)))
       (when (third element)
	     (write-out " not null")))))


;;;
;;;; SORT LIST ONLY
;;;
;;; Print out sort list with no sort prepended
;;;
(defun write-out-sort-list (sort)
  (unless (null sort) (write-out-sequence (listify sort) 'write-out-sort-item)))

;;;
;;;; SORT
;;;
;;; Print out the sort clause to a retrieve statement or the 'on' clause
;;; of a modify statement, if one was given.
;;; The names are assumed to be consistent with one of the column headings.
;;; The argument is  name | ({name | (name sortorder)}+) .
;;; The sortorder must be an 'asc' or 'desc'.
;;;
(defun write-out-sort (sort &optional (intro " order by "))
  (unless (null sort)
	  (write-out intro)
	  (write-out-sequence (listify sort) 'write-out-sort-item)))

(defun write-out-sort-item (item)
  (if (consp item)
      (progn
       (write-out-name (first item))
       (write-out " ")
       (write-out-name (second item)))
      (write-out-name item)))

;;;
;;;; =-LIST
;;;
;;; Each element is a list of length two.
;;; The two items are printed separated by a '='.
;;; Used by create, format, and modify.
;;; Target lists are handled separately, thus both elements
;;; are printed as items, not expressions.
;;;
(defun write-out-=-list (list)
  (write-out-list list 'write-out-=-element))

(defun write-out-=-element (element)
  (write-out-item (first element))
  (write-out "=")
  (write-out-item (second element)))

;
; ************************
; Ingres (ESQL) interface
; ************************
;

;;;
;;;; SEND VALUES TO INGRES
;;;
(defun send-values-to-ingres (vals &key procedure &aux
			       (fix (make-array 1 :element-type 'fixnum))
			       (flo (make-array 1 :element-type 'double-float)))
  (mapc
   (if procedure
       #'(lambda (x)
	     (typecase (second x)
		       (string (ii-LQprvProcValio
				(string-downcase (string (first x)))
				0 0 1 30 0 (second x)))
		       (integer (setf (aref fix 0) (coerce (second x) 'fixnum))
				(ii-LQprvProcValio
				 (string-downcase (string (first x)))
				 0 0 1 32 4 fix))
		       (float (setf (aref flo 0) (double-float (second x)))
			      (ii-LQprvProcValio
			       (string-downcase (string (first x)))
			       0 0 1 31 8 flo))))
       #'(lambda (x)
	     (typecase x
		       (string (ii-putdomio 0 1 30 0 x))
		       (integer (setf (aref fix 0) (coerce x 'fixnum))
				(ii-putdomio 0 1 32 4 fix))
		       (float (setf (aref flo 0) (double-float x))
			      (ii-putdomio 0 1 31 8 flo)))))
   vals))
;
; *********
;  cursors
; *********
;

;;;
;;;; RECORD-CURSOR-NAME
;;;
;;; Record a cursor name and the select associated with it for use when
;;; the cursor is opened (db-open).
;;;
(defun record-cursor-name (name select update-type)
  (when (assoc name *cursor-list*)
	(cerror "replace old declaration."
		"cursor ~s already exists." name)
	(forget-cursor-name 'c1))
  (setq *cursor-list* (acons name (list select update-type) *cursor-list*)))

;;;
;;;; FORGET-CURSOR-NAME
;;;
;;; Remove a cursor from the list.
;;;
(defun forget-cursor-name (name)
  (setq *cursor-list*
	(delete-if #'(lambda (x) (eq (car x) name)) *cursor-list*)))

;;;
;;;; LOOKUP-CURSOR-NAME
;;;
;;; Lookup a cursor name and return the select associated with it for use when
;;; the cursor is opened (db-open).
;;;
(defun lookup-cursor-name (name)
  (cdr (assoc name *cursor-list*)))

;;;
;;;; UPDATE-TYPE
;;;
;;; Check the `for-update' keyword arg to db-declare-cursor to determine the
;;; type of updating (if any) a cursor will be opened for.
;;;
(defun update-type (for-update)
  (cond ((null for-update) nil)
	((member (car for-update) '(deferred direct))
	 (car for-update))
	(t nil)))

;;;
;;;; CURSOR-ID
;;;
;;; Return a list: the cursor name (truncated to max-ingres-name-length),
;;; and two numbers.
;;;
(defun cursor-id (cursor-name &aux (string-name (string cursor-name)))
  (list (subseq string-name 0
		(min max-ingres-name-length (length string-name)))
	0 0))

;;;
;;;; DETERMINE-TARGET-TYPES
;;;
;;; Given a target list, return a list of equal length containing the
;;; elements :string, :integer, and :float.
;;;
(defun determine-target-types (target-list)
  (mapcar #'(lambda (element)
		    (if (listp element)
			(cond ((eq (first element) :integer)
			       :integer)
			      ((eq (first element) :float)
				   :float)
			      (t
			       :string))
			:string))
	  target-list
	  ))

;;;
;;;; RETRIEVE-RESULT-FROM-INGRES
;;;
;;; This is used by db-select to return results in list form
;;; Use a tail pointer to efficiently construct the list.
;;;
(defun retrieve-result-from-ingres (target-list &aux tail-ptr result)
  (setf result (cons nil nil))	; dummy cons to make things simpler
  (setf tail-ptr result)	; the last cons appended to the list
  (mapc-retrieve #'(lambda (x &aux (c (cons x nil)))
			   (setf (rest tail-ptr) c)
			   (setf tail-ptr c))	; non-nil return value
		 target-list)
  (rest result))

    
;;;
;;;; MAPCAR-RETRIEVE
;;;
;;; This is used by db-mapcar-retrieve to return results in list form.
;;; MAPCAR-RETRIEVE applies a function to every element in the list.
;;; Use a tail pointer to efficiently construct the list.
;;;
(defun mapcar-retrieve (function target-list &aux tail-ptr result)
  (setf result (cons nil nil))	; dummy cons to make things simpler
  (setf tail-ptr result)	; the last cons appended to the list
  (mapc-retrieve-nobreak #'(lambda (x &aux
				      (c (cons (funcall function x) nil)))
				   (setf (rest tail-ptr) c)
				   (setf tail-ptr c))	; non-nil return value
			 target-list)
  (setq retrieve-in-progress NIL)
  (rest result))

;;;
;;;; MAPCAN-RETRIEVE
;;;
;;; This is used by db-mapcan-retrieve to return results in a nconc'ed list.
;;; MAPCAN-RETRIEVE applies a function to every element in the list, and
;;; the results are nconc'ed together.
;;;
(defun mapcan-retrieve (function target-list &aux result)
  (setf result (cons nil nil))
  (mapc-retrieve-nobreak #'(lambda (x &aux
				      (c (nconc result (funcall function x))))
				   (setf result c))		
			 target-list)
  (setq retrieve-in-progress NIL)
  (rest result))

;;;
;;;; MAPC-RETRIEVE
;;;
;;; Once the query has been written out, apply the given function
;;; to a list containing each returned tuple.
;;; The target list is passed so that the number of returned domains and
;;; their types can be determined.
;;; If the function wants to kill the retrieve, it should return nil.
;;;
(defun mapc-retrieve (function target-list
			       &aux (target-types
				     (determine-target-types target-list)))
  (ii-retinit 0 0)
  (when (eql 0 (ii-errtest))
	(do ((tuple-count 0 (1+ tuple-count))
	     (cur-tuple nil)
	     )
	    ((eql 0 (ii-nextget))
	     (ii-flush 0 0)
	     )
	    (if (setf cur-tuple (retrieve-tuple target-types))
		(unless (funcall function cur-tuple)
			(ii-break)
			(return))))	; break loop
	)
  (setq retrieve-in-progress NIL))

;;;
;;;; MAPC-RETRIEVE-NOBREAK
;;;
;;; This function takes a function and a target list and
;;; applies to it to every tuple of a query.  Unlike mapc-retrieve, it
;;; does not kill the retrieve when the function returns nil.
;;; The query must first be written out before this function is used.

(defun mapc-retrieve-nobreak (function target-list
				      &aux (target-types
				      (determine-target-types target-list)))
  (ii-retinit 0 0)
  (when (eql 0 (ii-errtest))
	(do ((tuple-count 0 (1+ tuple-count))
	     (cur-tuple nil)
	     )
	    ((eql 0 (ii-nextget))
	     (ii-flush 0 0)
	     )
	    (setf cur-tuple (retrieve-tuple target-types))
	    (funcall function cur-tuple)))
  (setq retrieve-in-progress NIL))


;;;
;;;; RETRIEVE-TUPLE
;;;
;;; Given that the retrieve loop has been initialized, snarf
;;; the required results from the backend.  Result is a list of the
;;; same length as the target-types argument.
;;; Any errors cause the return value to be nil.
;;;
(defun retrieve-tuple (target-types &aux result)
  (setf result
	(mapcar
	 #'(lambda (item-type &aux str-len)
		   (case item-type
			 (:float (ii-retf))
			 (:integer (ii-reti))
			 (:string
			  (setf str-len (ii-rets *retrieve-string-buffer*))
			  (subseq *retrieve-string-buffer* 0 str-len))
			 (t
			  (error "Unknown target type: ~s" item-type))))
	 target-types))
  (if (eql 0 (ii-errtest))
      result))

;;;
;;;; FETCH-TUPLE
;;;
;;; Given that the fetch has been initialized, snarf
;;; the required results from the backend.  Result is a list of the
;;; same length as the target-types argument.
;;; Any errors cause the return value to be nil.
;;;
(defun fetch-tuple (target-types &aux result)
  (mapcar
   #'(lambda (item-type &aux str-len)
	     (case item-type
		   (:float (ii-csGetf))
		   (:integer (ii-csGeti))
		   (:string
		    (setf str-len (ii-csGets *retrieve-string-buffer*))
		    (subseq *retrieve-string-buffer* 0 str-len))
		   (t
		    (error "Unknown target type: ~s" item-type))))
   target-types))

;;;
;;;; SEND-QUERY
;;;
(defun send-query (targetlist from distinct qual group-by having union order-by
			      for-update direct)
  (when (and (null group-by) having)
	(error "having clause without group-by clause in select.~%"))
  (write-out "select ")
  (if distinct (write-out "distinct "))
  (write-out-target-sequence targetlist)
  (write-out-from-list from)
  (write-out-qual qual " where ")
  (write-out-group-by group-by)
  (write-out-qual having " having ")
  (unless (null union)
	  (write-out " union ")
	  (nested-command (apply (car union) (rest union))))
  (write-out-sort order-by)
  (when for-update
	(write-out " for update of ")
	(write-out-sequence for-update 'write-out-name))
  (unless (nested-command-mode)
	  (setq retrieve-in-progress T)))

;;;
;;;; COMPUTE-*-TARGETLIST
;;;
(defun compute-*-targetlist (from &aux (da (make-sqlda)))
  (mapcan
   #'(lambda (x)
	     (let* ((relname (car (listify x)))
		    (corrname (string (if (listp x) (second x) x)))
		    (targl (targlist '* relname)))
		   (mapcar #'(lambda (y)
				     (setf (second y)
					   (concatenate 'string
						    corrname "." (second y)))
				     y)
			   targl)))
   from))

;;;
;;;; COMPUTE-TARGETLIST
;;;
;;;
(defun compute-targetlist (cols from &rest ignored)
  (cond ((not expand-target-lists)
	 cols)
	((equal (listify cols) '(*))
	 (compute-*-targetlist (listify from)))
	(t
	 (targlist cols from))))


(defun targlist (cols from &aux (desc (make-sqlda)) (result nil) ptr)
  (setq cols (listify cols))
  (setq from (listify from))
  (setq ptr cols)
  (write-out-to-string (nested-command (db-select cols from)))
  (temp-not-nested
   (db-prepare 'clingstmt write-out-string)
   (db-describe 'clingstmt :desc desc))
  (dotimes (i (sqlda-sqld desc) result)
	   (setq result (append result
				(if (equal cols '(*))
				    (list (sqlvar desc i))
				    (prog1
				     `((,(sqltype desc i)
					 ,(car ptr)
					 ,(sqllen desc i)))
				     (setq ptr (cdr ptr))))))))

;;;
;;;; SQLVAR
;;;
(defun sqlvar (sqlda n)
  (when (>= n (sqlda-sqld sqlda))
	(error "no ~:R sqlvar" n))
  (list (sqltype sqlda n)
	(sqlname sqlda n)
	(sqllen sqlda n)))

;;;
;;;; SQLTYPE
;;;
(defun sqltype (sqlda n)
  (when (>= n (sqlda-sqld sqlda))
	(error "no ~:R column" n))
  (case (abs (sqlda-sqlvar-sqltype sqlda n))
	(30 :integer)
	(31 :float)
	(20 :string)	; char
	(21 :varchar)
	(3  :string)	; date
	(5  :float)))	; money

;;;
;;;; SQLLEN
;;;
(defun sqllen (sqlda n)
  (when (>= n (sqlda-sqld sqlda))
	(error "no ~:R column" n))
  (sqlda-sqlvar-sqllen sqlda n))

;;;
;;;; SQLNAME
;;;
(defun sqlname (sqlda n)
  (when (>= n (sqlda-sqld sqlda))
	(error "no ~:R column" n))
  (let* ((namelen (sqlda-sqlvar-sqlname-sqlname1 sqlda n))
	 (name (make-string namelen)))
	(dotimes (i namelen)
		 (setf (elt name i)
		       (code-char (sqlda-sqlvar-sqlname-sqlnamec sqlda n i))))
	name))

;;;
;;;; SQLERRM
;;;
;;; get the error message from the sqlca
;;;
(defun sqlerrm (&optional (sqlca sqlca-ptr)
			  &aux (len (sqlca-sqlerrm-sqlerrml sqlca))
			  (str (make-string 70 :initial-element #\Space)))
  (dotimes (i len (subseq str 0 len))
	   (setf (char str i) (code-char (sqlca-sqlerrm-sqlerrmc sqlca i)))))

;;;
;;;; INGERRM
;;;
;;; get the error message from ingres
;;;
(defun ingerrm (&optional (buf *retrieve-string-buffer*))
  (setq buf (make-string 2001))
  (subseq buf 0 (ii-eqiqs buf "errortext")))

;;;
;;;; DO INQUIRE
;;;
(defun do-inquire (obj &aux (buf (make-string 2001)))
  (case obj
	((errortext messagetext)
	 (subseq buf 0 (ii-eqiqs buf (string obj))))
	((errorno messagenumber rowcount endquery transaction)
	 (ii-eqiqi (string obj)))))

;;;
;;;; SQLSTATUS
;;;
;;; check sqlca.sqlcode and sqlca.sqlwarn for any
;;; out-of-the-ordinary conditions
;;;
(defun sqlstatus (&optional (sqlca sqlca-ptr)
			    &aux (code (sqlca-sqlcode sqlca)) result)
  (cond ((= code 0)
	 ;; possible warning
	 (if (eql (sqlca-sqlwarn-sqlwarn0 sqlca) (char-code #\W))
	     'warning))
	((> code 0)
	 ;; exception condition
	 (cond ((= code 100) 'not-found)
	       ((= code 700) 'message)
	       (t 'unknown-exception)))
	((< code 0)
	 'error)))

;;;
;;;; SQLPRINT
;;;
(defun sqlprint ()
  (if (eq (sqlstatus sqlca-ptr) 'error)
      (format t "~&CLING: INGRES error:~%~A" (ingerrm))))

;;;
;;;; DISPATCH-WHENEVER
;;;
(defun dispatch-whenever (stat)
  (case stat
	('error (when *error-handler* (funcall *error-handler*)))
	('warning (when *warn-handler* (funcall *warn-handler*)))
	('message (when *msg-handler* (funcall *msg-handler*)))
	('not-found (when *not-found-handler* (funcall *not-found-handler*))))
  nil)

(defun nested-command-mode ()
  (> nesting-level 0))

;;;
;;;; SQLCA INIT
;;;
(defun sqlca-init ()
  (unless (nested-command-mode)
	  (ii-sqInit sqlca-ptr)))



































































































