;;;
;;; Copyright (c) 1990 Regents of the University of California
;;; 
;;; $Author: bsmith $
;;; $Source: /pic2/picasso/new/widgets/misc/RCS/scroll-bar.cl,v $
;;; $Revision: 1.1 $
;;; $Date: 1991/08/04 19:08:32 $
;;;

(in-package "PT")

;;;
;;;   -- ABSTRACT --
;;;
;;; A scroll-bar is a widget conceptually containing two buttons and 
;;; a slide-bar.  It may be either horizontal or vertical.  For efficiency,
;;; it is coded as a single window rather than a collection.  A vertical
;;; scroll bar appears below:
;;;
;;;   ---------<- lower-limit
;;;   |       |
;;;   |       |<= button here calls prev-page-func
;;;   |       |
;;;   | +---+ |<- slider-location
;;;   | |   | |
;;;   | |   | |<= indicator.
;;;   | |   | |<= button here calls move-func (see text).
;;;   | |   | |
;;;   | +---+ |
;;;   |       |<= button here calls next-page-func
;;;   |       |
;;;   |-------|<- upper-limit
;;;   | +---+ |
;;;   | | ^ | |<= button here calls prev-line-func
;;;   | +---+ |
;;;   |-------|
;;;   | +---+ |
;;;   | | v | |<= button here calls next-line-func
;;;   | +---+ |
;;;   ---------
;;;
;;; 
;;; 
;;; Slots of interest are:
;;;	lower-limit: The value corresponding to the top of the scroll-bar
;;;	upper-limit: The value corresponding to the bottom of the scroll-bar
;;;	slider-location: The value corresponding to the top of the indicator
;;;	slider-size: The length of the slider in the indicator
;;;
;;; The callback functions are:
;;;	prev-line-func:	Eval'd when the prev-line-button is pressed
;;;	next-line-func:	Eval'd when the next-line-button is pressed
;;;	prev-page-func:	Eval'd when moused above slider.
;;;	next-page-func:	Eval'd when moused below slider.
;;;	moved-func:	Eval'd when moused in slider.
;;; All the functions should update the slots of the scroll-bar as
;;; appropriate.  The scroll-bar will be automatically repainted when
;;; the function returns.
;;;
;;; The programming of the dynamic drag function is a bit tricky, and
;;; deserves some discussion.  When the moved-function is eval'd, it
;;; usually sets up any internal structures it needs for fast scrolling,
;;; and the call the drag-scroll-bar function, passing it the scroll-bar
;;; instance (*sb*) a function (*func*), and the xcl event that triggered
;;; this sequence.  When the mouse moves, *func* is called with three
;;; parameters: the scroll-bar instance (*sb*). the value of the *data*
;;; slot in that instance, and the event.  At the time of the call, the
;;; slider-location of the scroll-bar instance contains the new position
;;; of the slider.
;;;
;;;   -- EXAMPLE --
;;;
;;; Make a vertical and a horizontal scroll bar and keep them synchronized.
;;; Provide functions for scrolling up by lines and pages, and an example of
;;; dynamic drag programming.
;;;
;;; (defun up-line (sb other-sb)
;;;   (decf (slider-location sb))
;;;   (setf (slider-location other-sb) (slider-location sb)))
;;; 
;;; (defun up-page (sb other-sb)
;;;   (decf (slider-location sb) (slider-size sb))
;;;   (setf (slider-location other-sb) (slider-location sb)))
;;;
;;; (defun sb-move (sb ev)
;;;   (let* ((other-sb (data sb))
;;;	     (func #'(lambda (sb data ev)
;;;		       (setf (slider-location other-sb) (slider-location sb))
;;;		       (do-repaint other-sb))))
;;;	    (drag-scroll-bar sb func ev)))
;;;
;;; (setq vsb (make-scroll-bar
;;; 	  :x-offset 20 :width 20 :y-offset 20 :height 100
;;; 	  :orientation :vertical
;;; 	  :lower-limit 1
;;; 	  :upper-limit 400
;;; 	  :slider-size 20
;;; 	  :slider-location 1
;;;	  :moved-func '(sb-move self event)
;;; 	  :prev-line-func '(up-line self (data self))
;;; 	  :prev-page-func '(up-page self (data self))))
;;;    
;;; (setq hsb (make-scroll-bar
;;; 	  :x-offset 20 :width 100 :y-offset 20 :height 20
;;; 	  :orientation :horizontal
;;; 	  :lower-limit 1
;;; 	  :upper-limit 400
;;; 	  :slider-size 20
;;; 	  :slider-location 1
;;;	  :moved-func '(sb-move self event)
;;; 	  :prev-line-func '(up-line self (data self))
;;; 	  :prev-page-func '(up-page self (data self))))
;;;
;;; (setf (data hsb) vsb)
;;; (setf (data vsb) hsb)
;;;
;;;   -- NOTES --
;;;
;;; Due to the sizes of the buttons, a scroll bar width (height for
;;; horizontal scrollbars) of 18-20 pixels looks best.  It
;;; will work for most any width/height chosen, but will
;;; look strange for widths/heights less than 18 pixels.
;;;

;;;
;;; define the scroll-bar field class
;;;
(defclass scroll-bar (widget)
  ((orientation                 ; :vertical or :horizontal
    :initarg :orientation                  
    :initform :vertical
    :type atom
    :accessor orientation)
   (lower-limit	                ; programmer defined lower bound of slider
    :initarg :lower-limit	                 
    :initform 0.0
    :type float
    :accessor lower-limit)
   (upper-limit                 ; ... upper bound
    :initarg :upper-limit                  
    :initform 100.0
    :type float
    :accessor upper-limit)
   (slider-size                 ; size in same units as lower and upper bound
    :initarg :slider-size                  
    :initform 25.0
    :type float
    :accessor slider-size)
   (slider-location             ; location in same units ...
    :initarg :slider-location              
    :initform 0.0
    :type float
    :accessor slider-location)
   (data			; Data to pass to function below
    :initarg :data			 
    :initform nil
    :type t
    :accessor data)
   (moved-func           ; function to call when slider is moved
    :initarg :moved-func            
    :initform nil
    :type t
    :accessor moved-func)
   (prev-page-func       ; function to call when area inside slide bar
    :initarg :prev-page-func        
    :initform nil
    :type t
    :accessor prev-page-func)
   (next-page-func       ; ... when area after slider is buttoned
    :initarg :next-page-func        
    :initform nil
    :type t
    :accessor next-page-func)
   (prev-line-func       ; function to call when area inside slide bar
    :initarg :prev-line-func        
    :initform nil
    :type t
    :accessor prev-line-func)
   (next-line-func       ; ... when area after slider is buttoned
    :initarg :next-line-func        
    :initform nil
    :type t
    :accessor next-line-func)
   (pause-seconds	 ; Pause time for button-hold scrolling
    :initarg :pause-seconds	  
    :initform 0.1
    :type float
    :accessor pause-seconds)
   (button-pos 
    :initarg :button-pos  
    :initform nil 
    :type keyword 
    :reader button-pos)

   ;---------------------------------------------------------------
   ; Internal use only
   ; Graphic contexts used by scroll-bars
   (gc-spec :initform 
	    '((gc-res "default")
	      (white-gc (:foreground "white"))
	      (black-gc (:foreground "black"))
	      (weave-gc (:paint "weave"))
	      (gray50-gc (:paint "gray50"))
	      (gray25-gc (:paint "gray25"))))

   (button-down-gc :initform nil :type vector :reader button-down-gc)
   (button-up-gc :initform nil :type vector :reader button-up-gc)
   (white-gc :initform nil :type vector :reader white-gc)
   (black-gc :initform nil :type vector :reader black-gc)
   (gray50-gc :initform nil :type vector :reader gray50-gc)
   (gray25-gc :initform nil :type vector :reader gray25-gc)
   (weave-gc :initform nil :type vector :reader weave-gc)

   ; Overide some defaults
   (name :initform "A Slide-Bar")
   (width :initform 20)
   (cursor :initform (default-cursor))
   (border-width :initform 0)
   (background :initform "weave")

   ; Private slots  -- Don't touch these

   (event-mask			; The event mask for this slider.
    :initform '(:exposure :button-motion :button-press :button-release)
    :type list)
   (prev-inverted :initform nil :type t :accessor prev-inverted)
   (next-inverted :initform nil :type t :accessor next-inverted)
   (begin                 ; coordinate of beginning of slider
    :initform 0           ; saved to catch events easily
    :type integer         ; y dimension for vertical, x for horizontal
    :accessor begin)
   (len                   ; length of slider in pixels
    :initform 0
    :type integer
    :accessor len)
   (button-start	  ; Start of area to display buttons, in pixels
    :initform 0
    :type integer
    :accessor button-start)
   (button-len		  ; length of area to display buttons in, in pixels
    :initform 40
    :type integer
    :accessor button-len)
   (slider-start	 ; start of area to display slider in
    :initform 0
    :type integer
    :accessor slider-start)
   (slider-len		 ; length of area to display slider in
    :initform 0
    :type integer
    :accessor slider-len)))

(defun init-sb-bitmaps (sb)
  (declare (ignore sb))
  (make-image :name "pan-up" :file "pan-up.bitmap")
  (make-image :name "pan-down" :file "pan-down.bitmap")
  (make-image :name "pan-left" :file "pan-left.bitmap")
  (make-image :name "pan-right" :file "pan-right.bitmap"))

(defun make-scroll-bar (&rest keys)
  (apply #'make-instance 'scroll-bar :allow-other-keys t keys))

(defun sb-move (self event)
  (drag-scroll-bar self nil event))

(defun sb-prev-page (self)
  (if (>= (- (slider-location self) (slider-size self))
	  (lower-limit self))
      (decf (slider-location self) (slider-size self))
      (setf (slider-location self) (lower-limit self))))

(defun sb-next-page (self)
  (if (<= (+ (slider-location self) (slider-size self)
	     (slider-size self))
	  (upper-limit self))
      (incf (slider-location self) (slider-size self))
      (setf (slider-location self) (- (upper-limit self) (slider-size self)))))

(defun sb-prev-line (self)
  (if (>= (1- (slider-location self)) (lower-limit self))
      (decf (slider-location self))))

(defun sb-next-line (self)
  (if (<= (+ (slider-location self) 1 (slider-size self))
	  (upper-limit self))
      (incf (slider-location self))))

(defmethod (setf orientation) (value (self scroll-bar))
  (unless (eq value (slot-value self 'orientation))
	  (setf (slot-value self 'orientation) value)
	  (if (vertical-p self)
	      (setf (base-size self) '(23 42))
	      (setf (base-size self) '(42 23)))
	  (recache-scroll-bar self)
	  (repaint self))
  value)

(defmethod (setf buttons) (value (self scroll-bar))
  (cond ((or (eq value :left ) (eq value :top))
	 (setq value :top-left))
	((or (eq value :right) (eq value :bottom))
	 (setq value :bottom-right))
	((or (eq value :bottom-left)
	     (eq value :bottom-right)
	     (eq value :top-left)
	     (eq value :top-right))
	 nil)
	(t
	 (warn "Illegal value passed to (setf (buttons ~s) ~s)~%" self value)
	 (setq value :top-left)))
  (setf (slot-value self 'button-pos) value)
  (recache-scroll-bar self)
  (repaint self)
  value)

;;;
;;; Recache the values of all the internal slots. Used whenever limits,
;;; position, size of slider changes, orientation changes, or button position
;;; changes.
;;;
(defun recache-scroll-bar (self)
  (let* ((ll (lower-limit self))
	 (ul (upper-limit self))
	 (loc (slider-location self))
	 (size (slider-size self))
	 (w (width self))
	 (h (height self))
	 (vert (vertical-p self))
	 (bl 40)
	 (bs 2)
	 (sl 0)
	 (ss 2)
	 (begin 0)
	 (len 0))

	;; Set the button parameters that need changing
	(if vert
	    (if (buttons-bottom self)
		(setf bs (- h bl 2))
		(setf ss (+ bl 2)))
	    (if (buttons-right self)
		(setf bs (- w bl 2))
		(setf ss (+ bl 2))))

	;; Reset the slider parameters
	(if vert
	    (setf sl (- h 4 bl))
	    (setf sl (- w 4 bl)))
	(when (/= ll ul)
	      (setf len (round (* size sl) (- ul ll))
		    begin (+ ss (round (* sl (- loc ll)) (- ul ll)))))
	;; If the size is 0, make slider length of entire region
	(cond ((and (= ul ll) (zerop size))
	       (setf begin ss len sl))
	      ((zerop size)
	       (setf begin ss len 0)))
	(setf (begin self) begin
	      (len self) (max 0 len)
	      (button-start self) bs
	      (button-len self) (max 0 bl)
	      (slider-start self) ss
	      (slider-len self) (max 0 sl))))

;; This is for consistency, and for use my table-fields.
(defmethod value ((self scroll-bar)
		  &key 
		  &allow-other-keys)
  (slider-location self))

(defmethod (setf value) (val (self scroll-bar))
  (setf (slider-location self) val))

(defmethod (setf lower-limit) (value (self scroll-bar))
  (setf (slot-value self 'lower-limit) value)
  (recache-scroll-bar self)
  (repaint-slider self)
  value)

(defmethod (setf upper-limit) (value (self scroll-bar))
  (setf (slot-value self 'upper-limit) value)
  (recache-scroll-bar self)
  (repaint-slider self)
  value)

(defmethod (setf slider-size) (value (self scroll-bar))
  (if (> 0 value) (setq value 0))
  (setf (slot-value self 'slider-size) value)
  (recache-scroll-bar self)
  (repaint-slider self)
  value)

(defmethod (setf slider-location) (value (self scroll-bar))
  (unless (eq value (slider-location self))
	  (let ((old-begin (begin self))
		(len (len self))
		(ss (slider-start self))
		(sl (slider-len self))
		(new-begin 0))
	       (setf (slot-value self 'slider-location) value)
	       (recache-scroll-bar self)
	       (setq new-begin (begin self))
	       (when (exposed-p self)
		     (if (and (<= ss old-begin (- (+ ss sl) len))
			      (<= ss new-begin (- (+ ss sl) len)))
			 (move-slider self old-begin new-begin len)
			 (repaint-slider self)))))
  value)

(defun move-slider (self old new len)
  (let ((res (res self))
	(w (width self))
	(h (height self))
	(gc (gray50-gc self)))
       (declare (integer old new len w h id bd-id))
       (cond ((vertical-p self)
	      (if (and (> len 1) (< 4 w))
		  (xlib:copy-area res gc 2 old (- w 4) (1- len) res 2 new))
	      (if (> old new)
		  (clear-region self 2 (+ len new) (- w 4) (- old new))
		  (clear-region self 2 old (- w 4) (- new old))))
	     (t
	      (if (and (> len 1) (< 4 h))
		  (xlib:copy-area res gc old 2 (1- len) (- h 4) res new 2))
	      (if (> old new)
		  (clear-region self (+ len new) 2 (- old new) (- h 4))
		  (clear-region self old 2 (- new old) (- h 4)))))))

;;;
;;; initialize a scroll-bar instance
;;;
(defmethod new-instance ((self scroll-bar)
			 &key
			 (buttons :bottom-right)
			 (name nil)
			 (moved-func '(sb-move self event))
			 (prev-page-func '(sb-prev-page self))
			 (next-page-func '(sb-next-page self))
			 (prev-line-func '(sb-prev-line self))
			 (next-line-func '(sb-next-line self))
			 &allow-other-keys)

  (init-sb-bitmaps self)

  (call-next-method)
  
  (setf (slot-value self 'moved-func) moved-func
	(slot-value self 'prev-page-func) prev-page-func
	(slot-value self 'next-page-func) next-page-func
	(slot-value self 'prev-line-func) prev-line-func
	(slot-value self 'next-line-func) next-line-func)

  (if (vertical-p self)
      (setf (base-size self) '(23 42))
      (setf (base-size self) '(42 23)))

  (if (null name)
      (if (vertical-p self)
	  (setf (name self) "A Vertical Scroll-Bar")
	  (setf (name self) "A Horizontal Scroll-Bar")))

  ;; Set the buttons for side effects!
  (setf (buttons self) buttons)
  self)

(defmethod resize-window-handler ((self scroll-bar))
  (recache-scroll-bar self))

(defmethod do-attach ((self scroll-bar) &aux f-up f-down)
  (call-next-method)
  (if (black-and-white-display-p)
      (progn
       (if (= 0 (pixel (get-color "white")))
	   (setq f-down xlib::boole-andc1 f-up xlib::boole-ior)
	   (setq f-down xlib::boole-ior f-up xlib::boole-andc1))
       (setf (slot-value self 'button-up-gc)
	     (make-gc self `(:foreground "black" 
					 :background "white"
					 :function ,f-up)))
       (setf (slot-value self 'button-down-gc)
	     (make-gc self `(:foreground "black" 
					 :background "white"
					 :function ,f-down))))
      (progn
       (setf (slot-value self 'button-up-gc)
	     (make-gc self '(:foreground "black" :background "gray50")))
       (setf (slot-value self 'button-down-gc)
	     (make-gc self '(:foreground "white" :background "gray25"))))))

(defmethod repaint-slider ((self scroll-bar))
  (unless (exposed-p self)
	  (return-from repaint-slider))
  (let* ((h (height self))
	 (w (width self))
	 (vert (vertical-p self))
	 (res (res self))
	 (begin (begin self))
	 (len (len self))
	 (bs (button-start self))
	 (bl (button-len self))
	 (ss (slider-start self))
	 (sl (slider-len self))
	 (top (buttons-top self))
	 (bottom (buttons-bottom self))
	 (right (buttons-right self))
	 (left (buttons-left self))
	 (gray50-gc (gray50-gc self))
	 (black-gc (black-gc self))
	 (white-gc (white-gc self))
	 (center (+ begin (round len 2))))
	
	;; Clear the background.
	(if vert
	    (clear-region self 2 ss (- w 4) sl)
	    (clear-region self ss 2 sl (- h 4)))
	
	;; Set the background of the slider
	(if vert
	    (if (< 8 w)
		(xlib:draw-rectangle res gray50-gc 4 begin (- w 8) len t))
	    (if (< 8 h)
		(xlib:draw-rectangle res gray50-gc begin 4 len (- h 8) t)))
	
	;; Draw in the trim on the slider...
	(if vert
	    (draw-3d-border self black-gc white-gc 2 begin (- w 4) len)
	    (draw-3d-border self black-gc white-gc begin 2 len (- h 4)))
	
	;; Draw center stripe...
	(if vert
	    (if (< 8 w)
		(progn
		 (xlib:draw-rectangle res white-gc 4 (1+ center) (- w 8) 1 t)
		 (xlib:draw-rectangle res black-gc 4 (1- center) (- w 8) 2 t)))
	    (if (< 8 h)
		(progn
		 (xlib:draw-rectangle res white-gc (1+ center) 4 1 (- h 8) t)
		 (xlib:draw-rectangle res black-gc (1- center) 4 2 (- h 8) t))))
	
	;; Do we need to redraw the buttons?
	(if (or (and (or bottom right) (>= (+ begin len) bs))
		(and (or top left) (<= begin (+ bs bl))))
	    (repaint-sb-buttons self)
	    ;; Just draw in the trim on the window
	    (draw-3d-border self black-gc white-gc 0 0 w h :invert t))))

(defun repaint-sb-buttons (self)
  (let* ((h (height self))
	 (w (width self))
	 (vert (vertical-p self))
	 (res (res self))
	 (up-bm-res (res (get-image "pan-up")))
	 (down-bm-res (res (get-image "pan-down")))
	 (left-bm-res (res (get-image "pan-left")))
	 (right-bm-res (res (get-image "pan-right")))
	 (bm-w 12)
	 (black-gc (black-gc self))
	 (white-gc (white-gc self))
	 (gray50-gc (gray50-gc self))
	 (gray25-gc (gray25-gc self))
	 (button-up-gc (button-up-gc self))
	 (button-down-gc (button-down-gc self))
	 (bl (button-len self))
	 (bs (button-start self))
	 (prev-inv (prev-inverted self))
	 (next-inv (next-inverted self)))
	
	;; Set the background of the buttons
	(if vert
	    (xlib:draw-rectangle res gray50-gc 0 bs w bl t)
	    (xlib:draw-rectangle res gray50-gc bs 0 bl h t))
	
	;; Set the background of the prev-line-button if inverted
	(if prev-inv
	    (if vert
		(if (< 4 w)
		    (xlib:draw-rectangle res gray25-gc 2 (+ bs 2) (- w 4) 16 t))
		(if (< 4 h)
		    (xlib:draw-rectangle res gray25-gc (+ bs 2) 2 16 (- h 4) t))
		))
	
	;; Set the background of the next-line-button if inverted
	(if next-inv
	    (if vert
		(if (< 4 w)
		   (xlib:draw-rectangle res gray25-gc 2 (+ bs 20) (- w 4) 16 t))
		(if (< 4 h)
		   (xlib:draw-rectangle res gray25-gc (+ bs 20) 2 16 (- h 4) t))
		))
	
	;; Plop in the prev bitmap 
	(if vert
	    (xlib:put-image res (if prev-inv button-down-gc button-up-gc)
			    up-bm-res 
			    :x (round (- w bm-w) 2) 
			    :y (+ bs 5)
			    :bitmap-p t)
	    (xlib:put-image res (if prev-inv button-down-gc button-up-gc) 
			    left-bm-res
			    :x (+ bs 4) 
			    :y (round (- h bm-w) 2) 
			    :bitmap-p t))
	
	;; Plop in the next bitmaps
	(if vert
	    (xlib:put-image res (if next-inv button-down-gc button-up-gc)
			    down-bm-res 
			    :x (round (- w bm-w) 2)
			    :y (+ bs 23)
			    :bitmap-p t)
	    (xlib:put-image res (if next-inv button-down-gc button-up-gc)
			    right-bm-res 
			    :x (+ bs 23)
			    :y (round (- h bm-w) 2)
			    :bitmap-p t))
	
	;; Draw in the trim on the prev button...
	(if vert
	    (draw-3d-border self black-gc white-gc 2 (+ bs 2) (- w 4) 16)
	    (draw-3d-border self black-gc white-gc (+ bs 2) 2 16 (- h 4)))
	
	;; Draw in the trim on the next button...
	(if vert
	    (draw-3d-border self black-gc white-gc 2 (+ bs 20) (- w 4) 16)
	    (draw-3d-border self black-gc white-gc (+ bs 20) 2 16 (- h 4)))
	
	;; Draw in the trim on the window
	(draw-3d-border self black-gc white-gc 0 0 w h :invert t)))

(defmethod do-repaint ((self scroll-bar)
		       &key 
		       &allow-other-keys)
  (let* ((h (height self))
	 (w (width self)))
	
	;; Clear the background.
	(clear-region self 0 0 w h)
	
	(repaint-slider self)
	(repaint-sb-buttons self)))

;;; =================================================================

(defhandler drop-on-floor ((self scroll-bar) &rest args
			   &default ((:button-release) (:pointer-motion)))
  (declare (ignore self args)))

(defun next-prev-line-event-loop (slot self event)
  (let ((pause-sec (pause-seconds self)))
       (declare (integer ev et id))
       (loop
	(execute slot self event)
	(my-sleep pause-sec)
	(if (eq 
	     :abort
	     (event-sync
	      :discard-after-process t
	      :handler
	      #'(lambda (&rest args &key event-key &allow-other-keys)
			(cond ((eq event-key :button-release)
			       :abort)
			      ((eq event-key :motion-notify))
			      (t 
			       (apply #'dispatch-event args)
			       nil)))))
	    (return)))))

(defhandler select ((self scroll-bar) &rest args &key x y &allow-other-keys
		    &default :button-press)
  (let* ((vert (vertical-p self))
	 (horiz (not vert))
	 (w (width self))
	 (h (height self))
	 (pos (if vert y x))
	 (bs (button-start self))
	 (bl (button-len self))
	 (len (len self))
	 (bl2 (/ bl 2))
	 (begin (begin self))
	 (top (buttons-top self))
	 (bottom (buttons-bottom self))
	 (right (buttons-right self))
	 (left (buttons-left self)))
	
	(cond ((or (and vert top (<= 0 pos bl2))
		   (and horiz left (<= 0 pos bl2))
		   (and vert bottom (<= bs pos (+ bs bl2)))
		   (and horiz right (<= bs pos (+ bs bl2))))
	       (when (prev-line-func self)
		     (setf (prev-inverted self) t)
		     (repaint-sb-buttons self)
		     (next-prev-line-event-loop 'prev-line-func self args)
		     (setf (prev-inverted self) nil)
		     (repaint-sb-buttons self)))

	      ((or (and vert top (<= 0 pos bl))
		   (and horiz left (<= 0 pos bl))
		   (and vert bottom (<= (+ bs bl2) pos h))
		   (and horiz right (<= (+ bs bl2) pos w)))
	       (when (next-line-func self)
		     (setf (next-inverted self) t)
		     (repaint-sb-buttons self)
		     (next-prev-line-event-loop 'next-line-func self args)
		     (setf (next-inverted self) nil)
		     (repaint-sb-buttons self)))
	       
	       ((< pos begin)
		(if (prev-page-func self) (execute 'prev-page-func self)))
	       
	       ((> pos (+ begin len))
		(if (next-page-func self) (execute 'next-page-func self)))
	       
	       (t
		(when (moved-func self)
		      (execute 'moved-func self args)
		      (repaint self)))))
  nil)

;;;
;;;   XXXXXXXXX -- Need to fix up 'execute' code to pass args as keywords.
;;;
(defun drag-scroll-bar (sb func args)
  (let* ((res (res sb))
	 (black-gc (gc-res sb))
	 (weave-gc (weave-gc sb))
	 (data (data sb))
	 (et nil)
	 (moved t)				;; Did we actually move?
	 (sb-len (1+ (len sb)))
	 (sb-loc 0.0)
	 (sb-begin (begin sb))
	 (bs (button-start sb))
	 (vert (vertical-p sb))
	 (w (width sb))
	 (h (height sb))
	 (l (- (if vert h w) 42))	;; size of slider along long dim
	 (ll (lower-limit sb))
	 (ul (upper-limit sb))
	 ;; conversion factor between pixels and slider coords
	 (factor (if (zerop l) 0 (/ (- ul ll) l)))
	 ;; Min/max legal value of event position
	 (min-pos (if (= 2 bs) 40 2))
	 (max-pos (+ min-pos (- l -2 sb-len)))
	 (ev-x (getf args :x))
	 (ev-y (getf args :y))
	 ;; Diff between mouse position and top of slider.
	 (delta (- sb-begin (if vert ev-y ev-x))))
	(declare (integer sb-len sb-begin w h l max-pos ev-x ev-y delta)
		 (number sb-loc ll ul factor))
	
	;;; main loop
	(event-loop :handler
#'(lambda (&rest args)
	  (setq et (getf args :event-key))
	  (cond ((eq et :motion-notify)
		 (setq ev-x (getf args :x)
		       ev-y (getf args :y))
		 (cond (vert
			;; Adjust coords and validate
			(setq ev-y (clamp min-pos (+ ev-y delta) max-pos))
			;; Did we actually move?
			(setq moved (not (= sb-begin ev-y)))
			(when moved
			      ;; Do fancy drag graphics
			      ;; Move-area params: (src-x src-y w h dst-x dst-y)
			      (xlib:copy-area res black-gc 2 sb-begin
					      (- w 4) (1- sb-len) res 2 ev-y)
			      ;; Fill in missing background
			      (if (> sb-begin ev-y)
				  (xlib:draw-rectangle 
				   res weave-gc 2 (+ sb-len ev-y)
				   (- w 4) (- sb-begin ev-y) t)
				  (xlib:draw-rectangle
				   res weave-gc 2 sb-begin
				   (- w 4) (- ev-y sb-begin) t))
			      ;; Update slider...
			      (setq sb-begin ev-y)
			      (setq sb-loc (+ ll (* factor ev-y)))
			      (setf (slot-value sb 'slider-location) sb-loc)))
		       (t
			;; Horizontal -- same as vert
			(setq ev-x (clamp min-pos (+ ev-x delta) max-pos))
			(setq moved (not (= sb-begin ev-x)))
			(when moved
			      (xlib:copy-area res black-gc sb-begin 2
					      (1- sb-len) (- h 4)
					      res ev-x 2)
			      (if (> sb-begin ev-x)
				  (xlib:draw-rectangle 
				   res weave-gc 
				   (+ sb-len ev-x) 2
				   (- sb-begin ev-x) (- h 4) t)
				  (xlib:draw-rectangle 
				   res weave-gc sb-begin 2
				   (- ev-x sb-begin) (- h 4) t))
			      (setq sb-begin ev-x)
			      (setq sb-loc (+ ll (* factor ev-x)))
			      (setf (slot-value sb 'slider-location) sb-loc))))
		 ;; Call users callback function
		 (if (and moved func) (funcall func sb data)))
		((eq et :button-release)
		 ;; Button released: Clean up and update slider data.
		 (setq ev-x (getf args :x)
		       ev-y (getf args :y))
		 (setq ev-y (clamp min-pos (+ ev-y delta) max-pos)
		       ev-x (clamp min-pos (+ ev-x delta) max-pos))
		 (decf ev-y min-pos)
		 (decf ev-x min-pos)
		 (if vert
		     (setq sb-loc (+ ll (* factor ev-y)))
		     (setq sb-loc (+ ll (* factor ev-x))))
		 (event-sync :windows sb)
		 (setf (slider-location sb) sb-loc)
		 (return-from drag-scroll-bar))
		(t
		 (apply #'dispatch-event args)))
	  nil))))
