;;;
;;; Postgres/CommonLISP interface
;;;
;;; Copyright (c) 1986 Regents of the University of California
;;; 
;;; Permission to use, copy, modify, and distribute this software and its
;;; documentation for any purpose and without fee is hereby granted,
;;; provided that the above copyright notice appear in all copies and
;;; that both that copyright notice and this permission notice appear in
;;; supporting documentation, and that the name of the University of
;;; California not be used in advertising or publicity pertaining to
;;; distribution of the software without specific, written prior
;;; permission.  The University of California makes no representations
;;; about the suitability of this software for any purpose.  It is
;;; provided "as is" without express or implied warranty.
;;; 
;;; $Author: bsmith $
;;; $Source: /pic2/picasso/new/widgets/libpq/RCS/pqfn.cl,v $
;;; $Revision: 1.1 $
;;; $Date: 1991/08/04 19:08:18 $
;;;

(in-package libpq)

;;;
;;; Connect to the backend
;;;
(defun connect-to-backend (&key (no-error-p nil))
  ;; If the communication is not established
  (when (null *pqportset*) 
	(if (null *initstr*)
	    (if (null (read-initstr no-error-p))
		(return-from connect-to-backend nil)))
	;; If the backend cannot be reached, signals an error
	(if (equal (cpconnect *pqhost* *pqport*) -1)
	    (if no-error-p
		(return-from connect-to-backend nil)
	      (error "Fatal -- No POSTGRES backend to connect to")))
	;; send the initstr to the backend
	(pqdebug (format nil "~%"))
	(pqdebug (format nil "Initstr sent to the backend:  ~A" *initstr*))
	(cputstr *initstr*) 
	(cpflush) 
	(setq *pqportset* t)))

;;; 
;;; Send the arguments to the backend
;;; 
(defun send-args (args)
  ;; Send the number of arguments
  (cputint (length args) 4)
  ;; Send all the arguments 
  (dolist (a args)
	  ;; Each arg is a pair (value-length value)
	  (let ((length (car a))
		(value (cadr a)))
	    ;; Send the length
	    (cputint length 4)
	    ;; Hack
	    ;; How can I get the binary value given the variable?
	    (typecase value
		      (integer (cputint value (abs length)))
		      (string (cputstr value))
		      (t (error "Only arguments of types integer and string are allowed."))))))

;;;
;;; Excute a function at the backend.
;;;
(defun pqfn (fnid result-length &rest args)
  "Execute the function \fBfnid\fP with \fBargs\fP"
  (connect-to-backend)
  ;; Send the function to the backend, details documented in the protocol
  ;; 'F' indicates a function
  (cputnchar "F" 1)
  ;; Xactid of the transaction in which this function is executed
  ;; Does not mean anything now, may be used in the future
  (cputint *pqxactid* 4)
  ;; The function id
  (cputint fnid 4)
  ;; The length of the return value.
  (cputint result-length 4)
  ;; Send the arugments
  (send-args args)
  ;; Flush
  (cpflush)
  ;; Process return values
  (process-return-values result-length))
  
;;;
;;; Process the returning values from the backend
;;; If an error message is returned from the backend, an error is signaled.
;;; If the function is executed successfully, two values are returned,
;;; the first is a T, the second is a list ("V" value) or ("P" portal-name),
;;; where value is an array of fixnum and each element is a byte.
;;;
(defun process-return-values (result-length &optional no-error-p)
  ;; Result-length is sent by the backend, may need to
  ;; check for consistency.
  (let ((fnid 0)
	(id "?")
	(errormsg (make-string error-msg-length :initial-element '#\a)))
    ;; Get the identifier
    (cgetpid id 0 1)
    (setq fnid (cgetpint 4))
    (cond 
     ;; Error
     ((equal id "E")
      (let ((strlen (cgetpstr errormsg error-msg-length)))
	(error "Error when executing the function: ~A"
	       (subseq errormsg 0 strlen))))
     ;; Return value
     ((equal id "V")
      (cgetpid id 0 1)
      (cond
       ;; G stands for an ordinary value
       ((equal id "G")
	(let ((value-length (cgetpint 4))
	      (value nil))
	  (cond
	   ;; Return value is an integer
	   ((and (<= value-length 4) (> value-length 0))
	    (setq value (cgetpint value-length)))
	   ;; Return value is a variable length string
	   ((member value-length '(-6 -1)) ; for now it is a string
	    (setq value (make-string 100 :initial-element '#\a))
	    (setq value-length (cgetpstr value 100)))
	   ((> value-length 4)
	    (setq value (make-array  100 :element-type '(unsigned-byte 8)))
	    (cgetpbytes value 100))
	   ;; Cases not considered now
	   (t (error "Returned value invalid.")))
	  ;; Read in the ending "0"
	  (cgetpid id 0 1)
	  (list "V" value)))
       ((equal id "C") ;; end of portal
	(cgetpint 4)
	(let ((command (make-string command-length :initial-element '#\c)))
	  (Cgetpstr command command-length)
	  (process-return-values result-length no-error-p)))
       ;; Return a portal
       ((or (equal id "P") (equal id "A"))
	(cgetpint 4)  ; getting xactid
	(process-portal no-error-p)
	(process-return-values result-length no-error-p))
       ;; No return value
       ((equal id "0")
	(list "V" nil))
       ;; Error otherwise
       (t (error 
	   "Protocol Error: Unexpected identifier ~S from the backend" id))))
     (t (error
	 "Protocol Error: Unexpected id: ~S fnid: ~S from the backend" id fnid)))))
