;;;
;;; Postgres/CommonLISP interface
;;;
;;; Copyright (c) 1986 Regents of the University of California
;;; 
;;; Permission to use, copy, modify, and distribute this software and its
;;; documentation for any purpose and without fee is hereby granted,
;;; provided that the above copyright notice appear in all copies and
;;; that both that copyright notice and this permission notice appear in
;;; supporting documentation, and that the name of the University of
;;; California not be used in advertising or publicity pertaining to
;;; distribution of the software without specific, written prior
;;; permission.  The University of California makes no representations
;;; about the suitability of this software for any purpose.  It is
;;; provided "as is" without express or implied warranty.
;;; 
;;; $Author: bsmith $
;;; $Source: /pic2/picasso/new/widgets/libpq/RCS/portal.cl,v $
;;; $Revision: 1.1 $
;;; $Date: 1991/08/04 19:08:18 $
;;;

(in-package libpq)

;;;
;;; Portal hash table maintenance functions
;;;

(defun putptable (pname portal-array)
  "Add an entry to the portal hash table given a portal name and a portal 
array"
  (setf (gethash pname *portal-table*) portal-array))

(defun rmptable (pname)
  "Delete an entry from the portal hash table given a portal name"
  (remhash pname *portal-table*))

(defun getptable (pname)
  "Return a portal array from the portal hash table given a portal name"
  (gethash pname *portal-table*))

;;;
;;; Sets up a portal for dumping data
;;;
(defun portal-setup(pname)
  "Set up a buffer for \fBpname\fP"
  (if (getptable pname)
      (portal-close pname))
  (putptable pname (getpblock)))

;;;
;;; Closes a portal, free up the buffer space
;;;
(defun portal-close(pname)
  "Free the buffer of \fBpname\fP and remove \fBpname\fP from *portal-table*"
  (let ((parray (getptable pname)))
    (when parray
	  ;; free up the buffer space
	  (let ((pblock parray)
		old-pblock
		(ntype-info (aref parray 1))
		(pblock-ptr 2)
		tblock
		iblock
		ntuple
		nfields)
	    (dotimes (tcnt ntype-info)
		     (when (= pblock-ptr (1- PBsize))
			   (setq old-pblock pblock)
			   (setq pblock (aref pblock pblock-ptr))
			   (setq pblock-ptr 0)
			   (putpblock old-pblock))
		     (setq ntuple (aref pblock pblock-ptr))
		     (setq pblock-ptr (1+ pblock-ptr))
		     (when (= pblock-ptr (1- PBsize))
			   (setq old-pblock pblock)
			   (setq pblock (aref pblock pblock-ptr))
			   (setq pblock-ptr 0)
			   (putpblock old-pblock))
		     (setq nfields (aref pblock pblock-ptr))
		     (setq pblock-ptr (1+ pblock-ptr))
		     (when (= pblock-ptr (1- PBsize))
			   (setq old-pblock pblock)
			   (setq pblock (aref pblock pblock-ptr))
			   (setq pblock-ptr 0)
			   (putpblock old-pblock))
		     (setq tblock (aref pblock pblock-ptr))
		     (setq pblock-ptr (1+ pblock-ptr))
		     ;; free the tuple blocks
		     (free-type tblock nfields)
		     ;; free the tuples
		     (dotimes (icnt ntuple)
			      (when (= pblock-ptr (1- PBsize))
				    (setq old-pblock pblock)
				    (setq pblock (aref pblock pblock-ptr))
				    (setq pblock-ptr 0)
				    (putpblock old-pblock))
			      (setq iblock (aref pblock pblock-ptr))
			      (setq pblock-ptr (1+ pblock-ptr))
			      (free-tuple iblock nfields)))
	    (if pblock
		(putpblock pblock)))))
    (rmptable pname)
  ;; code have to be added to free up the buffer space
  nil)

;;;
;;; Free type blocks
;;;
(defun free-type(tblock nfields)
  (let ((tblock-ptr 0)
	old-tblock)
    (dotimes (fcnt (1- nfields))
	     (setq tblock-ptr (+ tblock-ptr 3))
	     (when (> tblock-ptr (- TBsize 4))
		   (setq old-tblock tblock)
		   (setq tblock (aref tblock (1- TBsize)))
		   (setq tblock-ptr 0)
		   (puttblock old-tblock)))
    (if tblock 
	(puttblock tblock))))

;;;
;;; Free a tuple
;;;
(defun free-tuple(iblock nfields)			      
  (let ((iblock-ptr 0)
	old-iblock
	vblock
	vblock-ptr)
    (dotimes (fcnt nfields)
	     (when (> iblock-ptr (- IBsize 4))
		   (setq old-iblock iblock)
		   (setq iblock (aref iblock (1- IBsize)))
		   (setq iblock-ptr 0)
		   (putiblock old-iblock))
	     (setq vblock (aref iblock iblock-ptr))
	     (setq vblock-ptr (aref iblock (1+ iblock-ptr)))
	     (cond ((= vblock-ptr -1)
		    ;; a list of vblocks
		    (dolist (v vblock)
			    (putvblock v)))
		   (t 
		    ;; free the vblock, no problem if the same vblock has
		    ;; already being freed (putvblock calls pushnew)
		    (putvblock vblock)))
	     (incf iblock-ptr 3))
    (if iblock
	(putiblock iblock))))
			  

;;;
;;; Returns the hash value for a portal name
;;;
(defun pqparray(portal-name &optional no-error-p)
  "Returns the portal-array for \fBportal-name\fP"
  (let ((portal-array (getptable portal-name)))
    (if (or portal-array no-error-p)
	portal-array
      (error "There is no buffer for portal-name: ~A" portal-name))))
;;;
;;; Close a portal, free up the buffer space
;;;
(defun pqclose(portal-name)
  (portal-close portal-name))

(defun close-all-portals ()
  (maphash #'(lambda (key val)
	       (declare (ignore val))
	       (portal-close key))
	   *portal-table*))

;;;
;;; The internal structure of a portal buffer (using pblocks):
;;; -------------------------------------|
;;; | #tuples-in-this-buffer             |
;;; |------------------------------------|
;;; | #tuple-groups                      |
;;; |------------------------------------|
;;; | #tuples-in-group-0 (n0)            |
;;; |------------------------------------|
;;; | #fields-of-the-tuples-in-group-0   |
;;; |------------------------------------|
;;; | type-block-for-group-0             |    
;;; |------------------------------------|
;;; | index-block-of-tuple-0             |
;;; |------------------------------------|
;;; | index-block-of-tuple-1             |
;;; |------------------------------------|
;;; |           .                        |
;;; |           .                        |
;;; |           .                        | 
;;; |------------------------------------|
;;; | index-block-of-tuple-(1- n0)       |
;;; |------------------------------------|
;;; | #tuples-in-group-1 (n1)            |
;;; |------------------------------------|
;;; | #fields-of-the-tuples-in-group-1   |
;;; |------------------------------------|
;;; | type-block-for-group-1             |
;;; |------------------------------------|
;;; | index-block-of-tuple-n0            |
;;; |------------------------------------|
;;; |           .                        |
;;; |           .                        |
;;; |           .                        |
;;; |------------------------------------|
;;; | pointer-to-next-portal-block       |
;;; --------------------------------------
;;;
;;; Structure of a type-block (using tblocks)
;;; --------------------------------------
;;; | name-of-field-0                    |
;;; |------------------------------------|
;;; | type(adtid)-of-field-0             |
;;; |------------------------------------|
;;; | adtsize-of-type-0                  |
;;; |------------------------------------|
;;; | name-of-field-1                    |
;;; |------------------------------------|
;;; |           .                        |
;;; |           .                        |
;;; |           .                        |
;;; |------------------------------------|
;;; | pointer-to-next-type-block         |
;;; -------------------------------------
;;;
;;; Structure of an index-block (using iblocks):
;;; --------------------------------------
;;; | value-block-of-field-0             |
;;; |------------------------------------|
;;; | start-of-value-0-in-value-block    |
;;; |------------------------------------|
;;; | length-of-value-0                  |
;;; |------------------------------------|
;;; | value-block-of-field-1             |
;;; |------------------------------------|
;;; |           .                        |
;;; |           .                        |
;;; |           .                        |
;;; |------------------------------------|
;;; | pointer-to-next-value-block        |
;;; -------------------------------------
;;;
;;; A value block is simply a string (vblock).
;;;

;;;
;;; Returns the number of tuples in a portal
;;;
(defun pqntuples(portal-array &optional no-error-p)
  "Return the number of tuples in \fBportal-array\fP"
  (if (null-portal-array portal-array no-error-p)
      nil
    (aref portal-array 0)))

;;;
;;; Check if portal-array is NIL
;;;
(defun null-portal-array (portal-array &optional no-error-p)
  "Return T if \fBportal-array\fP is NIL and \fBno-error-p\fP is T"
  (if (null portal-array)
      (if no-error-p
	  T
	(error "The portal-array is NIL"))
    nil))

;;;
;;; Return the number of fields in the tuple from the portal array
;;;
(defun pqnfields(portal-array tuple-index &optional no-error-p)
  "Return the number of fields in the tuple \fBtuple-index\fP in \fBportal-array"
  ;; if portal-array is nil, signal an error or return nil depending on the
  ;; value of no-error-p
  (if (null-portal-array portal-array no-error-p)
      (return-from pqnfields nil))
  (let ((ntuple (aref portal-array 0))
	(pblock portal-array)
	(pblock-ptr 2))
    ;; test for 
    (cond ((or (minusp tuple-index)
	       (>= tuple-index ntuple))
	   ;; bad tuple index (negative or > number of tuples)
	   (if no-error-p
	       (return-from pqnfields nil)
	     (error "Tuple-index ~D is out of bound" tuple-index)))
	  ((= (aref portal-array 1) 1)
	   ;; only one tuple group -- return number of fields
	   (return-from pqnfields (aref portal-array 3))))
    ;; get number of tuples in portal block
    (setq ntuple (aref portal-array 2))
    ;; loop to find portal block containing tuple-index
    (loop
     ;; test if w/in portal block
     (if (< tuple-index ntuple)
	 ;; yes
	 (return))
     ;; decrement tuple index by number of tuples in block
     (decf tuple-index ntuple)
     ;; increment portal block pointer to next portal block
     (incf pblock-ptr 3)
     ;; test if portal block pointer is > portal block size
     (when (> pblock-ptr (- PBsize 2))
	   ;; get next portal block
	   (setq pblock (aref pblock (1- PBsize)))
	   ;; set portal block pointer to the beginning of new portal block
	   (setq pblock-ptr (- (1+ pblock-ptr) PBsize)))
     ;; get number of tuples in portal block
     (setq ntuple (aref pblock pblock-ptr)))
    ;; increment portal block pointer to number of fields in tuple
    (incf pblock-ptr)
    ;; test if portal block pointer is > portal block size
    (when (> pblock-ptr (- PBsize 2))
	  ;; get next portal block
	  (setq pblock (aref pblock (1- PBsize)))
	  ;; set portal block pointer to start of next portal block
	  (setq pblock-ptr (- (1+ pblock-ptr) PBsize)))
    ;; return number of fields
    (aref pblock pblock-ptr)))

;;;
;;; Find the type block for a tuple
;;;
(defun find-type-block (portal-array tuple-index &optional no-error-p)
  "Returns the type block for \fBtuple-index\fP in \fBportal-array\fP"
  (if (null-portal-array portal-array no-error-p)
      (return-from find-type-block nil))
  (let ((ntuple (aref portal-array 0))
	(pblock portal-array)
	(pblock-ptr 2)
	(nfields 0))
    ;; test for bad tuple index (negative or > all tuples)
    (if (or (minusp tuple-index)
	    (>= tuple-index ntuple))
	;; return nil -- error
	(if no-error-p
	    (return-from find-type-block nil)
	  (error "Tuple-index ~D is out of bound" tuple-index)))
    ;; get number of tuples in portal block
    (setq ntuple (aref portal-array 2))
    ;; loop until tuple index is < number of tuples
    (loop
     ;; test if w/in portal block
     (if (< tuple-index ntuple)
	 ;; yes
	 (return))
     ;; decrement tuple index by number of tuples in portal block
     (decf tuple-index ntuple)
     ;; increment portal block pointer to next tuple block
     (incf pblock-ptr 3)
     ;; test if portal block pointer > portal block size
     (when (> pblock-ptr (- PBsize 2))
	   ;; get next portal block
	   (setq pblock (aref pblock (1- PBsize)))
	   ;; set portal block pointer to start of new portal block
	   (setq pblock-ptr (- (1+ pblock-ptr) PBsize)))
     ;; get number of tuples in portal block
     (setq ntuple (aref pblock pblock-ptr)))
    ;; increment portal block pointer to number of fields in tuple
    (incf pblock-ptr)
    ;; test if portal block pointer is > portal block size
    (when (> pblock-ptr (- PBsize 2))
	  ;; get next portal block
	  (setq pblock (aref pblock (1- PBsize)))
	  ;; set portal block pointer to start of new portal block
	  (setq pblock-ptr (- (1+ pblock-ptr) PBsize)))
    ;; get the number of fields
    (setq nfields (aref pblock pblock-ptr))
    ;; increment portal block pointer to type block for tuple
    (incf pblock-ptr)
    ;; test if portal block pointer is > portal block size
    (when (> pblock-ptr (- PBsize 2))
	  ;; get next portal block
	  (setq pblock (aref pblock (1- PBsize)))
	  ;; set portal block pointer to start of next portal block
	  (setq pblock-ptr (- (1+ pblock-ptr) PBsize)))
    ;; return tblock
    (aref pblock pblock-ptr)))

;;;
;;; Synonym for find-type-block
;;;
(setf (symbol-function 'pqttype) (symbol-function 'find-type-block))

;;;
;;; Return the field number of a given field name w/in a tuple
;;;
(defun pqfnumber (portal-array tuple-index field-name &optional no-error-p)
  "Return the field number of a \fBfield-name in the tuple \fBtuple-index\fP in \fBportal-array\fP"
  ;; get type block for tuple
  (let ((tblock (find-type-block portal-array tuple-index no-error-p))
	(tblock-ptr 0)
	(nfields (pqnfields portal-array tuple-index no-error-p)))
    ;; if there is an error, return nil
    (if (or (null tblock) (null nfields))
	(return-from pqfnumber nil))
    ;; loop for field name
    (dotimes (fcnt nfields)
	     ;; test if field name matches
	     (if (string= (aref tblock tblock-ptr) field-name)
		 ;; return field position
		 (return-from pqfnumber fcnt))
	     ;; increment to next type block
	     (incf tblock-ptr 3)
	     ;; test if type block pointer is > type block size
	     (when (> tblock-ptr (- TBsize 4))
		    ;; get next type block
		    (setq tblock (aref tblock (1- TBsize)))
		    ;; set type block pointer to start of next type block
		    (setq tblock-ptr 0)))
    ;; if field-name does not exist, return nil or signal an error 
    ;; depending on the value of no-error-p
    (if no-error-p
	nil
      (error "Field-name ~A does not exist" field-name))))

;;;
;;; Return the name of a given field number w/in a tuple
;;;
(defun pqfname (portal-array tuple-index field-number &optional no-error-p)
  "Return the field-name of \fBfield-number\fP in the tuple \fBtuple-index\fP in \FBportal-array\fP"
  ;; get type block for tuple
  (let ((tblock (find-type-block portal-array tuple-index no-error-p))
	(tblock-ptr 0)
	(nfields (pqnfields portal-array tuple-index no-error-p)))
    ;; if there is an error, return nil
    (if (or (null tblock) (null nfields))
	(return-from pqfname nil))
    ;; if field-number is out-of-bound, return nil or signal an error 
    ;; depending on the value of no-error-p
    (if (or (minusp field-number) 
	    (>= field-number nfields))
	(if no-error-p
	    (return-from pqfname nil)
	  (error "The field-nubmer ~D is out of bound" field-number)))
    ;; loop to field number
    (dotimes (fcnt field-number)
	     ;; increment to next type block
	     (incf tblock-ptr 3)
	     ;; test if type block pointer is > type block size
	     (when (> tblock-ptr (- TBsize 4))
		   ;; get next type block
		   (setq tblock (aref tblock (1- TBsize)))
		   ;; set type block pointer to start of next type block
		   (setq tblock-ptr 0)))
    ;; return field name
    (aref tblock tblock-ptr)))

;;;
;;; Return the type of a field in a tuple of a portal array
;;;
(defun PQftype (portal-array tuple-index field-number &optional no-error-p)
  "Return the type of the field with \fBfield-number\fP in the tuple \fBtuple-index\fP in \fBportal-array\fP"
  ;; get type block for tuple
  (let ((tblock (find-type-block portal-array tuple-index no-error-p))
	(tblock-ptr 0)
	(nfields (pqnfields portal-array tuple-index no-error-p)))
    ;; if there is an error, return nil
    (if (or (null tblock) (null nfields))
	(return-from pqftype nil))
    ;; if field-number is out-of-bound, return nil or signal an error 
    ;; depending on the value of no-error-p
    (if (or (minusp field-number) 
	    (>= field-number nfields))
	(if no-error-p
	    (return-from pqftype nil)
	  (error "The field-nubmer ~D is out of bound" field-number)))
    ;; loop to field number
    (dotimes (fcnt field-number)
	     ;; increment to next type block
	     (incf tblock-ptr 3)
	     ;; test if type block pointer is > type block size
	     (when (> tblock-ptr (- TBsize 4))
		   ;; get next type block
		   (setq tblock (aref tblock (1- TBsize)))
		   ;; set type block pointer to start of next type block
		   (setq tblock-ptr 0)))
    ;; return field type
    (aref tblock (1+ tblock-ptr))))

;;;
;;; Determine if two tuple have the same type information
;;;
(defun pqsametype(portal-array tuple-index1 tuple-index2 &optional no-error-p)
  "Return T if the corresponding fields of the two tuples have the same type"
  (let ((tblock1 (find-type-block portal-array tuple-index1 no-error-p))
	(tblock2 (find-type-block portal-array tuple-index2 no-error-p)))
    ;; if there is an error, return nil
    (if (or (null tblock1) (null tblock2))
	(return-from pqsametype nil))
    (eq tblock1 tblock2)))

;;;
;;; Return the value of a field in a tuple of a portal array
;;;
(defun pqgetvalue(portal-array tuple-index field-number &optional no-error-p)
  "Returns the value of the field with \fBfield-number\fP in the tuple \fBtuple-index\fP in \fBportal-array\fP"
  (if (null-portal-array portal-array no-error-p)
      (return-from pqgetvalue nil))
  (let ((ntuple (aref portal-array 0))
	(pblock portal-array)
	(pblock-ptr 2)
	(iblock nil)
	(iblock-ptr 0)
	(nfields 0))
    ;; test for bad tuple index (negative or > all tuples)
    (if (or (minusp tuple-index)
	    (>= tuple-index ntuple))
	;; return nil or signal an error depending on no-error-p
	(if no-error-p
	    (return-from PQgetvalue nil)
	  (error "The tuple-index ~D is out of bound" tuple-index)))
    ;; get number of tuples in portal group
    (setq ntuple (aref portal-array 2))
    ;; loop until find the tuple group containing the tuple
    (loop
     ;; test if w/in portal group
     (if (< tuple-index ntuple)
	 ;; yes
	 (return))
     ;; decrement tuple index by number of tuples in this portal group
     (decf tuple-index ntuple)
     ;; increment portal block pointer to next tuple group
     (incf pblock-ptr 3)
     ;; test if portal block pointer > portal block size
     (when (> pblock-ptr (- PBsize 2))
	   ;; get next portal block
	   (setq pblock (aref pblock (1- PBsize)))
	   ;; set portal block pointer to start of new portal block
	   (setq pblock-ptr (- (1+ pblock-ptr) PBsize)))
     ;; get number of tuples in portal group
     (setq ntuple (aref pblock pblock-ptr)))
    ;; increment portal block pointer to number of fields in tuple
    (incf pblock-ptr)
    (when (> pblock-ptr (- PBsize 2))
	  ;; get next portal block
	  (setq pblock (aref pblock (1- PBsize)))
	  ;; set portal block pointer to start of new portal block
	  (setq pblock-ptr (- (1+ pblock-ptr) PBsize)))
    ;; get the number of fields
    (setq nfields (aref pblock pblock-ptr))
    ;; test if field number is > number of fields for tuple
    (if (or (minusp nfields) (> field-number nfields))
	;; return nil or signal an error depending on no-error-p
	(if no-error-p
	    (return-from pqgetvalue nil)
	  (error "The tuple-index ~D is out of bound" tuple-index)))
    ;; increment portal block pointer to index block for tuple
    ;; skip the type block for this group, thus plus 2
    (incf pblock-ptr (+ tuple-index 2))
    ;; loop until find the portal block that contains the tuple
    (loop
     ;; test if within this portal block
     (if (< pblock-ptr (- PBsize 1))
	 ;; yes
	 (return))
     ;; get next portal block
     (setq pblock (aref pblock (1- PBsize)))
     ;; set portal block pointer to start of new portal block
     (setq pblock-ptr (- (1+ pblock-ptr) PBsize)))
    ;; get index block for tuple
    (setq iblock (aref pblock pblock-ptr))
    ;; initialize index block pointer
    (setq iblock-ptr 0)
    ;; loop to field number
    (dotimes (fcnt field-number)
	     ;; increment to next index block
	     (incf iblock-ptr 3)
	     ;; test if index block pointer is > index block size
	     (when (> iblock-ptr (- IBsize 4))
		   ;; get next index block
		   (setq iblock (aref iblock (1- IBsize)))
		   ;; set index block pointer to start of next index block
		   (setq iblock-ptr 0)))
    ;; get value from value block
    (let* ((vblock (aref iblock iblock-ptr))
	   (vblock-ptr (aref iblock (incf iblock-ptr)))
	   (vlen (aref iblock (incf iblock-ptr))))
      ;; test if value block is null
      (cond ((null vblock)
	     ;; return empty string -- error
	     "")
	    ((>= vblock-ptr 0)
	     ;; value within one block, return the value
	     (subseq vblock vblock-ptr (+ vblock-ptr vlen)))
	    (t 
	     ;; large value, span several blocks
	     (let ((result "")
		   (vblock-list vblock))
	       (setq vblock (car vblock-list))
	       (setq vblock-list (cdr vblock-list))
	       (loop
		(when (<= vlen VBsize)
		      (setq result (format nil "~A~A" result 
					   (subseq vblock 0 vlen)))
		      (return))
		(setq result (format nil "~A~A" result vblock))
		(setq vblock (car vblock-list))
		(setq vblock-list (cdr vblock-list))
		(setq vlen (- vlen VBsize)))
	       result))))))
