(in-package "PT")

;;;***********************************************************
;;;
;;;  HYPERDOC class, methods, and support functions
;;;

(defdbclass hyperdoc (node-set owned-obj)
  ;; anything else need to go in here??
  ((description :initform "A network of nodes and links")
   (start-node :initarg :start-node :initform nil :type node 
               :accessor start-node
               :documentation "Node to open when hyperdoc is first selected")
   (links :initarg :links :initform nil :type list :accessor links)
   (paths :initform nil :type list :accessor paths
	  :documentation "Paths defined through this hyperdocument")
   (default-path :initarg :path :initform "" :type string :accessor default-path
              :documentation "Directory to look in for nodes in this hyperdoc")
   )
  (:documentation "Set of nodes and links that comprise a hyperdocument")
)

(defun make-hyperdoc (&rest args)
  (apply #'make-instance 'hyperdoc args))

(defmethod new-instance ((h hyperdoc) &key name nodes start-node
			 dataset (owner *user*) perms
                         &allow-other-keys)
  (if (and name (string= dataset ""))
      (setf (dataset h)
            (concatenate 'string *hyperdoc-info-directory*
                         (string name)
                         ".data")))
  (unless perms (add-perms h owner '(all)))
  ;; if no start-node is given, use the first of the nodes specified,
  ;; if any:
  (if nodes
      (dolist (n nodes)
              (add-obj n h)))
  (if (null start-node)
      (if nodes (setf (start-node h) (car nodes))))
  (store h))

(defmethod store ((h hyperdoc))
   ;;  (setf (gethash (name h) *hyperdocs*) h)
  ;; I'm changing this back to a list to simplify variable and slot
  ;; bindings in various widgets:
  (pushnew h #!*hyperdocs*))

(defmethod unstore ((h hyperdoc))
  ;;  (remhash (name h) *hyperdocs*)
  (db-unstore-object h)
  (setf #!*hyperdocs* (remove h #!*hyperdocs*)))

(defun get-hyperdoc-names (&optional (hyperdoc-list #!*hyperdocs*))
  (if (not (typep hyperdoc-list 'cons)) (setq hyperdoc-list hyperdoc-list))
  ;; this returns a list of strings enclosed in lists, as required by tables:  
  (mapcar #'(lambda (x) (list (string (name x)))) hyperdoc-list))
#|
  (maphash #'(lambda (key hyperdoc) 
               ;; collect all hyperdoc names into a list of lists [for
               ;; display in table]:
               (setq names (append names (list (list (string (name hyperdoc)))))))
           hyperdoc-list)
  names)
|#

(defmethod bookmarks ((h hyperdoc))
  (apply #'append (mapcar #'bookmarks (nodes h))))

(defmethod open-hyperdoc ((h hyperdoc) &optional (make-current? t))
  (pushnew h #!*open-hyperdocs*)
  (when make-current? (select-hyperdoc h)))

(defmethod select-hyperdoc ((h hyperdoc))
  "makes hyperdoc h the current hyperdoc"
  (setf #!*current-hyperdoc* h)
   (if (start-node h) (open-node (start-node h))))

(defun open-hyperdocs (hyperdoc-list)
  "open all hyperdocs on list, making first one current"
  (open-hyperdoc (car hyperdoc-list) t)
  (dolist (h (cdr hyperdoc-list))
          (open-hyperdoc h nil)))

(defun close-hyperdoc (h)
  (when h
        (if (or (stringp h) (symbolp h)) (setq h (get-hyperdoc-named h)))
        (setf #!*open-hyperdocs*
              (remove h #!*open-hyperdocs*))
        (setf #!*current-hyperdoc* 
              (if #!*open-hyperdocs* 
                  (car #!*open-hyperdocs*) 
                nil))
        (dolist (n (nodes h))
                (close-node n))))

(defmethod browse ((h hyperdoc))
  "invokes graphical browser on hyperdoc"
  (when *widgets-loaded*
	(with-feedback "Generating graphical representation..."
             (call (find-po-named '("new-hip" "browser" . "panel")) :hyperdoc h))))

(defmethod links ((obj t))
  nil)
(defmethod nodes ((obj t))
  nil)

(defmethod edit ((h hyperdoc))
  (let ((result (call (find-po-named '("new-hip" "edit-hyperdoc" . "dialog"))
                                     :hyperdoc h)))
    (when result
          (update h result)
	  (modify h))))

(defmethod rename ((h hyperdoc) new-name)
  (when (not (= (name h) new-name))
        (let* ((data-file (dataset h))
               (new-file (concatenate 'string 
                                      (subseq data-file 0 (search (name h) data-file))
                                      (string new-name)
                                      ".data")))
          (rename-file data-file new-file)
          (setf (dataset h) new-file)
          (setf (name h) new-name)
	  (modify h))))

(defmethod copy ((h hyperdoc))
  (let* ((copy-name (read-from-string (get-string :prompt "Enter name for copy: ")))
         (copy (make-hyperdoc :name copy-name 
                              :default-path (default-path h)
                              :nodes (nodes h)
                              :links (links h)
                              :start-node (start-node h)
                              :description (format nil "Copy of hyperdoc ~a"
                                                   (name h)))))
    ;;  Alternately, we could make copies of all the nodes and links,
    ;;  but for now I'm just making a shared copy
    ;;    (dolist (n (nodes h))
    ;;      (add-obj (copy n) copy))  ;; also adds links      
    copy))


(defmethod import-node ((h hyperdoc) (n node) 
                        &key (import-links t) (by-copy nil))
  (if (member n (nodes h))
      ;; add-obj actually checks for this, but since it's a slightly
      ;; different situation I thought perhaps I'd deal with it here, too...
      (announce-error 
       (format nil "Node ~a already belongs in hyperdoc ~a" (name n) (name h)))) 
  (with-feedback (format nil "Importing node ~a into hyperdoc ~a"
			 (name n) (name n))
		 (if by-copy
		     (add-obj (copy n) h import-links)
		   (add-obj n h import-links))))

