
;;;
;;;  FILE-IO.CL
;;;  This file contains routines for saving HIP objects to files and
;;;  recreating them from that stored information.  These capabilities
;;;  are used to save link information about nodes, for use in editing
;;;  that info outside of HIP, and also to save hyperdoc structure
;;;  outside of the database.

(in-package "PT")

;;;  Saving/loading hyperdocs:

(defmethod write-hyperdoc ((h hyperdoc))
  (with-open-file (str (dataset h) :direction :output :if-exists
		       :new-version :if-does-not-exist :create)
	  (format str ";; Hyperdoc ~a~%" (name h))
	  (format str "(~%")
	  ;;  save: owner, perms, descr, keywds, nodes, start-node,
	  ;;  links, default-path
	  (dolist (attr '(name owner perms description keywords default-path))
		  (format str " (~a . ~s) ~%"  attr (slot-value h attr)))
	  (when (start-node h)
		(format str " (START-NODE . ~a) ~%"  (name (start-node h))))
	  (when (nodes h)
		(format str " (NODES . (")
		(dolist (val (mapcar #'(lambda (n) (list (name n) (linkset n)))
				     (nodes h)))
			(format str "~s~%" val))
		(format str "))~%"))
	  (when (links h)
		(format str " (LINKS . (")
		(dolist (val (mapcar #'label (links h)))
			(format str "~s~%" val))
		(format str "))~%"))
	  (when (paths h)
		(format str " (PATHS . (")
		(dolist (p (paths h)) (write-path p str))
		(format str "))~%"))
	  (format str ")~%"))
  ;; save all node info, too:
  (dolist (n (nodes h)) (if (modified? n) (write-node n)))
  (feedback (format nil "Wrote file ~s" (dataset h)))
  (unmodify h))

(defun load-hyperdoc-from-file ()
  (let ((file (call (find-po-named '("new-hip" "load-hyperdoc" . "dialog")))))
    (when file
	  (load-hyperdoc file))))

;; NOTE: need to add check that the desired hyperdoc has not already
;; been loaded, and if so maybe we should update it with the data from
;; the file...  
(defmethod load-hyperdoc (file &aux data)
  (if (symbolp file)              ;; passed in doc name instead of pathname
      (setf file (concatenate 'string (string file) ".data")))
  (if (not (find #\/ file :test #'char=))  ;; passed in file name only
      (setf file (concatenate 'string
			      *hyperdoc-info-directory*
			      (string file))))
  (format t "~%Loading hyperdoc from ~a..." file)
  (with-open-file (str file :direction :input :if-does-not-exist :error)
       (setq data (read str)))
  (let ((hd (make-hyperdoc :name (cdr (assoc 'name data))
		 :owner (cdr (assoc 'owner data))
		 :allow-other-keys t
		 :perms (cdr (assoc 'perms data))
		 :description (cdr (assoc 'description data))
		 :keywords (cdr (assoc 'keywords data))
		 :dataset file
		 :default-path (cdr (assoc 'default-path data))))
	(old-current-doc (ch)))  ;; store value of current-hyperdoc
    ;; fetch all the nodes (will automatically fetch the links, too):
    (setf #!*current-hyperdoc* hd)  ;; temporarily, to set context for
				    ;; load operations
    (dolist (node-data (cdr (assoc 'nodes data)))
	    (load-node (car node-data) (second node-data) hd))
    ;; now we can look up the start-node:
    (setf (start-node hd) (get-node-named (cdr (assoc 'start-node data))))
    ;; finally, now that we have all the nodes and markers, we can fetch
    ;; the real endpoints of all the links:
    (dolist (l (links hd))
        ;; if its dest didn't exist when the link was created, get it now:
        (if (and (listp (dest l)) (get-node (second (dest l))))
	    (set-dest l 
		      (get-link-marker-in-node (car (dest l)) (second (dest l))))))
    (dolist (p (cdr (assoc 'paths data)))
	    (load-path p hd))
    (setf #!*current-hyperdoc* old-current-doc)
    (open-hyperdoc hd)
    hd))

;;;  Saving/loading nodes:

;; (defmethod write-node ((wn wip-node))
;;  (warn "Can't save wip link data yet..."))

(defmethod write-node ((n node))
  (let ((file (linkset n)))
    (with-open-file (str file :direction :output 
			 :if-exists :new-version :if-does-not-exist :create)
       (format str ";; Node ~a~%" (name n))
       (format str "(~%")
       ;;  save: name, type, owner, perms, descr, keywds, links, dataset
       (dolist (attr '(name type owner perms description keywords dataset))
	       (format str " (~a . ~s) ~%"  attr (slot-value n attr)))
       (format str " (HYPERDOCS . ~a) ~%" (mapcar #'name (hyperdocs n)))
       (save-link-info n str)
       (format str ")~%"))
    (feedback (format nil "Wrote file ~s" file))))

(defun load-node (name file hyperdoc)
  (cond ((get-node name)
	 (format t "Node ~a already loaded..." name)
	 (add-obj (get-node name) hyperdoc))
	(t
	 (with-feedback (format nil "~%Loading node ~a..." name)
	   (let* ((data (if (probe-file file)
			    (with-open-file (str file :direction :input) 
					(read str))))
		  (node (if data
			      (make-node (cdr (assoc 'type data))
					 :name name
					 :allow-other-keys t
					 :owner (cdr (assoc 'owner data))
					 :perms (cdr (assoc 'perms data))
					 :description (cdr (assoc 'description data))
					 :keywords (cdr (assoc 'keywords data))
					 :dataset (cdr (assoc 'dataset data))
					 :hyperdocs (remove-if #'null 
							       (mapcar #'get-hyperdoc 
								       (cdr (assoc 'hyperdocs data))))))))
			  #|
			      ((((make-node 'wip
					 :name name
					 :dataset file))))
                          |#
	     (when data
		   (dolist (lm-data (cdr (assoc 'link-markers data)))
			   (load-link-marker lm-data node))))))))


;;;  Saving/loading links and link markers:

(defun save-link-info (node str)
  (let ((link-markers (link-markers node)))
    (when link-markers
	  (format str " (LINK-MARKERS  ")
	  (dolist (b link-markers)
		  (write-link-marker b str))
	  (format str "~%  )~%"))))

(defun write-link-marker (b str)
  "writes list representing link-marker b to str, which must be an open
   output stream"
  (let ((links (links-from b)))
    (format str "~%    (")
    (format str "~%      (NAME . ~a)" (name b))
    (format str "~%      (LABEL . ~s)" (label b))
    (format str "~%      (REGION . ~a)" (region b))
    (when links
	  (format str "~%      (LINKS  ")
	  (dolist (l links)
		  (write-link l str))
	  (format str "~%      )")))
  (format str "~%     )")
)

(defun load-link-marker (lm-data parent)
  (let ((lm (make-link-marker :name (cdr (assoc 'name lm-data))
			      :allow-other-keys t
			      :label (cdr (assoc 'label lm-data))
			      :region (cdr (assoc 'region lm-data))
			      :parent parent)))
    (dolist (link-data (cdr (assoc 'links lm-data)))
	    (load-link link-data lm))
    lm))


(defun write-link (l str)
  "writes list representing link l to str, which must be an open
   output stream"
  (format str "~%       (")
  (format str "~%         (NAME . ~s)" (name l))
  (format str "~%         (LABEL . ~s)" (label l))
  (format str "~%         (TYPE . ~a)" (if (link-type l) (name (link-type l))))
  ;; for destination marker, save its (gensym-ed) name and name of parent node;
  ;; when this link is recreated, we'll look up the marker...not
  ;; pretty, but I don't want to rely on OBJIDs yet
  (format str "~%         (DEST . ~s)" (list (name (dest l))
					    (name (parent (dest l)))))
  (format str "~%       )")
)


(defun load-link (link-data source)
  (let* ((dest-info (cdr (assoc 'dest link-data)))
	 (dest-node (get-node-named (second dest-info)))
	 (dest (if dest-node (get-link-marker-in-node (car dest-info) dest-node))))
    (make-link :name (cdr (assoc 'name link-data))
	       :label (cdr (assoc 'label link-data))
	       :allow-other-keys t
	       :type (get-link-type (cdr (assoc 'type link-data)))
	       :source source
	       :hyperdocs (hyperdocs (parent source))
	       ;; if dest not yet created, just store the raw info for
	       ;; later lookup:
	       :dest (or dest dest-info))))

	
(defun write-path (path str)
  (format str "(~a ~s ~%               ~s)~%" 
	  (name path) (description path)
	  (mapcar #'name (nodes path))))

(defun load-path (path-data hyperdoc)
  (make-path :name (car path-data) 
	     :description (second path-data)
	     :nodes (mapcar #'(lambda (n) (get-node n hyperdoc))
			    (third path-data))))
