;;
;;  CLASS-METHODS.CL
;;  This file contains a bunch of methods that were cluttering up
;;  class-defs, mostly involving deleting objects and removing various
;;  types of objects from various other types.

(in-package "PT")

;;;  WIPE-OUT is a complete delete on an object, removing it from
;;;  the database.   Defined on markers, nodes, links, and hyperdocs. 
;;;  REMOVE-FROM-HYPERDOCS just removes the object from one or more
;;;  hyperdocs but doesn't delete it unless it will no longer belong
;;;  to any hyperdoc after being removed from those specified.
;;;  Defined on links and nodes.
;;;  REMOVE-XXX is the function called by the above to do the actual
;;;  removal.  Defined on all combinations that make sense.
;;;  DO-DELETE is a top-level function that queries the user as to
;;;  whether the object should be wiped out or just removed from the
;;;  current hyperdoc, then calls the appropriate method.

(defmethod wipe-out ((m marker))
  (when (parent m)
	(remove-obj m (parent m))
	(unstore m)))

(defmethod wipe-out ((lm link-marker))
  (call-next-method)
  (dolist (l (append (links-from lm) (links-into lm)))
	  (wipe-out l))
)

(defmethod wipe-out ((l link))
  "effectively deletes link by removing it from everybody who has
   contact with it" 
  (with-feedback (format nil "Deleting link ~a" (label l))
        (progn
          (remove-obj l (source l))
          (when (typep (dest l) 'link-marker)
		(remove-obj l (dest l) :to))
          (dolist (h (hyperdocs l))
                  (remove-obj l h))
          (unstore l))))  ;; remove from global set of links

(defmethod wipe-out ((n node))
  "effectively deletes node by removing it from everybody who has
   contact with it" 
  (with-feedback (format nil "Deleting node ~a" (name n))
         (progn
           ;; if node is being viewed, close its panel
           (close-node n)
           ;; get rid of all link-marker in the node, which in
	   ;; turn will wipe out the links:
           (dolist (b (link-markers n))
                  (wipe-out b))
	   (dolist (h (hyperdocs n)) 
		   (remove-obj n h))
	   (unstore n))))
		    
(defmethod wipe-out ((h hyperdoc))
  (when (ask-user-to-confirm 
         (format nil "Do you really want to delete hyperdocument ~a?" (name h)))
        (close-hyperdoc h)
        (dolist (n (nodes h))
                (remove-from-hyperdocs n (list h) :confirm? nil))
        (unstore h)))

(defmethod wipe-out ((p path))
  (dolist (l (links p))
	  (wipe-out l))
  (remove-type (get-link-type (name p)))
  (remove-obj p (car (hyperdocs p))))

;;;  Marker/node methods:

(defmethod add-obj ((b bookmark) (n node) &optional detail)
  "adds bookmark b to list of bookmarks defined in node n"
  (declare (ignore detail))
  (setf (bookmarks n) 
        (sort (copy-list (pushnew b (bookmarks n)))
              #'tuple-< 
              :key #'offset))
  (modify n))

(defmethod add-obj ((lm link-marker) (n node) &optional detail)
  "adds link-marker ln to list of link-markers defined in node n"
  (declare (ignore detail))
  (if (find (label lm) (link-markers n) :test #'equal :key #'label)
      ;; should change this to prompt the user for a unique name:
      (setf (label lm) (string (gensym "bookmark-label-"))))
  (if (null (region lm))  ;; i.e., it's a path marker
      (push lm (link-markers n))
    (let ((path-markers (remove-if-not #'null (link-markers n) :key #'region))
	  (real-markers (remove-if #'null (link-markers n) :key #'region)))
      (setf (link-markers n) 
	    (append path-markers
		    (sort (copy-list (push lm real-markers))
			  #'tuple-< 
			  :key #'offset)))))
  (modify n))

(defmethod add-obj ((m marker) (vn video-node) &optional detail)
  (declare (ignore detail))
  (call-next-method)
  (when (not (gethash (read-from-string (label m))
		      (disk-index (videodisk vn))))
	(enter-index (videodisk vn) 
		     (read-from-string (label m)) ;; key
		     (list (car (offset m)) (car (endpt m)) t t nil))))

  ;; do we want remove-obj to wipe out the index entry for the block in
  ;; a video node??

(defmethod remove-obj ((m marker) (n node) &optional detail)
  (declare (ignore detail))
  (let ((slot (if (bookmarkp m) 'bookmarks 'link-markers)))
    (setf (slot-value n slot) (remove m (slot-value n slot)))
    (setf (parent m) nil)
    (modify n)))


;;;  Link/link-marker methods:

(defmethod add-obj ((l link) (lm link-marker) &optional (dir :from))
  "adds link l to list of links coming from or to link-marker lm"
  ;; add check for label uniqueness?
  (case dir
	(:from (progn
		 (pushnew l (links-from lm))
		 (modify (parent lm))))
	(:to (pushnew l (links-into lm)))))


(defmethod remove-obj ((l link) (lm link-marker) &optional (dir :from))
  "removes link l from list of links coming from/into link-marker lm"
  (let ((slot (if (eq dir :from) 'links-from 'links-into)))
    (when (member l (slot-value lm slot))
        (setf (slot-value lm slot) (remove l (slot-value lm slot))))
    ;; see if we need this marker any more:
    (when (and (null (links-from lm))
	       (null (links-into lm)))
	  (wipe-out lm))
    (when (and (eq dir :from) (parent lm))
	  (remove-obj l (parent lm)))))


;;;  Link/node methods:

(defmethod remove-obj ((l link) (n node) &optional detail)
  (declare (ignore detail))
  (when (member l (links-from n))
	;; links aren't stored explicitly on nodes, so all we have to
	;; do is let the node know it's been changed:
        (modify n)))


;;;  Link/hyperdoc methods:

(defmethod remove-from-hyperdocs ((l link) hyperdocs &key &allow-other-keys)
  (if (and (equal hyperdocs (hyperdocs l))  ;; removing from all its hyperdocs
	   (ask-user-to-confirm "Link will not belong to any HyperDocuments after this, and should be deleted completely.  Proceed?"))
      (wipe-out l)
    (dolist (h hyperdocs)
            (remove-obj l h))))
        
(defmethod add-obj ((l link) (h hyperdoc) &optional detail)
  (declare (ignore detail))
  (pushnew l (links h))
  (pushnew h (hyperdocs l))
  (dolist (n (list (source-node l) (dest-node l)))
	  (if (and n (not (member n (nodes h))))
	      (add-obj n h)))
  (modify h))

(defmethod remove-obj ((l link) (h hyperdoc) &optional detail)
  (declare (ignore detail))
  "removes link l from list of links owned by hyperdoc h"
  (with-feedback (format nil "Removing link ~a from ~a" (name l) (name h))
                 (when (member l (links h))  ;; this is pretty extraneous...
                       ;; I don't trust "delete"...
                       (setf (hyperdocs l) (remove h (hyperdocs l)))
                       (setf (links h) (remove l (links h)))
		       ;; if no other links from h reach the dest of
		       ;; l, remove that node as well:
		       (if (and (dest-node l) 
				(notany #'(lambda (x) (member x (links h)))
					(remove l (links-into (dest-node l)))))
			   (remove-obj (dest-node l) h))))
  (modify h))


;;;  Node/hyperdoc methods:

(defmethod add-obj ((n node) (h hyperdoc) &optional (add-links? t))
  (cond ((and (not (member n (nodes h)))
	      (find (name n) (nodes h) :test #'equal :key #'name))
	 ;; should change this to prompt the user for a unique name:
	 (announce-error (format nil "A node of name ~a already exists in  ~a; ~%Node names must be distinct within a hyperdoc" 
			      (name n) (name h))))
	(t
	 (pushnew n (nodes h))
	 ;;  Need to see if the node is already identified with the
	 ;;  hyperdoc, which will happen if we're reloading the node
	 ;;  from a file:
	 (unless (find (name h) (hyperdocs n) :test #'equal :key #'name)
		 (pushnew h (hyperdocs n)))
	 (modify h)
	 (if add-links?
	     (dolist (l (links-from n))
		     (add-obj l h))))))

(defmethod remove-obj ((n node) (h hyperdoc) &optional detail)
  (declare (ignore detail))
  "removes node n from list of nodes owned by hyperdoc h"
  (with-feedback (format nil "Removing node ~a from ~a" (name n) (name h))
                 (when (member n (nodes h))
		       (close-node n)
                       (setf (hyperdocs n) (remove h (hyperdocs n)))
                       (setf (nodes h) (remove n (nodes h)))
		       (modify h)
		       (dolist (l (append (links-from n) (links-into n)))
			       (remove-obj l h)))))



(defmethod remove-from-hyperdocs ((n node) hyperdocs 
                                  &key (confirm? t) &allow-other-keys)
  (if (and (equal hyperdocs (hyperdocs n))  ;; removing from all its hyperdocs
	    (or (not confirm?)
		(ask-user-to-confirm 
	     (format nil "Node ~a will not belong to any hyperdocs after this, and will be deleted completely.  Proceed?" (name n)))))
      (wipe-out n nil)  ;; don't ask for confirmation again
    (dolist (h hyperdocs)
	  (remove-obj n h))))

;;
;; Path/hyperdoc methods:
;;

(defmethod add-obj ((p path) (h hyperdoc) &optional detail)
  (declare (ignore detail))
  (setf (hyperdocs p) (list h))
  (pushnew p (paths h)))  ;; might also need to set perms/owner?

(defmethod remove-obj ((p path) (h hyperdoc) &optional detail)
  (declare (ignore detail))
  (setf (hyperdocs p) nil)
  (setf (paths h) (remove p (paths h))))
  
;;
;;  DO-DELETE function:
;;

(defun do-delete (obj)
  (let ((option (call (find-po-named '("new-hip"  "confirm-delete" . "dialog")) 
		      :obj obj)))
    (when option
	  (if (zerop option) ;; wipe out
	      (wipe-out obj)
	    (remove-from-hyperdocs obj (list #!*current-hyperdoc*))))))
