
;;;;  HYPERMEDIA-MIXIN.CL
;;;;  This file contains the definition and methods for the mixin
;;;;  class used to add hypermedia functionality to other widget
;;;;  classes.  For example, the widget class hyper-text-widget would
;;;;  inherit from scrolling-text-widget and hypermedia-mixin.

(in-package "PT")

(defclass hypermedia-mixin (pmc)
  ((node :type node :initarg :node :initform nil :accessor node
	 :documentation "Node associated with this widget")
   (selected-marker :type 'marker :accessor selected-marker :initform nil
		    :documentation "Holds last marker selected by user")
   (show-markers :type atom :initform t :accessor markers-shown?
		   ;; could make this t, nil, or 'dynamic...
	       :documentation "when t, markers are displayed")
   (name :type string :initarg :name :initform "A hypermedia node viewer")
   (event-mask :initform '(:button-press :button-1-motion :keypress
					 :exposure :double-click))
   )
  (:documentation "Mixin class providing hypermedia behavior"))


(defmethod new-instance ((self hypermedia-mixin)
			 &key &allow-other-keys)
  (call-next-method)
)


;;;
;;;  Generic function specifications:
;;;  These methods should all be specialized by classes using the mixin.
;;;  (one of these days maybe I'll actually make these defgenerics...)
;;;

(defmethod widget ((self hypermedia-mixin))
  "returns the actual widget doing the I/O under the hyper-stuff"
  nil)

(defmethod setup-node ((self hypermedia-mixin) (n node))
  "reads in dataset of node, setting values of appropriate slots"
  nil)

(defmethod set-position ((self hypermedia-mixin) 
			 &key &allow-other-keys)
  "sets current position in widget to pos (a list of coords)"
  ;; specialized versions of this should just call the appropriate
  ;; method for the underlying widget
  nil)  

(defmethod get-current-position ((self hypermedia-mixin))
  "returns coords of current cursor position in widget"
  nil)

(defmethod get-mark-region ((self hypermedia-mixin))
  "returns currently-selected region in underlying widget, in the form
   of a list of tuples representing points"
  nil)

(defmethod exposed-markers ((self hypermedia-mixin))
  "returns list of markers currently visible in underlying widget"
  (visible-markers (node self)))

(defmethod unmark-marker ((self hypermedia-mixin) (b marker))
  "removes any highlighting (*not* outlining) from marker region"
  nil)

(defmethod scroll-to ((self hypermedia-mixin) offset)
  "scrolls widget to offset, which should be a cons of (row . col) or
   (x . y), etc." 
   (declare (ignore offset))
  nil)


;;
;; Methods to access attributes of the underlying node:
;;

(defmethod links ((self hypermedia-mixin))
  (links (node self)))

(defmethod link-markers ((self hypermedia-mixin))
  (link-markers (node self)))

(defmethod markers ((self hypermedia-mixin))
  (markers (node self)))

(defmethod bookmarks ((self hypermedia-mixin))
  (bookmarks (node self)))

;;;
;;; Methods to ensure that resetting the node propagates all other
;;; necessary changes to the object and the previously-displayed node:
;;;

(defmethod (setf node) :before ((n node) (self hypermedia-mixin))
  ;; make sure the old node doesn't still think it's being displayed:
  (if (node self) (setf (viewer (node self)) nil)))

(defmethod (setf node) :after ((n node) (self hypermedia-mixin))
  (setf (viewer n) self)
  (setup-node self n))

;;;
;;;  The repaint method has to make sure markers are displayed, if
;;;  appropriate.
;;;
(defmethod do-repaint ((self hypermedia-mixin) 
		       &key  &allow-other-keys)
  (call-next-method)
  (hm-repaint self))

(defun hm-repaint (self)
  (if (markers-shown? self) 
      (draw-markers self)
    ;; in some cases, self and (widget self) are the same thing, so we
    ;; need to check to avoid infinite loops while ensuring that all
    ;; children get repainted:
    (cond ((not (eql self (widget self)))
	   (repaint (widget self)))
	  ((slot-exists-p self 'children)
	   (dolist (c (children self)) (repaint c))))))

(defmethod draw-markers ((self hypermedia-mixin))
  (dolist (b (exposed-markers self))
       (draw-marker b (widget self))))

;;;
;;;  Methods for controlling display of markers:
;;;  (This will need to change if we want markers shown dynamically...)
;;;

(defmethod show-markers ((self hypermedia-mixin))
  (setf (markers-shown? self) t)
  ;;  I'm assuming that when a user asks to see all the markers, she
  ;;  no longer wants a particular one to be selected:
  (setf (selected-marker self) nil)
  (hm-repaint self))

(defmethod hide-markers ((self hypermedia-mixin))
  ;; subclasses should specialize the rest...
  (setf (markers-shown? self) nil)
  (hm-repaint self))

(defmethod toggle-marker-display ((self hypermedia-mixin))
  (if (markers-shown? self)
      (hide-markers self)
    (show-markers self)))

(defmethod select-link-marker ((self hypermedia-mixin) (lm link-marker))
  "outlines the designated link-marker, hiding all others"
  (make-current (my-panel self))  ;; make this panel the current one
  (hide-markers self)
  (draw-marker lm (widget self))
  (setf (selected-marker self) lm))

(defmethod unselect-current-marker ((self hypermedia-mixin))
  (setf (selected-marker self) nil))

(defmethod get-candidate-markers ((self hypermedia-mixin))
  "returns list of markers that contain current position in widget"
  (let ((pos (get-current-position self)))
    (when pos
	  (remove-if-not #'(lambda (b) (and (tuple-> pos (offset b))
					   (tuple-< pos (endpt b))))
			(remove-if #'(lambda (x) 
				       (null (region x)))
				   (markers (node self)))))))

(defmethod get-current-marker ((self hypermedia-mixin))
  "Returns the link-marker containing the current point in widget, if
   any. If more than one, should return the closest and smallest
   enclosing link-marker." 
  ;; else go on to determine the currently indicated one:
  (let ((pos (get-current-position self)))
    (when pos
	  (let* ((candidates (get-candidate-markers self))
		 ;; find the shortest distance from any marker offset to point:
		 (min-distance 
		  (when candidates
			(apply #'min 
			       (mapcar #'(lambda (x) (tuple-distance (offset x) pos))
				       candidates)))))
	    (if candidates
		;; get rid of any that start farther than the mininum distance, and take
		;; the smallest of those that remain:
		(car 
		 (sort 
		       (remove-if #'(lambda (x) (> (tuple-distance (offset x) pos)
						   min-distance))
				  candidates)
		       #'<
		       :key #'area))
	      nil)))))

(defmethod get-current-link-marker ((self hypermedia-mixin))
  ;; first see if markers are hidden and one has been selected:
  (if (and (not (markers-shown? self))
	   (selected-marker self))
      (selected-marker self)
    (let ((m (get-current-marker self)))
      (if (typep m 'link-marker)
	  m
	nil))))

(defmethod get-current-link-marker ((self widget))
  (if (typep (parent self) 'hypermedia-mixin)
      (get-current-link-marker (parent self))))

(defmethod get-current-bookmark ((self hypermedia-mixin))
  (let ((m (get-current-marker self)))
    (if (bookmarkp m)
	m
      nil)))

;;;
;;;  Misc. useful methods and functions:
;;;

(defmethod my-panel ((self hypermedia-mixin))
  "returns panel containing widget"
  (do ((p (parent self) (parent p)))
      ((eq (class-of p) (find-class 'panel)) p)))

(defmethod my-panel ((self text-widget))
  (if (typep (parent self) 'hypermedia-mixin)
      (my-panel (parent self))))

(defmethod return-from-node ((hm hypermedia-mixin))
  ;; retraces the path taken to arrive at the node being displayed
  (let ((link (opener (node hm))))
    (when link
	(let ((dir (if (member link (links-into (node hm))) :back :forw)))
	  (close-node (node hm))
	  (follow link :dir dir)))))

(defmethod create-bookmark ((self hypermedia-mixin))
  (make-bookmark :parent (node self)
		 :region (get-mark-region self)
		 :label (call (find-po-named '("new-hip" "create-marker" . "dialog")))))


;;;
;;;  Event handlers for navigational stuff:
;;;

;;; Control/left-mouse brings up a menu of links from the current link-marker:
(defhandler show-link-menu ((self hypermedia-mixin)
			    &rest args 
			    &default 
			    (:button-press :detail :left-button :state :control))
    (show-link-menu self args))

(defun show-link-menu (self &key x y &allow-other-keys)
  (let* ((lm ;; if there is a current link marker, use it; else use
	  ;; all of them in this node:
	  (or (get-current-link-marker self) (link-markers (node self))))
	 (links (if lm (filter (append (links-into lm) (links-from lm))) nil)))
    (when (and x y)
	  (set-position self :pos (list x y)))
    (if links
	(access-links-from lm)
      (announce-error "No valid links from current position"))))

;;;  Control/middle-mouse automatically follows the first link from
;;;  the current link-marker:
(defhandler follow-default-link ((self hypermedia-mixin)
				 &rest args
				 &default 
				 (:button-press :detail :middle-button :state :control))
  (follow-default-link self args))

(defun follow-default-link (self &rest args)
  (declare (ignore args))
  (let* ((lm (get-current-link-marker self))
	 (links (if lm (filter (links-from lm)) nil)))
    (when links
	(follow (car links))
	(feedback (format nil "~%Following link ~a.." (label (car links))))
	)
    ))
