
;;;
;;;  HYPER-TABLE-WIDGET
;;;  Widget for displaying tabular data with linking capabilities
;;;     

(in-package "PT")

;;;  Notes for me on matrix-fields:
;;;   relevant variables are current-field(s), row-index, col-index,
;;;   data-rows, data-cols; up/down/right/left-func (scrolling)

(defclass hyper-table-widget (hypermedia-mixin table-field)
  ((field-types :initform nil :initarg :field-types :accessor field-types 
	       :documentation "Type of data displayed in each column")
   (orig-values :initform nil :accessor orig-values)
   )
  (:documentation "Table field widgets with hypermedia behavior"))

(defun make-hyper-table-widget (&rest args)
  (apply #'make-instance 'hyper-table-widget :allow-other-keys t args))

(defmethod new-instance ((self hyper-table-widget) 
			 &key node 
			 (show-markers t)
			 &allow-other-keys)
  (call-next-method)
  ;; might need to define and install new up-func, etc. on scroll bars
  ;; to make scrolling work right if we define a repaint on these
  (when node
	(setup-node self node)
	(if show-markers
	    (draw-markers self)))
  ;; temporary kludge to get these guys to listen to the right events:
  ;;  (might be able to replace these with defhandlers again, since
  ;;  table-field has a class-event-map)
  (register-callback (widget self) #'show-link-menu 
		     :button-press :detail :left-button :state :control)
  (register-callback (widget self) #'follow-default-link
		     :button-press :detail :middle-button :state :control)
  (register-callback self #'hyper-select-unique
		     :button-press :detail :left-button)
  (register-callback self #'hyper-select-multiple
		     :button-press :detail :right-button))
  

(defmethod widget ((self hyper-table-widget))
  (matrix-field self))

;;;
;;;  Method for setting up and saving node correctly:
;;;


(defmethod setup-node ((self hyper-table-widget) (n table-node) 
		       &aux data vals)
  (with-open-file (str (dataset n) :direction :input
		       :if-does-not-exist :create)
       (setq data (read str)))
  (setf (name self) (cadr (assoc 'name data))
	(col-titles (matrix-field self)) (cadr (assoc 'headers data))
	(field-types self) (cadr (assoc 'types data)))
  (setq vals (cdr (assoc 'values data)))
  (setf (orig-values self) vals
	;; (data-cols (widget self)) (apply #'max (mapcar #'length vals))
	(value self) (mapcar #'(lambda (x) (mapcar #'princ-to-string x)) vals))
  (unmodify n)
  self)


(defmethod setup-node ((self hyper-table-widget) (n wip-node) &aux vals)
  (setq vals (mapcar #'list (wip-slot-names n) (wip-slot-values n))) 
  (setf
   (name self) (format nil "RUN ~S LOG ~S" (run-id n) (log-id n))
   (col-titles self) '("Field" "Value")
   (row-titles self) nil
   (field-types self) '(string string)
   (orig-values self) vals
   (value self) (mapcar #'(lambda (x) (mapcar #'princ-to-string x)) vals)
   )
  (unmodify n)
  self)


;;;
;;; Methods for setting and querying the current state of the widget:
;;;

(defmethod get-current-position ((self hyper-table-widget))
  ;; not sure what to do about the situation when there's more than
  ;; one index selected...
  (car (current-indices (widget self))))

(defmethod set-position ((self hyper-table-widget) &key pos
			 &allow-other-keys)
  (setf (current-indices (widget self)) (list pos)))
(defmethod set-position ((self matrix-field) &key 
			 &allow-other-keys)
 nil)

(defmethod get-mark-region ((self hyper-table-widget))
  "returns marked region as ((br  bc) (er  ec))"
  (let* ((indices (current-indices (matrix-field self)))
	 (upper-left (if indices (apply #'tuple-min indices)))
	 (lower-right (if indices (apply #'tuple-max indices))))
    (and indices
	 (list upper-left lower-right))))

(defmethod unmark-marker ((self hyper-table-widget) (b marker))
  (setf (current-indices (widget self)) nil))

(defmethod exposed-markers ((self hyper-table-widget))
  (when (node self)
	(let* (;; add checks for column position, too
	       (tr (row-index (matrix-field self)))
	       (br (+ tr (1- (visible-rows self)))))
	  (remove-if-not #'(lambda (b) (or (and (>= (car (offset b)) tr)
						(<  (car (offset b)) br))
					    (and (>= (car (endpt b)) tr)
						 (< (car (endpt b)) br))))
			 (visible-markers (node self))))))


(defmethod scroll-to ((self hyper-table-widget) offset)
  "offset is (row col) in the table widget"
  (let*  ((mf (matrix-field self))
	  (row (car offset))
	  (col (cadr offset))  ;; ignored for now
	  (r (row-index mf)))
	 (declare (ignore col))
    (if (< r row)
	(funcall (down-func mf) mf (- row r)))
      (funcall (up-func mf) mf (- r row))))

;;;
;;;  Event handlers:
;;;

;;;  These just ensure that the panel is made current whenever the
;;;  user selects one of the table entries:

(defhandler select-unique ((self hyper-table-widget)
			     &rest args &key child
			     &allow-other-keys
			     &default
			     (:button-press :detail :left-button))
  (declare (ignore args))
  (hyper-select-unique self :child child))

(defun hyper-select-unique (self &key child &allow-other-keys)
  (matrix-field-select-unique (widget self) :child child)
  (make-current (my-panel self)))

(defhandler select-multiple ((self hyper-table-widget)
			     &rest args &key child
			     &allow-other-keys
			     &default
			     (:button-press :detail :right-button))
  (declare (ignore args))
  (hyper-select-multiple self :child child))

(defun hyper-select-multiple (self &key child &allow-other-keys)
  (matrix-field-select-multiple (widget self) :child child)
  (make-current (my-panel self)))
  
;;;
;;;  The following methods provide access to attributes of the
;;;  underlying table widget:
;;;

(defmethod row ((self hyper-table-widget))
  (row (matrix-field self)))
(defmethod (setf row) (row (self hyper-table-widget))
  (setf (row (matrix-field self)) row))

(defmethod column ((self hyper-table-widget))
  (column (matrix-field self)))
(defmethod (setf column) (column (self hyper-table-widget))
  (setf (column (matrix-field self)) column))

;;;
;;; Methods on table-field and matrix-field:
;;;

(defmethod do-repaint :after ((self matrix-field) &key &allow-other-keys)
  (if (and (eq (class-of (parent self)) (find-class 'hyper-table-widget))
	   (markers-shown? (parent self)))
      (hm-repaint (parent self))))

(defmethod max-col ((self matrix-field))
  (1- (data-cols self)))

(defmethod row ((self matrix-field))
  (if (current-indices self) 
      (caar (last (current-indices self)))
    (row-index self)))

(defmethod column ((self matrix-field))
  (if (current-indices self) 
      (cadar (last (current-indices self)))
    (col-index self)))


;;; Methods for outlining marker regions:

(defmethod marker-outline-points (marker (widget matrix-field))
  "returns a set of (row, col) points defining the region around block of table"
  (let* ((beg (offset marker))
	 (end (endpt marker))
	 (x1 (car beg)) (y1 (cadr beg))
	 (x2 (car end)) (y2 (cadr end)))
    (list beg (list x1 y2) end (list x2 y1))))

(defmethod translate-point (pt (widget matrix-field) &optional (adjust :tl) (outer? nil))
  "translates pt to coordinates of widget, adjusting so actual point
   will be outside of text region"
  ;; adjust values are :tl, :tr, :bl, :br (for top/bottom,
  ;; right/left)
  ;;;  For tables, pt will be a (row, col) offset into the matrix, so
  ;;;  we have to adjust both for field offset and size when doing the adjustment.
  (let* ((top (row-index widget))
	 (left (col-index widget))
	 (bottom (1- (rows widget)))
	 (right (1- (cols widget)))
	 ;; truncate points at visible boundaries of table:
	 (r (max top (min bottom (car pt))))
	 (c (max left (min right (cadr pt))))
	 (off-left (< c left))
	 (off-top (< r top))
	 (entry (aref (field-table widget) 
		      (max 0 (- r top)) (max 0 (- c left))))
	 (wi (getf entry :width))
	 (hi (getf entry :height))
	 (x-off (if off-left -4 (getf entry :x)))
	 (y-off (if off-top -5 (getf entry :y)))
	 (wi-adjust (if outer? -3 -2))
	 (hi-adjust (if outer? -4 -2))
	 ;; remember that the y-val of the point indicates l-r
	 ;; position (column), and the x-val is up-down position
	 ;; (row), so we have to switch and then translate them:
	 (new-x  (- x-off left))  ;; these may not be right...
	 (new-y  (- y-off top)))
    ;; adjust x and y so lines go in proper spaces between rows/cols
    (case adjust
	  (:tl (list (- new-x wi-adjust) (- new-y hi-adjust)))
	  (:tr (list (+ new-x wi wi-adjust) (- new-y hi-adjust)))
	  (:bl (list (- new-x wi-adjust) (+ new-y hi hi-adjust)))
	  (:br (list (+ new-x wi wi-adjust) (+ new-y hi hi-adjust)))
	  (otherwise (list new-x new-y))
	  )))

;;;
;;;  Scrolling functions that ensure hm-repaint gets called:
;;;

(defun htw-scroll-up (mf n)
  (if (uniform-rows mf)
      (mf-uni-scroll-up mf n)
    (mf-var-scroll-up mf n))
  (hm-repaint (parent mf)))

(defun htw-scroll-down (mf n)
  (if (uniform-rows mf)
      (mf-uni-scroll-down mf n)
    (mf-var-scroll-down mf n))
  (hm-repaint (parent mf)))

(defun htw-scroll-right (mf n)
  (if (uniform-cols mf)
      (mf-uni-scroll-right mf n)
    (mf-var-scroll-right mf n))
  (hm-repaint (parent mf)))

(defun htw-scroll-left (mf n)
  (if (uniform-cols mf)
      (mf-uni-scroll-left mf n)
    (mf-var-scroll-left mf n))
  (hm-repaint (parent mf)))
