
;;;
;;;  HYPER-COLLECTION-WIDGET
;;;  Widget for displaying tabular data with linking capabilities
;;;     

(in-package "PT")

(defclass hyper-collection-widget (hypermedia-mixin collection-widget)
  ( 
   (current-region :initform nil :accessor current-region)
   (event-mask :initform '(:button-press :button-1-motion :exposure))
    )
  (:documentation "Collection widgets with hypermedia behavior"))

(defun make-hyper-collection-widget (&rest args)
  (apply #'make-instance 'hyper-collection-widget :allow-other-keys t args))

(defmethod new-instance ((self hyper-collection-widget) 
			 &key node 
			 (show-markers t)
			 &allow-other-keys)
  (call-next-method)
  (setf (gm self) 'packed-gm)
  (when node
	(setup-node self node)
	(if show-markers
	    (draw-markers self)))
  )
  

(defmethod widget ((self hyper-collection-widget))
  self)

;;;
;;;  Method for setting up and saving node correctly:
;;;

(defmethod setup-node ((self hyper-collection-widget) (n node-record))
  ;; get rid of current fields:
  (dolist (c (children self))
	  (delete-child self c))
  ;; fields is a list of (name . value) pairs indicating the components
  ;; of the record and their types
  (dolist (f (fields n))
	  (typecase (cdr f)  
		#|
		(number 
		 (setf (parent 
			(make-num-entry :editable nil
					:label-type :left 
					:label (create-label (car f))
					:gray t
					:value (cdr f)
					:geom-spec '(:top 20 :top-pad 5)))
		       self))
                |#
		((or symbol string number) 
		 (setf (parent
			(make-text-gadget :label-type :left 
					  :label (create-label (car f))
					  :horiz-just :center
					  :vert-just :center
					  :gray t
					  :background *panel-bg*
					  :foreground "black"
					  :value (string (cdr f))
					  :width (* 2 8 (length
							 (string (cdr f))))
					  :geom-spec '(:top 20 :top-pad 5)))
		       self))
		(list (let ((tf (make-table-field
				:label-type :frame
				:label (create-label (car f))
				:cols 2
				:gray t
				:geom-spec '(:top 100 :top-pad 5))))
			(setf (parent tf) self)
			(setf (value tf)
			      (mapcar #'(lambda (x) (mapcar #'princ-to-string x))
				      (cdr f)))))))
  (unmodify n)
  self)

(defun create-label (label-text)
  (format nil "~12@a:  " label-text))

(defmethod save ((nr node-record) &optional hw)
  (declare (ignore hw))
  (call-next-method))

(defmethod save ((n wip-node) &optional hw)
  (declare (ignore hw))
  (format t "WIP nodes are read-only.  Ignoring save.~%")
  (call-next-method))

;;;
;;; Methods for setting and querying the current state of the widget:
;;;

(defmethod set-position ((self hyper-collection-widget)
			 &key &allow-other-keys)
  nil)

(defmethod get-current-position ((self hyper-collection-widget))
  (current-region self))
(defmethod get-mark-region ((self hyper-collection-widget))
  (current-region self))

;;;  Because marker regions only roughly correspond to the regions of
;;;  the collection's children, we need a slightly different method
;;;  for figuring out the current marker:
(defmethod get-candidate-markers ((self hyper-collection-widget))
  (find-if #'(lambda (x) (close-enough (region x) (current-region self)))
	   (markers (node self))))
  
(defun close-enough (reg1 reg2)
  "returns t if regions start at the same place and are the same
   height"
  (and (equal (car reg1) (car reg2))
       (= (cadr (second reg1)) (cadr (second reg2)))))

(defmethod unmark-marker ((self hyper-collection-widget) (b marker))
  (dolist (c (children self))
	  (if (inverted c) (revert c))))

(defmethod exposed-markers ((self hyper-collection-widget))
  (when (node self)
	(visible-markers (node self))))

(defmethod scroll-to ((self hyper-collection-widget) offset)
  (declare (ignore offset))
  nil)

;;;
;;;  Event handlers:
;;;

;;;  These just ensure that the panel is made current whenever the
;;;  user selects one of the collection entries:

(defhandler select-unique ((self hyper-collection-widget)
			     &rest args &key x y
			     &allow-other-keys
			     &default
			     (:button-press :detail :left-button))
  (declare (ignore args))
  (collection-select-unique self :x x :y y))

(defun collection-select-unique (self &key x y &allow-other-keys)
  (let* ((child (find-if #'(lambda (c) (contains-point (region c) (list x y)))
			 (children self)))
	 (cr (if child (region child))))
    (when cr
      (setf (current-region self) 
	    (list (list (car cr) (second cr))
		  (list (+ (car cr) (third cr)) 
			(+ (second cr) (fourth cr)))))
      (dolist (c (children self)) (if (inverted c) (revert c)))
      (revert child))
    (make-current (my-panel self))))

(defun revert (window)
  (invert window)
  (repaint window))

(defmethod do-repaint :after ((self collection-widget) &key &allow-other-keys)
  (if (and (eq (class-of self) (find-class 'hyper-collection-widget))
	   (markers-shown? self))
      (hm-repaint self)))


;;; Methods for outlining marker regions:

(defmethod marker-outline-points (marker (widget collection-gadget))
  "returns a set of (row, col) points defining the region around block of table"
  (let* ((beg (offset marker))
	 (end (endpt marker))
	 (x1 (car beg)) (y1 (cadr beg))
	 (x2 (car end)) (y2 (cadr end)))
    (list beg (list x2 y1) end (list x1 y2))))

(defmethod translate-point (pt (widget collection-gadget) 
			       &optional (adjust :tl) (outer? nil))
  "translates pt to coordinates of widget, adjusting so actual point
   will be outside of text region"
  ;; adjust values are :tl, :tr, :bl, :br (for top/bottom,
  ;; right/left)
  (let* ((r (car pt))
	 (c (cadr pt))
	 (wi-adjust (if outer? -4 -3))
	 (hi-adjust (if outer? -4 -2))
	 )
    ;; adjust x and y so lines go in proper spaces between rows/cols
    (case adjust
	  (:tl (list (- r wi-adjust) (- c hi-adjust)))
	  (:tr (list (+ r wi-adjust) (- c hi-adjust)))
	  (:bl (list (- r wi-adjust) (+ c hi-adjust)))
	  (:br (list (+ r wi-adjust) (+ c hi-adjust)))
	  (otherwise (list r c))
	  )))
