(in-package "PT")

(defun general-search (initial-node-list &key
		       (depth-limit nil)
		       traverse-fn
		       process-fn
		       enqueue-fn
		       dequeue-fn)
  "This is the general search routine which uses the indicated functions 
   to traverse and process each node in the graph.  
   INITIAL-NODE-LIST is a list of nodes that constitute the root(s) of the graph
   DEPTH-LIMIT is a number indicating the highest level value to be
      processed, or NIL for no limit. 
   TRAVERSE-FN is a function which takes a single node and returns a
       list of immediate neighbors of it 
   PROCESS-FN is a function which processes the current node
   ENQUEUE-FN is a function which adds a list of new nodes to the
      proper places in the search-queuen 
   DEQUEUE-FN is a function which removes the next node to be processed from 
      the search-queue.
   All of these functions should directly manipulate the global
     search-queue named *search-queue*." 

  ;; Initialize the search-queue to the root(s) of the graph to be
  ;; traversed (after wrapping them up) 
  (setf *search-queue* 
	(mapcar #'(lambda (root-item)
			  (make-search-queue-wrapper :item root-item :level 0))
		initial-node-list))

  ;; Loop through each state to be visited, getting the next node of the queue
  (do* ((wrapped-node (funcall dequeue-fn) (funcall dequeue-fn))
	(level (when wrapped-node
		 (search-queue-wrapper-level wrapped-node))
	       (when wrapped-node
		 (search-queue-wrapper-level wrapped-node)))
	(current-node  (when wrapped-node
			 (search-queue-wrapper-item wrapped-node))
		       (when wrapped-node
			 (search-queue-wrapper-item wrapped-node))))

      ;; Stop when no more nodes remain to be processed
      ((null wrapped-node) t)

    ;;Process this node
    (funcall process-fn current-node)

    ;; Then add its neighbors to the queue
    (when (or (null depth-limit)
	      (< level depth-limit))

      ;; Note that TRAVERSE-FN returns a list of children, 
      ;; and we need to wrap them up before adding them to the list
      (funcall enqueue-fn 
	       (mapcar #'(lambda (root-item)
				 (make-search-queue-wrapper :item root-item 
							    :level (1+ level)))
		       (funcall traverse-fn current-node))))))

(defun depth-first-enqueue (node-list)
  "Function to add nodes in the correct order for depth-first `
   traversals of a graph."
  (setf *search-queue* (append node-list *search-queue*)))

(defun breadth-first-enqueue (node-list)
  "Function to add nodes in the correct order for breadth-first 
   traversals of a graph."
  (setf *search-queue* (append *search-queue* node-list)))

(defun normal-dequeue ()
  "Function to remove the first element from the search-queue, 
   for use when queue is already sorted."
  (pop *search-queue*))

(defun make-prioritized-node-list (node-list)
  "This takes a list of nodes and returns a prioritized list suitable
    for initializing or merging with the search-queue." 
  (sort (mapcar #'(lambda (node)
			   (cons (funcall *best-first-eval-fn* node)
				 node))
		       node-list)
	#'<
	:key #'car))

(defun best-first-enqueue (node-list)
  "Function to add nodes in the correct order for best-first traversals of 
   a graph, where best is determined by passing each node to the 
   function stored on *best-first-eval-fn* which is supposed to give a 
   numerical ranking, SMALLER numbers having HIGHER priority."

  ;; This works by storing the nodes on the list in the form (priority . node).  
  ;; This means that the dequeue function must strip out the node when
  ;; removing something from the list. 
  (setf *search-queue*
	;; Merge the new list of nodes with the existing node list,
	;; preserving the priority-based ordering 
	(merge 'list
	       (make-prioritized-node-list node-list)
	       *search-queue*
	       #'<
	       :key #'car)))

(defun best-first-dequeue ()
  "Function to remove the first element from the search queue, where
   the actual node is the cdr of a dotted pair." 
  (cdr (pop *search-queue*)))
  
(defun depth-first-search (initial-node-list &key
			   depth-limit
			   traverse-fn
			   process-fn)
  "Syntatic sugar for GENERAL-SEARCH so that it is clearer that a function
   calling this one is doing a depth-first search."
  (general-search initial-node-list
		  :depth-limit depth-limit
		  :traverse-fn traverse-fn
		  :process-fn process-fn
		  :enqueue-fn #'depth-first-enqueue
		  :dequeue-fn #'normal-dequeue))

(defun breadth-first-search (initial-node-list &key
			     depth-limit
			     traverse-fn
			     process-fn)
  "Syntatic sugar for GENERAL-SEARCH so that it is clearer that a function
   calling this one is doing a breadth-first search."
  (general-search initial-node-list
		  :depth-limit depth-limit
		  :traverse-fn traverse-fn
		  :process-fn process-fn
		  :enqueue-fn #'breadth-first-enqueue
		  :dequeue-fn #'normal-dequeue))

(defun best-first-search (initial-node-list &key
			  depth-limit
			  traverse-fn
			  process-fn
			  priority-fn)
  "Syntatic sugar for GENERAL-SEARCH so that it is clearer that a function
   calling this one is doing a best-first search."
  (setf *best-first-eval-fn* priority-fn)
  (general-search (make-prioritized-node-list initial-node-list)
		  :depth-limit depth-limit
		  :traverse-fn traverse-fn
		  :process-fn process-fn
		  :enqueue-fn #'best-first-enqueue
		  :dequeue-fn #'best-first-dequeue))

