;;;
;;; Copyright (c) 1990 Regents of the University of California
;;; 
;;; $Author: bsmith $
;;; $Source: /pic2/picasso/new/widgets/check-radio/RCS/button-group.cl,v $
;;; $Revision: 1.1 $
;;; $Date: 1991/08/04 19:07:04 $
;;;

(in-package "PT")

;;;
;;;  A button-group consists of a sequence of paired image-table and label
;;;  where one or more pairs can be made "active" by clicking on it.
;;;  the value slot of the button-group is either an index corresponding
;;;  to the unique active image-label pair, or a list of indices which 
;;;  correspond to all active pairs.
;;;
;;;  For example, the following code creates a button-group with
;;;  3 items horizontally, with the label-table below the buttons.
;;;
;;;	(make-button-group :items '("Equipment" "Utilities" "Lots")
;;;				 :orientation :horizontal
;;;				 :label-just :bottom)
;;;
;;;  Creation options:
;;;	items		A list of specs, one per button.  A spec is either a
;;;			label or a property list (eg. '("hello" :font "8x13")).
;;;	font		The font to be used for all label-table (if not specified
;;;			in the property-lists.
;;;	orientation	:horizontal or :vertical
;;;
;;;  Font, label-just and orientation may be setf'd to change their values.
;;;

(defclass button-group (widget)
  ((name :initform "A Button Group")
   (label-table 
    :initarg :label-table  
    :initform nil
    :type vector
    :reader label-table)
   (active-image
    :initarg :active-image 
    :initform nil
    :type image
    :reader active-image)
   (inactive-image
    :initarg :inactive-image 
    :initform nil
    :type image
    :reader inactive-image)
   (image-table
    :initarg :image-table 
    :initform nil
    :type vector)
   (orientation 
    :initarg :orientation  
    :initform :vertical 
    :type keyword 
    :reader orientation)
   (event-mask :initform '(:exposure :button-press))
   (font :initform "8x13")
   (gc-spec :initform '(gc-res (:font "8x13")))))

(defun make-button-group (&rest keys)
  (apply #'make-instance 'button-group :allow-other-keys t keys))

;;;
;;; Define setf accessor methods...
;;;
(defmethod (setf value) (value (self button-group) &aux oldval) 
  (setq oldval (slot-value self 'value))
  (when (and (not (equal value oldval)) 
	     (or (listp value)
		 (<= 0 value (length (label-table self)))))
	(let ((image-table (slot-value self 'image-table))
	      (active-image (slot-value self 'active-image))
	      (inactive-image (slot-value self 'inactive-image)))
	     (unless (listp value) (setq value (list value)))
	     (setf (slot-value self 'value) value)
	     (dotimes (im (length (slot-value self 'image-table)))
		      (rplaca (svref image-table im)
			      (if (member im value)
				  active-image
				  inactive-image)))
	     (repaint self))))

(defmethod update-value ((self button-group) &aux val)
  (setq val (slot-value self 'value))
  (let ((image-table (slot-value self 'image-table))
	(active-image (slot-value self 'active-image))
	(inactive-image (slot-value self 'inactive-image)))
       (unless (listp val) (setq val (list val)))
       (dotimes (im (length (slot-value self 'image-table)))
		(rplaca (svref image-table im)
			(if (member im val)
			    active-image
			    inactive-image)))
       (repaint self)))

(defmethod (setf orientation) (value (self button-group))
  (unless (eq value (orientation self))
	  (setf (slot-value self 'orientation) value)
	  (resize-window-handler self)))

(defmethod vertical ((self button-group))
  (eq (slot-value self 'orientation) :vertical))

(defmethod resize-window-handler ((self button-group) 
				  &aux orient n label-table image-table)
  (setq label-table (label-table self)
	image-table (slot-value self 'image-table))
  (setq orient (orientation self)
	n (length label-table))
  (if (eq (slot-value self 'orientation) :vertical) 
      (do* ((i 0 (1+ i))
	    (l (svref label-table i))
	    (im (svref image-table i))
	    (w (max 1 (- (width self) 30))) 
	    (h (height self))
	    (y (round (* (/ i n) h)))
	    (ht (round (/ h n))))
	   ((>= i n))
	   (setq l (svref label-table i) 
		 im (svref image-table i)
		 y (round (* (/ i n) h)) 
		 ht (round (/ h n)))
	   (setf (getf (cdr l) :x) 30
		 (getf (cdr l) :y) y
		 (getf (cdr l) :width) w
		 (getf (cdr l) :height) ht)
	   (setf (getf (cdr im) :y) y
		 (getf (cdr im) :height) ht))
      (do* ((i 0 (1+ i))
	    (l (svref label-table i))
	    (im (svref image-table i))
	    (w (width self)) 
	    (h (max 1 (- (height self) 30)))
	    (x (round (* (/ i n) w)) (round (* (/ i n) w)))
	    (wt (round (/ w n)) (round (/ w n))))
	   ((>= i n))
	   (setq l (svref label-table i) 
		 im (svref image-table i))
	   (setf (getf (cdr l) :x) x
		 (getf (cdr l) :y) 30
		 (getf (cdr l) :width) wt
		 (getf (cdr l) :height) h)
	   (setf (getf (cdr im) :x) x
		 (getf (cdr im) :width) wt))))

(defmethod dim-item ((self button-group) item)
  (when (<= 1 item (length (gadgets self)))
	(dim (elt (gadgets self) (1- item)))
	(repaint (elt (gadgets self) (1- item)))))

;;;
;;; New instance method for button-group
;;;
(defmethod new-instance ((self button-group)
			 &key
			 (items 		'(""))
			 (font			nil)
			 (orientation 		:vertical)
			 (base-width		nil)
			 (base-height		nil)
			 (base-size		nil)
			 &allow-other-keys
			 &aux len ltbl itbl inactive-image (maxh 0) (maxw 0)
			 imh imw val h w)
  (call-next-method)
  
  (setq inactive-image (slot-value self 'inactive-image))
  (setq font (slot-value self 'font))
  (setq imh (height (active-image self))
	imw (width (active-image self)))
  ;;	create tables
  (setq len (length items))
  (setq ltbl (make-array (list len) :element-type 'list)
	itbl (make-array (list len) :element-type 'list))
  
  ;;	create label and image synthetic gadgets
  (cond ((not (eq orientation :vertical))
	 (do* ((i 0 (1+ i))
	       (il items (cdr il))
	       (l (car il) (car il))
	       (f nil))
	      ((>= i len))
	      ;;	label. . .
	      (unless (and (listp l) (not (stringp (cadr l))))
		      (setq l (list l)))
	      (setf (svref ltbl i) l)
	      ;;	get dimensions
	      (setq val (car l)) 
	      (cond ((stringp val)
		     (setq f (getf (cdr l) :font))
		     (unless f (setq f font)
			     (when (stringp f) 
				   (setq f (get-font f))))
		     (setq w (text-width val :font f)
			   h (font-height f)))
		    ((color-p val)
		     ;; Unless specified, make height same as height of
		     ;; active image and square.
		     (setq w (getf (cdr l) :width)
			   h (getf (cdr l) :height))
		     (if (null h) (setq h (height (active-image self))))
		     (if (null w) (setq w h)))
		    ((not (listp val))
		     (setq w (width val)
			   h (height val)))
		    ((null val)
		     (setq w 0 h 0))
		    (t
		     (setq w 0 h 0)
		     (setq f (getf (cdr l) :font))
		     (unless f (setq f font)
			     (when (stringp f) 
				   (setq f (get-font f))))
		     (dolist (v val)
			     (cond ((stringp v)
				    (incf w (text-width v :font f))
				    (incf h (font-height f)))
				   ((not (listp v))
				    (incf w (width v))
				    (incf h (height v)))
				   ((null v)
				    nil)))))
	      
	      ;;	Update maxs. . .
	      (when (> h maxh)
		    (setq maxh h))
	      (setq maxw (+ maxw (max w imw) 10))
	      
	      ;;	image. . . 
	      (setf (svref itbl i)
		    (list inactive-image :y 0 :height 30 :bitmap-p t)))
	 (setq maxh (+ maxh imh 10)))
	(t
	 (do* ((i 0 (1+ i))
	       (il items (cdr il))
	       (l (car il) (car il))
	       (f nil))
	      ((>= i len))
	      ;;	label. . .
	      (unless (and (listp l) (not (stringp (cadr l))))
		      (setq l (list l)))
	      (setf (svref ltbl i) l)
	      ;;	get dimensions
	      (setq val (car l)) 
	      (cond ((stringp val)
		     (setq f (getf (cdr l) :font))
		     (unless f (setq f font)
			     (when (stringp f) 
				   (setq f (get-font f))))
		     (setq w (text-width val :font f)
			   h (font-height f)))
		    ((color-p val)
		     ;; Unless specified, make height same as height of
		     ;; active image and square.
		     (setq w (getf (cdr l) :width)
			   h (getf (cdr l) :height))
		     (if (null h) (setq h (height (active-image self))))
		     (if (null w) (setq w h)))
		    ((not (listp val))
		     (setq w (width val)
			   h (height val)))
		    ((null val)
		     (setq w 0 h 0))
		    (t
		     (setq w 0 h 0)
		     (setq f (getf (cdr l) :font))
		     (unless f (setq f font)
			     (when (stringp f) 
				   (setq f (get-font f))))
		     (dolist (v val)
			     (cond ((stringp v)
				    (incf w (text-width v :font f))
				    (incf h (font-height f)))
				   ((not (listp v))
				    (incf w (width v))
				    (incf h (height v)))
				   ((null v)
				    nil)))))
	      
	      ;;	Update maxs. . .
	      (when (> w maxw)
		    (setq maxw w))
	      (setq maxh (+ maxh (max h imh) 10))
	      
	      ;;	image. . .
	      (setf (svref itbl i)
		    (list inactive-image :x 0 :width 30 :bitmap-p t)))
	 (setq maxw (+ maxw imw 15))))
  
  ;; 	set base-size
  (setf (base-size self) 
	(cond (base-size base-size)
	      (base-width (list base-width maxh))
	      (base-height (list maxw base-height))
	      (t (list maxw maxh))))
  
  ;;	set slots
  (setf (slot-value self 'label-table) ltbl
	(slot-value self 'image-table) itbl)
  
  ;;	update-value
  (update-value self)
  
  ;;	return self
  self)

;;;
;;;	Register event handlers
;;;
(defhandler select ((self button-group) &key x y &allow-other-keys 
		    &aux image-table active-image inactive-image index val
		    &default :button-press)
  (setq image-table (slot-value self 'image-table)
	active-image (slot-value self 'active-image)
	inactive-image (slot-value self 'inactive-image)
	val (slot-value self 'value))
  
  ;;	find which pair was clicked, if any
  (if (not (eq (orientation self) :vertical)) 
      (let* ((im (cdr (svref image-table 0)))
	     (w (getf im :width))
	     (h (getf im :height)))
	    (when (> y h) (return-from button-group-select))
	    (setq index (truncate x w))
	    (setq im (svref image-table index))
	    (rplaca im (if (eq (car im) active-image)
			   inactive-image
			   active-image))
	    (clear-region self (getf (cdr im) :x) 0 w h)
	    (apply #'put im))
      (let* ((im (cdr (svref image-table 0)))
	     (w (getf im :width))
	     (h (getf im :height)))
	    (when (> x w) (return-from button-group-select))
	    (setq index (truncate y h))
	    (setq im (svref image-table index))
	    (rplaca im (if (eq (car im) active-image)
			   inactive-image
			   active-image))
	    (clear-region self 0 (getf (cdr im) :y) w h)
	    (apply #'put im)))
  
  ;;	set value appropriately
  (setq val (slot-value self 'value))
  (setf (repaint-flag self) nil)
  (setf (value self) 
	(if (member index val) (remove index val) (cons index val)))
  (setf (repaint-flag self) t))

(defmethod do-attach ((self button-group) 
		      &aux label-table image-table gc l font)
  (call-next-method)
  (setq label-table (slot-value self 'label-table)
	image-table (slot-value self 'image-table)
	gc (slot-value self 'gc-res))
  (dotimes (i (length label-table))
	   (setq l (svref label-table i))
	   (setq font (getf (cdr l) :font))
	   (when font
		 (if (font-p font)
		     (font-attach font)
		     (progn
		      (setq font (get-font font))
		      (when font (font-attach font)))))
	   (setf (getf (cdr l) :gc) gc 
		 (getf (cdr l) :window) self)
	   (if (color-p (car l))
	       (attach (car l)))
	   (setq l (svref image-table i))
	   (setf (getf (cdr l) :gc) gc
		 (getf (cdr l) :window) self
		 (getf (cdr l) :width) 30))
  (resize-window-handler self))

(defmethod do-detach ((self button-group) &aux label-table image-table l font)
  (call-next-method)
  (setq label-table (slot-value self 'label-table))
  (setq image-table (slot-value self 'image-table))
  (dotimes (i (length label-table))
	   (setq l (svref label-table i))
	   (setq font (getf (cdr l) :font))
	   (when font (font-detach font))
	   (if (color-p (car l))
	       (detach (car l)))))

;;;
;;;	Repaint method puts synthetic gadgets on screen
;;;
(defmethod do-repaint ((self button-group) 
		       &key 
		       &allow-other-keys
		       &aux image-table label-table)
  (setq image-table (slot-value self 'image-table)
	label-table (slot-value self 'label-table))
  (dotimes (i (length label-table))
	   (apply #'put (svref image-table i))
	   (apply #'put (svref label-table i))))
