(in-package "PT")

;; a rgb-image is a class created to store the raw image information 
;; obtained from a GIF file.  
;; - data is the raster array of raw pixel values.
;; - red, green and blue are the arrays of intensities for each pixel value.
;; - used is a boolean array indicating which pixel values are actually 
;;   used in data.
;; - num-colors is the number of entries in red, green, blue and used.
;; - width and height are the size of data.
;;
;; The image slot is a caching slot used for cheating a little bit.
;; Here's how it works:
;; - As X is defined, each window has its own colormap.  Remember
;;   that a colormap is a table that maps pixel values to color cells.
;; - This means that each time a color image is drawn in a window,
;;   all the colors used by the image have to be allocated in the colormap
;;   of that window, and the pixel values in the raster of the image have
;;   to be adjusted to reflect the color representation in the colormap.
;; - If this is indeed the case, a different image has to be created for

;;   each window because the colormaps of windows could be different.
;;   For example, on one window's colormap, the pixel 6 could correspond
;;   to red, while on another window's colormap, 6 corresponds to blue.
;;   Using the same image for these two windows would make a picture look
;;   pretty weird.
;; - Because of the current hardware reality, what happens most of the
;;   time is that deep down inside all windows share the same colormap.
;;   After all, there's not a lot of hardware around that supports a different
;;   set of colors for each X window.
;; - Knowing the above, we can cheat a little bit.  The idea is that 
;;   we only have to allocate the colors and modify the raster the first
;;   time around.  Then we save the image and use it again and again since
;;   we know that the colormaps are the same.
;; - The morally conscious can disable this cheat feature by specifying
;;   :fast nil in most functions.
;;


(defstruct rgb-image
  (data nil :type vector)
  (red nil :type vector)
  (green nil :type vector)
  (blue nil :type vector)
  (used nil :type vector)
  (num-colors nil :type integer)
  (width nil :type integer)
  (height nil :type integer)
  (image nil :type vector)
  )

;; this function bu default uses (current-colormap), which is the colormap
;; of the root window.
(defun rgb-image-to-image (rgb-image
			   &key
			   (colormap (res (current-colormap)))
			   (fast t))
  (cond ((null rgb-image) nil)
	((and fast (rgb-image-image rgb-image))
	 (rgb-image-image rgb-image))
	(t (let* ((rr (rgb-image-red rgb-image))
		  (rg (rgb-image-green rgb-image))
		  (rb (rgb-image-blue rgb-image))
		  (pt (if (true-color-p) nil
			  (allocate-colors colormap rr rg rb
					   (rgb-image-used rgb-image)
					   (rgb-image-num-colors rgb-image))))
		  (rw (rgb-image-width rgb-image))
		  (rh (rgb-image-height rgb-image))
		  (rd (rgb-image-data rgb-image))
		  ;; (new-data (copy-seq rd))
		  (image (xlib:create-image :bits-per-pixel 
				   	    (if (true-color-p) 32 8)
					    ;; :bytes-per-line (* 4 rw)
					    :data
					    (fix-pixel-values rd pt
							      rb rg rr)
					    :height rh
					    :width rw
					    :format :z-pixmap
					    :depth 
				   	    (if (true-color-p) 32 8)
					    ))
		  )
	     (setf (rgb-image-image rgb-image) image)
	     image))))

;; The colormap of drawable is used to allocate the colors.
;; there is no "cheating" at all here.
(defun put-rgb-image (drawable
		      gc
		      rgb-image
		      &key
		      (src-x 0)
		      (src-y 0)
		      (x 0)
		      (y 0)
		      (width (rgb-image-width rgb-image))
		      (height (rgb-image-height rgb-image))
		      )
  (let* ((colormap (xlib:window-colormap drawable))
	 (rr (rgb-image-red rgb-image))
	 (rg (rgb-image-green rgb-image))
	 (rb (rgb-image-blue rgb-image))
	 (pt (if (true-color-p) nil
		 (allocate-colors colormap rr rg rb
				  (rgb-image-used rgb-image)
				  (rgb-image-num-colors rgb-image))))
	 (rw (rgb-image-width rgb-image))
	 (rh (rgb-image-height rgb-image))
	 (rd (rgb-image-data rgb-image))
	 ;; (new-data (copy-seq rd))
	 (image (xlib:create-image :bits-per-pixel 
				   (if (true-color-p) 32 8)
				   ;; :bytes-per-line (* 4 rw)
				   :data (fix-pixel-values rd pt rr rg rb)
				   :height rh
				   :width rw
				   :format :z-pixmap
				   :depth 32))
	 )
    (declare (ignore foo))
    (xlib:put-image (res drawable) gc image
		    :x x :y y
		    :width width :height height
		    :src-x src-x :src-y src-y)
    image))
	
(defun create-rgb-image (gif-file &aux width height rgb-image)
  (when (null gif-file)
	(return-from create-rgb-image nil))
  (when gif-file
	(multiple-value-setq (width height)
			     (probe-gif gif-file))
	(if (or (null width) (null height))
	    (error "create-rgb-image: couldn't open \`~s\`." gif-file))
	(setq rgb-image 
	      (make-rgb-image
	       :data (make-array (if (true-color-p)
				     (list (+ 4 (* 4 width height)))
				     (list (* width height)))
				 :element-type 'xlib::card8)
	       :red (make-array '(256) :element-type 'xlib::card8)
	       :green (make-array '(256) :element-type 'xlib::card8)
	       :blue (make-array '(256) :element-type 'xlib::card8)
	       :used (make-array '(256) :element-type 'xlib::card8)
	       :num-colors nil
	       :width width
	       :height height
	       :image nil))
	(setf (rgb-image-num-colors rgb-image)
	      (load-gif gif-file
			(rgb-image-data rgb-image)
			(rgb-image-red rgb-image)
			(rgb-image-green rgb-image)
			(rgb-image-blue rgb-image)
			(rgb-image-used rgb-image)
			(if (true-color-p) 1 0)
			))
	)
  rgb-image)
	 

