;;;
;;; Copyright (c) 1990 Regents of the University of California
;;; 
;;; $Author: bsmith $
;;; $Source: /pic2/picasso/new/toolkit/resource/image.cl,v $
;;; $Revision: 1.1 $
;;; $Date: 1991/08/04 19:03:13 $
;;;

(in-package "PT")

;;;
;;; image class
;;;

(defclass image (pmc)
  ((res
    :initform nil
    :type vector
    :reader res)
   (name
    :initform "an image"
    :initarg :name
    :type string
    :accessor name)
   (data
    :initform nil
    :initarg :data
    :type t
    :reader data)
   (bitmap-p
    :initform nil
    :type atom
    :reader bitmap-p)
   (colormap
    :initform nil
    :type t
    :reader colormap)
   (ref-count
    :initform 0
    :type integer)))

;;;	
;;;	Accessors. . .
;;;

(defmethod width ((self image)
		  &key 
		  &allow-other-keys)
  (when (attached-p self)
	(xlib:image-width (res self))))

(defmethod height ((self image))
  (when (attached-p self)
	(xlib:image-height (res self))))

(defmethod depth ((self image))
  (when (attached-p self)
	(xlib:image-depth (res self))))

(defun get-image (&optional name spec)
  (declare (ignore spec))
  (if name
      (gethash name *global-image-hashtab*)
      nil))

(defun remove-image (&optional name spec)
  (declare (ignore spec))
  (if name
      (remhash name *global-image-hashtab*)
      nil))

(defun make-image (&rest args &key file source gif-file name &allow-other-keys
			 &aux im)
  (cond ((or file gif-file)
	 (cond ((and name (setq im (get-image name))) 
		im)
	       (t
		(unless name
			(setf (getf args :name)
			      (setq name (or file gif-file))))
		(apply #'make-instance 'image :allow-other-keys t args))))
	(source
	 (cond ((and name (setq im (get-image name)))
		im)
	       (t
		(unless name
			(setq name source))
		(apply #'make-instance 'image :allow-other-keys t args))))
	(name
	 (let ((im (get-image name)))
	      (unless im 
		      (warn "make-image: no image named \'~s\'." name))
	      im))
	(t (warn "make-image: no image named \'~s\'." name))))

;;;
;;; image initialization method
;;;

(defmethod new-instance ((self image)
			 &key
			 (name nil)
			 ;; generate image from file
			 (file nil)
			 (gif-file nil)
			 ;; generate image from data
			 (data nil)
			 ;; generate image from window
			 (source nil)
			 (src-x 0)
			 (src-y 0)
			 (width nil)
			 (height nil)
			 ;; attach?
			 (attach-p t)
			 ;; allow specification of slots
			 &allow-other-keys
			 &aux im)
  (unless (colormap-p (colormap self))
	  (setf (slot-value self 'colormap) (default-colormap)))
  
  ;; switch on keywords
  (cond (file
	 (setf (slot-value self 'data) file)
	 (setf (slot-value self 'bitmap-p) t))
	(gif-file 
	 ;; create rgb-image from file
	 (unless (setq gif-file (find-library-file gif-file))
		 (error "image.new-instance: bad file ~S" gif-file))
	 (setq gif-file (namestring gif-file))
	 (setf (slot-value self 'data)
	       (setq im (create-rgb-image gif-file)))
	 (unless name (setq name (pathname-name data))))
	(data
	 (setf (slot-value self 'data) data))
	(source
	 ;; test src-x
	 (if (not (and (numberp src-x)
		       (plusp src-x)
		       (< src-x (width source))))
	     ;; signal error
	     (error "image.new-instance: invalid src-x ~s" src-x))
	 ;; test y-offset
	 (if (not (and (numberp src-y)
		       (plusp src-y)
		       (< src-y (height source))))
	     ;; signal error
	     (error "image.new-instance: invalid src-y ~s" src-y))
	 ;; test width
	 (if (not (and (numberp width)
		       (plusp width)
		       (< width (height source))))
	     ;; signal error
	     (error "image.new-instance: invalid width ~s" width))
	 ;; test height
	 (if (not (and (numberp height)
		       (plusp height)
		       (< height (height source))))
	     ;; signal error
	     (error "image.new-instance: invalid height ~s" height))
	 ;; allocate resource id
	 (setq im (xlib:get-image (res source)
				  src-x
				  src-y
				  width
				  height))
	 ;; test for failure
	 (unless im
		 ;; signal error
		 (error "image.new-instance: can't create image?"))
	 ;; store resource id
	 (setf (slot-value self 'res) im)
	 (setq attach-p nil)
	 (setf (slot-value self 'ref-count) 1)))
  
  (if name 
      (setf (gethash name *global-image-hashtab*) self))
  
  (if attach-p
      (do-attach self))
  
  ;; return self
  self)

(defmethod do-attach ((self image) &aux data im)
  ;; don't reattach if ref-count is positive
  (when (> (slot-value self 'ref-count) 0)
	(incf (slot-value self 'ref-count))
	(return-from do-attach))
  
  (setq data (data self))
  (cond ((stringp data)
	     ;; create image from file
	     (unless (setq data (find-library-file data))
		     (error "image.do-attach: bad file ~S" data))
	     (setf (slot-value self 'res)
		   (setq im (xlib:read-bitmap-file data))))
	((pathnamep data)
	     (unless (setq data (find-library-file data))
		     (error "image.do-attach: bad file ~S" data))
	     (setf (slot-value self 'res)
		   (setq im (xlib:read-bitmap-file data))))
	((typep data 'rgb-image)
	     ;; check colormap
	     (if (not (colormap-p (colormap self)))
		 (error "image.do-attach: can't attach gif-type image ~S with no colormap" self))
	     
	     (setq im (rgb-image-to-image
		       data :colormap (res (colormap self))))
	     (unless im
		     ;; signal error
		     (error "image.do-attach: can't create image!"))
	     ;; store resource id
	     (setf (slot-value self 'res) im))
	(t
	     (error "image.do-attach: can't attach image ~S" self)))
  
  (if (= 1 (depth self))
      (setf (slot-value self 'bitmap-p) t))
  
  (setf (slot-value self 'ref-count) 1))

(defmethod do-detach ((self image))
  ;; don't detach if ref-count is 0
  (when (> (slot-value self 'ref-count) 1) 
	(decf (slot-value self 'ref-count))
	(return-from do-detach))
;;  (xlib:free-colors (res (colormap self)) (list (pixel self)))
  (setf (slot-value self 'res) nil
	(slot-value self 'ref-count) 0))

;;;
;;; image operation methods
;;;

(defmethod put ((self image)
		&key
		(window 	(root-window))
		(gc 		nil)
		(x 		nil)
		(y 		nil)
		(x-offset 	0)
		(y-offset 	0)
		(width 		nil)
		(height 	nil)
		(pix-width 	-1 widthp)
		(pix-height 	-1 heightp)
		(horiz-just 	:center)
		(vert-just 	:center)
		(bitmap-p 	nil)
		(inverted 	nil)
		(dimmed 	nil)
		(gc-invert 	nil)
		(gc-dimmed 	nil)
		&allow-other-keys &aux res)
  ;; test width specified
  (if (null x)
      (setq x (repaint-x window)))
  (if (null y)
      (setq y (repaint-y window)))
  (when (null widthp)
	;; get width from image
	(setq pix-width (width self)))
  (when (null heightp)
	;; get width from image
	(setq pix-height (height self)))
  (when (and inverted gc-invert)
	(setq gc gc-invert))
  (unless gc (setq gc (gc-res window)))
  (if (and dimmed (not gc-dimmed))
      (setq gc-dimmed gc))
  (unless bitmap-p (setq bitmap-p (bitmap-p self)))
  ;; adjust x and y according to justification requirements
  (let ((w (if width (+ width x) (slot-value window 'width)))
	(h (if height (+ height y) (slot-value window 'height))))
       (cond ((eql vert-just :bottom)
	      (setq y (- h pix-height)))
	     ((eql vert-just :center)
	      (incf y (round (/ (- h y pix-height) 2)))))
       (cond ((eql horiz-just :right)
	      (setq x (- w pix-width)))
	     ((eql horiz-just :center)
	      (incf x (round (/ (- w x pix-width) 2))))))
  ;; put image
  (cond ((and dimmed (black-and-white-display-p (display window)))
	 (setq res (res window))
	 (xlib:draw-rectangle res gc-dimmed x y pix-width pix-height t)
	 (xlib:put-image res gc (res self) 
			 :x x :y y :width pix-width :height pix-height
			 :src-x x-offset :src-y y-offset :bitmap-p bitmap-p)
	 (xlib:draw-rectangle res gc-dimmed x y pix-width pix-height t))
	(t
	 (if dimmed
	     (setq gc gc-dimmed))
	 (xlib:put-image (res window) gc (res self) 
			 :x x :y y :width pix-width :height pix-height
			 :src-x x-offset :src-y y-offset :bitmap-p bitmap-p)))
  ;; return image
  self)
