;;;
;;; Copyright (c) 1990 Regents of the University of California
;;; 
;;; $Author: smoot $
;;; $Source: /pic2/picasso/toolkit/base/RCS/window.cl,v $
;;; $Revision: 1.3 $
;;; $Date: 1991/11/26 22:58:01 $
;;;

(in-package "PT")

;;;
;;; Window states: A window can be in one of three states: exposed,
;;; concealed or pending. These are returned by the status method. If
;;; a window is pending, it can be for one of three reasons: 
;;;
;;;	a) it's not attached
;;;	b) it's been shrunk to zero size by a geometry manager("pended")
;;;	c) it's parent is not visible.
;;;
;;; These are orthogonal blocks -- they can be caused by the server, the
;;; geometry manager or the application, respectively. We can use the
;;; attached-p macro to detect conditiopn (a), and (b) and (c) are 
;;; stored as a bit-vector in the window "state" slot. The "status" slot
;;; holds one of :exposed or :concealed, and the status method returns one
;;; of :concealed, :exposed or :pending by combining the values stored in
;;; the state and status slots appropriately.
;;;
;;; The macros "invisible-p", "exposed-p", "attached-p", "pending-p",
;;; "concealed-p", "pended-p" should be used to determine the status of
;;; a window.
;;;

;;;
;;; window class
;;;

(defclass window (pmc)
  (
   ;; Information to support the window-hierarchy

   (res 
    :initform nil
    :type vector
    :reader res)
   (parent
    :initarg :parent 
    :initform nil
    :type window
    :accessor parent)
   (display
    :initarg :display 
    :initform nil
    :type display
    :reader display)
   (screen
    :initarg :screen 
    :initform nil
    :type screen
    :reader screen)
   (lexical-parent
    :initarg :lexical-parent 
    :initform nil
    :type window
    :accessor lexical-parent)
   
   ;; Information about a window geometry. The setf methods on these slots
   ;; cause a repaint method to be invoked.
   (x-offset
    :initarg :x-offset 
    :initform 0
    :type integer
    :reader x-offset)
   (y-offset
    :initarg :y-offset 
    :initform 0
    :type integer
    :reader y-offset)
   (width
    :initarg :width 
    :initform 1
    :type integer
    ;; hack below. 
    ;; :reader width
    )
   (height
    :initarg :height 
    :initform 1
    :type integer
    :reader height)
   
   ;; The windows resize-hint.  Used by geometry managers. The setf methods
   ;; on these slots cause the resize-hint-changed method to be invoked.
   (base-width
    :initarg :base-width 
    :initform 1
    :type integer
    :reader base-width)
   (base-height
    :initarg :base-height 
    :initform 1
    :type integer
    :reader base-height)
   (width-increment
    :initarg :width-increment 
    :initform 1
    :type integer
    :reader width-increment)
   (height-increment
    :initarg :height-increment 
    :initform 1
    :type integer
    :reader height-increment)
   (width-height-ratio
    :initarg :width-height-ratio 
    :initform nil
    :type float
    :reader width-height-ratio)
   
   ;; Drawing information about a window.  Used by most any drawing function.
   (gc-spec			;; list of gc-specifications where each spec
    :initform nil		;; is (<slot-name> . <gc-description>).  The
    :type list			;; described gc is created and stuffed into
    :reader gc-spec		;; the named slot automatically when the window
    :allocation :class)		;; is attached.

   (gc-res			;; xlib gc to use for all graphics ops
    :initform nil		;; since a window doesn't have a resource
    :type vector		;; of its own (unless it's an x-window)
    :reader gc-res)
   (font 
    :initarg :font  
    :initform nil
    :type t
    :accessor font)
   (background
    :initarg :background 
    :initform "white"
    :type color
    :reader background)
   (inverted-background
    :initarg :inverted-background 
    :initform "black"
    :type color
    :reader inverted-background)
   (dimmed-background
    :initarg :dimmed-background 
    :initform "white"
    :type color
    :reader dimmed-background)
   (foreground
    :initarg :foreground 
    :initform "black"
    :type color
    :reader foreground)
   (dimmed-foreground
    :initarg :dimmed-foreground 
    :initform "gray50"
    :type color
    :reader dimmed-foreground)
   (inverted-foreground
    :initarg :inverted-foreground 
    :initform "white"
    :type color
    :reader inverted-foreground)

   (dimmed
    :initarg :dimmed 
    :initform nil
    :type t
    :accessor dimmed)
   (inverted
    :initarg :inverted 
    :initform nil
    :type t
    :accessor inverted)
   (colormap
    :initarg :colormap 
    :initform nil 
    :type colormap
    :accessor colormap)
   
   ;; Info to optimize repainting...
   (repaint-flag
    :initarg :repaint-flag 
    :initform t
    :type t
    :accessor repaint-flag)
   (partial-repaint-p
    :initform nil
    :type atom
    :reader partial-repaint-p)
   (intern-res-cache
    :initform nil
    :type list)
   (intern-gc
    :initform t
    :type vector)
   (cached-background
    :initform nil
    :type integer)

   ;; Misc information.
   (value
    :initarg :value
    :initform nil
    :type t
    ;; :accessor value
    )
   (name
    :initarg :name 
    :initform "A Window"
    :type string
    :accessor name)
   (doc 
    :initarg :doc  
    :initform ""
    :type string
    :accessor doc)
   (objid
    :initarg :objid 
    :initform nil
    :type t
    :reader objid)
   (state
    :initarg :state 
    :initform 0
    :type integer
    :accessor state)
   (status
    :initarg :status 
    :initform :exposed
    :type symbol
    :accessor status)
   (attach-when-possible
    :initarg :attach-when-possible 
    :initform t
    :type atom
    :accessor attach-when-possible)
   (geom-spec
    :initarg :geom-spec 
    :initform nil
    :type t
    :accessor geom-spec)
   (mf-selectable-widget
    :initarg :mf-selectable-widget 
    :initform t
    :type atom
    :accessor mf-selectable-widget)

   (hot-spot
    :initform :border		;; one of :window, :label, or :border
    :initarg :hot-spot
    :type symbol
    :reader hot-spot)
   (bordered-label
    :initform nil
    :initarg :bordered-label
    :type symbol
    :reader bordered-label)

   ;; Label info
   (label
    :initarg :label 
    :initform nil
    :type t
    :reader label)
   ;; Currently One of (:left :frame :bottom nil)
   (label-type
    :initarg :label-type 
    :initform nil
    :type symbol
    :reader label-type)
   (label-x
    :initarg :label-x 
    :initform nil
    :type integer
    :reader label-x)
   (label-y
    :initarg :label-y 
    :initform nil
    :type integer
    :reader label-y)
   (label-attributes
    :initarg :label-attributes 
    :initform nil
    :type list
    :reader label-attributes)
   (intern-label-gc
    :initform t 
    :type vector)

   ;; Border info
   (border-width
    :initarg :border-width 
    :initform 0
    :type t
    :reader border-width)
   ;; One of (:box :frame :shadow :black-frame nil)
   (border-type
    :initarg :border-type 
    :initform nil
    :type symbol
    :reader border-type)
   (border-attributes
    :initarg :border-attributes 
    :initform nil
    :type list
    :accessor border-attributes)
   (intern-border-gcs
    :initform t
    :type t)))

(defun make-window (&rest keys)
  (apply #'make-instance 'window :allow-other-keys t keys))

;; 
;; hack to add &allow-other-keys to standard reader method.
;;
(defmethod width ((self window) &key &allow-other-keys)
  (slot-value self 'width))

(defmethod value ((self window) &key &allow-other-keys)
  (slot-value self 'value))

(defmethod (setf value) (val (self window))
  (setf (slot-value self 'value) val))

;;;
;;; Setf methods for slots in the window class
;;;

(defmethod (setf status) (value (self window))
  (let ((old-status (managed-p self))
	(new-status nil))
       (setf (slot-value self 'status) value)
       (setq new-status (managed-p self))
       (if (not (eq new-status old-status))
	   (status-changed self))
       ;; return status
       new-status))

;;;
;;; Setf methods that affect the resize-hint.
;;;

(defmethod (setf hot-spot) (val (self window))
  (setf (slot-value self 'hot-spot) val)
  (resize-hint-changed self))

(defmethod (setf bordered-label) (val (self window))
  (setf (slot-value self 'bordered-label) val)
  (resize-hint-changed self))

(defmethod (setf base-width) (value (self window))
  (set-resize-hint self :base-width value)
  (slot-value self 'base-width))

(defmethod (setf base-height) (value (self window))
  (set-resize-hint self :base-height value)
  (slot-value self 'base-height))

(defmethod (setf width-increment) (value (self window))
  (set-resize-hint self :width-increment value)
  (width-increment self))

(defmethod (setf height-increment) (value (self window))
  (set-resize-hint self :height-increment value)
  (height-increment self))

(defmethod (setf base-size) (value (self window))
  (set-resize-hint self :base-width (first value)
		   :base-height (second value))
  (list (slot-value self 'base-width) (slot-value self 'base-height)))

(defmethod (setf increment-size) (value (self window))
  (set-resize-hint self :width-increment (first value)
		   :height-increment (second value))
  (increment-size self))

(defmethod (setf width-height-ratio) (value (self window))
  (set-resize-hint self :width-height-ratio value)
  (width-height-ratio self))

(defmethod (setf resize-hint) (value (self window))
  (set-resize-hint self 
		   :base-width (first value)
		   :base-height (second value)
		   :width-increment (third value)
		   :height-increment (fourth value)
		   :width-height-ratio (fifth value)))

(defmethod (setf geom-spec) :after (value (self window))
  (declare (ignore value))
  (geom-spec-changed self))

;;;
;;; Setf methods to futz with a windows geometry.
;;;

(defmethod (setf res) (value (self window))
  ;; warn
  (declare (ignore value))
  (warn "Can't manually change the resource of a window"))

(defmethod (setf x-offset) (value (self window))
  (configure self :x-offset value)
  value)

(defmethod (setf y-offset) (value (self window))
  (configure self :y-offset value)
  value)

(defmethod (setf width) (value (self window))
  (configure self :width value)
  value)

(defmethod (setf height) (value (self window))
  (configure self :height value)
  value)

(defmethod (setf location) (value (self window))
  (configure self
	     :x-offset (first value)
	     :y-offset (second value))
  value)

(defmethod (setf size) (value (self window))
  (configure self
	     :width (first value)
	     :height (second value))
  value)

(defmethod (setf region) (value (self window))
  (configure self
	     :x-offset (first value)
	     :y-offset (second value)
	     :width (third value)
	     :height (fourth value))
  value)

;;;
;;;	Setf methods on colors update gc-res if attached. . .
;;;

(defmethod (setf font) (value (self window))
  (cond ((attached-p self) 
	 (let ((old (slot-value self 'font))) 
	      (when (font-p old) 
		    (font-detach old))) 
	 (when value
	       (unless 
		(or (and (stringp value) 
			 (setq value (make-font :name value 
						:display (display self)))) 
		    (font-p value))
		(error "window.setf.font: invalid font ~s." value)) 
	       (setf (slot-value self 'font) value)
	       (font-attach value)
	       (let ((gc (gc-res self)))
		    (when (xlib:gcontext-p gc) 
			  (setf (xlib:gcontext-font gc) (res value))))))
	(t (if (or (stringp value) (font-p value) (null value))
	       (setf (slot-value self 'font) value)
	       (error "window.setf.font: invalid font ~s." value)))))

(defmethod (setf background) (val (self window) &aux gc old attp) 
  (setq old (slot-value self 'background)
	attp (attached-p self))
  (if (xlib:gcontext-p (setq gc (slot-value self 'intern-gc)))
      (cond ((stringp val)
	     (if (not attp)
		 (setf (slot-value self 'background) val)
		 (if (setq old (get-paint val self))
		     (setf (background self) old)
		     (error "window.setf.background: can't find paint \`~s\`."
			    val))))
	    ((color-p val) 
	     (when attp
		   (color-attach val)
		   (setf (xlib:gcontext-foreground gc) (pixel val) 
			 (xlib:gcontext-fill-style gc) :solid)
		   (when (xlib:gcontext-p (setq gc (slot-value self 'gc-res)))
			 (setf (xlib:gcontext-background gc) (pixel val))
			 (xlib:force-gcontext-changes gc))
		   (when (paint-p old)
			 (do-detach old)))
	     (setf (slot-value self 'background) val)
	     (clear self))
	    ((integerp val)
	     (when attp
		   (setf (xlib:gcontext-foreground gc) val 
			 (xlib:gcontext-fill-style gc) :solid)
		   (when (xlib:gcontext-p (setq gc (slot-value self 'gc-res)))
			 (setf (xlib:gcontext-background gc) val)
			 (xlib:force-gcontext-changes gc))
		   (when (paint-p old)
			 (do-detach old)))
	     (setf (slot-value self 'background) val)
	     (clear self))
	    ((image-p val)
	     (setq val (make-tile :image val :window self))
	     (when attp
		   (setf (xlib:gcontext-tile gc) (res val)
			 (xlib:gcontext-fill-style gc) :tiled))
	     (setf (slot-value self 'background) val)
	     (clear self)
	     (when (and attp (paint-p old))
		   (do-detach old)))
	    ((tile-p val) 
	     (setf (window val) self)
	     (when attp
		   (setf (xlib:gcontext-stipple gc) (res val)
			 (xlib:gcontext-foreground gc) (pixel (foreground val))
			 (xlib:gcontext-background gc) (pixel (background val))
			 (xlib:gcontext-fill-style gc) :opaque-stippled))
	     (setf (slot-value self 'background) val)
	     (clear self)
	     (when (and attp (paint-p old))
		   (do-detach old)))
	    ((null val) 
	     (when (and attp (setq val (slot-value self 'cached-background))
			(xlib:gcontext-p (setq gc (slot-value self 'gc-res))))
		   (setf (xlib:gcontext-background gc) val))
	     (setf (slot-value self 'background) nil)
	     (clear self)
	     (when (and attp (paint-p old))
		   (do-detach old)))
	    ((eq val :parent-relative) 
	     (when (and attp (setq val (slot-value self 'cached-background))
			(xlib:gcontext-p (setq gc (slot-value self 'gc-res))))
		   (setf (xlib:gcontext-background gc) val))
	     (setf (slot-value self 'background) :parent-relative)
	     (clear self)
	     (when (and attp (paint-p old))
		   (do-detach old)))
	    (t
	     (error "window.setf.background: invalid background \`~s\`." 
		    val)))
      (setf (slot-value self 'background) val)))

(defmethod (setf inverted-background) (value (self window))
  (cond
   ((or (paint-p value) (numberp value) (get-paint value self) (null value)
	(eq :parent-relative value))
    (setf (slot-value self 'inverted-background) value))
   (t
    (error "window.setf-inverted-background: invalid color ~s" value))))

(defmethod (setf dimmed-background) (value (self window))
  (cond
   ((or (paint-p value) (numberp value) (get-paint value self) (null value)
	(eq :parent-relative value))
    (setf (slot-value self 'dimmed-background) value))
   (t
    (error "window.setf-dimmed-background: invalid color ~s" value))))

(defmethod (setf foreground) (val (self window) &aux gc old attp) 
  (setq old (slot-value self 'foreground)
	attp (attached-p self))
  (if (xlib:gcontext-p (setq gc (gc-res self)))
      (cond ((stringp val)
	     (if (not attp)
		 (setf (slot-value self 'foreground) val)
		 (if (setq old (get-paint val self)) 
		     (setf (foreground self) old) 
		     (error 
		      "window.setf.foreground: couldn't find paint \`~s\`." 
		      val))))
	    ((color-p val) 
	     (when attp
		   (color-attach val)
		   (setf (xlib:gcontext-foreground gc) (pixel val) 
			 (xlib:gcontext-fill-style gc) :solid)
		   (xlib:force-gcontext-changes gc)
		   (when (paint-p old)
			 (do-detach old)))
	     (setf (slot-value self 'foreground) val))
	    ((integerp val)
	     (when attp
		   (setf (xlib:gcontext-foreground gc) val 
			 (xlib:gcontext-fill-style gc) :solid
			 (xlib:gcontext-tile gc) nil)
		   (when (paint-p old)
			 (do-detach old)))
	     (setf (slot-value self 'foreground) val))
	    ((image-p val)
	     (setq val (make-tile :image val :window self))
	     (when attp
		   (setf (xlib:gcontext-tile gc) (res val)
			 (xlib:gcontext-fill-style gc) :tiled))
	     (setf (slot-value self 'foreground) val)
	     (when (and attp old (paint-p old))
		   (do-detach old)))
	    ((tile-p val) 
	     (setf (window val) self)
	     (when attp
		   (setf (xlib:gcontext-stipple gc) (res val)
			 (xlib:gcontext-foreground gc) (pixel (foreground val))
			 (xlib:gcontext-background gc) (pixel (background val))
			 (xlib:gcontext-fill-style gc) :opaque-stippled))
	     (setf (slot-value self 'foreground) val) 
	     (when (and attp old (paint-p old))
		   (do-detach old)))
	    (t
	     (error "window.setf.foreground: invalid paint \`~s\`." val)))
      (setf (slot-value self 'foreground) val)))

(defmethod (setf inverted-foreground) (value (self window))
  (cond
   ((or (paint-p value) (numberp value) (get-paint value self))
    (setf (slot-value self 'inverted-foreground) value))
   (t
    (error "window.setf-inverted-foreground: invalid color ~s" value))))

(defmethod (setf dimmed-foreground) (value (self window))
  (cond
   ((or (paint-p value) (numberp value) (get-paint value self))
    (setf (slot-value self 'dimmed-foreground) value))
   (t
    (error "window.setf-dimmed-foreground: invalid color ~s" value))))

(defmethod (setf inverted) (val (self window) &aux sv)
  (setq sv (slot-value self 'inverted))
  (unless (or (and val sv) (and (null val) (null sv)))
	  (psetf (background self) (inverted-background self)
		 (inverted-background self) (background self))
	  (psetf (foreground self) (inverted-foreground self)
		 (inverted-foreground self) (foreground self)))
  (setf (slot-value self 'inverted) val))

(defmethod (setf dimmed) (val (self window) &aux sv)
  (setq sv (slot-value self 'dimmed))
  (unless (or (and val sv) (and (null val) (null sv)))
	  (psetf (foreground self) (dimmed-foreground self)
		 (dimmed-foreground self) (foreground self)))
  (setf (slot-value self 'dimmed) val)
  (when (and (attached-p self) (black-and-white-display-p (display self))
	     (xlib:gcontext-p (setq sv (slot-value self 'gc-res))))
	(setf (xlib:gcontext-function sv) 
	      (if val 8 2))))

;;;
;;;	Label accessors
;;;

(defmethod (setf label-type) (val (self window) &aux oldval)
  (unless (eq val (setq oldval (slot-value self 'label-type)))
	  (setf (slot-value self 'label-type) val)
	  (when (exposed-p self)
		(label-clear oldval self))
	  (label-init val self)
	  (when (exposed-p self)
		(label-repaint val self)
		(flush-display (display self)))))

(defmethod (setf label) (val (self window) &aux ltype oldval)
  (unless (eq val (setq oldval (slot-value self 'label)))
	  (setq ltype (slot-value self 'label-type))
	  (when (exposed-p self)
		(label-clear ltype self))
	  (setf (slot-value self 'label) val)
	  (when (exposed-p self)
		(label-repaint ltype self))))

(defmethod (setf label-x) (val (self window) &aux ltype oldval)
  (unless (eq val (setq oldval (slot-value self 'label-x)))
	  (setq ltype (slot-value self 'label-type))
	  (when (exposed-p self)
		(label-clear ltype self))
	  (setf (slot-value self 'label-x) val)
	  (when (exposed-p self)
		(label-repaint ltype self))))

(defmethod (setf label-y) (val (self window) &aux ltype oldval)
  (unless (eq val (setq oldval (slot-value self 'label-y)))
	  (setq ltype (slot-value self 'label-type))
	  (when (exposed-p self)
		(label-clear ltype self))
	  (setf (slot-value self 'label-y) val)
	  (when (exposed-p self)
		(label-repaint ltype self))))

(defmethod label-position ((self window))
  (list (slot-value self 'label-x) (slot-value self 'label-y)))

(defmethod (setf label-position) (val (self window) &aux ltype oldval)
  (unless (equal val (setq oldval (list (slot-value self 'label-x) 
					(slot-value self 'label-y))))
	  (setq ltype (slot-value self 'label-type))
	  (when (exposed-p self)
		(label-clear ltype self))
	  (setf (slot-value self 'label-x) (car val)
		(slot-value self 'label-y) (cadr val))
	  (when (exposed-p self)
		(label-repaint ltype self))))

(defmethod (setf label-attributes) (val (self window))
  (setf (slot-value self 'label-attributes) val)
  (label-notify-change (slot-value self 'label-type) self))

(defmethod label-font ((self window))
  (getf (slot-value self 'label-attributes) :font))

(defmethod (setf label-font) (val (self window) &aux sval)
  (setq sval (slot-value self 'label-attributes))
  (setf (getf sval :font) val)
  (setf (slot-value self 'label-attributes) sval)
  (label-notify-change (slot-value self 'label-type) self))

;;;
;;;	Border accessors
;;;

(defmethod (setf border-type) (val (self window) &aux oldval)
  (unless (eq val (setq oldval (slot-value self 'border-type)))
	  (setf (slot-value self 'border-type) val)
	  (when (attached-p self)
		(when (exposed-p self)
		      (border-clear oldval self))
		(border-init val self)
		(when (exposed-p self)
		      (border-repaint val self)
		      (flush-display (display self))))))

(defmethod gray ((self window))
  (eq (border-type self) :frame))

(defmethod (setf gray) (val (self window))
  (setf (border-type self)
	(if val :frame :box)))

(defmethod (setf border-width) (val (self window) 
				    &aux oldval sup lbw tbw rbw bbw btp)
  (setq oldval (slot-value self 'border-width)
	sup (parent self))
  (unless (eq val oldval) 
	  (unless (or (integerp val) (= (length val) 4))
		  (error "window.setf.border-width: invalid width \'~s\'" val))
	  (setf (slot-value self 'border-width) val)
	  (when (exposed-p self)
		(if (consp val)
		    (setq lbw (first val)
			  tbw (second val)
			  rbw (third val)
			  bbw (fourth val))
		    (setq lbw val
			  tbw val
			  rbw val
			  bbw val))
		(setq btp (border-type self))
		(border-clear btp self)
		(border-resize btp self)
		(unless (opaque-window-p self)
			(do-repaint self))
		(border-repaint btp self))))


;;;
;;; Method to change the parent of a window.  Uses add-child and
;;; delete-child macros, so cannot be included in window-def.cl
;;;

(defmethod (setf parent) (value (self window))
  ;;
  ;; If I was attached, and my parent is being set to null or a detached
  ;; window, detach.
  ;;
  ;; If I was detached, and my parent is being set to something attached,
  ;; attach.
  ;;
  (if (and (attached-p self) (or (null value) (not (attached-p value))))
      (let ((old-sup (parent self)))
	   (setf (slot-value self 'parent) value)
	   (detach self)
	   (setf (slot-value self 'parent) old-sup))
      (if (and (not (attached-p self)) (window-p value) (attached-p value))
	  (let ((old-sup (parent self)))
	       (setf (slot-value self 'parent) value)
	       (attach self)
	       (setf (slot-value self 'parent) old-sup))))

  (when (parent self)
	(delete-child (parent self) self)
	(setf (slot-value self 'parent) nil))
  (when value
	(setf (slot-value self 'parent) value)
	(add-child value self))

  ;; Update my status...
  (update-state self))

;;;
;;; window initialization method
;;;

(defmethod new-instance ((self window)
			 &key
			 (screen nil)
			 (gray nil)
			 (label-type nil)
			 (label-font nil)
			 (display nil)

			 ;; Geometry keywords
			 (x-offset nil)
			 (y-offset nil)
			 (width nil)
			 (height nil)
			 (location nil)
			 (size nil)
			 (region nil)

			 ;; Resize hint keywords
			 (base-width nil)
			 (base-height nil)
			 (base-size nil)
			 (width-increment nil)
			 (height-increment nil)
			 (increment-size nil)
			 (resize-hint nil)

			 ;; Color keywords
			 (background nil)
			 (foreground nil)
			 (font nil)
			 (colormap nil)
			 (border-width nil bwp)
			 &allow-other-keys
			 &aux def)
  "Initialize a new instance of the window class"
  
  ;; get defaults
  (if (and (not background) (setq def (get-default self "background")))
      (setf (slot-value self 'background) def))
  (if (and (not foreground) (setq def (get-default self "foreground")))
      (setf (slot-value self 'foreground) def))
  (if (and (not font) (setq def (get-default self "font")))
      (setf (slot-value self 'font) def))

  (unless screen
	  (setf (slot-value self 'screen)
		(setq screen (current-screen))))
  (unless display
	  (setf (slot-value self 'display)
		(setq screen (display screen))))
  (unless colormap
	  (setf (slot-value self 'colormap)
		(setq colormap (colormap (root-window screen)))))

  ;; Handle the name and objid tables
  (let ((objid (slot-value self 'objid)))
       (unless objid
	       (setf (slot-value self 'objid)
		     (setq objid (generate-objid))))
       (setf (gethash objid *objid-window-table*) self))

		 
  ;; ====== Interpret the resize-hint =======
  ;; Set the base-width
  (setf (slot-value self 'base-width)
	(cond (base-size (round (first base-size)))
	      (resize-hint (round (first resize-hint)))
	      (base-width (round base-width))
	      (t (round (slot-value self 'base-width)))))
  
  ;; Set the base-height
  (setf (slot-value self 'base-height)
	(cond (base-size (round (second base-size)))
	      (resize-hint (round (second resize-hint)))
	      (base-height (round base-height))
	      (t (round (slot-value self 'base-height)))))
  
  ;; Set the width-increment
  (setf (slot-value self 'width-increment)
	(cond (increment-size (round (first increment-size)))
	      (resize-hint (round (third resize-hint)))
	      (width-increment (round width-increment))
	      (t (round (slot-value self 'width-increment)))))
  
  ;; Set the height-increment
  (setf (slot-value self 'height-increment)
	(cond (increment-size (round (second increment-size)))
	      (resize-hint (round (fourth resize-hint)))
	      (height-increment (round height-increment))
	      (t (round (slot-value self 'height-increment)))))
  
  ;; ====== Interpret the geometry =======
  ;; set the x-offset
  (setf (slot-value self 'x-offset)
	(cond (location (round (first location)))
	      (region (round (first region)))
	      (x-offset (round x-offset))
	      (t (round (slot-value self 'x-offset)))))
  
  ;; set the y-offset 
  (setf (slot-value self 'y-offset)
	(cond (location (round (second location)))
	      (region (round (second region)))
	      (y-offset (round y-offset))
	      (t (round (slot-value self 'y-offset)))))
  
  ;; set the width
  (setf (slot-value self 'width)
	(cond (size (round (first size)))
	      (region (round (third region)))
	      (width (round width))
	      (t (round (slot-value self 'width)))))
  (if (< (slot-value self 'width) (slot-value self 'base-width))
      (setf (slot-value self 'width) (slot-value self 'base-width)))
  
  ;; set the height
  (setf (slot-value self 'height)
	(cond (size (round (second size)))
	      (region (round (fourth region)))
	      (height (round height))
	      (t (round (slot-value self 'height)))))
  (if (< (slot-value self 'height) (slot-value self 'base-height))
      (setf (slot-value self 'height) (slot-value self 'base-height)))
  
  ;; Register the gc-specs
  (register-window-gcs self)

  ;; ====== Verify the value in the status slot. =======
  (if (null (slot-value self 'status))
      (setf (slot-value self 'status) :exposed))

  ;; add font to label-attributes fields
  (when label-font
	(setf (getf (slot-value self 'label-attributes) :font)
	      label-font))
  (when gray
	(setf (slot-value self 'border-type) :frame)
	(unless label-type
		(setf (slot-value self 'label-type) :frame)))

  (when bwp
	(setf (getf (border-attributes self) :border-width)
	      border-width))

  (update-state self)

  ;; return self
  self)

;;;
;;; new-instance after method creates children at creation time
;;; NOTE:  This method has been place in collection-gadget-method.cl to
;;; eliminate circular dependencies. . .
;;;
;;; (defmethod new-instance :after ((self window)))
;;;

(defmethod set-resize-hint ((self window) 
			    &key
			    (base-width (base-width self))
			    (base-height (base-height self))
			    (width-increment (width-increment self))
			    (height-increment (height-increment self))
			    (width-height-ratio (width-height-ratio self)))
  (when (<= base-width 0)
;	(warn "window.set-resize-hint: invalid base-width ~s. Resetting to 1~%"
;	      base-width)
	(setq base-width 1))
  (when (<= base-height 0)
;	(warn "window.set-resize-hint: invalid base-height ~s. Resetting to 1~%"
;	      base-height)
	(setq base-height 1))
  (if (< width-increment 0)
      (warn "window.set-resize-hint: invalid width-increment ~s"
	    width-increment))
  (if (< height-increment 0)
      (warn "window.set-resize-hint: invalid height-increment ~s"
	    height-increment))
  (when (and width-height-ratio (< width-height-ratio 0))
	(warn "window.set-resize-hint: invalid width-height-ratio ~s"
	      width-height-ratio))
  ;; change the slot
  (unless (and (= base-width (base-width self))
	       (= base-height (base-height self))
	       (= width-increment (width-increment self))
	       (= height-increment (height-increment self))
	       (eql width-height-ratio (width-height-ratio self)))
	  (setf (slot-value self 'base-width) base-width)
	  (setf (slot-value self 'base-height) base-height)
	  (setf (slot-value self 'width-increment) width-increment)
	  (setf (slot-value self 'height-increment) height-increment)
	  (setf (slot-value self 'width-height-ratio) width-height-ratio)
	  (resize-hint-changed self)))

(defmethod resize ((self window) w h)
  (configure self :width w :height h)
  self)

(defmethod move ((self window) x y)
  (configure self :x-offset x :y-offset y)
  self)

(defmethod reshape ((self window) x y w h)
  (configure self :x-offset x :y-offset y :width w :height h)
  self)

(defmethod configure ((self window)
		      &key
		      (x-offset (slot-value self 'x-offset))
		      (y-offset (slot-value self 'y-offset))
		      (width (width self))
		      (height (height self))
		      (auto-unpend t)
		      &allow-other-keys 
		      &aux gc)
  
  ;; Check for rationals, etc that aren't integers.
  (if (numberp x-offset) (setq x-offset (truncate x-offset)))
  (if (numberp y-offset) (setq y-offset (truncate y-offset)))
  (if (numberp width) (setq width (truncate width)))
  (if (numberp height) (setq height (truncate height)))
  
  (cond
   
   ;; validity checks
   
   ((not (and (integerp x-offset) (integerp y-offset) 
	      (integerp width) (integerp height)))
    (warn "window.reshape: invalid argument(s) ~s ~s ~s ~s"
	  x-offset y-offset width height)
    nil)
   
   (t    
    ;; only configure when something changes
    (when (or (/= x-offset (slot-value self 'x-offset))
	      (/= y-offset (slot-value self 'y-offset))
	      (/= width (slot-value self 'width))
	      (/= height (slot-value self 'height)))
	  
	  (setf (slot-value self 'x-offset) x-offset)
	  (setf (slot-value self 'y-offset) y-offset)
	  (setf (slot-value self 'width) width)
	  (setf (slot-value self 'height) height)
	  (if (or (<= width 0) (<= height 0))
	      (pend self)
	      (progn
	       (if (and auto-unpend (pended-p self)) (unpend self))
	       (resize-window-handler self)))
	  t)

    ;;  set clip-mask of gc-res if appropriate
    (when (and (> width 0) (> height 0)
	       (not (x-window-p self))
	       (xlib:gcontext-p (setq gc (slot-value self 'gc-res))))
	  (setf (xlib:gcontext-clip-x gc) (repaint-x self)
		(xlib:gcontext-clip-y gc) (repaint-y self)
		(xlib:gcontext-clip-mask gc) (list 0 0 width height))))))

(defmethod clear-region ((self window) x y w h)
  (when (and (exposed-p self)
	     (plusp x) (plusp y) (plusp w) (plusp h))
	(if (background self) 
	    (xlib:draw-rectangle (res self) (slot-value self 'intern-gc) 
				 x y w h t)
	    (xlib:clear-area (res self) 
			     :x x ;;; old: (+ x (repaint-x self))
			     :y y ;;; old: (+ y (repaint-y self))
			     :width w
			     :height h
			     :exposures-p nil))))

(defmethod do-repaint-region ((self window) x y w h &key clear)
  (declare (ignore x y w h clear)))

(defmethod resize-window-handler ((self window))
  t)

(defun update-state (window &aux sup)
  "Update the state slot after parent has changed it's state."
  (if (eq (slot-value window 'status) :exposed)
      (if (and (window-p (setq sup (parent window))) 
	       (exposed-p sup) 
;;	       (viewable-p sup)
	       )
	  (make-uninvisible window)
	  (make-invisible window))))
      
;;;
;;; Window query methods
;;;

(defmethod status ((self window))
  (if (eq (slot-value self 'status) :concealed)
      :concealed
      (if (and (zerop (state self)) (attached-p self))
	  :exposed
	  :pending)))

(defmethod base-size ((self window))
  (list (slot-value self 'base-width)
	(slot-value self 'base-height)))

(defmethod increment-size ((self window))
  (list (slot-value self 'width-increment)
	(slot-value self 'height-increment)))

(defmethod resize-hint ((self window))
  (list
   (slot-value self 'base-width)
   (slot-value self 'base-height)
   (slot-value self 'width-increment)
   (slot-value self 'height-increment)
   (slot-value self 'width-height-ratio)))

(defmethod location ((self window))
  (list (slot-value self 'x-offset) (slot-value self 'y-offset)))

(defmethod size ((self window))
  (list (slot-value self 'width) (slot-value self 'height)))

(defmethod region ((self window))
  (list (slot-value self 'x-offset) (slot-value self 'y-offset)
	(slot-value self 'width) (slot-value self 'height)))

(defun window-left-pad (self &aux bw)
  (when (listp (setq bw (border-width self)))
	(setq bw (car bw)))
  (max bw (label-left-pad (label-type self) self)))

(defun window-top-pad (self &aux bw)
  (when (listp (setq bw (border-width self)))
	(setq bw (second bw)))
  (max bw (label-top-pad (label-type self) self)))

(defun window-right-pad (self &aux bw)
  (when (listp (setq bw (border-width self)))
	(setq bw (third bw)))
  (max bw (label-right-pad (label-type self) self)))

(defun window-bottom-pad (self &aux bw)
  (when (listp (setq bw (border-width self)))
	(setq bw (fourth bw)))
  (max bw (label-bottom-pad (label-type self) self)))

(defun region-offset (self &key x y width height 
			   (borders nil bp) (label nil lp)
			   &aux bw bl bt br bb ll lt lr lb)
  (cond ((and bp lp))
	(lp
	 (setq borders (eq (hot-spot self) :border)))
	(bp
	 (setq label (eq :window (hot-spot self))))
	(t
	 (case (hot-spot self)
	       (:border
		(setq borders t label nil))
	       (:label
		(setq label t borders nil))
	       (t
		(setq label nil borders nil)))))
  (cond ((null borders)
	 (setq bl 0 bt 0 br 0 bb 0))
	((and (eq (border-type self) :box) (x-window-p self))
	 (setq bw (border-width self))
	 (setq bl 0 bt 0 br (* 2 bw) bb (* 2 bw)))
	(t
	 (setq bw (border-width self))
	 (if (listp bw) 
	     (setq bl (first bw) 
		   bt (second bw)
		   br (third bw)
		   bb (fourth bw))
	     (setq bl bw
		   bt bw
		   br bw
		   bb bw))))
  (if (and label (label-type self))
      (multiple-value-setq (ll lt lr lb) (label-pad (label-type self) self))
      (setq ll 0 lt 0 lr 0 lb 0))
  (setq bl (+ bl ll)
	bt (+ bt lt))
  (setq br (+ bl (+ br lr))
	bb (+ bt (+ bb lb)))
  (if (null x)
      (setq x (x-offset self)))
  (if (null y)
      (setq y (y-offset self)))
  (if (null width)
      (setq width (width self)))
  (if (null height)
      (setq height (height self)))
  (values bl bt br bb x y width height))

(defun virtual-region (self &key x y width height 
			   (borders nil bp) (label nil lp)
			   &aux bl bt br bb)
  (cond ((and bp lp))
	(lp
	 (setq borders (eq (hot-spot self) :border)))
	(bp
	 (setq label (eq :window (hot-spot self))))
	(t
	 (case (hot-spot self)
	       (:border
		(setq borders t label nil))
	       (:label
		(setq label t borders nil))
	       (t
		(setq label nil borders nil)))))
  (multiple-value-setq (bl bt br bb x y width height)
		       (region-offset self :x x :y y 
				      :width width :height height
				      :borders borders :label label))
  (list (- x bl) (- y bt) (+ width br) (+ height bb)))

(defun virtual-width (self &key width
			   (borders nil bp) (label nil lp)
			   &aux x y height bl bt br bb)
  (cond ((and bp lp))
	(lp
	 (setq borders (eq (hot-spot self) :border)))
	(bp
	 (setq label (eq :window (hot-spot self))))
	(t
	 (case (hot-spot self)
	       (:border
		(setq borders t label nil))
	       (:label
		(setq label t borders nil))
	       (t
		(setq label nil borders nil)))))
  (multiple-value-setq (bl bt br bb x y width height)
		       (region-offset self :width width
				      :borders borders :label label))
  (+ width br))

(defun virtual-height (self &key height
			   (borders nil bp) (label nil lp)
			   &aux x y width bl bt br bb)
  (cond ((and bp lp))
	(lp
	 (setq borders (eq (hot-spot self) :border)))
	(bp
	 (setq label (eq :window (hot-spot self))))
	(t
	 (case (hot-spot self)
	       (:border
		(setq borders t label nil))
	       (:label
		(setq label t borders nil))
	       (t
		(setq label nil borders nil)))))
  (multiple-value-setq (bl bt br bb x y width height)
		       (region-offset self :height height
				      :borders borders :label label))
  (+ height bb))

;;;  returns region given actual-region
(defun actual-region (self &key x y width height 
			  (borders nil bp) (label nil lp)
			  &aux bl bt br bb)
  (cond ((and bp lp))
	(lp
	 (setq borders (eq (hot-spot self) :border)))
	(bp
	 (setq label (eq :window (hot-spot self))))
	(t
	 (case (hot-spot self)
	       (:border
		(setq borders t label nil))
	       (:label
		(setq label t borders nil))
	       (t
		(setq label nil borders nil)))))
  (multiple-value-setq (bl bt br bb x y width height)
		       (region-offset self :x x :y y
				      :width width :height height
				      :borders borders :label label))
  (list (+ x bl) (+ y bt) (- width br) (- height bb)))

;;;
;;;	Functions to handle registration, retrieval, and creation of gcs
;;;

(defun register-window-gcs (win &aux gc-spec desc name)
  (unless (setq gc-spec (gc-spec win))
	  (return-from register-window-gcs))
  (unless (consp (car gc-spec))
	  (setq gc-spec (list gc-spec)))
  (dolist (spec gc-spec)
	  (setq desc (cadr spec))
	  (cond ((listp desc)
		 (setq name 
		       (if (stringp (car desc)) 
			   (prog1 
			    (car desc)
			    (setq desc (cdr desc)))
			   "default"))
		 (setq name (list name (string (car spec)) 
				  (string (class-name (class-of win)))))
		 (when (get-gc name)
		       (return-from register-window-gcs))
		 (register-gc name desc (car name)))
		(t 
		 (setf (gethash (list desc (string (car spec)) 
				      (string (class-name (class-of win))))
				*global-gc-hashtab*)
		       (get-gc desc))))))

(defun update-window-gc (win desc &aux el name cache)
  (cond ((stringp (setq name (getf desc :background))) 
	 (setq el (make-color :name name :colormap (colormap win) :attach-p t))
	 (setf (getf desc :background)
	       (cond ((color-p el)
		      (color-attach el)
		      (push el cache)
		      (pixel el))
		     (t (warn 
			 "update-window-gc: can't find paint ~s for window ~s." 
			 el win)
			0))))
	((color-p name) 
	 (setf (getf desc :background) (pixel name))))
  (cond ((stringp (setq name (getf desc :foreground))) 
	 (setq el (make-color :name name :colormap (colormap win) :attach-p t))
	 (setf (getf desc :foreground)
	       (cond ((color-p el)
		      (color-attach el)
		      (push el cache)
		      (pixel el))
		     (t (warn 
			 "update-window-gc: can't find paint ~s for window ~s." 
			 el win)
			1))))
	((color-p name) 
	 (setf (getf desc :foreground) (pixel name))))
  (cond ((stringp (setq name (getf desc :tile)))
	 (setq el (get-image name))
	 (setf (getf desc :tile)
	       (cond ((image-p el)
		      (setq el (make-tile :window win :image el))
		      (push el cache)
		      (res el))
		     (t (warn
			 "update-window-gc: can't find image ~s for window ~s." 
			 el win))))) 
	((image-p name) 
	 (setq el (make-tile :window win :image name))
	 (push el cache)
	 (setf (getf desc :tile) (res el))))
  (cond ((stringp (setq name (getf desc :stipple)))
	 (setq el (get-image name))
	 (setf (getf desc :stipple)
	       (cond ((image-p el)
		      (setq el (make-tile :window win :image el))
		      (res el))
		     (t (warn
			 "update-window-gc: can't find image ~s for window ~s." 
			 el win))))) 
	((image-p name) 
	 (setq el (make-tile :window win :image name))
	 (push el cache)
	 (setf (getf desc :stipple) (res el))))
  (cond ((stringp (setq name (getf desc :paint)))
	 (setq el (get-paint name))
	 (remf desc :tile)
	 (cond ((image-p el)
		(setq el (make-tile :window win :image el))
		(setf (getf desc :tile) (res el)
		      (getf desc :fill-style) :tiled)
		(push el cache))
	       ((color-p el)
		(color-attach el)
		(setf (getf desc :foreground) (pixel el)
		      (getf desc :fill-style) :solid)
		(push el cache))
	       (t (warn
		   "update-window-gc: can't find paint ~s for window ~s." 
		   el win)))
	 (remf desc :paint))
	((image-p name)
	 (setf (getf desc :tile) (res (make-tile :window win :image name)))
	 (remf desc :paint)
	 (setf (getf desc :fill-style) :tiled)
	 (push name cache))
	((color-p name)
	 (color-attach name)
	 (setf (getf desc :foreground) (pixel name))
	 (remf desc :paint)
	 (setf (getf desc :fill-style) :solid)
	 (push name cache)))
  (when (setq el (getf desc :font))
	(when (stringp el)
	      (setq el (make-font :name el :attach-p t 
				  :display (display win))))
	(setf (getf desc :font)
	      (cond ((font-p el)
		     (do-attach el)
		     (push el cache)
		     (res el))
		    (t (warn
			"update-window-gc: can't find font ~s for window ~s." 
			el win)))))
  (setf (slot-value win 'intern-res-cache)
	(append cache (slot-value win 'intern-res-cache)))
  desc)

(defun make-window-gcs (win &aux gc-spec name desc)
  ;;	Create internal gc
  (when (slot-value win 'intern-gc)
	(setf (slot-value win 'intern-gc)
	      (xlib:create-gcontext :drawable (res win))))
  
  (unless (setq gc-spec (gc-spec win))
	  (return-from make-window-gcs))
  (unless (consp (car gc-spec))
	  (setq gc-spec (list gc-spec)))
  
  ;;	Create gcs in gc-spec
  (dolist (spec gc-spec)
	  (setq desc (cadr spec))
	  (setq name 
		(cond ((listp desc)
		       (if (stringp (car desc)) 
			   (prog1 
			    (car desc)
			    (setq desc (cdr desc)))
			   "default"))
		      ((stringp desc) desc)
		      (t "default")))
	  (setq desc
		(if (eq name desc)
		    (get-gc name) 
		    (get-gc (list name (string (car spec)) 
				  (string (class-name (class-of win)))))))
	  ;;	Attach associated resources
	  (setq desc (update-window-gc win desc))
	  (remf desc :paint)
	  (setf (slot-value win (car spec)) 
		(apply #'xlib:create-gcontext 
		       (nconc (list :drawable (res win)) desc)))))

(defun destroy-window-gcs (win &aux gc-spec gc)
  (unless (setq gc-spec (gc-spec win))
	  (return-from destroy-window-gcs))
  (unless (consp (car gc-spec))
	  (setq gc-spec (list gc-spec)))

  ;;	Destroy gcs in gc-spec
  (dolist (spec gc-spec)
	  (setq gc (slot-value win (car spec)))
	  (when (xlib:gcontext-p gc)
		(xlib:free-gcontext gc))
	  (setf (slot-value win (car spec)) nil))

  ;;	Detach associated resources
  (dolist (el (slot-value win 'intern-res-cache))
	  (when el (do-detach el)))
  (setf (slot-value win 'intern-res-cache) nil)
  
  ;;	Destroy internal gc
  (when (setq gc (slot-value win 'intern-gc))
	(xlib:free-gcontext gc)
	(setf (slot-value win 'intern-gc) t)))

;;;
;;; window methods that affect the status
;;;

(defmethod do-attach :after ((self window))
  (if (stat-exposed-p self)
      (status-changed self)))

(defmethod do-detach :after ((self window))
  (if (not (concealed-p self))
      (status-changed self)))

(defmethod do-attach ((self window) &aux gc fg bg lx ly)
  (setq lx (label-x self)
	ly (label-y self))
  (setf (slot-value self 'label-x) 0
	(slot-value self 'label-y) 0)
  (setf (repaint-flag self) nil)
  (make-window-gcs self)
  (when (and (setq gc (slot-value self 'gc-res)) 
	     (or (null (background self)) 
		 (eq :parent-relative (background self))))
	(setf (slot-value self 'cached-background)
	      (xlib:gcontext-background gc)))
  (setq bg (slot-value self 'background)
	fg (slot-value self 'foreground))
  (setf (slot-value self 'background) nil
	(slot-value self 'foreground) nil)
  (setf (background self) bg
	(foreground self) fg)
  (setf (font self) (slot-value self 'font))
  (border-init (border-type self) self)
  (label-init (label-type self) self)
  (if lx (setf (label-x self) lx))
  (if ly (setf (label-y self) ly))
  (setf (repaint-flag self) t)
  t)

(defmethod do-detach ((self window)) 
  (mapc #'(lambda (x) (when (paint-p x) (detach x)))
	(list 
	 (slot-value self 'foreground)
	 (slot-value self 'background)))
  (destroy-window-gcs self))

(defmethod do-pend ((self window))
  (setf (state self) (logior (state self) *pended*)))

(defmethod do-unpend ((self window))
  (setf (state self) (logandc2 (state self) *pended*)))

(defmethod do-make-invisible ((self window) 
			      &key (x-unmap t))
  (declare (ignore x-unmap))
  (setf (state self) (logior (state self) *not-visible*)))

(defmethod do-make-uninvisible ((self window)
				&key 
				&allow-other-keys)
  (setf (state self) (logandc2 (state self) *not-visible*)))

(defmethod do-conceal ((self window) 
		       &key 
		       &allow-other-keys)
  ;; Setf will do validation, propagation
  (setf (status self) :concealed))

(defmethod do-expose ((self window)
		      &key 
		      &allow-other-keys)
  ;; Setf method will do the status-change, if appropriate
  (setf (status self) :exposed))

;;;
;;; window operation methods
;;;

(defmethod destroy ((self window))
  (when (attached-p self)
	(detach self)
	(delete-child (parent self) self)))

(defmethod clear ((self window)
		  &key
		  &allow-other-keys)
  ;; repaint the background if not null
  (when (exposed-p self)
	(if (background self) 
	    (xlib:draw-rectangle (res self) (slot-value self 'intern-gc)
				 (repaint-x self)
				 (repaint-y self)
				 (width self) (height self)
				 t)
	    (xlib:clear-area (res self) 
			     :x (repaint-x self) 
			     :y (repaint-y self)
			     :width (width self)
			     :height (height self)
			     :exposures-p nil))))

;;; replaced by function below
;;;(defmethod repaint :around ((self window)
			    ;;;&key
			    ;;;(clear t)
			    ;;;&allow-other-keys)
  ;;;(declare (ignore clear))
  ;;;(if (and (repaint-flag self) (exposed-p self))
      ;;;(call-next-method)))

(defmethod do-repaint ((self window)
		       &key 
		       &allow-other-keys)
  t)

(defun repaint (w &key (clear t))
  (when (repaint-flag w)
	(when (exposed-p w)
	      (when clear (clear w))
	      (do-repaint w)
;;;	      (border-repaint (border-type w) w)
	)))

(defun repaint-region (win x y w h &key (clear t))
  (when (repaint-flag win)
	(when (exposed-p win)
	      (when clear (clear-region win x y w h))
	      (if (or (not (partial-repaint-p win))
		      (synth-p win) )
		  (progn 
		    (do-repaint win :clear clear))
		(progn	
		  (do-repaint-region win x y w h :clear (collection-p win))
;;;	          (border-repaint (border-type win) win)
		  ))
	      )))

(defmethod invert ((self window))
  (setf (inverted self) (not (inverted self))))

(defmethod dim ((self window))
  (setf (dimmed self) (not (dimmed self))))

(defmethod widget ((self window))
  self)

(defun pg (root &key (depth 10000) (level 0))
  (if (not (zerop depth))
      (let ((blanks (make-string level :initial-element #\space)))
	   (format t "~a~s~%" blanks root)
	   (format t "    ~aRegion: ~S~%" blanks (region root))
	   (format t "    ~aVirtual-Region: ~S~%" blanks (virtual-region root))
	   (format t "    ~aResize-hint: ~S~%" blanks (resize-hint root))
	   (format t "    ~aStatus: ~S~%" blanks (status root))
	   (format t "    ~aGeom-spec: ~S~%" blanks (geom-spec root))
	   (format t "    ~aAttached: ~S~%" blanks (attached-p root))
	   (if (collection-p root)
	       (format t "    ~aGm: ~S~%" blanks (gm root)))
	   (when (collection-p root)
		 (format t "    ~aChildren:~%" blanks)
		 (dolist (ch (children root))
			 (pg ch :depth (1- depth) :level (+ level 4))))))
  root)

(defun pt (root &key (depth 10000) (level 0))
  (if (not (zerop depth))
      (let ((blanks (make-string level :initial-element #\space)))
	   (format t "~a~s~%" blanks root)
	   (format t "    ~aRegion: ~S~%" blanks (region root))
	   (format t "    ~aResize-hint: ~S~%" blanks (resize-hint root))
	   (format t "    ~aStatus: ~S~%" blanks (status root))
	   (format t "    ~aState: ~S~%" blanks (state root))
	   (format t "    ~aAttached: ~S~%" blanks (attached-p root))
	   (if (gadget-p root)
	       (format t "    ~aValue: ~S~%" blanks (value root)))
	   (when (collection-p root)
		 (format t "    ~aChildren:~%" blanks)
		 (dolist (ch (children root))
			 (pt ch :depth (1- depth) :level (+ level 4))))))
  root)

(defun reload-picasso-object-named (name-form &optional (destroy-old t) 
					      &aux old new)
  (setq old (gethash name-form *global-dict*))
  (remhash name-form *global-dict*)
  (setq new (find-picasso-object-named name-form))
  (setf (lexical-parent new) (lexical-parent old))
  (when (and destroy-old old)
	(format t "Destroying old. . .~%")
	(destroy old))
  new)
