(in-package "PT")

;;;
;;; Macros to determine a windows status.
;;;

(defmacro exposed-p (window)
  "Return whether the window is exposed (visible on screen)."
  `(eq (status ,window) :exposed))

(defmacro stat-exposed-p (window)
  "Return whether the window's status is :exposed (wants to be visible)."
  `(eq (slot-value ,window 'status) :exposed))

(defmacro pending-p (window)
  "Return whether the window wants to be visible, but is not for some reason"
  `(eq (status ,window) :pending))

(defmacro concealed-p (window)
  "Return whether the window is concealed (not visible on screen)."
  `(eq (status ,window) :concealed))

(defmacro invisible-p (window)
  "Return whether the window is invisible (exposed, but parent not)"
  `(not (zerop (logand (state ,window) *not-visible*))))

(defmacro pended-p (window)
  "Return whether the window has been pended (exposed, but too small)"
  `(not (zerop (logand (state ,window) *pended*))))

(defmacro active-p (window)
  "Return whether the window is active (has a parent)"
  `(exposed-p ,window))

(defmacro attached-p (self)
  "Return whether the object is attached (has an X server rep)"
  `(not (null (res ,self))))

(defmacro detached-p (self)
  "Return whether the object is attached (has an X server rep)"
  `(null (res ,self)))

(defmacro window-attached-p (window)
  "Return whether the object is attached (has an X server rep)"
  `(not (null (res ,window))))

(defmacro window-detached-p (window)
  "Return whether the window is not attached (has no X server rep)"
  `(null (res ,window)))

(defmacro managed-p (window)
  "Return whether the window's geometry is being managed."
  `(and (stat-exposed-p ,window) (attached-p ,window)))

(defmacro exposed-of (window-list)
  "Return a list of exposed children windows."
  `(mapcan #'(lambda (ch) (if (exposed-p ch) (list ch))) ,window-list))

(defmacro stat-exposed-of (window-list)
  "Return a list of children windows that are or want to be visible."
  `(mapcan #'(lambda (ch) (if (stat-exposed-p ch) (list ch))) ,window-list))

(defmacro managed-of (window-list)
  "Return a list of children windows that are managed."
  `(mapcan #'(lambda (ch) (if (managed-p ch) (list ch))) ,window-list))

(defmacro invisible-of (window-list)
  "Return a list of children windows that are invisible."
  `(mapcan #'(lambda (ch) (if (invisible-p ch) (list ch))) ,window-list))

(defmacro pending-of (window-list)
  "Return a list of children windows that are not on the screen, but exposed."
  `(mapcan #'(lambda (ch) (if (pending-p ch) (list ch))) ,window-list))

(defmacro concealed-of (window-list)
  "Return a list of children windows that are concealed."
  `(mapcan #'(lambda (ch) (if (concealed-p ch) (list ch))) ,window-list))

(defmacro attached-of (window-list)
  "Return a list of children windows that are attached."
  `(mapcan #'(lambda (ch) (if (attached-p ch) (list ch))) ,window-list))

(defmacro detached-of (window-list)
  "Return a list of children windows that are not attached."
  `(mapcan #'(lambda (ch) (if (detached-p ch) (list ch))) ,window-list))

(defmacro gadgets-of (window-list)
  "Return a list of children windows that are gadgets."
  `(mapcan #'(lambda (ch) 
		     (unless (typep ch 'x-window) (list ch))) ,window-list))

(defmacro exposed-gadgets-of (window-list)
  "Return a list of children windows that are gadgets and exposed."
  `(mapcan #'(lambda (ch) (unless (or (typep ch 'x-window)
				      (not (exposed-p ch)))
				  (list ch))) ,window-list))

;;;
;;; macros which act as around methods for common methods. Thses all affect
;;; a windows status.
;;;
;;; several have associated defun forms to ease tracing for debugging
;;; and to allow these to be passed to apply or mapcar
;;;

(defmacro viewable-p (window)
  `(and (attached-p ,window) 
	(eq :viewable (xlib:window-map-state (res ,window)))))

(defmacro macro-attach (window)
  `(if (or (not (typep ,window 'window)) 
	   (and (typep (parent ,window) 'window)
		(attached-p (parent ,window))))
       (do-attach ,window)))

(defun attach (w)
  (macro-attach w))

(defmacro macro-conceal (window &rest keywords)
  `(unless (concealed-p ,window) 
	   (do-conceal ,window ,@keywords)
	   (when (and (viewable-p (parent ,window))
		      (attached-p ,window))
		 (border-clear (border-type ,window) ,window)
		 (label-clear (label-type ,window) ,window))))

(defun conceal (w &key (transparent nil) (x-unmap t))
      (macro-conceal w :transparent transparent :x-unmap x-unmap))

(defmacro macro-expose (window &rest keywords)
  `(unless (and (exposed-p ,window) (viewable-p ,window))
	   (do-expose ,window ,@keywords)))

(defun expose (w &key (x-map t))
  (macro-expose w :x-map x-map))

(defmacro macro-detach (window)
  `(if (attached-p ,window)
       (do-detach ,window)))

(defun detach (w)
  (macro-detach w))

(defmacro macro-pend (window)
  `(unless (pended-p ,window)
	   (do-pend ,window)))

(defun pend (w) 
  (macro-pend w))

(defmacro macro-unpend (window)
  `(if (pended-p ,window)
       (do-unpend ,window)))

(defun unpend (w) 
  (macro-unpend w))

(defmacro macro-make-invisible (window &key (x-unmap t))
  `(do-make-invisible ,window :x-unmap ,x-unmap))

(defun make-invisible (w &key (x-unmap t))
  (macro-make-invisible w :x-unmap x-unmap))

(defmacro macro-make-uninvisible (window)
  `(if (and (invisible-p ,window) (exposed-p (parent ,window)))
       (do-make-uninvisible ,window)))

(defun make-uninvisible (w)
  (macro-make-uninvisible w))
