(in-package "CLING")

(defconstant max-ingres-name-length 24)

(defvar *cling-available* nil)

(defvar time-in-cling 0)
(defvar ptime-in-cling 0)

(defvar current-db-name nil)
(defvar retrieve-in-progress nil)
(defvar show-query t)

(defvar to-ingres nil)
(defvar to-string nil)
(defvar write-out-string "")

(defvar *binary-ops* '(+ - * / ** > < >= <= = != and or))

;; Association list of cursor names and their select statements and update
;; types.
(defvar *cursor-list* nil)

;; Records name of the cursor open for deferred update. (max 1)
(defvar *deferred-update* nil)

(defvar *retrieve-string-buffer* (make-string 2001))

;;;
;;;; EXPAND-TARGET-LISTS
;;;
;;; Set to nil to prevent target lists from being expanded to include column
;;; type and to eliminate `*'s.
;;;
(defvar expand-target-lists t)

(defvar *error-handler* nil)
(defvar *warn-handler* nil)
(defvar *msg-handler* nil)
(defvar *not-found-handler* nil)
(defvar *autocommit* nil)

(defvar nesting-level 0)

(defvar sqlca-ptr nil)

(defvar *libq-loaded* nil)

(defvar *libq-argument-check* t)

;; *libq-object-location* and *libq-foreign-files* in headers/compat.cl
(defconstant *libq-system-libraries* '("m"))

(defvar sqlda nil)

(defconstant sqlda-num-vars 128)  ;; if changed, change the defcstruct below

;;;
;;; Increment the timers to keep an idea of how much time is being spent
;;; running this garbage.
;;; Is this really of interest?
;;;

(defmacro record-time (&rest body)
  `(let ((start-time (get-internal-real-time))
         (start-ptime (get-internal-run-time))
         (final-value (progn ,@body)))
        (setq time-in-cling             ;### integer math will truncate this!
              (+ time-in-cling (/ (- (get-internal-real-time) start-time)
                                internal-time-units-per-second)))
        (setq ptime-in-cling
              (+ ptime-in-cling
                 (/ (- (get-internal-run-time) start-ptime)
                    internal-time-units-per-second)))
        final-value))

;;;
;;; This is a call to IIwritio with an option to print the text to
;;; stdout for debugging and/or to a variable write-out-string.
;;;

(defmacro write-out (object)
  `(progn
        (if to-ingres
                (ii-writio 0 0 1 32 0 ,object))
        (if show-query
                (princ ,object))
        (if to-string
                (setq write-out-string
                          (concatenate 'string write-out-string ,object)))))

;;;
;;; This sends all write-out's to write-out-string and not to ingres.
;;; Clears write-out-string before new output is added.
;;;

(defmacro write-out-to-string (&rest body)
  `(let ((old-ti to-ingres)
                 (old-ts to-string))
                (unwind-protect
                 (progn
                  (setq to-ingres nil)
                  (setq to-string t)
                  (setq write-out-string "")
                  ,@body)
                 (setq to-ingres old-ti)
                 (setq to-string old-ts))))


;;;
;;; IIsyncup is called after everything except a retrieve that returns data
;;; to the user.
;;;
(defmacro end-command ()
  `(when *cling-available*
     (unless (nested-command-mode)
	     (ii-syncup 0 0)
	     (if show-query (terpri))
	     (dispatch-whenever (sqlstatus sqlca-ptr)))))

(defmacro listify (arg)
  `(if (listp ,arg)
    ,arg
    (list ,arg)))

(defmacro autocommit ()
  `(when *cling-available*
	 (if *autocommit* (do-db-commit))))

(defmacro nested-command (&rest body)
  `(let ((old-level nesting-level))
        (unwind-protect
         (progn
          (incf nesting-level)
          ,@body)
         (setq nesting-level old-level))))

(defmacro temp-not-nested (&rest cmds)
  `(let ((old-val nesting-level)
         (cmd-val nil))
        (setq nesting-level 0)
        (setq cmd-val (progn ,@cmds))
        (setq nesting-level old-val)
        cmd-val))


;; sqlca structure          ; in C space not LISP space in allegro
#+allegro
(defcstruct (sqlca :malloc)
  (sqlcaid 8     :char)          ; char sqlcaid[8]
  (sqlcabc              :long)
  (sqlcode              :long)
  (sqlerrm              (sqlerrml       :short) ; struct {...} sqlerrm
                                (sqlerrmc 70    :char))  ; char sqlerrmc[70]
  (sqlerrp 8    :char)          ; char sqlerrp[8]
  (sqlerrd 6    :long)          ; long sqlerrd[6]
  (sqlwarn              (sqlwarn0       :char)  ; struct {...} sqlwarn
                                (sqlwarn1       :char)
                                (sqlwarn2       :char)
                                (sqlwarn3       :char)
                                (sqlwarn4       :char)
                                (sqlwarn5       :char)
                                (sqlwarn6       :char)
                                (sqlwarn7       :char))
  (sqlext  8    :char))         ; char sqlext[8]

#+lucid
(def-foreign-struct sqlerrmtype
  (sqlerrml :type :signed-16bit)
  (sqlerrmc :type (:array :character (70))))

#+lucid
(def-foreign-struct sqlwarntype
  (sqlwarn0 :type :character)
  (sqlwarn1 :type :character)
  (sqlwarn2 :type :character)
  (sqlwarn3 :type :character)
  (sqlwarn4 :type :character)
  (sqlwarn5 :type :character)
  (sqlwarn6 :type :character)
  (sqlwarn7 :type :character))

#+lucid
(def-foreign-struct sqlca
  (sqlcaid :type (:array :character (8))) 
  (sqlcabc :type :signed-32bit)
  (sqlcode :type :signed-32bit)
  (sqlerrm :type sqlerrmtype) 
  (sqlerrp :type (:array :character (8))) 
  (sqlerrd :type (:array :signed-32bit (6)))
  (sqlwarn :type sqlwarntype)
  (sqlext  :type (:array :character (8))))

#+allegro
(defcstruct sqlname-struct
  (sqlname1             :short)
  (sqlnamec 34  :char))

#+lucid
(def-foreign-struct sqlname-struct
  (sqlname1 :type :signed-16bit)
  (sqlnamec :type (:array :character (34))))

#+allegro
(defcstruct sqlvar-struct
  (sqltype              :short)
  (sqllen               :short)
  (sqldata      *       :char)
  (sqlind       *       :short)
  (sqlname              sqlname-struct))

#+lucid
(def-foreign-struct sqlvar-struct
  (sqltype :type :signed-16bit)
  (sqllen  :type :signed-16bit)
  (sqldata :type (:pointer :character))
  (sqlind  :type (:pointer :signed-16bit))
  (sqlname :type sqlname-struct))

#+allegro
(defcstruct sqlda
                             (sqldaid     8       :char)
                             (sqldabc             :long)
                             (sqln                :short)
                             (sqld                :short)
                             (sqlvar 128 sqlvar-struct))

#+lucid
(def-foreign-struct sqlda
  (sqldaid :type (:array :character (8)))
  (sqldabc :type :signed-32bit)
  (sqln    :type :signed-16bit)
  (sqld    :type :signed-16bit)
  (sqlvar :type (:array sqlvar-struct (128))))
