(in-package "PT")
#+cling (use--package "CLING")

;;;
;;; Define the class hierarchy for the facility database schema.
;;;

(defclass utility (fac-object)
  ((class :initarg :class :initform "Unknown" :type string :accessor class)
   (id :initarg :id :initform 0 :type integer :accessor id)
   (tuple :initarg :tuple :initform nil :type list :accessor tuple)
   ))

(defun utility-p (x) (typep x 'utility))

(defclass connection ()
  ((util-id :initarg :util-id :initform 0 :type integer :accessor util-id)
   (src-id :initarg :src-id :initform 0 :type t :accessor src-id)
   (dst-id :initarg :dst-id :initform 0 :type t :accessor dst-id)
   (tuple :initarg :tuple :initform nil :type list :accessor tuple)
   ))

(defun connection-p (x) (typep x 'connection))

;;; =============================================================
;;;
;;; Convert a utility tuple to a CLOS utility object.
;;;
(defun make-util-from-tuple (tuple)
  ;; Tuples are of form (id name class)
  (let ((id (first tuple))
	(name (trim-str (second tuple)))
	(class (trim-str (third tuple))))
       (make-instance 'utility :id id :tuple tuple :name name :class class)))

;;;
;;; Read all the utilties from the database -- faked, for now.
;;; Return a list of the objects read.
;;;
#-cling
(defun read-db-util (fac-id)
  "Read the utilities from the database for this facility and build the utility objects."
  (let* ((in-file (open (picasso-path "lib/db/util") :direction :input))
	 (rv (mapcar #'make-util-from-tuple (do-read in-file nil nil))))
	(close in-file)
	;; Set the fac-id of each.
	(dolist (obj rv) (setf (fac-id obj) fac-id))
	rv))

#+cling
(defun read-db-util (fac-id)
  "Read the utilities from the database for this facility and build the utility objects."
  (let ((rv (db-mapl-retrieve 
	     #'make-util-from-tuple
	     '((:integer utility.id)
	       (:string utility.kind)
	       (:string utility.class)))))

       ;; Set the fac-id.
       (dolist (obj rv) (setf (fac-id obj) fac-id))
       rv))

(defun save-db-util (ulist)
  (let ((tuple-list (mapcar #'tuple ulist))
	(out-file (open (picasso-path "lib/db/util") :direction :output)))
       (format out-file "(")
       (dolist (tup tuple-list)
	       (format out-file "~s~%" tup))
       (format out-file ")")
       (close out-file)))

;;; =============================================================
;;;
;;; Convert a connection tuple to a CLOS connection object.
;;;
(defun make-connection-from-tuple (tuple)
  ;; Tuples are of form (src dst)
  (let* ((src (first tuple))
	 (dst (second tuple)))
	(make-instance 'connection :src-id src :dst-id dst)))

;;;
;;; Read all the connections from the database -- faked, for now.
;;; Return a list of the objects read.
;;;
#-cling
(defun read-db-connection (fac-id)
  "Read the connections from the database for this facility and build the conncetion objects."
  (declare (ignore fac-id))
  (let* ((in-file (open (picasso-path "lib/db/connection") :direction :input))
	 (rv (mapcar #'make-connection-from-tuple (do-read in-file nil nil))))
	(close in-file)
	rv))

#+cling
(defun read-db-connection (fac-id)
  "Read the connections from the database for this facility and build the conncetion objects."
  (let ((rv (db-mapl-retrieve
	     #'make-connection-from-tuple
	     '((:integer connection.startjunct)
	       (:integer connection.endjunct)))))

       ;; Set the fac-id.
       rv))

(defun save-db-connection (clist)
  (let ((tuple-list (mapcar #'tuple clist))
	(out-file (open (picasso-path "lib/db/connection") :direction :output)))
       (format out-file "(")
       (dolist (tup tuple-list)
	       (format out-file "~s~%" tup))
       (format out-file ")")
       (close out-file)))

;;; =============================================================
;;;
;;; Resolve all the graphical objects
;;;
(defun resolve-connections (connect-list junct-tab)
  (dolist (c connect-list)
	  (let* ((src (gethash (src-id c) junct-tab))
		 (dst (gethash (dst-id c) junct-tab))
		 v1 v2)
		(if (null src)
		    (warn "Unable to find junction with id=~s" (src-id c))
		    (setq v1 (vertex-id src)))
		(if (null dst)
		    (warn "Unable to find junction with id=~s" (dst-id c))
		    (setq v2 (vertex-id dst)))
		(setf (src-id c) v1
		      (dst-id c) v2
		      (util-id c) (util-id src)))))

(defun resolve-utils (util-list con-list)
  (dolist (u util-list)
	  (let (lines sv dv)
	       (dolist (con con-list)
		       (when (eq (util-id con) (id u))
			     (setq sv (src-id con)
				   dv (dst-id con))
			     (push (new-sgg-line sv dv) lines)))
	       (new-sgg-object :id u
			       :line-ids lines
			       :annot-ids nil
			       :symbol-ids nil
			       :selectable t
			       :visible t
			       :color "white"
			       :line-width 0))))

(defun find-util-join (key)
  (let* ((util #!util-list))
        (cond ((= key *util-equip*)
               (find-util-equip util #!eq-junct-list #!sel-equip))
              ((= key *util-junct*)
               (find-util-junct util #!sel-junct))
              (t
               (warn "Internal error: illegal util-join op.")
               nil))))

