(in-package "PT")
#+cling(use-package "CLING")

;;;
;;; Define the class hierarchy for the facility database schema.
;;;

(defclass equipment (fac-object)
  ((id :initarg :id :initform 0 :type integer :accessor id)
   (class :initarg :class :initform "Unknown" :type string :accessor class)
   (make :initarg :make :initform "Unknown" :type string :accessor make)
   (descrip :initarg :descrip :initform "" :type string :accessor descrip)
   (tuple :initarg :tuple :initform nil :type list :accessor tuple)
   (faults-tuple :initarg :tuple :initform nil :type list 
		 :accessor faults-tuple)
   (picture-keys :initform nil :type list :accessor picture-keys)))

(defun equipment-p (x) (typep x 'equipment))

;;; =============================================================
;;;
;;; Convert a equip tuple to a CLOS equipment object.  Also registers
;;; the associated graphic object.
;;;
(defun make-equip-from-tuple (tuple)
  ;; Tuples are of form (id class make instance descrip x1 y1 x2 y2 x3 y3 x4 y4)
  (let* ((id (first tuple))
	 (class (trim-str (second tuple)))
	 (make (trim-str (third tuple)))
	 (name (trim-str (fourth tuple)))
	 (descrip (trim-str (fifth tuple)))
	 (x1 (nth 5 tuple))
	 (y1 (nth 6 tuple))
	 (x2 (nth 7 tuple))
	 (y2 (nth 8 tuple))
	 (x3 (nth 9 tuple))
	 (y3 (nth 10 tuple))
	 (x4 (nth 11 tuple))
	 (y4 (nth 12 tuple))
	 (v1 (new-sgg-vertex x1 y1))
	 (v2 (new-sgg-vertex x2 y2))
	 (v3 (new-sgg-vertex x3 y3))
	 (v4 (new-sgg-vertex x4 y4))
	 (av (new-sgg-vertex (/ (+ x1 x2 x3 x4) 4) (/ (+ y1 y2 y3 y4) 4)))
	 (aw (- (max x1 x2 x3 x4) (min x1 x2 x3 x4)))
	 (ah (- (max y1 y2 y3 y4) (min y1 y2 y3 y4)))
	 (line1 (new-sgg-line v1 v2))
	 (line2 (new-sgg-line v2 v3))
	 (line3 (new-sgg-line v3 v4))
	 (line4 (new-sgg-line v4 v1))
	 (annot (new-sgg-annotation name av aw ah))
	 (rv nil))
	(setq rv (make-instance 'equipment :id id :tuple tuple
				:class class :make make
				:name name :descrip descrip))
	(new-sgg-object :id rv 
			:line-ids `(,line1 ,line2 ,line3 ,line4)
			:annot-ids `(,annot)
			:symbol-ids nil
			:selectable t
			:visible t
			:color "white"
			:line-width 0)
	rv))

;;;
;;; Read all the equipment from the file *picasso-home*/lib/db/equip
;;; Return a list of the objects read.
;;;
#-cling
(defun read-db-equip (fac-id)
  "Read the equipment from the database for this facility and build the equipment objects."
  (let* ((in-file (open (picasso-path "lib/db/equip") :direction :input))
	 (rv (mapcar #'make-equip-from-tuple
		     (do-read in-file nil nil))))
	(close in-file)
	;; Set the fac-id of each. We may also wish to set the color, etc.
	(dolist (obj rv) (setf (fac-id obj) fac-id))
	rv))

#+cling
(defun read-db-equip (fac-id)
  "Read the equipment from the database for this facility and build the equipment objects."
  (let ((rv (db-mapl-retrieve
	     #'make-equip-from-tuple
	     '((:integer equipment.id)
	       (:string equip_name.class)
	       (:string equip_name.make)
	       (:string equip_name.name)
	       (:string equip_name.descrip)
	       (:float equipment.x1)
	       (:float equipment.y1)
	       (:float equipment.x2)
	       (:float equipment.y2)
	       (:float equipment.x3)
	       (:float equipment.y3)
	       (:float equipment.x4)
	       (:float equipment.y4))
	     :where '(= equipment.id equip_name.id))))
       ;; Set the fac-id of each. We may also wish to set the color, etc.
       (dolist (obj rv) (setf (fac-id obj) fac-id))
       rv))

;;;
;;; Save all the equipment to the file *picasso-home*/lib/db/equip
;;;
(defun save-db-equip (elist)
  (let ((tuple-list (mapcar #'tuple elist))
	(out-file (open (picasso-path "lib/db/equip") :direction :output)))
       (format out-file "(")
       (dolist (tup tuple-list)
	       (format out-file "~s~%" tup))
       (format out-file ")")
       (close out-file)))

(defun find-picture-keys (disk equip)
  (let ((instance (read-from-string (name equip)))
	pics)
       (maphash #'(lambda (k v) 
                     (declare (ignore v))
                     (if (eql (car k) instance) (push k pics))) 
		(video:disk-index disk))
       (setf (picture-keys equip) (sort pics #'< :key #'cadr))))

(defun find-eq-join (key)
  (let* ((equip #!equip-list))
        (cond ((= key *eq-space*)
               (find-eq-space equip #!sel-space))
              ((= key *eq-util*)
               (find-eq-util equip #!eq-junct-list #!sel-util))
              ((= key *eq-lots*)
               (find-eq-lot equip #!sel-lot #!eq-run-map))
              ((= key *eq-junct*)
               (find-eq-junct equip #!sel-junct))
              (t
               (warn "Internal error: illegal eq-join op.")
               nil))))

