;;;
;;; Copyright (c) 1990 Regents of the University of California
;;; 
;;; $Author: picasso $
;;; $Source: RCS/text-gadget.cl,v $
;;; $Revision: 1.1 $
;;; $Date: 90/07/18 09:25:43 $
;;;

(in-package 'pt  :nicknames '(picasso-toolkit) :use '(lisp excl pcl))

;;;
;;; Definition of the text-gadget class
;;;

(defclass text-gadget (gadget)
  ((horiz-just
    :initarg :horiz-just :initform :center
    :type keyword
    :accessor horiz-just)
   (vert-just
    :initarg :vert-just :initform :center
    :type keyword
    :accessor vert-just)
   (font
    :initarg :font :initform *default-font-name*
    :type font
    :accessor font)
   (mask
    :initarg :mask :initform nil
    :type t
    :accessor mask)
   (self-adjusting
    :initarg :self-adjusting :initform nil
    :type atom
    :accessor self-adjusting)

   ;; Override some parents defaults...
   (gc-spec :initform '(gc-res "default"))
   (foreground :initform "black")
   (dimmed-foreground :initform "gray50")
   (inverted-foreground :initform "white")
   (inverted-background :initform "black")

   ;;  Internal
   (width-cache
    :initform nil
    :type list)))

(defun make-text-gadget (&rest keys)
  (apply #'make-instance 'text-gadget :allow-other-keys t keys))

;;;
;;; text-gadget setf methods
;;;

(defun recompute-text-gadget-cache (self)
  (let ((font-res (res (font self)))
	(sw-list nil))
       (dolist (str (value self))
	       (setq sw-list (nconc sw-list
				    (list (xlib:text-width font-res str)))))
       (setf (slot-value self 'width-cache) sw-list)
       (if (self-adjusting self)
	   (let ((base-width 0)
		 (base-height 0))
		(if sw-list
		    (setq base-width (apply #'max sw-list)))
		(setq base-height (* (length sw-list) 
				     (font-height (font self))))
		(setf (base-size self) (list (+ 4 (max base-width 1))
					     (+ 4 (max base-height 1))))))))

(defun break-up-string (str)
  "Return a list of strings corresponding to breaking this string at all
  newlines."
  (let ((value nil))
       (do ((pos (position #\newline str) (position #\newline str)))
	   ((null str))
	   (cond
	    (pos
	     (setf value (nconc value (list (subseq str 0 pos))))
	     (setf str (subseq str (1+ pos))))
	    (t
	     (setf value (nconc value (list str)))
	     (setf str nil))))
       value))

(defmethod (setf value) ((value list) (self text-gadget))
  (let ((text nil))
       (dolist (str value)
	       (setf text (nconc text (break-up-string str))))
       (setf (slot-value self 'value) text))
  (when (attached-p self)
	(recompute-text-gadget-cache self)
	(repaint self)))

(defmethod (setf value) ((value string) (self text-gadget))
  (setf (slot-value self 'value) (break-up-string value))
  (when (attached-p self)
	(recompute-text-gadget-cache self)
	(repaint self)))

(defmethod (setf dimmed) (val (self text-gadget))
  (call-next-method)
  (setf (slot-value self 'dimmed) val)
  (repaint self))

(defmethod (setf font) (value (self text-gadget))
  (when value
	(cond ((attached-p self) 
	       (let ((old (slot-value self 'font))) 
		    (when (font-p old) 
			  (font-detach old))) 
	       (unless (or (and (stringp value) 
				(setq value (make-font :name value
						       :display (display self)))) 
			   (font-p value))
		       (error "text-gadget.setf.font: invalid font ~s." value)) 
	       (setf (slot-value self 'font) value)
	       (font-attach value)
	       (setf (xlib:gcontext-font (gc-res self)) (res value))
	       (recompute-text-gadget-cache self))
	      (t (if (or (stringp value) (font-p value))
		     (setf (slot-value self 'font) value)
		     (error "text-gadget.setf.font: invalid font ~s." value))))))

(defmethod new-instance ((self text-gadget)
			 &key
			 (value "")
			 (base-width nil)
			 (base-height nil)
			 (base-size nil)
			 &allow-other-keys &aux df)
  (declare (ignore value))
  (call-next-method)
  (if (null (or base-width base-height base-size))
      (setf (self-adjusting self) t))
  (when base-size
	(setq base-width (car base-size)
	      base-height (cadr base-size)))
  (cond ((and base-width base-height) 
	 (setf (base-size self) (list base-width base-height)))
	(base-width
	 (setf (base-width self) base-width))
	(base-height
	 (setf (base-height self) base-height)))
  (when (stringp (setq df (dimmed-foreground self)))
	(setf (dimmed-foreground self) (get-paint df self)))
  self)

(defmethod do-attach ((self text-gadget))
  (call-next-method)
  (setf (dimmed self) (slot-value self 'dimmed)))

(defmethod do-repaint ((self text-gadget))
  (let* ((font (font self))
	 (width (width self))
	 (height (height self))
	 (rx (repaint-x self))
	 (ry (repaint-y self))
	 (mask (mask self))
	 (sh (font-ascent font))
	 (len (length (value self)))
	 (gc (gc-res self))
	 (res (res self))
	 (vert-just (vert-just self))
	 (horiz-just (horiz-just self))
	 (top-offset 
	  (case vert-just
		(:top 0)
		(:center (round (/ (- height (* sh (- len 2))) 2)))
		(:bottom (- height (* sh (1- len))))))
	 (x 0)
	 (y 0)
	 (h (font-height font)))
	(clear self)
	(do* ((str-list (value self) (cdr str-list))
	      (str (car str-list) (car str-list))
	      (sw-list (slot-value self 'width-cache) (cdr sw-list))
	      (sw (car sw-list) (car sw-list))
	      (count 0 (1+ count)))
	     ((null str-list))
	     (setq y (+ top-offset (* h count)))
	     (case horiz-just
		   (:left (setq x 0))
		   (:right (setq x (- width sw)))
		   (:center (setq x (round (/ (- width sw) 2)))))
	     (setq x (+ x rx))
	     (setq y (+ y ry))
	     
	     ;; put text in specified color(s)
	     (if (dimmed self)
		 (if mask 
		     (draw-gray-text-mask res gc str x y sw h)
		     (draw-gray-text res gc str x y sw h))
		 (if mask
		     (xlib:draw-glyphs res gc x y str)
		     (xlib:draw-image-glyphs res gc x y str))))))

;;;
;;;  export symbols for package
;;;
(export '(font
	  horiz-just
	  make-text-gadget
	  mask
	  repaint
	  text-gadget
	  vert-just)
  (find-package 'pt))
