;;;
;;; Copyright (c) 1990 Regents of the University of California
;;; 
;;; $Author: seitz $
;;; $Source: RCS/text-buffer-gadget-methods.cl,v $
;;; $Revision: 1.4 $
;;; $Date: 90/07/26 17:49:30 $
;;;

(in-package 'pt  :nicknames '(picasso-toolkit) :use '(lisp excl pcl))

;;
;; METHODS FOR TEXT-BUFFER-GADGET
;;

(defmethod new-instance ((self text-buffer-gadget)
			 &key
			 (value nil)
			 (rows nil)
			 (columns nil)
			 (base-width nil)
			 (base-height nil)
			 (base-size nil)
			 (cursor-mode :overwrite)
			 (background (inverted-foreground self))
			 (inverted-background (foreground self))
			 (dimmed-background (background self))
			 &allow-other-keys
			 &aux font)
  (when base-size
	(setq base-width (first base-size))
	(setq base-height (second base-size)))
  
  (setf (slot-value self 'buffer) (make-buffer))
  (do-propagate 'buffer self)
  (call-next-method)
  (setf (font self) 
	(setq font (slot-value self 'font)))
  (when (null rows)
	(if base-height 
	    (setq rows (truncate base-height (font-ascent font)))
	    (setq rows 24)))
  
  (when (null columns)
	(if base-width
	    (setq columns (truncate base-width (font-width font)))
	    (setq columns 80)))
  
  (if (not (or base-width base-height))
      (setf (self-adjusting self) t))
  (setf (rows self) rows)
  (setf (columns self) columns)
  (setf (cursor-mode self) cursor-mode)
  (when value (setf (value self) value)))

(defmethod value ((self text-buffer-gadget))
  (buf-data (buffer self)))

(defmethod mark ((self text-buffer-gadget)
		 &key
		 (mark-row (mark-row self))
		 (mark-column (mark-column self)))
  (let* ((row (row self))
	 (column (column self))
	 (mr (mark-row self))
	 (mc (mark-column self))
	 (old-mr (if mr mr row))
	 (old-mc (if mc mc column)))
	(cond ((or (null mark-row) (null mark-column))
	       (unmark self))
	      ((or (> mark-row row)
		   (and (= mark-row row) (>= mark-column column)))
	       (setf (slot-value self 'mark-row) mark-row)
	       (setf (slot-value self 'mark-column) mark-column)
	       (do-propagate 'mark-row self)
	       (do-propagate 'mark-column self)
	       (tbg-repaint self
			    :start-row (min mark-row old-mr)
			    :end-row (max mark-row old-mr)))
	      (t
	       ;; mark point is before cursor; swap and redo.
	       ;; avoid unnecessary repaints.
	       (setf (repaint-flag self) nil)
	       (setf (row self) mark-row)
	       (setf (column self) mark-column)
	       ;; set mark point after setting row and column since their setf
	       ;; methods will unmark.
	       (setf (slot-value self 'mark-row) old-mr)
	       (setf (slot-value self 'mark-column) old-mc)
	       (do-propagate 'mark-row self)
	       (do-propagate 'mark-column self)
	       ;; re-activate repaints and do a big one covering everything.
	       (setf (repaint-flag self) t)
	       (tbg-repaint self 
			    :start-row mark-row
			    :end-row old-mr)))))

(defmethod unmark ((self text-buffer-gadget))
  (let ((row (row self))
	(column (column self))
	(mrow (mark-row self))
	(mcolumn (mark-column self)))
       (when (and mrow mcolumn)
	     (setf (slot-value self 'mark-row) nil)
	     (setf (slot-value self 'mark-column) nil)
	     (do-propagate 'mark-row self)
	     (do-propagate 'mark-column self)
	     (tbg-repaint self :start-row row :end-row mrow))))

(defmethod delete-mark ((self text-buffer-gadget))
  "This will zap the marked text into non-existance."
  (let* ((row (row self))
	 (column (column self))
	 (mark-row (mark-row self))
	 (mark-column (mark-column self))
	 (buffer (buffer self))
	 (n (- mark-row row)))
	(setf (repaint-flag self) nil)
	(dotimes (foo n)
		 (when (< column (buf-columns buffer row))
		       (kill-line buffer row column))
		 (kill-line buffer row column))
	(setf (repaint-flag self) t)
	(cond ((> n 0)
	       (delete-char buffer row column
			    :count (1+ mark-column))
	       (unmark self)
	       (tbg-repaint self :start-row row))
	      (t ; i.e. (= n 0)
		 (delete-char buffer row column
			      :count (1+ (- mark-column column)))
		 (unmark self)
		 (tbg-repaint self :start-row row :end-row row)))))

(defmethod copy-mark ((self text-buffer-gadget))
  "This will copy the marked text into the x-cut-buffer number 0"
  (let* ((row (row self))
	 (column (column self))
	 (mark-row (mark-row self))
	 (mark-column (mark-column self))
	 (d (res (display self)))
	 (buffer (buffer self)))
	(cond ((= mark-row row) 
	       (let ((str (value buffer :row row :column column)))
		    (setf (xlib:cut-buffer d)
			  (copy-seq (my-subseq str 0 
					       (min (1+ (- mark-column column))
						    (length str)))))))
	      (t
	       ;;i.e. (> mark-row row)
	       ;; first line first
	       (setf (xlib:cut-buffer d) (value buffer :row row :column column))
	       (add-bytes d (string #\Newline))
	       
	       ;; now the lines in between the first and the last
	       (do ((n (1+ row) (1+ n)))
		   ((= n mark-row))
		   (add-bytes d (value buffer :row n))
		   (add-bytes d (string #\Newline)))
	       
	       ;; finally, the last line
	       (let ((str (value buffer :row mark-row)))
		    (add-bytes d (my-subseq str 0 (min (1+ mark-column)
						       (length str)))))))))

(defmethod resize-window-handler ((self text-buffer-gadget))
  "window has been resized; reset rows and columns.
  This method is called whenever some external force (possibly a geometry
manager) changes the size of the window of the gadget."
  (let* ((rows (rows self))
	 (columns (columns self))
	 (font (font self))
	 (fw (font-width font))
	 (fh (font-height font))
	 (width (width self))
	 (height (height self))
	 (new-rows (truncate height fh))
	 (new-columns (truncate width fw)))
	(setf (rows self) new-rows)
	(setf (columns self) new-columns)
	(tbg-repaint self)))

(defmethod do-repaint-region ((self text-buffer-gadget)
			      x y width height
			      &key
			      (clear t)
			      &allow-other-keys)
  "Called whenever there is an expose-region event. 
  x, y, width and height are in pixels."
  (when clear (clear-region self x y width height))
  (let* ((top-of-screen (top-of-screen self))
	 (left-of-screen (left-of-screen self))
	 (font (font self))
	 (fw (width font))
	 (fh (height font))
	 (start-row (+ top-of-screen (truncate y fh)))
	 (end-row (+ start-row (truncate height fh)))
	 (start-column (+ left-of-screen (truncate x fw)))
	 (end-column (+ start-column (truncate width fw))))
	(tbg-repaint self
		     :start-row start-row :end-row end-row
		     :start-column start-column :end-column end-column)))


;;
;; the do-repaint method is THE time sink in text-buffer-gadget.
;; Any speed optimizing should start here.
;;

(defmethod do-repaint ((self text-buffer-gadget))
  (tbg-repaint self))

(defun tbg-repaint (self
		    &key
		    (start-row nil)
		    (end-row nil)
		    (start-column nil)
		    (end-column nil)
		    &allow-other-keys)
  
  (if (not (and (exposed-p self) (repaint-flag self)))
      (return-from tbg-repaint))
  
  (let* ((dimmed (dimmed self))
	 (inverted (inverted self))
	 (gc (gc-res self))
	 (mark-row (mark-row self))
	 (mark-column (mark-column self))
	 (last-line (1- (rows (buffer self))))
	 (top-of-screen (top-of-screen self))
	 (left-of-screen (left-of-screen self))
	 (top-pad (top-pad self))
	 (rows (rows self))
	 (columns (columns self))
	 (row (row self))
	 (column (column self))
	 (font (font self))
	 (fw (font-width font))
	 (fh (font-height font)))
	;;
	;; start-row, end-row, start-column, end-column are the inclusive
	;; boundaries of the area to be repainted.
	;;
	
	(if start-row (setq start-row (max top-of-screen start-row))
	    (setq start-row top-of-screen))
	
	(if end-row
	    (setq end-row (min end-row (+ top-of-screen rows) last-line))
	    ;; end-row may be set to a value greater than last-line, but that
	    ;; is necessary since we want to clean the junk that remains after
	    ;; the last line if, for example, we scroll to the end of the buffer.
	    (setq end-row (+ top-of-screen rows)))
	
	(if start-column (setq start-column (max left-of-screen start-column))
	    (setq start-column left-of-screen))
	
	(if end-column
	    (setq end-column (min end-column (+ left-of-screen columns)))
	    (setq end-column (+ left-of-screen columns)))
	
	;; Try this out for debugging...
	(setq end-column (+ left-of-screen columns))
	(setq start-column left-of-screen)

	;;
	;; perform the actual repaint job (i.e. tell a function to do it.)
	;;
	
	(repaint-function self start-row start-column end-row end-column)
	
	;;
	;; when the mark point is non-nil, do not paint cursor since cursor
	;; position is the start of the marked range.
	;;
	;; both the marked range and the cursor are repainted in (not inverted)
	;; mode.  therefore it doesn't make sense do it when the text is dimmed.
	
	(unless dimmed
		;; if mark is off if either mark slot is nil.
		(cond ((and mark-row mark-column) 
		       ;; repaint row, mark-row and everything in between.
		       (cond ((= row mark-row)
			      (repaint-function self row column row mark-column
						t))
			     ((> mark-row row)
			      (repaint-function self row column row end-column
						t)
			      (repaint-function
			       self
			       mark-row start-column
			       mark-row (min mark-column end-column))
			      (when (> mark-row (1+ row))
				    (repaint-function
				     self
				     (1+ row) start-column (1- mark-row) 
				     end-column t)))))
		      ((current self)  ;no mark, paint cursor
		       (case (cursor-mode self)
			     (:insert
			      ;; vertical bar for insert mode.
			      (xlib:draw-rectangle 
			       (res self) gc 
			       (* fw (- column left-of-screen)) 
			       (+ top-pad (* fh (- row top-of-screen)))
			       2 fh 
			       t))
			     
			     (:overwrite
			      (repaint-function self row column row column
						t))))))))

(defun repaint-function (tbg start-row start-column end-row end-column
			     &optional (invert nil))
  "Streamlined repaint function. Does no boundary checks."
  (when (and (>= end-row start-row)
	     (>= end-column start-column))
	(let* ((top-of-screen (top-of-screen tbg))
	       (left-of-screen (left-of-screen tbg))
	       (top-pad (top-pad tbg))
	       (dimmed (dimmed tbg))
	       (res (res tbg))
	       (gc (gc-res tbg))
	       (font (font tbg))
	       (fh (font-height font))
	       (fw (font-width font))
	       (buffer (buffer tbg))
	       (width (width tbg))
	       (num-columns (- end-column start-column -1)) ; this -1 makes
	       ; boundaries inclusive.
	       (w (* fw num-columns)))
	      (when invert
		    (let ((fg (xlib:gcontext-foreground gc)))
			 (setf (xlib:gcontext-foreground gc)
			       (xlib:gcontext-background gc)
			       (xlib:gcontext-background gc) fg)))
	      (do* ((curr-row start-row (1+ curr-row))         ;row to be repainted
		    (y (+ top-pad (font-ascent font)
			  (* (- curr-row top-of-screen) fh))   ;y coordinate.
		       (+ y fh))
		    (x (* (- start-column left-of-screen) fw)) ;x coordinate
		    (str (string-adjust (tbg-get-value buffer
					       :row curr-row
					       :column start-column)
					num-columns)
			 (string-adjust (tbg-get-value buffer         
					       :row curr-row
					       :column start-column)
					num-columns)))
		   ((> curr-row end-row))
		   (if dimmed 
		       (draw-gray-text-mask res gc str x y width w fh)
		       (xlib:draw-image-glyphs res gc x y str)))
	      (when invert
		    (let ((fg (xlib:gcontext-foreground gc)))
			 (setf (xlib:gcontext-foreground gc)
			       (xlib:gcontext-background gc)
			       (xlib:gcontext-background gc) fg))))))

;; this is used by the repaint method.
(defvar *tbg-space-str* (make-string 200 :initial-element #\Space))
(defun string-adjust (str n)
  "return the first n characters of str.  pad with spaces is string is
  not long enough."
  (let ((len (length str)))
       (if (<= n len)
	   (subseq str 0 n)
	   (concatenate 'string
			str (subseq *tbg-space-str* 0 (- n len))))))

(defmethod new ((self text-buffer-gadget))
  (setf (row self) 0)
  (setf (column self) 0)
  (new (buffer self))
  (tbg-repaint self))

(defmethod put ((self text-buffer-gadget)
		s
		&key
		(overwrite nil)
		(repaint t)
		&aux new-row new-column)
  "insert string s into text-buffer-gadget at current cursor position,
  and update cursor position.  Repaint new text if repaint is t."
  (let ((row (row self))
	(column (column self)))
       (multiple-value-setq (new-row new-column)
			    (put (buffer self) s
				 :row row
				 :column column
				 :overwrite overwrite))
       
       (setf (row self) new-row)
       (setf (column self) new-column)
       
       (when repaint
	     (if overwrite
		 (if (= row new-row)
		     (tbg-repaint self
				  :start-row row :end-row row
				  :start-column column :end-column new-column)
		     ;; (/= row new-row)
		     (tbg-repaint self :start-row row :end-row new-row))
		 ;; insert mode
		 (if (= row new-row)
		     (tbg-repaint self
				  :start-row row :end-row row
				  :start-column column)
		     ;; (/= row new-row)
		     (tbg-repaint self :start-row row))))))

(defmethod load-file ((self text-buffer-gadget)
		      filename
		      &key
		      (count -1))
  (new self)
  (put-file self filename :count count))

(defmethod put-file ((self text-buffer-gadget)
		     filename
		     &key
		     (count -1))
  (put-file (buffer self) filename
	    :count count :repaint nil
	    :row (row self) :column (column self))
  (tbg-repaint self))

(defmethod save-file ((self text-buffer-gadget)
		      filename)
  (save-file (buffer self) filename))

(defmethod append-to-file ((self text-buffer-gadget)
			   filename)
  (append-to-file (buffer self) filename))

(defmethod search-forward ((self text-buffer-gadget) pattern)
  "Search for first forward occurence of pattern.  If found, return t
  and position cursor just past the last letter of the occurence. 
  If the pattern cannot be matched or if the pattern is
  the empty string, then return nil."
  (if (= (length pattern) 0) (values nil)
      (let* ((row (row self))
	     (column (column self))
	     (buffer (buffer self))
	     (last-row (1- (rows buffer)))
	     (fail (make-fail-array pattern))
	     pos)
	    
	    ;; first look at remainder of current line.
	    (setq pos (search-string (value buffer :row row :column (1+ column)) 
				     pattern fail))
	    
	    (cond (pos 
		   ;; successful 
		   (setf (column self) (+ column pos 1))
		   t)
		  (t
		   ;; look at rest of lines.
		   (do* ((row-count (1+ row) (1+ row-count))
			 (this-line (value buffer :row row-count)
				    (value buffer :row row-count)))
			((or (> row-count last-row)
			     (setq pos (search-string this-line pattern fail)))
			 ;; if pos is nil then search-string didn't find 
			 ;; anything. 
			 ;; Set the cursor position only if (/= pos nil)
			 (cond (pos
				;; successful.
				(setf (row self) row-count)
				(setf (column self) pos)
				t)
			       (t 
				;; search failed.
				nil)))
			()))))))

(defmethod search-backward ((self text-buffer-gadget) pattern)
  (if (= (length pattern) 0) (values nil)
      (let* ((row (row self))
	     (column (column self))
	     (buffer (buffer self))
	     pos)
	    
	    ;; first look at current line.
	    (setq pos (search-string-backward (subseq (value buffer :row row) 
						      0 column) 
					      pattern))
	    
	    (cond (pos 
		   ;; successful 
		   (setf (column self) pos)
		   t)
		  (t
		   ;; look at rest of lines.
		   (do* ((row-count (1- row) (1- row-count))
			 (this-line (value buffer :row row-count)
				    (value buffer :row row-count)))
			((or (< row-count 0)
			     (setq pos (search-string-backward this-line pattern)))
			 ;; if pos is nil then search-string didn't find 
			 ;; anything. 
			 ;; Set the cursor position only if (/= pos nil)
			 (cond (pos
				;; successful.
				(setf (row self) row-count)
				(setf (column self) pos)
				t)
			       (t 
				;; search failed.
				nil)))
			()))))))


(defun tbg-scroll-right (self n) 
  "try to scroll right by n columns.  However, will not go past last
  character in current line."
  (let* ((column (column self))
	 (row (row self))
	 (left-of-screen (min (+ n (left-of-screen self))
			      (buf-columns (buffer self) row)))
	 (font (font self))
	 (fw (width font))
	 (fh (height font))
	 (columns (columns self))
	 (rows (rows self))
	 (w (* columns fw))
	 (h (* rows fh))
	 (res (res self))
	 (right-of-screen (+ left-of-screen columns -1)))
	(setf (left-of-screen self) left-of-screen)
	(when (< column left-of-screen)
	      (setf (column self) left-of-screen))
	(if (< n columns)
	    (let ((offset (* n fw)))
		 (xlib:copy-area res (gc-res self) offset 0 w h res 0 0)))
	(tbg-repaint self :start-column (- right-of-screen n))))

(defun tbg-scroll-left (self n)
  (let* ((column (column self))
	 (row (row self))
	 (left-of-screen (max (- (left-of-screen self) n) 0))
	 (font (font self))
	 (columns (columns self))
	 (rows (rows self))
	 (fw (width font))
	 (fh (height font))
	 (w (* columns fw))
	 (h (* rows fh))
	 (res (res self))
	 (right-of-screen (+ left-of-screen columns -1)))	 
	(setf (left-of-screen self) left-of-screen)
	(when (> column right-of-screen)
	      (setf (column self) right-of-screen))
	(if (> n columns)
	    (let ((offset (* n fw)))
		 (xlib:copy-area res (gc-res self) 0 0 w h res offset 0)))
	(tbg-repaint self :end-column (+ left-of-screen n))))

(defun tbg-scroll-down (self n)
  "Scroll down n lines."
  (let* ((last-line (1- (rows (buffer self))))
	 (top-of-screen (top-of-screen self))
	 (new-top-of-screen (min last-line (+ n top-of-screen)))
	 (row (row self))
	 (column (column self))
	 (res (res self))
	 (font (font self))
	 (fh (font-height font))
	 (fw (font-width font))
	 (rows (rows self))
	 (columns (columns self))
	 (w (* columns fw))
	 (h (* rows fh))
	 (bottom-of-screen (+ new-top-of-screen rows -1))
	 (fast-scroll (fast-scroll self)))
	;; set the top-of-screen: if fast-scrolling, no propagations occur.
	(if fast-scroll
	    (setf (slot-value self 'top-of-screen) new-top-of-screen)
	    (setf (top-of-screen self) new-top-of-screen))
	(when (> new-top-of-screen row)
	      (setf (row self) new-top-of-screen))
	(setq n (- new-top-of-screen top-of-screen))
	(cond ((< n rows)
	       (let ((offset (* n fh))) ; offset is in pixels.
		    (xlib:copy-area res (gc-res self)
				    0 offset w (- h offset) res 0 0)
		    (tbg-repaint self :start-row (- bottom-of-screen n))))
	      (t
	       (tbg-repaint self)))))

(defun tbg-scroll-up (self n) 
  "scroll up n lines."
  (let* ((last-line (1- (rows (buffer self))))
	 (top-of-screen (top-of-screen self))
	 (new-top-of-screen (max 0 (- top-of-screen n)))
	 (row (row self))
	 (column (column self))
	 (res (res self))
	 (font (font self))
	 (fh (font-height font))
	 (fw (font-width font))
	 (rows (rows self))
	 (columns (columns self))
	 (w (* columns fw))
	 (h (* rows fh))
	 (bottom-of-screen (+ new-top-of-screen rows -1))
	 (fast-scroll (fast-scroll self)))
	;; set the top-of-screen: if fast-scrolling, no propagations occur.
	(if fast-scroll
	    (setf (slot-value self 'top-of-screen) new-top-of-screen)
	    (setf (top-of-screen self) new-top-of-screen))
	(when (> row bottom-of-screen)
	      (setf (row self) bottom-of-screen))
	(setq n (- top-of-screen new-top-of-screen))
	(cond ((< n rows)
	       (let ((offset (* n fh)))
		    (xlib:copy-area res (gc-res self) 
				    0 0 w (- h offset) res 0 offset)
		    (tbg-repaint self :end-row (+ top-of-screen n))))
	      (t
	       (tbg-repaint self)))))

;;
;; X-Cut-Buffer functions.
;;
(defun add-bytes (d str)
  (setf (xlib:cut-buffer d)
	(concatenate 'string (xlib:cut-buffer d) str)))


;;;
