;;;
;;; Copyright (c) 1990 Regents of the University of California
;;;
;;; $Author: bsmith $
;;; $Source: RCS/text-buffer-gadget-defs.cl,v $
;;; $Revision: 1.4 $
;;; $Date: 90/07/23 18:42:50 $
;;;

(in-package 'pt  :nicknames '(picasso-toolkit) :use '(lisp excl pcl))

;;;
;;; text-buffer-gadget
;;
;; This is a gadget specifically created for use by text-widget.
;; it consists of an instance of the buffer class, with methods
;; and functions to display it on the screen, with some fancy extras.
;; These extras are: insert and overwrite cursor, marking portions of
;; text, different fonts (only FIXED-WIDTH fonts supported), etc.
;;;

;; Description of slots.
;; buffer - an instance of the buffer class.
;; row, column - current position of the cursor in the buffer.  This is NOT
;;               the position of the cursor on the screen.  The position of
;;               the cursor on the screen is:
;;               (- left-of-screen column), (- top-of-screen row).
;; rows, columns - the size of the window in rows and columns of the window
;;                 through which the text-widget is displayed.
;; mark-row, mark-column - end of marked portion of text.
;;                         If no text is marked, mark-row and mark-column
;;                         contain nil.
;; top-of-screen - the first row of buffer that is displayed in the window
;;                 (i.e. the row that appears on the top of the window.)
;; left-of-screen - leftmost column of buffer displayed in the window.
;; cursor-mode - one of :overwrite or :insert.  The :overwrite cursor is a
;;               solid reversed block.  The :insert cursor is a vertical bar.
;; font - font in which the text-widget is displayed.
;; gray - if t, displays the text in gray. If nil, displays text normally.

(defclass text-buffer-gadget (gadget)
  ((name :initform "A Text-Buffer-Gadget")
   (event-mask :initform '(:exposure))
   (font :initform "8x13")
   (background :initform "white")
   (gc-spec :initform '(gc-res (:font "8x13")))
   (buffer
    :initarg :buffer 
    :initform nil 
    :type t 
    :accessor buffer)
   (row
    :initarg :row 
    :initform 0 
    :type integer 
    :accessor row)
   (column
    :initarg :column 
    :initform 0 
    :type integer 
    :accessor column)
   (mark-row
    :initarg :mark-row 
    :initform nil 
    :type integer 
    :accessor mark-row)
   (mark-column
    :initarg :mark-column 
    :initform nil 
    :type integer 
    :accessor mark-column)
   (rows
    :initarg :rows 
    :initform 24 
    :type integer 
    :accessor rows)
   (columns
    :initarg :columns 
    :initform 80 
    :type integer 
    :accessor columns)
   (top-of-screen
    :initarg :top-of-screen 
    :initform 0 
    :type integer 
    :accessor top-of-screen)
   (left-of-screen
    :initarg :left-of-screen 
    :initform 0 
    :type integer 
    :accessor left-of-screen)
   (cursor-mode
    :initarg :cursor-mode 
    :initform :overwrite 
    :type keyword 
    :accessor cursor-mode)
   (fast-scroll
    :initarg :fast-scroll 
    :initform nil 
    :type t 
    :accessor fast-scroll)
   (cursor-moved
    :initarg :cursor-moved 
    :initform nil 
    :type t 
    :accessor cursor-moved)
   (data
    :initarg :data 
    :initform nil 
    :type t 
    :accessor data)
   (top-pad
    :initarg :top-pad 
    :initform 0 
    :type integer 
    :accessor top-pad)
   (self-adjusting
    :initarg :self-adjusting 
    :initform nil 
    :type atom
    :accessor self-adjusting)
   (current
    :initarg :current 
    :initform t 
    :type atom 
    :accessor current)))


;;;
;;; text-buffer-gadget setf methods
;;;

(defmethod (setf value) (val (self text-buffer-gadget)) 
  (typecase val
	    (string 
	     (new self)
	     (put self val :repaint nil)
	     (setf (column self) 0))
	    (vector
	     (new self)
	     (setf (buf-data (buffer self)) val)
	     (setf (column self) 0))
	    (PCL::IWMC-CLASS
	     (warn "text.setf.value: Illegal value ~S" val))
	    (list
	     (new self)
	     (mapc #'(lambda (x) (put self x) (put self (string #\Newline)))
		   val)
	     (setf (row self) 0)
	     (setf (column self) 0))
	    (number
	     (new self)
	     (put self (princ-to-string val) :repaint nil))
	    (t
	     (warn "text.setf.value: Illegal value ~S" val))))

(defmethod (setf invert) (val (self text-buffer-gadget)) 
  (setf (slot-value self 'invert) val)
  (repaint self))

(defmethod (setf cursor-mode) (cursor-mode (self text-buffer-gadget))
  (let ((row (row self))
	(column (column self)))
       (setf (slot-value self 'cursor-mode) cursor-mode)
       (tbg-repaint self
		    :start-row row :end-row row
		    :start-column column :end-column column)))

(defmethod (setf font) (font (self text-buffer-gadget)) 
  "font changes affect the number of rows and columns."
  ;; change the slot value
  (call-next-method)
  (when (attached-p self)
	(let* ((fw (font-width font))
	       (fh (font-height font))
	       (width (width self))
	       (height (height self))
	       (rows (rows self))
	       (columns (columns self))
	       (new-rows (truncate height fh))
	       (new-columns (truncate width fw))
	       (data (data self)))
	      (setf (rows self) new-rows)
	      (setf (columns self) new-columns)
	      (setf (width-increment self) fw)
	      (setf (height-increment self) fh)
	      (repaint self))))

(defmethod (setf top-of-screen) (value (self text-buffer-gadget))
  (setf (slot-value self 'top-of-screen) (round value))
  (tbg-repaint self))

(defmethod (setf left-of-screen) (value (self text-buffer-gadget))
  (setf (slot-value self 'left-of-screen) (round value)))

;; Doing a setf-rows or setf-columns will attempt to resize the 
;; window to fit the number of rows or columns.  However, since the
;; the geometry manager does not guarantee a window resize, the setf methods
;; for rows and columns do not guarantee to actually do anything.
;; The best way to do things (and the only way guaranteed to work) is
;; to created the gadget with the desired number of rows and columns.

(defmethod (setf columns) (value (self text-buffer-gadget))
  "Initiate a request for changing the number of columns displayed on
  the screen by the gadget."
  ;; test for inactive window
  (setf value (round value))
  (unless (eq value (slot-value self 'columns))
	  (setf (slot-value self 'columns) value)
	  (if (self-adjusting self)
	      ;; change the width of the window
	      ;; by setting the resize hint a call to resize-window-handler will
	      ;; hopefully be generated.
	      (setf (base-width self) (* value (width (font self)))))))

(defmethod (setf rows) (value (self text-buffer-gadget))
  "Initiate a request for changing the number of rows displayed on 
  the screen by the gadget."
  ;; test for inactive window
  (setf value (round value))
  (unless (eq value (slot-value self 'rows))
	  (setf (slot-value self 'rows) value)
	  (if (self-adjusting self)
	      ;; change the height of the window
	      ;; The way to change the number of rows is to change the
	      ;; resize-hint of the window and let the resize-window-handler
	      ;; actually change the number of rows.
	      (setf (base-height self) (* value (font-height (font self)))))))

;; setting the row and/or column will also scroll the screen and redraw
;; automatically.
;; The cursor will never be left outside of the screen.
;; If the values given for the new row or new column
;; are out of bounds, the setf method will correct them to place them back
;; in bounds.

;; something to do: add a method to set cursor position.  This way we'll
;; be able to set both row and column with one call.

(defmethod (setf row) (n (self text-buffer-gadget)) 
  (setq n (round n))
  (let* ((top-of-screen (top-of-screen self))
	 (rows (rows self))
	 (old-row (row self))
	 (new-row (max 0 (min n (1- (rows (buffer self))))))
	 (buffer-columns (buf-columns (buffer self) new-row))
	 (old-column (column self))
	 (new-column (min old-column buffer-columns)))
	(unmark self)
	(setf (cursor-moved self) t)
	(when (not (= old-row new-row))
	      (setf (column self) new-column)
	      (setf (slot-value self 'row) new-row)
	      (cond ((< new-row top-of-screen)
		     (tbg-scroll-up self (- top-of-screen new-row)))
		    ((> new-row (+ top-of-screen rows -1))
		     (tbg-scroll-down self (- new-row (+ top-of-screen rows -1))))
		    (t
		     (tbg-repaint self 
				  :start-row old-row  :end-row old-row
				  :start-column new-column :end-column new-column)
		     (tbg-repaint self 
				  :start-row new-row  :end-row new-row
				  :start-column new-column :end-column new-column))))
	new-row))

(defmethod (setf column) (n (self text-buffer-gadget)) 
  (setq n (round n))
  (let* ((left-of-screen (left-of-screen self))
	 (row (row self))
	 (old-column (column self))
	 (columns (columns self))
	 (new-column (max 0 (min n (buf-columns (buffer self) row)))))
	(unmark self)
	(setf (cursor-moved self) t)
	(when (not (= old-column new-column))
	      (setf (slot-value self 'column) new-column)
	      (cond ((< new-column left-of-screen)
		     (when (and (>= old-column left-of-screen)
				(< old-column (+ left-of-screen columns)))
			   (tbg-repaint self 
					:start-row row :end-row row
					:start-column old-column
					:end-column old-column))
		     (tbg-scroll-left self (- left-of-screen new-column)))
		    ((> new-column (+ left-of-screen columns -1))
		     (when (and (>= old-column left-of-screen)
				(< old-column (+ left-of-screen columns)))
			   (tbg-repaint self 
					:start-row row :end-row row
					:start-column old-column
					:end-column old-column))
		     (tbg-scroll-right self
				       (- new-column (+ left-of-screen columns -1))))
		    (t
		     (tbg-repaint self 
				  :start-row row  :end-row row
				  :start-column (min new-column old-column)
				  :end-column (max new-column old-column)))))
	new-column))

;; 
;; marking means highlighting a portion of text to which we want to do
;; something special (like delete.)
;; The portion of text marked is in between the cursor position and
;; the mark position (determined by mark-row and mark-column).
;; The way to mark text is by calling the mark method.  The way to unmark
;; text is to call the unmark method.  The setf methods for mark-row and
;; mark-column do not work properly and are only still around because
;; some old code still uses them.  Eventually they may be eliminated.

(defmethod (setf mark-row) (n (self text-buffer-gadget))
  (setq n (round n))
  (let ((row (row self))
	(column (column self))
	(old-mark-row (mark-row self))
	(mark-column (mark-column self)))
       (cond ((or (null mark-column)
		  (and (null old-mark-row) (null n)))
	      ;; mark block already off -- just return nil.
	      (setf (slot-value self 'mark-row) n)
	      nil)
	     ((null n) ;but not (null old-mark-row)
	      ;; mark block was on; now turn it off.
	      (setf (slot-value self 'mark-row) n)
	      (tbg-repaint self :start-row row :end-row old-mark-row))
	     ;; the mark point must always be after the cursor.
	     ((or (> n row)
		  (and (= n row) (>= mark-column column)))
	      (setf (slot-value self 'mark-row) n)
	      (tbg-repaint self 
			   :start-row (if old-mark-row (min n old-mark-row) row)
			   :end-row (if old-mark-row (max n old-mark-row) n)))
	     ;; mark point is before cursor. move cursor to mark point
	     ;; and set mark-point to cursor.
	     ((or (< n row)
		  (and (= n row) (< mark-column column)))
	      ;; don't do intermediate repaints, just a big complete one
	      ;; at the end.
	      (setf (repaint-flag self) nil) 
	      (setf (row self) n)
	      (setf (column self) mark-column)
	      (setf (slot-value self 'mark-row) row)
	      (setf (mark-column self) column)
	      (setf (repaint-flag self) t)
	      (tbg-repaint self :start-row n :end-row old-mark-row)))))

(defmethod (setf mark-column) (n (self text-buffer-gadget))
  (setq n (round n))
  (let ((row (row self))
	(column (column self))
	(mark-row (mark-row self))
	(old-mark-column (mark-column self)))
       (cond ((or (null mark-row)
		  (and (null old-mark-column) (null n)))
	      ;; mark block already off -- just return nil
	      (setf (slot-value self 'mark-column) n)
	      nil)
	     ((null n) ; but not (null old-mark-column)
	      ;; in this case, block was on, so turn it off.
	      (setf (slot-value self 'mark-column) n)
	      (tbg-repaint self :start-row row :end-row mark-row))
	     ;; remember that method (setf mark-row)
	     ;; guarantees (>= mark-row row) always.
	     ((and (= mark-row row) (< n column))
	      ;; mark point is before cursor, so we switch
	      ;; the cursor position and mark point.
	      
	      (setf (repaint-flag self) nil) ; avoid extra repaints.
	      (setf (column self) n)
	      (setf (slot-value self 'mark-column) column)
	      ;; setf-column zaps mark-row and mark-column
	      (setf (mark-row self) mark-row) 
	      (setf (repaint-flag self) t)
	      (tbg-repaint self 
			   :start-row row :end-row row
			   :start-column n :end-column old-mark-column))
	     (t
	      ;; i.e. (or (> mark-row row)
			  ;;          (and (= mark-row row) (>= n column)))
	      ;; new mark point is after cursor; just move it.
	      (setf (slot-value self 'mark-column) n)
	      (tbg-repaint self :start-row mark-row :end-row mark-row)))))
