;;;
;;; Copyright (c) 1990 Regents of the University of California
;;; 
;;; $Author: bsmith $
;;; $Source: RCS/buffer.cl,v $
;;; $Revision: 1.4 $
;;; $Date: 90/07/30 10:10:19 $
;;;

(in-package 'pt  :nicknames '(picasso-toolkit) :use '(lisp excl pcl))

;;;
;;; buffer class
;;;

; buffers store text in a two-dimensional array.
; The size limit of the text is system dependent, and determined
; by array-total-size-limit.

;; DESCRIPTION OF SLOTS
;;
;; buf-data - holds the array where the text is actually stored.
;;

(defclass buffer ()
  ((buf-data
    :initarg :buf-data 
    :initform nil
    :type vector
    :accessor buf-data)
   (rows
    :initarg :rows
    :initform nil
    :type integer
    :accessor rows)))

(defun make-buffer (&rest keys)
  (apply #'make-instance 'buffer :allow-other-keys t keys))

(defmethod new-instance ((self buffer)
			 &key (ignore nil) &allow-other-keys)
  "Initial dimension is 1 (one line) because many 
  buffers will be used by entry-widgets and we don't want
  to waste too much space."
  (setf (buf-data self) (make-array '(1)
				    :initial-element ""
				    :element-type 'string
				    :adjustable t
				    :fill-pointer 1))
  (setf (rows self) 1))

(defmethod (setf value) (val (self buffer))
  (setf (buf-data self) val))

;; METHODS ON BUFFER

(defun detab (str &key (tab-stop 8) 
		  &aux
		  (tab-count (count #\Tab str))
		  (strlen (length str)))
  (if (zerop tab-count)
      str
      (let ((col 0)
	    (char #\Space)
	    (out-str (make-string (+ strlen (* (1- tab-stop) tab-count))
				  :initial-element #\Space)))
	   (dotimes (i strlen)
		    (setq char (schar str i))
		    (if (eq char #\Tab)
			(incf col (- tab-stop (mod col tab-stop)))
			(setf (schar out-str col) char
			      col (1+ col))))
	   out-str)))

(defmethod new ((self buffer))
  "Clear all the contents of self.  Instead of actually erasing 
  whatever was in (buf-data self), we just create a new array for
  buf-data and let the old one garbage-collect."
  (setf (buf-data self) (make-array '(1)
				    :initial-element ""
				    :element-type 'string
				    :adjustable t
				    :fill-pointer 1))
  (setf (rows self) 1))

(defmethod buf-columns ((self buffer) row)
  "number of columns in row.
  Columns are numbered from 0 to (1- (columns buffer))"
  (length (aref (buf-data self) row)))

(defmethod value ((self buffer) 
		  &key
		  (row 0)
		  (column 0))
  (tbg-get-value self :row row :column column))
  
(defun tbg-get-value (self &key (row 0) (column 0))
  "return the string that starts at row, column in the buffer.
  If no column is specified, start at column 0.  If no row is specified,
  use first row in buffer."
  
  (let ((buf-data (buf-data self)))
       (if (and (< row (length buf-data))
		(>= row 0))
	   (if (and (< column (length (aref buf-data row)))
		    (>= column 0))
	       (my-subseq-string (aref (buf-data self) row) column)
	       "")
	   "")))

(defmethod put-char ((self buffer) ch
		     &key
		     (row (1- (rows self)))
		     (column (buf-columns self row))
		     (overwrite nil overwritep))
  "Insert the character immediately before the cursor position.
  A return character will cause a break in the current line.
  Returns updated position of row and column."
  (let ((buf-data (buf-data self)))
       (declare (type vector buf-data))
       (when (and overwrite overwritep
		  (not (char= ch #\Return)))
	     ;; It's a kludge, but the easiest thing to do at the moment.
	     (delete-char self row column))
       (cond ((or (char= ch #\Return) (char= ch #\Newline))
	      ;; Create one more row and alter data.
	      (when (not (and overwrite overwritep))
		    (setf (buf-data self) (make-vector-hole buf-data row))
		    (setf (aref (buf-data self) row)
			  (adjust (my-subseq (aref (buf-data self) row)
					     0 column)))
		    (setf (aref (buf-data self) (1+ row))
			  (adjust (my-subseq (aref (buf-data self) (1+ row))
					     column)))
		    (setf (rows self) (length buf-data)))
	      ; alter the cursor position
	      (setf row (1+ row))
	      (setf column 0))
	     
	     (t 
	      ; just insert a character.
	      (setf (aref buf-data row)
		    (insert-string (aref buf-data row) (string ch) column))
	      ;alter the cursor position
	      (setf column (1+ column))
	      nil)))
  (values row column))

(defmethod put ((self buffer) s 
		&key
		(row (1- (rows self)))
		(column (buf-columns self row))
		(overwrite nil))
  "insert or overwrite string s at position row, column of self.
  There may be newlines in s.  Tabs are removed from s"
  (do ((l (position #\newline s) (position #\newline s)))
      ((or (null s) (null l)))
      (when (> l 0)
	    (put-func self (my-subseq s 0 l) row column :overwrite overwrite)
	    (setq column (+ column l)))
      (put-char self '#\Return :row row :column column :overwrite overwrite)
      (setq row (1+ row))
      (setq column 0)
      (setq s (my-subseq s (1+ l))))
  (when s 
	(put-func self s row column :overwrite overwrite)
	(setq column (+ column (length s))))
  (values row column))


;; the functions below are used by the put method.

(defun put-func (buffer s row column &key (overwrite nil))
  "string s may not contain newlines"
  (if overwrite
      (buffer-replace-string buffer s row column)
      (buffer-insert-string buffer s row column)))

(defun buffer-insert-string (buffer str row column)
  "Insert the given string s to buffer at row, col. 
  The string s must not contain a carriage return."
  (setf (aref (buf-data buffer) row)
	(insert-string (value buffer :row row) str column))
  buffer)

(defun buffer-replace-string (buffer str row column)
  "Overwrite the given string s at buffer position row, column. 
  The string s must not contain a carriage return."
  (setf (aref (buf-data buffer) row)
	(replace-string (value buffer :row row) str column))
  buffer)

(defmethod delete-char ((self buffer) row column
			&key
			(count 1))
  "Delete the character immediately ahead (or on) the cursor position.
  Deletion at the end of a line causes concatenation with the
  next line.  When :count is greater than the number of characters
  in the line, delete to the end of line. 
  Returns updated row and column."
  (let* ((curr-line (value self :row row)))
	(cond ((< column (length curr-line))
	       ; the "normal" case
	       (setf (aref (buf-data self) row)
		     (adjust (concatenate 'string
					  (my-subseq curr-line 0 column)
					  (my-subseq curr-line
						     (+ column count))))))
	      ((< row (1- (length (buf-data self))))
	       ; end of line case
	       (setf (aref (buf-data self) row)
		     (concatenate 'string
				  curr-line
				  (aref (buf-data self) (1+ row))))
	       (vector-squash (buf-data self) (1+ row))
	       (setf (rows self) (length (buf-data self))))))
  (values row column))

(defmethod delete-backward ((self buffer) row column)
  "Delete the character immediately before the cursor position.
  Deleting backward at the beginning of a line causes a join
  of the two lines involved.  Return updated row, column."
  (let ((buf-data (buf-data self))
	(temp 0))
       (declare (type integer temp)
		(type vector buf-data))
       
       (cond ((and (= column 0) (> row 0))
	      ; the normal beginning-of-line case
	      (setq temp (buf-columns self (1- row)))
	      (setf (aref buf-data (1- row))
		    (adjust (concatenate 'string
					 (aref buf-data (1- row))
					 (aref buf-data row))))
	      (setf (buf-data self) (vector-squash buf-data row))
	      
	      ; now set the cursor position right
	      (setf row (1- row))
	      (setf column temp)
	      (setf (rows self) (length buf-data)))
	     
	     ((and (= column 0) (= row 0))
	      ; beginning of buffer (do nothing)
	      nil)
	     
	     (t ; the "normal" case
		(setf (aref buf-data row)
		      (adjust (concatenate 
			       'string
			       (my-subseq (aref buf-data row) 0 (1- column))
			       (my-subseq (aref buf-data row) column))))
		; set the cursor position
		(setf column (1- column)))))
  (values row column))

(defmethod kill-line ((self buffer) row column)
  "Delete (kill) the line after the cursor position.
  Return updated row, col"
  (let* ((buf-data (buf-data self))
	 (curr-line (value self :row row)))
	(declare (type vector buf-data)
		 (type string curr-line))
	(cond ((< column (buf-columns self row))
	       ; the "normal" case
	       (setf (aref (buf-data self) row)
		     (adjust (my-subseq curr-line 0 column))))
	      ((< row (1- (rows self)))
	       ; end of line, but not last line.
	       (setf (aref (buf-data self) (1+ row))
		     (concatenate 'string
				  (aref buf-data row)
				  (aref buf-data (1+ row))))
	       (setf (buf-data self) (vector-squash buf-data row))
	       (setf (rows self) (length buf-data)))))
  (values row column))

(defmethod load-file ((self buffer) filename
		      &key 
		      (count -1))
  "Load the first count lines of text data into self.  All previous
  contents of self are lost.  Returns 0, 0 for row, column."
  (new self)
  (put-file self filename :count count :row 0 :column 0)
  (values 0 0))

(defmethod put-file ((self buffer) filename
		     &key
		     (count -1)
		     (row (1- (rows self)))
		     (column (buf-columns self row)))
  "Read the first count lines of text data from a file and insert it into
  the buffer at row, column.  If count=-1, then all lines of the file are
  read in (this is the default).  If row and column are not specified, 
  they default to the end of buffer.  Returns updated row and column.
  Returns nil if the file does not exist."
  (let ((input-stream (open filename :direction :input
			    :if-does-not-exist nil))
	(buf-data (buf-data self))
	(temp-buf-data (make-array '(20) :element-type 'string
				   :initial-element ""
				   :adjustable t
				   :fill-pointer 0))
	(len 0)
	(temp-len 0))
       (when (and input-stream (> (length filename) 0)) ;; else Crash!
	     (do ((curr-line (read-line input-stream nil nil)
			     (read-line input-stream nil nil))
		  (i 1 (1+ i)))
		 ((or (and (/= count -1) (> i count)) (null curr-line)))
		 (vector-push-extend (detab curr-line) temp-buf-data))
	     (close input-stream))
       ;; Copy the data in place...
       (setq temp-len (length temp-buf-data))
       (setq len (length buf-data))
       (unless (zerop temp-len)
	       (adjust-array buf-data (+ temp-len len) :fill-pointer t)
	       (do* ((src (1- len) (1- src))
		     (dest (+ src temp-len) (1- dest)))
		    ((< src row))
		    (setf (aref buf-data dest) (aref buf-data src)))
	       (dotimes (i temp-len)
			(setf (aref buf-data (+ i row))
			      (aref temp-buf-data i)))
	       (setf (rows self) (length (buf-data self)))))
  (values row column))

(defmethod append-to-file ((self buffer) filename)
  "Take the data from a buffer and append it to a file."
  (let* ((buf-data (buf-data  self))
	 (length (length buf-data))
	 (output-stream (open filename 
			      :direction :output
			      :if-exists :append
			      :if-does-not-exist :create)))
	(terpri output-stream)
	(dotimes (i length)
		 (princ (aref buf-data i) output-stream)
		 (terpri output-stream))
	(close output-stream)))

(defmethod save-file ((self buffer) filename)
  "Take the data from a buffer and save it to a file."
  (let* ((buf-data (buf-data  self))
	 (length (length buf-data))
	 (output-stream (open filename :direction :output
			      :if-exists :supersede
			      :if-does-not-exist :create)))
	(dotimes (i length)
		 (princ (aref buf-data i) output-stream)
		 (terpri output-stream))
	(close output-stream)))


;;;
;;; useful bunch of functions for buffer.
;;;

(defun my-subseq (v start &optional (end nil end-supplied))
  "Return the subsequence of a vector from start to (end-1).
  The default for end is the length of the vector.  Handles
  boundary cases by returning nil (e.g. (my-subseq v 10 0) ==> nil )."
  
  (if (not end-supplied)
      (setq end (length v)))
  
  ; check for nil cases
  (if (or (>= start end)
	  (< start 0)
	  (> end (length v)))
      nil
      (subseq v start end)))

(defun my-subseq-string (s start &optional (end nil end-supplied))
  "Return the string from start to (end-1) of the string s.
  If start<0, then we start from 0.  Similarly, if end>(length s), then
  we end at (1- (length s)).  Otherwise, return the null string."
  (if (or (not end-supplied)
	  (> end (length s)))
      (setq end (length s)))
  (setq start (max 0 start))
  (if (>= start end)
      ""
      (subseq s start end)))

(defun adjust (s)
  "Return s if s is non-nil; return an empty string if s is nil.
  To be primarily used for string operations."
  (cond (s)
	(t "")))

(defun make-vector-hole (v i)
  "Make a vector one element longer,
  shift elements from (i+1) on one element to the right,
  and make element i equal the (i+1)th element."
  ;; NOTE:  this function is undefined for vectors whose length
  ;;        is equal to zero.
  (let ((last-line (vector-push-extend "" v)))
       (do ((n last-line (1- n)))
	   ((<= n i) v)
	   (setf (aref v n) (aref v (1- n))))))

(defun vector-squash (v i)
  "Delete the ith element of v and shift the end elements
  one to the left."
  (let ((last-line (1- (length v))))
       (do ((n i (1+ n)))
	   ((>= n last-line))
	   (setf (aref v n) (aref v (1+ n))))
       (vector-pop v))
  v)

(defun insert-string (s1 s2 i)
  "Insert s2 at the ith spot of s1.  s1 and s2 must
  be strings and i is an integer."
  (concatenate 'string
	       (my-subseq s1 0 i)
	       s2
	       (my-subseq s1 i)))

(defun replace-string (s1 s2 i)
  "Overwrite s1 with s2 starting at i."
  (concatenate 'string
	       (my-subseq s1 0 i)
	       s2
	       (my-subseq s1 (+ i (length s2)))))

(defun make-fail-array (pattern)
  "Construct the fail array for the Knuth-Morris-Pratt string
  searching algorithm."
  (let* ((pattern-end (1- (length pattern)))   ; last position in pattern
	 (fail (make-array `(,(1+ pattern-end)))))
	; set up the fail array
	(setf (aref fail 0) -1)
	(do* ((k 1 (1+ k))
	      (r (aref fail (1- k)) (aref fail (1- k))))
	     ((> k pattern-end))  ; get out when we have filled the array
	     
	     (do ()
		 ((or (<= r -1)
		      (char= (aref pattern r)
			     (aref pattern (1- k))))
		  (setf (aref fail k) (1+ r)))
		 (setq r (aref fail r))))
	fail))

(defun search-string (text-string pattern fail)
  "Given a string and a pattern to search for, return the position
  of the first character of the matched pattern.  Search starts at
  the beginning of text-string and uses the Knuth-Morris-Pratt
  algorithm.  nil is returned if the search fails."
  (let* ((text-end (1- (length text-string)))  ; last position in text-string
	 (pattern-end (1- (length pattern)))   ; last position in pattern
	 )
	
	; now look for the pattern
	(do ((j 0)
	     (k 0))
	    ((or (> j text-end)
		 (> k pattern-end))
	     ; return either the position (if match) or nil (no match)
	     (if (> k pattern-end)
		 (- j pattern-end 1)
		 nil))
	    
	    ; basically an if-then-else
	    (cond ((or (= k -1)
		       (char= (aref text-string j)
			      (aref pattern k)))
		   (setq j (1+ j))
		   (setq k (1+ k)))
		  (t
		   (setq k (aref fail k)))) )))

(defun search-string-backward (text-string pattern)
  "brute force search algorithm.  Returns position of first character
  in matched pattern, or nil if search fails."
  (let ((pattern-length (length pattern)))
       (do ((i (1- (length text-string)))
	    (j (1- pattern-length)))
	   ((or (< i 0) (< j 0)) ;end clause
	    (if (< j 0) ;was pattern found?
		(1+ i)  ;yes, return its location.
		;; no, return nil
		nil))
	   (cond ((char= (aref text-string i) (aref pattern j))
		  ;; letters match, keep looking
		  (setq i (1- i))
		  (setq j (1- j)))
		 (t ;; letters don't match.
		    (setq i (+ (- i j 2) pattern-length))
		    (setq j (1- pattern-length)))))))

(defun buffer-to-string (buf-data)
  "Convert an array of strings into one long string,
  compatible with the POSTGRES type text."
  (let ((last-line (1- (length buf-data)))
	(working-string ""))
       (do ((curr-line 0 (1+ curr-line)))
	   ((> curr-line last-line))
	   (setq working-string (concatenate 'string
					     working-string
					     (aref buf-data curr-line)
					     (string '#\Return))))
       working-string))


(defun string-to-buffer (text-string)
  "Convert to a string [a text string in POSTGRES] to
  an array of strings [the buf-data part of a buffer]."
  (let ((text-end (1- (length text-string))))
       (do* ((text-row 0 (1+ text-col)) ; marks the beginning of a line in text-string
	     (text-col 0 text-row)      ; marks the overall current position
	     (last-line 0 (1+ last-line)) ; current line number in buffer
	     (buffer (make-array '(0)
				 :element-type 'string
				 :adjustable t
				 :fill-pointer 0)))
	    ((> text-row text-end) buffer)
	    
	    (do ()
		((or (> text-col text-end)
		     (char= (aref text-string text-col) '#\Return)))
		(setq text-col (1+ text-col)))
	    
	    (vector-push-extend (my-subseq text-string
					   text-row text-col)
				buffer))))


(defun next-non-alphanumeric-pos (str n)
  "Returns the position of the first non-alphanumeric
  character after position n, or else if not found returns position past last
  character."
  (let ((last-char (1- (length str))))
       (if (> n last-char) (1+ last-char)
	   (do* ((count n (1+ count))
		 (ch (aref str count) (aref str count)))
		((or (not (alphanumericp ch)) (>= count last-char))
		 (if (not (alphanumericp ch)) count (1+ last-char)))))))

(defun next-alphanumeric-pos (str n)
  "Returns the position of the first alphanumeric
  character after position n, or else if not found returns position after
  last character."
  (let ((last-char (1- (length str))))
       (if (> n last-char) (1+ last-char)
	   (do* ((count n (1+ count))
		 (ch (aref str count) (aref str count)))
		((or (alphanumericp ch) (>= count last-char))
		 (if (alphanumericp ch) count (1+ last-char)))))))

(defun previous-non-alphanumeric-pos (str n)
  (let ((last-char (1- (length str))))
       (cond ((<= n 0) 0)
	     ((> n last-char) (1+ last-char))
	     (t (do* ((count (min last-char n) (1- count))
		      (ch (aref str count) (aref str count)))
		     ((or (not (alphanumericp ch)) (<= count 0))
		      (if (not (alphanumericp ch)) count 0)))))))

(defun previous-alphanumeric-pos (str n)
  (let ((last-char (1- (length str))))
       (cond ((<= n 0) 0)
	     (t (do* ((count (min last-char n) (1- count))
		      (ch (aref str count) (aref str count)))
		     ((or (alphanumericp ch) (<= count 0))
		      (if (alphanumericp ch) count 0)))))))

