;;;
;;; Copyright (c) 1990 Regents of the University of California
;;; 
;;; $Author seitz $
;;; $Source $
;;; $Revision $
;;; $Date $
;;;

(in-package 'pt  :nicknames '(picasso-toolkit) :use '(lisp excl pcl))

;;;
;;;___________________________________________________________________________
;;;
;;;				Macros
;;;___________________________________________________________________________
;;;

(defmacro mf-scroll-up (mf n)
  `(funcall (up-func ,mf) ,mf ,n))

(defmacro mf-scroll-down (mf n)
  `(funcall (down-func ,mf) ,mf ,n))

(defmacro mf-scroll-left (mf n)
  `(funcall (left-func ,mf) ,mf ,n))

(defmacro mf-scroll-right (mf n)
  `(funcall (right-func ,mf) ,mf ,n))

;;;
;;;	Macro to determine the object which REALLY holds the value for obj
;;;

(defmacro det-value-holder (obj)
  `(cond ((typep ,obj 'text-widget)
	  (buffer ,obj))
	 ((typep ,obj 'button)
	  (slot-value ,obj 'value))
	 ((and (typep ,obj 'collection-widget) (window-p (value ,obj)))
	  (value ,obj))
	 (t ,obj)))

;;;	
;;;	Function to fill an incomplete list data specification with nils
;;;

(defun fill-2d-list (l rows cols &aux len newl)
  (setq newl (mapcar #'(lambda (x &aux len) (setq len (length x))
			       (cond ((= len cols) x)
				     ((< len cols) 
				      (append x (make-list (- cols len))))
				     (t
				      (butlast x (- len cols)))))
		     l))
  (setq len (length l))
  (cond ((< len rows)
	 (append newl
		 (make-list (- rows len)
			    :initial-element (make-list cols))))
	((> len rows)
	 (butlast newl (- len rows)))
	(t newl)))

;;;	
;;;	Functions/Macros to sort fields by row/col
;;;

(defmacro un-current-fields-by-row (mf)
  `(let ((all-ch (mapcar #'geom-spec (children ,mf))) 
	 (cur-ch (slot-value ,mf 'current-indices)))
	;;	Return list of all fields not current
	(sort (set-difference all-ch cur-ch :test #'pair-comp) #'car-comp)))

(defmacro un-current-fields-by-col (mf)
  `(let ((all-ch (mapcar #'geom-spec (children ,mf))) 
	 (cur-ch (slot-value ,mf 'current-indices)))
	;;	Return list of all fields not current
	(sort (set-difference all-ch cur-ch :test #'pair-comp) #'cadr-comp)))

(defun un-current-elements (mf)
  (let* ((rows (data-rows mf))
	 (cols (data-cols mf))
	 (l nil)
	 (all-ch (dotimes (r rows l)
			  (dotimes (c cols)
				   (setq l (cons (list r c) l)))))
	 (cur-ch (slot-value mf 'current-indices)))
	(set-difference all-ch cur-ch :test #'pair-comp)))

(defun all-fields-by-row (mf)
  (let ((r (data-rows mf))
	(c (data-cols mf))
	(l nil))
       (dotimes (i r)
		(dotimes (j c)
			 (setq l (append l (list (list i j))))))
       l))

(defun all-fields-by-col (mf)
  (let ((r (data-rows mf))
	(c (data-cols mf))
	(l nil))
       (dotimes (i c)
		(dotimes (j r)
			 (setq l (append l (list (list j i))))))
       l))

(defun enumerate-row (mf r &aux l)
  (dotimes (i (data-cols mf))
	   (setq l (nconc l (list (list r i)))))
  l)

(defun enumerate-col (mf c &aux l)
  (dotimes (i (data-rows mf))
	   (setq l (nconc l (list (list i c)))))
  l)

(defun current-fields-by-row (mf)
  (sort (slot-value mf 'current-indices) #'car-comp))

(defun current-fields-by-col (mf)
  (sort (slot-value mf 'current-indices) #'cadr-comp))

(defun atom-car-comp (x y)
  (= x (car y)))

(defun atom-cadr-comp (x y)
  (= x (cadr y)))

(defun car-comp (x y) 
  (< (car x) (car y)))

(defun cadr-comp (x y) 
  (< (cadr x) (cadr y)))

(defun pair-comp (x y)
  (and (= (car x) (car y)) (= (cadr x) (cadr y))))

;;;
;;;	Function to copy arrays
;;;

(defun copy-2d-array (arr &aux rows cols cop)
  (setq cop (make-array (list (setq rows (array-dimension arr 0))
			      (setq cols (array-dimension arr 1)))))
  (dotimes (r rows)
	   (dotimes (c cols)
		    (setf (aref cop r c) (aref arr r c))))
  cop)

(defun copy-1d-array (arr &aux rows cop)
  (setq cop (make-array (list (setq rows (array-dimension arr 0)))))
  (dotimes (r rows)
	   (setf (aref cop r) (aref arr r)))
  cop)

;;;
;;;	Function to quickly set data
;;;

(defun mf-quick-set-data (self mf)
  (if (typep (data mf) 'pgclos::portal)
      (setf (slot-value (slot-value mf 'data) 'array) self)
      (setf (slot-value mf 'data) self))
  (setf (slot-value mf 'changed-indices) nil)
  (mf-propagate mf))

;;;
;;;	Functions to make a data row or column current
;;;

(defmethod make-row-current ((self matrix-field) row &aux l)
  (setq l nil)
  (dotimes (i (cols self))
	   (setq l (nconc l (list (list row i)))))
  (setf (current-indices self) (nunion (slot-value self 'current-indices) l
				       :test #'pair-comp)))

(defmethod make-row-uncurrent ((self matrix-field) row)
  (setf (current-indices self)
	(delete row (current-indices self) :test #'atom-car-comp)))

(defmethod make-col-current ((self matrix-field) col &aux l)
  (setq l nil)
  (dotimes (i (rows self))
	   (setq l (nconc l (list (list i col)))))
  (setf (current-indices self) (nunion (slot-value self 'current-indices) l
				     :test #'pair-comp)))

(defmethod make-col-uncurrent ((self matrix-field) col)
  (setf (current-indices self)
	(delete col (current-indices self) :test #'atom-cadr-comp)))

;;;
;;;	Determines if a widget is selectable
;;;

(defun mf-selectable (wid &aux em)
  (setq em (event-mask wid))
  (or (consp (first em)) (consp (second em)) (consp (third em))))

;;;
;;;___________________________________________________________________________
;;;
;;;				Table-field stuff
;;;___________________________________________________________________________
;;;

;;;
;;;	Functions to swap two rows/cols in a matrix (& title).  Depending on
;;;	if the matrix has uniform or variable rows/cols, a certain function
;;;	is needed.  
;;;

;;	Swaps uniform-height rows
(defun ta-uni-swap-rows (self row1 row2 &aux mf rt) 
  (mf-uni-swap-rows (setq mf (matrix-field self)) row1 row2)
  (when (setq rt (slot-value mf 'row-titles)) 
	(mf-uni-swap-rows rt row1 row2)))

;;	Swaps uniform-width cols
(defun ta-uni-swap-cols (self col1 col2 &aux mf ct)
  (mf-uni-swap-cols (setq mf (matrix-field self)) col1 col2)
  (when (setq ct (slot-value mf 'col-titles))
	      (mf-uni-swap-cols ct col1 col2)))

;;	Swaps variable-height rows
(defun ta-var-swap-rows (self row1 row2 &aux mf rt) 
  (gm-matrix-row-exchange (setq mf (matrix-field self)) row1 row2)
  (mf-uni-swap-rows mf row1 row2)
  (when (setq rt (slot-value mf 'row-titles)) 
	(gm-matrix-row-exchange rt row1 row2)
	(mf-uni-swap-rows rt row1 row2)))

;;	Swaps variable-width cols
(defun ta-var-swap-cols (self col1 col2 &aux mf ct)
  (gm-matrix-col-exchange (setq mf (matrix-field self)) col1 col2)
  (mf-uni-swap-cols mf col1 col2)
  (when (setq ct (slot-value mf 'col-titles))
	      (gm-matrix-col-exchange ct col1 col2)
	      (mf-uni-swap-cols ct col1 col2)))

;;;
;;;	Functions to push all rows/cols to the far top/left
;;;

(defun ta-scoot-rows (wid &rest args &aux mf ti ci ui rows cols oldarr oldti
			  newarr newti nr darows dacols oldda oldtida newda
			  newtida ticurs dai ndar uni-rows ta r curs rindex)
  (declare (ignore args))
  (setq mf (data wid))
  (setq ta (parent mf)
	ti (slot-value mf 'row-titles)
	rows (rows mf)
	cols (cols mf)
	rindex (row-index mf)
	darows (data-rows mf)
	dacols (data-cols mf))
  (setq ci (slot-value ti 'current-indices)
	oldti (field-table ti)
	oldtida (data-array ti)
	newti (make-array (list rows 1))
	newtida (make-array (list darows 1))
	ticurs nil
	ui (un-current-fields-by-row ti)
	dai (un-current-elements ti))
  (setq oldarr (field-table mf)
	curs (slot-value mf 'current-indices)
	oldda (data-array mf)
	newarr (make-array (list rows cols))
	newda (make-array (list darows dacols))
	uni-rows (uniform-rows mf)
	nr 0)
  
  ;;	Create new arrays with current rows first
  (if uni-rows
      (progn
       (dolist (cur ci)
	       (setq r (car cur))
	       ;;	data...
	       (dotimes (c dacols)
			(setf (aref newda nr c) (aref oldda r c))) 
	       (mapc #'(lambda (x)
			       (if (= (car x) r) 
				   (rplaca x nr)))
		     curs)
	       (setf (aref newtida nr 0) (aref oldtida r 0))
	       (setq ticurs (cons (list nr 0) ticurs))
	       (setq nr (1+ nr)))
       (setq ndar nr))
      (progn
       (dolist (cur ci)
	       (setq r (- (car cur) rindex))
	       ;;	matrix...
	       (dotimes (c cols)
			(setf (geom-spec 
			       (setf (aref newarr nr c) (aref oldarr r c)))
			      (list nr c)))
	       (mapc #'(lambda (x)
			       (if (= (car x) r) 
				   (rplaca x nr)))
		     curs)
	       ;; 	titles...
	       (setf (geom-spec
		      (setf (aref newti nr 0) (aref oldti r 0)))
		     (list nr 0)) 
	       (setq ticurs (cons (list nr 0) ticurs))
	       
	       ;;	data...
	       (setq r (+ r rindex))
	       (dotimes (c dacols)
			(setf (aref newda nr c) (aref oldda r c)))
	       (setf (aref newtida nr 0) (aref oldtida r 0))
	       (setq nr (1+ nr)))
       (setq ndar nr)
       
       ;;	Put in rest of rows
       (dolist (cur ui)
	       (setq r (car cur))
	       (dotimes (c cols) 
			(setf (geom-spec
			       (setf (aref newarr nr c) (aref oldarr r c)))
			      (list nr c)))
	       (setf (geom-spec
		      (setf (aref newti nr 0) (aref oldti r 0)))
		     (list nr 0))
	       (setq nr (1+ nr)))
       
       ;;	Notify geometry-manager of changes
       (gm-matrix-scoot-rows mf (mapcar #'car ci) (mapcar #'car ui))
       (gm-matrix-scoot-rows ti (mapcar #'car ci) (mapcar #'car ui))))
  
  ;;	Replace old arrays with new ones
  (setf (field-table mf) newarr
	(field-table ti) newti)
  
  ;;	Put in rest of data
  (dolist (cur dai)
	  (setq r (car cur))
	  (dotimes (c cols)
		   (setf (aref newda ndar c) (aref oldda r c)))
	  (setf (aref newtida ndar 0) (aref oldtida r 0))
	  (setq ndar (1+ ndar)))
  
  (setf (row-index mf) 0
	(row-index ti) 0
	(slider-location (vert-scroll-bar ta)) 0)
  
  ;;	Replace old data with new
  (mf-quick-set-data newda mf)
  (mf-quick-set-data newtida ti)
  (setf (current-indices mf) curs)
  (setf (current-indices ti) ticurs)
  
  (if uni-rows
      (progn
       (mf-repaint mf)
       (mf-repaint ti))
      (progn
       (force-repack mf)
       (force-repack ti))))

(defun ta-scoot-cols (wid &rest args &aux mf ti ci ui rows cols oldarr oldti
			  newarr newti nc darows dacols oldda oldtida newda
			  newtida ticurs dai ndac uni-cols ta c curs cindex)
  (declare (ignore args))
  (setq mf (data wid))
  (setq ta (parent mf)
	ti (slot-value mf 'col-titles)
	rows (rows mf)
	cols (cols mf)
	cindex (col-index mf)
	darows (data-rows mf)
	dacols (data-cols mf))
  (setq ci (slot-value ti 'current-indices)
	ui (un-current-fields-by-col ti)
	curs (slot-value mf 'current-indices)
	dai (un-current-elements ti))
  (setq oldarr (field-table mf)
	oldti (field-table ti)
	oldda (data-array mf)
	oldtida (data-array ti)
	newarr (make-array (list rows cols))
	newti (make-array (list 1 cols))
	newda (make-array (list darows dacols))
	newtida (make-array (list 1 dacols))
	ticurs nil
	uni-cols (uniform-cols mf)
	nc 0)
  
  ;;	Create new arrays with current cols first
  (if uni-cols
      (progn
       (dolist (cur ci)
	       (setq c (cadr cur))
	       ;;	data...
	       (dotimes (r darows)
			(setf (aref newda r nc) (aref oldda r c))) 
	       (mapc #'(lambda (x)
			       (if (= (cadr x) c) 
				   (setf (cadr x) nc)))
		     curs)
	       (setf (aref newtida 0 nc) (aref oldtida 0 c)) 
	       (setq ticurs (cons (list 0 nc) ticurs))
	       (setq nc (1+ nc)))
       (setq ndac nc))
      (progn
       (dolist (cur ci)
	       (setq c (- (cadr cur) cindex))
	       ;;	matrix...
	       (dotimes (r rows)
			(setf (geom-spec 
			       (setf (aref newarr r nc) (aref oldarr r c)))
			      (list r nc))) 
	       (mapc #'(lambda (x)
			       (if (= (cadr x) c) 
				   (setf (cadr x) nc)))
		     curs)
	       ;;	titles...
	       (setf (geom-spec
		      (setf (aref newti 0 nc) (aref oldti 0 c)))
		     (list 0 nc)) 
	       (setq ticurs (cons (list 0 nc) ticurs))
	       ;;	data...
	       (setq c (+ c cindex))
	       (dotimes (r darows)
			(setf (aref newda r nc) (aref oldda r c)))
	       (setf (aref newtida 0 nc) (aref oldtida 0 c))
	       (setq nc (1+ nc)))
       (setq ndac nc)

       ;;	Put in rest of cols
       (dolist (cur ui)
	       (setq c (cadr cur))
	       (dotimes (r rows) 
			(setf (geom-spec
			       (setf (aref newarr r nc) (aref oldarr r c)))
			      (list r nc)))
	       (setf (geom-spec
		      (setf (aref newti 0 nc) (aref oldti 0 c)))
		     (list 0 nc))
	       (setq nc (1+ nc)))
       
       ;;	Replace old matrices with new ones
       (setf (field-table mf) newarr
	     (field-table ti) newti)
       
       ;;	Notify geometry-manager of changes
       (gm-matrix-scoot-cols mf (mapcar #'cadr ci) (mapcar #'cadr ui))
       (gm-matrix-scoot-cols ti (mapcar #'cadr ci) (mapcar #'cadr ui))))

  ;;	Put in rest of data
  (dolist (cur dai)
	  (setq c (cadr cur))
	  (dotimes (r rows)
		   (setf (aref newda r ndac) (aref oldda r c)))
	  (setf (aref newtida 0 ndac) (aref oldtida 0 c))
	  (setq ndac (1+ ndac)))
  
  (setf (col-index mf) 0
	(col-index ti) 0
	(slider-location (horiz-scroll-bar ta)) 0)

  ;;	Replace old data with new
  (mf-quick-set-data newda mf)
  (mf-quick-set-data newtida ti)

  ;;	Repack/paint
  (if uni-cols
      (progn
       (mf-repaint mf)
       (mf-repaint ti))
      (progn
       (force-repack mf)
       (force-repack ti)))
  
  (setf (current-indices mf) curs
	(current-indices ti) ticurs))

;;;
;;;	Scrolling functions
;;;

(defun tf-up-row (ignore tf ev)
  (declare (ignore ignore ev))
  (let ((mf (matrix-field tf))
	(sb (vert-scroll-bar tf)))
       (mf-scroll-up mf 1)
       (setf (slider-location sb) (visible-row-index mf)
	     (slider-size sb) (visible-rows mf))))

(defun tf-down-row (ignore tf ev)
  (declare (ignore ignore ev))
  (let ((mf (matrix-field tf))
	(sb (vert-scroll-bar tf)))
       (mf-scroll-down mf 1)
       (setf (slider-location sb) (visible-row-index mf)
	     (slider-size sb) (visible-rows mf))))

(defun tf-down-page (ignore tf ev)
  (declare (ignore ignore ev))
  (let ((mf (matrix-field tf))
	(sb (vert-scroll-bar tf)))
       (mf-scroll-down mf (visible-rows mf))
       (setf (slider-location sb) (visible-row-index mf)
	     (slider-size sb) (visible-rows mf))))

(defun tf-up-page (ignore tf ev)
  (declare (ignore ignore ev))
  (let ((mf (matrix-field tf))
	(sb (vert-scroll-bar tf)))
       (mf-scroll-up mf (visible-rows mf))
       (setf (slider-location sb) (visible-row-index mf)
	     (slider-size sb) (visible-rows mf))))

(defun tf-move-row (sb tf ev)
  (drag-scroll-bar sb 
	#'(lambda (sb tf) 
		  (let* ((mf (matrix-field tf))
			 (old-pos (visible-row-index mf))
			 (new-pos (round (slider-location sb))))
			(if (> old-pos new-pos)
			    (mf-scroll-up mf (- old-pos new-pos))
			    (mf-scroll-down mf (- new-pos old-pos)))))
	  ev))

(defun tf-left-col (ignore tf ev)
  (declare (ignore ignore ev))
  (let ((mf (matrix-field tf))
	(sb (horiz-scroll-bar tf)))
       (mf-scroll-left mf 1)
       (setf (slider-location sb) (visible-col-index mf)
	     (slider-size sb) (visible-cols mf))))

(defun tf-right-col (ignore tf ev)
  (declare (ignore ignore ev))
  (let ((mf (matrix-field tf))
	(sb (horiz-scroll-bar tf)))
       (mf-scroll-right mf 1)
       (setf (slider-location sb) (visible-col-index mf)
	     (slider-size sb) (visible-cols mf))))

(defun tf-right-page (ignore tf ev)
  (declare (ignore ignore ev))
  (let ((mf (matrix-field tf))
	(sb (horiz-scroll-bar tf)))
       (mf-scroll-right mf (visible-cols mf))
       (setf (slider-location sb) (visible-col-index mf)
	     (slider-size sb) (visible-cols mf))))

(defun tf-left-page (ignore tf ev)
  (declare (ignore ignore ev))
  (let ((mf (matrix-field tf))
	(sb (horiz-scroll-bar tf)))
       (mf-scroll-left mf (visible-cols mf))
       (setf (slider-location sb) (visible-col-index mf)
	     (slider-size sb) (visible-cols mf))))

(defun tf-move-col (sb tf ev)
  (drag-scroll-bar sb 
	#'(lambda (sb tf) 
		  (let* ((mf (matrix-field tf))
			 (old-pos (visible-col-index mf))
			 (new-pos (round (slider-location sb))))
			(if (> old-pos new-pos)
			    (mf-scroll-left mf (- old-pos new-pos))
			    (mf-scroll-right mf (- new-pos old-pos)))))
	  ev))

;;;
;;;	Revert
;;;
(defun ta-revert (wid &rest args &aux mf)
  (declare (ignore args))
  (when (setq mf (data wid))
	(mf-revert mf)))

;;;
;;;	Set all fields to un-current
;;;

(defun ta-rows-uncurrent (wid &rest args &aux rt)
  (declare (ignore args))
  (when (setq rt (slot-value (data wid) 'row-titles))
	(setf (current-indices rt) nil)))

(defun ta-cols-uncurrent (wid &rest args &aux ct)
  (declare (ignore args))
  (when (setq ct (slot-value (data wid) 'col-titles))
	(setf (current-indices ct) nil)))

(defun ta-matrix-uncurrent (wid &rest args &aux mf)
  (declare (ignore args))
  (setq mf (matrix-field (parent wid)))
  (when mf (setf (current-indices mf) nil))
  (mf-repaint mf))

;;;
;;;	Methods to make the given row/col current
;;;

(defmethod make-row-current ((self table-field) row &aux mf rt)
  (make-row-current (setq mf (matrix-field self)) row)
  (when (setq rt (slot-value mf 'row-titles))
	(make-row-current rt row)))

(defmethod make-col-current ((self table-field) col &aux mf ct)
  (make-col-current (setq mf (matrix-field self)) col)
  (when (setq ct (slot-value mf 'col-titles))
	(make-col-current ct col)))

;;;
;;;___________________________________________________________________________
;;;
;;;				Matrix-field stuff
;;;___________________________________________________________________________
;;;

;;;
;;;	Insert/Delete Row/col
;;;

;;	n = row below which you wish to insert
(defmethod insert-row ((self matrix-field) &optional n &aux nw)
  (when (consp (setq nw (nomad-widget self)))
	(setq nw (getf nw :geom-spec))
	(setq nw (aref (field-table self) (car nw) (cadr nw)))
	(mf-deactivate nw self))
  (if (null n)
      (setq n (1- (data-rows self))))
  (cond ((uniform-rows self)
	 (let ((da (data self))
	       (rows (data-rows self))
	       (cols (data-cols self))
	       (inc nil)
	       (newda nil))
	      (when (= rows (num-elements da))
		    (if (setq inc (overflow-increment self))
			(progn
			 (setq newda (make-array (list (+ rows inc) cols)))
			 (do ((r n (1+ r)))
			     ((>= r rows))
			     (dotimes (c cols)
				      (setf (aref newda r c)
					    (aref da r c))))
			 (setf (slot-value self 'data)
			       (setq da newda)))
			(warn "Table-full!!!  Must make bigger data-table")))
	      (do ((r (+ n 2) (1+ r)))
		  ((> r rows))
		  (dotimes (c cols)
			   (setf (aref da r c) (aref da (1- r) c))))
	      (dotimes (c cols)
		       (setf (aref da (1+ n) c) nil))
	      (mf-propagate self))
	 ;;	Update changed-indices
	 (setf (slot-value self 'changed-indices)
	       (mapcar #'(lambda (x)
				 (if (> (car x) n)
				     (list (1+ (car x)) (cadr x))
				     x))
		       (changed-indices self)))
	 (setf (data-rows self) (1+ (data-rows self))
	       (data-cols self) (max (data-cols self) (cols self))))
	(t
	 (warn "Can't insert a row into this matrix"))))

;;	n = column left of which you wish to insert
(defmethod insert-col ((self matrix-field) &optional n &aux nw)
  (when (consp (setq nw (nomad-widget self)))
	(setq nw (getf nw :geom-spec))
	(setq nw (aref (field-table self) (car nw) (cadr nw)))
	(mf-deactivate nw self))
  (if (null n) 
      (setq n (1- (data-cols self))))
  (cond ((uniform-cols self)
	 (let ((da (data self))
	       (rows (data-rows self))
	       (cols (data-cols self))
	       (inc nil)
	       (newda nil)) 
	      (when (= cols (num-cols da))
		    (if (setq inc (overflow-increment self))
			(progn
			 (setq newda (make-array (list (+ rows inc) cols)))
			 (do ((c n (1+ c)))
			     ((>= c cols))
			     (dotimes (r rows)
				      (setf (aref newda r c)
					    (aref da r c))))
			 (setf (slot-value self 'data)
			       (setq da newda)))
			(warn "Table-full!!!  Must make bigger data-table")))
	      (do ((c (+ n 2) (1+ c)))
		  ((>= c cols))
		  (dotimes (r rows)
			   (setf (aref da r c) (aref da (1- r) c))))
	      (dotimes (r rows)
		       (setf (aref da r (1+ n)) nil))
	      (mf-propagate self))
	 ;;	Update changed-indices
	 (setf (slot-value self 'changed-indices)
	       (mapcar #'(lambda (x)
				 (if (> (cadr x) n)
				     (list (car x) (1+ (cadr x)))
				     x))
		       (changed-indices self)))
	 (setf (data-cols self) (1+ (data-cols self))
	       (data-rows self) (max (data-rows self) (rows self))))
	(t
	 (warn "Can't insert a column into this matrix"))))

(defmethod delete-row ((self matrix-field) n)
  (cond ((uniform-rows self)
	 (let ((da (data self))
	       (rows (data-rows self))
	       (cols (data-cols self)))
	      (do ((r (1+ n) (1+ r)))
		  ((>= r rows))
		  (dotimes (c cols)
			   (setf (aref da (1- r) c) (aref da r c))))
	      (dotimes (c cols)
		       (setf (aref da (1- rows) c) nil)))
	 (setf (data-rows self) (1- (data-rows self))))
	(t
	 (let ((ft (field-table self)))
	      (dotimes (c (cols self))
		       (conceal (aref ft n c)))))))

(defmethod delete-col ((self matrix-field) n)
  (cond ((uniform-cols self)
	 (let ((da (data self))
	       (rows (data-rows self))
	       (cols (data-cols self)))
	      (do ((c 0 (1+ c)))
		  ((>= c cols))
		  (dotimes (r rows)
			   (setf (aref da r c) (aref da r c)))))
	 (setf (data-cols self) (1- (data-cols self))))
	(t
	 (let ((ft (field-table self)))
	      (dotimes (r (rows self))
		       (conceal (aref ft r n)))))))

;;;
;;;	Scrolling functions
;;;
;;;	For optimization purposes, there are two types of scrolling
;;;	functions for each direction (up, down, left, right), one set to be
;;;	used for rows/columns of uniform height/width and another set for 
;;;	rows/cols of variable height/width.  All needed scrolling functions 
;;;	are cached when the matrix-field is instantiated.  Hence, determining 
;;;	which scrolling function to use is completely automatic, so activating
;;;	these macros will always call the right scrolling function.
;;;

;;	For uniform rows
(defun mf-uni-scroll-up (mf n &aux new-index old-index ct cur)
  (setq old-index (slot-value mf 'row-index))
  (setq new-index (max 0 (- old-index n)))
  (unless (= new-index old-index)
	  (setq cur (current-field mf))
	  (cond ((listp cur)
		 (setf (row-index mf) new-index) 
		 (mf-uni-propagate mf :up n) 
		 (update-indices mf))
		(t
		 (mf-deactivate cur mf) 
		 (setf (slot-value mf 'current-field) nil) 
		 (setf (row-index mf) new-index) 
		 (mf-uni-propagate mf :up n) 
		 (update-indices mf)
		 (setq ct (- (caar (current-indices mf)) old-index))
		 (if (and 
		      (>= ct 0)
		      (< ct (min (data-rows mf) (rows mf))))
		     (setf (slot-value mf 'current-field) cur))))
	  (repaint-inverts mf)
	  (if (setq ct (slot-value mf 'row-titles))
	      (mf-scroll-up ct n))))

;;	For uniform rows
(defun mf-uni-scroll-down (mf n &aux new-index max-index ri ct cur)
  (setq ri (slot-value mf 'row-index))
  (setq max-index (max 0 (- (data-rows mf) (visible-rows mf))))
  (setq new-index (min max-index (+ ri n)))
  (unless (= new-index ri)
	  (setq cur (current-field mf))
	  (cond ((listp cur) 
		 (setf (row-index mf) new-index) 
		 (mf-uni-propagate mf :down n) 
		 (update-indices mf))
		(t
		 (mf-deactivate (current-field mf) mf) 
		 (setf (slot-value mf 'current-field) nil) 
		 (setf (row-index mf) new-index) 
		 (mf-uni-propagate mf :down n) 
		 (update-indices mf)
		 (setq ct (- (caar (current-indices mf)) new-index))
		 (if (and 
		      (>= ct 0) 
		      (< ct (min (data-rows mf) (rows mf))))
		     (setf (slot-value mf 'current-field) cur))))
	  (repaint-inverts mf)
	  (if (setq ct (slot-value mf 'row-titles))
	      (mf-scroll-down ct n))))
  
;;	For uniform columns
(defun mf-uni-scroll-left (mf n &aux new-index old-index ct cur)
  (setq old-index (slot-value mf 'col-index))
  (setq new-index (max 0 (- old-index n)))
  (unless (= new-index old-index)
	  (setq cur (current-field mf))
	  (cond ((listp cur) 
		 (setf (col-index mf) new-index)
		 (mf-uni-propagate mf :left n)
		 (update-indices mf))
		(t
		 (mf-deactivate (current-field mf) mf) 
		 (setf (slot-value mf 'current-field) nil) 
		 (setf (col-index mf) new-index)
		 (mf-uni-propagate mf :left n)
		 (update-indices mf)
		 (setq ct (- (cadar (current-indices mf)) new-index))
		 (if (and 
		      (>= ct 0) 
		      (< ct (min (data-cols mf) (cols mf))))
		     (setf (slot-value mf 'current-field) cur))))
	  (repaint-inverts mf)
	  (if (setq rt (slot-value mf 'col-titles))
	      (mf-scroll-left rt n))))
  
;;	For uniform columns
(defun mf-uni-scroll-right (mf n &aux new-index max-index md ci ct cur)
  (setq ci (slot-value mf 'col-index))
  (setq md (gm-data mf))
  (setq max-index (- (data-cols mf) (md-visible-cols md)))
  (setq new-index (max 0 (min max-index (+ ci n))))
  (unless (= new-index ci)
	  (setq cur (current-field mf))
	  (cond ((listp cur) 
		 (setf (col-index mf) new-index)
		 (mf-uni-propagate mf :right n)
		 (update-indices mf))
		(t
		 (mf-deactivate (current-field mf) mf) 
		 (setf (slot-value mf 'current-field) nil) 
		 (setf (col-index mf) new-index)
		 (mf-uni-propagate mf :right n)
		 (update-indices mf)
		 (setq ct (- (cadar (current-indices mf)) new-index))
		 (if (and 
		      (>= ct 0) 
		      (< ct (min (data-cols mf) (cols mf))))
		     (setf (slot-value mf 'current-field) cur))))
	  (repaint-inverts mf)
	  (if (setq rt (slot-value mf 'col-titles))
	      (mf-scroll-right rt n))))

;;	For variable cols
(defun mf-var-scroll-right (mf n &aux new-index max-index md ci rt)
  (setq md (gm-data mf))
  (setq	ci (md-col-index md))
  (setq max-index (- (data-cols mf) (md-visible-cols md)))
  (setq new-index (max 0 (min max-index (+ ci n))))
  (unless (= new-index ci)
	  (mf-clear mf)
	  (setf (md-col-index md) new-index)
	  (setf (flag mf) t)
	  (force-repack mf)
	  (setf (flag mf) nil)
	  (mf-repaint mf)
	  (if (setq rt (slot-value mf 'col-titles))
	      (mf-scroll-right rt n))))

;;	For variable cols
(defun mf-var-scroll-left (mf n &aux new-index old-index md rt)
  (setq md (gm-data mf))
  (setq old-index (md-col-index md))
  (setq new-index (max 0 (- old-index n)))
  (unless (= new-index old-index)
	  (mf-clear mf)
	  (setf (md-col-index md) new-index)
	  (setf (flag mf) t)
	  (force-repack mf)
	  (setf (flag mf) nil)
	  (mf-repaint mf)
	  (if (setq rt (slot-value mf 'col-titles))
	      (mf-scroll-left rt n))))

;;	For variable rows
(defun mf-var-scroll-down (mf n &aux new-index max-index md ri ct)
  (setq md (gm-data mf))
  (setq ri (md-row-index md))
  (setq max-index (max 0 (- (data-rows mf) (md-visible-rows md))))
  (setq new-index (min max-index (+ ri n)))
  (unless (= new-index ri)
	  (mf-clear mf)
	  (setf (md-row-index md) new-index)
	  (setf (flag mf) t)
	  (force-repack mf)
	  (setf (flag mf) nil)
	  (mf-repaint mf) 
	  (if (setq ct (slot-value mf 'row-titles)) 
	      (mf-scroll-down ct n))))

;;	For variable rows
(defun mf-var-scroll-up (mf n &aux new-index old-index md ct)
  (setq md (gm-data mf))
  (setq old-index (md-row-index md))
  (setq new-index (max 0 (- old-index n)))
  (unless (= new-index old-index)
	  (mf-clear mf)
	  (setf (md-row-index md) new-index)
	  (setf (flag mf) t)
	  (force-repack mf)
	  (setf (flag mf) nil)
	  (mf-repaint mf)
	  (if (setq ct (slot-value mf 'row-titles))
	      (mf-scroll-up ct n))))

;;;
;;;	Functions/Macros to swap data for uniform rows/cols
;;;

;;	For uniform rows
(defun mf-uni-swap-rows (mf row1 row2 &aux da mc el1 el2 ci)
  (setq da (data mf))
  (setq mc (data-cols mf))
  (do ((col 0 (1+ col)))
      ((= col mc))
      (psetf (aref da row1 col) (setq el1 (aref da row2 col))
	     (aref da row2 col) (setq el2 (aref da row1 col))))
  (setq ci (slot-value mf 'current-indices))
  (setf (current-indices mf) 
	(mapcar #'(lambda (x &aux c) 
			  (setq c (car x))
			  (cond ((= c row1) (list row2 (cadr x)))
				((= c row2) (list row1 (cadr x)))
				(t x)))
		ci)))

;;	For uniform cols
(defun mf-uni-swap-cols (mf col1 col2 &aux da mr el1 el2 ci)
  (setq da (data mf))
  (setq mr (data-rows mf))
  (do ((row 0 (1+ row)))
      ((= row mr))
      (psetf (aref da row col1) (setq el1 (aref da row col2))
	     (aref da row col2) (setq el2 (aref da row col1))))
  (setq ci (slot-value mf 'current-indices))
  (setf (current-indices mf) 
	(mapcar #'(lambda (x &aux c) 
			  (setq c (cadr x))
			  (cond ((= c col1) (list (car x) col2))
				((= c col2) (list (car x) col1))
				(t x)))
		ci)))

;;;
;;;	Revert function
;;;
(defun mf-revert (mf &aux rv ti rda mdch)
  (when (setq rv (revert-table mf))
	(setq rv (copy-matrix-detail rv))
	(unless (and (uniform-rows mf) (uniform-cols mf))
		(setq mdch (md-children rv))
		(dotimes (r (md-num-rows rv))
			 (dotimes (c (md-num-cols rv))
				  (setf (geom-spec (aref mdch r c))
					(list r c)))))
	(setf (gm-data mf) rv)
	(when (setq rda (revert-data mf))
	      (mf-quick-set-data rda mf))
	(force-repack mf)
	(setf (current-indices mf) nil)
	(mf-propagate mf))
  (when (setq ti (slot-value mf 'row-titles))
	(when (setq rv (revert-table ti))
	      (setq rv (copy-matrix-detail rv))
	      (unless (and (uniform-rows ti) (uniform-cols ti))
		      (setq mdch (md-children rv))
		      (dotimes (r (md-num-rows rv))
			       (dotimes (c (md-num-cols rv))
					(setf (geom-spec (aref mdch r c))
					      (list r c)))))
	      (setf (gm-data ti) rv)
	      (when (setq rda (revert-data ti))
		    (mf-quick-set-data rda ti))
	      (force-repack ti)
	      (setf (current-indices ti) nil)
	      (mf-propagate ti)))
  (when (setq ti (slot-value mf 'col-titles))
	(when (setq rv (revert-table ti))
	      (setq rv (copy-matrix-detail rv))
	      (unless (and (uniform-rows ti) (uniform-cols ti))
		      (setq mdch (md-children rv))
		      (dotimes (r (md-num-rows rv))
			       (dotimes (c (md-num-cols rv))
					(setf (geom-spec (aref mdch r c))
					      (list r c)))))
	      (setf (gm-data ti) rv)
	      (when (setq rda (revert-data ti))
		    (mf-quick-set-data rda ti))
	      (force-repack ti)
	      (setf (current-indices ti) nil)
	      (mf-propagate ti))))

;;;
;;;	Functions to handle marking the current field
;;;

;;	Marks a field as being current
(defun mf-invert (ef &key (draw t) (bw nil) (bh nil) (mf nil) (sel nil)
		     &aux vl x y w h bd bdw bdh gs flag) 
  (unless mf
	  (setq mf (parent ef)))
  (unless bw
	  (setq bw (inter-col-pad mf)))
  (unless bh
	  (setq	bh (inter-row-pad mf)))
  (unless sel
	  (setq sel (selection mf)))
  (cond ((listp ef)
	 (case sel
	       (:entry
		(setq x (1+ (max 0 (- (getf ef :x) bw)))
		      y (1+ (max 0 (- (getf ef :y) bh)))
		      w (+ (getf ef :width) bw)
		      h (+ (getf ef :height) bh)))
	       (:row
		(setq x 1
		      y (1+ (max 0 (- (getf ef :y) bh)))
		      w (- (width mf) bw)
		      h (+ (getf ef :height) bh)))
	       ((:column :col)
		(setq x (1+ (max 0 (- (getf ef :x) bw)))
		      y 1
		      w (+ (getf ef :width) bw)
		      h (- (height mf) bh))))
	 (setq gs (getf ef :geom-spec)))
	(t
	 (setq flag t)
	 (setq bd (border-width ef))
	 (if (listp bd)
	     (setq bdw (+ (first bd) (third bd))
		   bdh (+ (second bd) (fourth bd)))
	     (setq bdw 
		   (setq bdh (+ bd bd))))
	 (setq gs (geom-spec ef))
	 (case sel
	       (:entry 
		(setq x (1+ (max 0 (- (x-offset ef) bw)))
		      y (1+ (max 0 (- (y-offset ef) bh)))
		      w (+ (width ef) bw bdw)
		      h (+ (height ef) bh bdh)))
	       (:row
		(setq x 1
		      y (1+ (max 0 (- (y-offset ef) bh)))
		      w (- (width mf) bw)
		      h (+ (height ef) bh bdh)))
	       ((:column :col)
		(setq x (1+ (max 0 (- (x-offset ef) bw)))
		      y 1
		      w (+ (width ef) bw bdw)
		      h (- (height mf) bh))))))
  
  (setq vl (list x y w h))
  (if flag
      (if (eq (third gs) :unselectable)
	  (if (fourth gs) 
	      (setf (fourth (geom-spec ef)) vl)
	      (nconc (geom-spec ef) (list vl)))
	  (if (third gs) 
	      (setf (third (geom-spec ef)) vl)
	      (nconc (geom-spec ef) (list vl)))) 
      (if (eq (third gs) :unselectable)
	  (if (fourth gs) 
	      (setf (fourth (getf ef :geom-spec)) vl)
	      (nconc (getf ef :geom-spec) (list vl)))
	  (if (third gs) 
	      (setf (third (getf ef :geom-spec)) vl)
	      (nconc (getf ef :geom-spec) (list vl)))))
  (when draw 
	(when (or (and (listp ef) (not (getf ef :exposed)))
		  (and (not (listp ef)) (not (exposed-p ef))))
	      (return-from mf-invert))
	(let ((gc (gc-res mf)))
	     (xlib:draw-rectangle (res mf) gc x y w h))))

;;	Marks a field as being uncurrent
(defun mf-uninvert (ef &key (bw nil) (bh nil) (mf nil) (sel nil)
		       &aux vl gs res gc f backg)
  (unless mf 
	  (setq mf (parent ef)))
  (unless bw
	  (setq bw (inter-col-pad mf)))
  (unless bh
	  (setq	bh (inter-row-pad mf)))
  (setq gs (if (listp ef) (getf ef :geom-spec) (geom-spec ef)))
  (if (eq (third gs) :unselectable)
      (setq vl (fourth gs))
      (setq vl (third gs)))
  (when vl
	(setq res (res mf)
	      gc (gc-res mf))
	(unless gc (return-from mf-uninvert))
	(setq f (xlib:gcontext-foreground gc))
	(setq backg (background mf))
	(cond ((color-p backg) 
	       (setf (xlib:gcontext-foreground gc) (pixel backg)))
	      (t
	       (setf (xlib:gcontext-tile gc) (res backg))))
	(xlib:draw-rectangle (res mf) (gc-res mf) (first vl) (second vl)
			     (third vl) (fourth vl))
	(setf (xlib:gcontext-foreground gc) f)
	
	;;	Redraw Grid
	(when (grid-lines mf)
	      (mf-draw-grid mf))))

;;	Updates the vertex-lists of the current-fields after a resize
(defun update-inverts (mf &aux bw bh)
  (setq bw (inter-col-pad mf)
	bh (inter-row-pad mf))
  (repaint-inverts mf :clear t :bw bw :bh bh)
  (dolist (cur (current-fields mf))
	  (cond ((listp cur) 
		 (when (getf cur :exposed)
		       (mf-invert cur :draw nil :bw bw :bh bh :mf mf)))
		((exposed-p cur)
		 (mf-invert cur :draw nil :bw bw :bh bh :mf mf)))))

;;	Repaints the borders around the current-fields
(defun repaint-inverts (mf &key (clear nil) (bw nil) (bh nil) 
			   &aux gs vl res gc f c)
  (unless bw
	  (setq bw (inter-col-pad mf)))
  (unless bh
	  (setq	bh (inter-row-pad mf)))
  (setq res (res mf)
	gc (gc-res mf))
  (case (selection mf) 
	(:entry
	 (dolist (c (current-fields mf))
		 (cond ((listp c) 
			(when (getf c :exposed)
			      (setq gs (getf c :geom-spec)) 
			      (if (eq (third gs) :unselectable)
				  (setq vl (fourth gs))
				  (setq vl (third gs)))
			      (unless vl (return-from repaint-inverts))
			      (when clear
				    (setq f (xlib:gcontext-foreground gc))
				    (setf (xlib:gcontext-foreground gc) 
					  (pixel (background mf))))
			      (xlib:draw-rectangle res gc 
						   (first vl) (second vl) 
						   (third vl) (fourth vl))
			      (when clear 
				    (setf (xlib:gcontext-foreground gc) f))))
		       ((exposed-p c) 
			(setq gs (geom-spec c)) 
			(if (eq (third gs) :unselectable)
			    (setq vl (fourth gs))
			    (setq vl (third gs))) 
			(unless vl (return-from repaint-inverts))
			(when clear
			      (setq f (xlib:gcontext-foreground gc))
			      (setf (xlib:gcontext-foreground gc) 
				    (pixel (background mf))))
			(xlib:draw-rectangle res gc 
					     (first vl) (second vl) 
					     (third vl) (fourth vl))
			(when clear 
			      (setf (xlib:gcontext-foreground gc) f))))))
	(:row
	 (dolist (c (current-fields mf))
		 (cond ((listp c) 
			(when (getf c :exposed)
			      (setq gs (getf c :geom-spec)) 
			      (if (eq (third gs) :unselectable)
				  (setq vl (fourth gs))
				  (setq vl (third gs))) 
			      (unless vl (return-from repaint-inverts))
			      (when clear
				    (setq f (xlib:gcontext-foreground gc))
				    (setf (xlib:gcontext-foreground gc) 
					  (pixel (background mf))))
			      (xlib:draw-rectangle res gc 
						   (first vl) (second vl) 
						   (third vl) (fourth vl))
			      (when clear 
				    (setf (xlib:gcontext-foreground gc) f))))
		       ((exposed-p c) 
			(setq gs (geom-spec c)) 
			(if (eq (third gs) :unselectable)
			    (setq vl (fourth gs))
			    (setq vl (third gs))) 
			(unless vl (return-from repaint-inverts))
			(when clear
			      (setq f (xlib:gcontext-foreground gc))
			      (setf (xlib:gcontext-foreground gc) 
				    (pixel (background mf))))
			(xlib:draw-rectangle res gc 
					     (first vl) (second vl) 
					     (third vl) (fourth vl))
			(when clear 
			      (setf (xlib:gcontext-foreground gc) f))))))
	((:col :column)
	 (dolist (c (current-fields mf))
		 (cond ((listp c) 
			(when (getf c :exposed)
			      (setq gs (getf c :geom-spec)) 
			      (if (eq (third gs) :unselectable)
				  (setq vl (fourth gs))
				  (setq vl (third gs))) 
			      (unless vl (return-from repaint-inverts))
			      (when clear
				    (setq f (xlib:gcontext-foreground gc))
				    (setf (xlib:gcontext-foreground gc) 
					  (pixel (background mf))))
			      (xlib:draw-rectangle res gc 
						   (first vl) (second vl) 
						   (third vl) (fourth vl))
			      (when clear 
				    (setf (xlib:gcontext-foreground gc) f))))
		       ((exposed-p c) 
			(setq gs (geom-spec c)) 
			(if (eq (third gs) :unselectable)
			    (setq vl (fourth gs))
			    (setq vl (third gs))) 
			(unless vl (return-from repaint-inverts))
			(when clear
			      (setq f (xlib:gcontext-foreground gc))
			      (setf (xlib:gcontext-foreground gc) 
				    (pixel (background mf))))
			(xlib:draw-rectangle res gc 
					     (first vl) (second vl) 
					     (third vl) (fourth vl))
			(when clear 
			      (setf (xlib:gcontext-foreground gc) f))))))))
