;;;
;;; Copyright (c) 1990 Regents of the University of California
;;; 
;;; $Author: seitz $
;;; $Source: RCS/matrix-field-def.cl,v $
;;; $Revision: 1.2 $
;;; $Date: 90/07/26 17:28:50 $
;;;

(in-package 'pt  :nicknames '(picasso-toolkit) :use '(lisp excl pcl))

;;;	
;;;	The matrix-field is designed to display data which is intrinsically
;;;	tabular (ie. can be organized in a row/column fashion) in structure.  
;;;	The properties of the matrix-field are split into three sections:
;;;	primary functionality, secondary features, and geometry management.
;;;	These parts are described by the matrix-field, table-field, and
;;;	matrix geometry-manager, respectively.  These parts are considerably
;;;	interconnected when they are used together, but are also designed to
;;;	work independent of one another to an extent.
;;;
;;;	-----------------------------------------------------------------------
;;;				Matrix-field
;;;	-----------------------------------------------------------------------
;;;
;;;	Display-format
;;;
;;;	The matrix-field displays a 2-dimensional array of data in a
;;;	2-dimensional array of fields (windows).  The mapping between data
;;;	and fields may or may not be one-to-one, as there may be more data 
;;;	objects than fields.  The display is completely governed by a few 
;;;	variables: rows: number of rows of fields, 
;;;		   cols: number of columns of fields,
;;;		   visible-rows: number of rows of fields currently visible, 
;;;		   visible-cols: number of columns of fields currently visible,
;;;		   row-index: index to data corresponding to top row visible, 
;;;		   col-index: index to data corr. to top column visible.
;;;		   geom-spec: each field has 
;;;				x-offset 	and a
;;;				y-offset
;;;		 	      into the matrix which are stored in the geom-spec
;;;			      slot in the field
;;;	
;;;	The data object displayed in the upper left corner of the matrix can
;;;	be referenced by (aref <data> (+ <row-index> <x-offset>)
;;;				      (+ <col-index> <y-offset>))
;;;	where the x-offset and y-offset are simply those of the field located
;;;	in the upper left corner of the matrix.  This data location is "bound"
;;;	to the field in the upper left corner of the matrix.  The keyword is
;;;	data location--the data object itself is NOT bound to this field, 
;;;	only the location of the object, specified by variable offsets.  Thus,
;;;	if the row-index, col-index, x-offset, or y-offset of the field change,
;;;	the data object displayed in the upper left corner will vary 
;;;	accordingly.
;;;	NOTE: the row and column indices of a field in the matrix are stored
;;;	in the geom-spec of the field as a list (x y).
;;;
;;;	Current-fields
;;;
;;;	A matrix is set up such that only one field can receive events (except
;;;	:expose-window) at a time.  This field may be accessed by the
;;;	"current-field" slot in its matrix.  Any number of fields may be marked
;;;	current at once, however, though all but the designated current-field
;;;	will receive only expose-window events.  The slot "current-fields" of 
;;;	a matrix keeps an up-to-date list of all fields that are marked
;;;	current.  At present, a field is marked current by putting a border
;;;	around it.  The last slot of interest is the "current-indices" slot
;;;	of a matrix which keeps an up-to-date list of all indices into the
;;;	data which correspond to the current-fields in the "current-fields"
;;;	slot.  Since it is a data item which is actually designated to be 
;;;	current and not the actual gadget/widget (the gadget/widget may change 
;;;	during scrolling), an ITEM MAY ONLY BE MARKED CURRENT BY SETTING THE
;;;	CURRENT-INDICES of the matrix.  For example, 
;;;		(setf (current-indices <matrix-field>) 
;;;		      '((0 0))) 
;;;	will set the data item which is displayed in the top-left corner of 
;;;	the matrix (only visible when the matrix is scrolled to the far top 
;;;	and far left--when row-index = col-index = 0) to be current.  Also,
;;;	whenever the data item is visible, the corresponding field which
;;;	displays this data item will be marked the current-field.
;;;	If the current-indices is set to be more than one offset pair, the
;;;	current-field slot will be initially nil.  For example,
;;;		for a matrix with row-index 1 and col-index
;;;		(setf (current-indices <matrix-field>)
;;;		      '((3 0) (3 1) (3 2) (3 3) . . .))
;;;	will mark the whole 2nd visible row to be current (1 + 2 = 3).
;;;	To make selecting data items by selecting fields simple, the following
;;;	functions are bound to the following button-clicks:
;;;	Left-click:	Select data item in field to be the current-field
;;;			(all other fields will be automatically deselected).
;;;	Right-click:	Add item in field to list of already current fields
;;;			or delete it from list it already current
;;;	Middle-click:	Make field THE current-field if already A current-field.
;;;	NOTE:  if the :unique-selection keyword is passed with a value of t
;;;	       on instantiation of the matrix, ALL button clicks will select
;;;	       the data item in the selected field to be the current-field and
;;;	       deselect all other fields.
;;;	See selection in instantiation documentation for SELECTING whole ROWS 
;;;	or COLUMNS.
;;; 
;;;	Initialization/Functionality 
;;; 
;;;	On instantiation, a matrix-field creates 
;;;	    (1) an array of data (if the data is not already in an array format)
;;;	    (2) an array of fields
;;;	    (3) a matrix-field for column titles (optional)
;;;	    (4) a matrix-field for row titles (optional)
;;;	    (5) a cache of scrolling functions (for optimization)
;;;	    (6) other stuff (to be discussed later)
;;;	The first two of these have already been discussed.  Numbers 3 & 4
;;;	may or may not be used (even if they are created) optionally.  The
;;;	columns titles are 1xm dimensional and the row-titles are nx1 
;;;	dimensional, where the whole matrix (without titles) is nxm dimensional.
;;;	The column and row title matrices can be accessed through the SLOT-
;;;	VALUES of row-titles and col-titles of the matrix.  
;;;	Number 5 is just a cache of functions to use for scrolling up, down,
;;;	left, and right.  Depending on whether the rows/columns of fields are of
;;;	uniform	height/width (respectively), different scrolling functions are 
;;;	more efficient than others.  The decision and cache is made at 
;;;	instantiation and both are updated if necessary whenever the base-size 
;;;	of a field changes (see section on matrix geometry management).  
;;;	See below for instantiation options.
;;;
;;;	Changing Data
;;;
;;;	The data of the matrix-field can be altered by doing a setf on data.
;;;	This does an mf-propagate which syncs all the fields in the matrix
;;;	with the data.  This can be a somewhat expensive operation if there
;;;	are many real (not synthetic) widgets (in the range of 50 or more) and 
;;;	can often be avoided
;;;	if just one or two rows or columns of data are modified.  After 
;;;	changing a row data in the data array of the matrix, the matrix must
;;;	be updated and which can be done by one of the functions:
;;;	mf-sync-row or mf-sync-col.
;;;
;;;	Synthetic fields
;;;
;;;	Since creation, scrolling, and resizing time become major bottlenecks 
;;;	with large matrices. Hence, I have invented things called "synthetic" 
;;;	fields which mimic real fields (widgets and gadgets) but with real time 
;;;	behavior.  Timings indicate that a table can be speeded up by between
;;;	1000 and 2000 percent by using synthetic fields instead of "real" ones.
;;;	
;;;	Short Implementation note:
;;;	Synthetic fields are not widgets or gadgets, but rather property lists 
;;;	which just get passed directly to a function which blasts the values 
;;;	directly onto the matrix window.  
;;;       	The implementation is all invisible except for a couple things.
;;;	The current-fields and current-field of a matrix are no longer 
;;;	necessarily windows (they can be property lists).  However, this 
;;;	shouldn't matter anyway because it's the CURRENT-INDICES slot which 
;;;	people should be using (for binds and the like).
;;;	A synthetic widget can be created using the :editable keyword
;;;	(see section on INSTANTIATION).  Basically, a synthetic field is a
;;;	synthetic widget if it's editable, a synthetic gadget if it's not.
;;;     Synthetic widgets have the property that whenever you select one, 
;;;	a real widget jumps to replace it on the screen (and disappears when it
;;;	is no longer current).  Thus, instead of creating widget for every 
;;;	place in the matrix (225 entries for a 15x15 table) we are now creating 
;;;	only one (I call it a "nomad-widget").  This explains the enhanced
;;;	performance of matrices with synthetic fields.
;;;	See section on INSTANTIATION for further documentation.
;;;       
;;;	-----------------------------------------------------------------------
;;;				Table Field
;;;	-----------------------------------------------------------------------
;;;
;;;	Functionality
;;;
;;;	Where the matrix-field contains the neccessary "hard" data, field 
;;;	objects, and functionality, the table-field represents the user 
;;;	interface to the matrix.  A table field layout consists of a primary
;;;	matrix-field, one or two (optional) title matrix-fields, two scroll-bars
;;;	(one for rows, one for columns), two tf-buttons (see table-field.cl 
;;;	for description of functionality), and one deselect button.  Consult 
;;;	table-field.cl & table-utils.cl for more in-depth description of 
;;;	layout & function.
;;;
;;;	-----------------------------------------------------------------------
;;;				Matrix Geometry Management
;;;	-----------------------------------------------------------------------
;;;
;;;	The matrix-field is designed to work with the matrix-gm geometry-
;;;	manager and thus should be used in association with matrix-gm to take 
;;;	advantage of all features (unless another manager is written to do so).
;;;	Unlike other geometry-managers & widgets, the matrix-gm and matrix-field
;;;	communicate explicitly to a small degree.  The matrix-gm notifies the
;;;	matrix-field whenever it is resized, as the row/col-indices may have
;;;	to be adjusted, and also whenever the base-size of a field in the 
;;;	matrix is altered, as the scrolling-cache may need to be updated.
;;;	However, the matrix-field may optionally be used with any other 
;;;	applicable geometry-manager, such as rubber-gm, with a few 
;;;	modifications of matrix-field.  The primary obstacle is that 
;;;	matrix-field currently uses the geom-spec of its children as indices 
;;;	into the data (see above on x,y-offsets).  These modifications will be 
;;;	described in other documentation.
;;;	See documentation on matrix-gm for information on the functionality
;;;	of matrix-gm.
;;;
;;;	
;;;	-----------------------------------------------------------------------
;;;				Matrix Instantiation
;;;	-----------------------------------------------------------------------
;;;
;;;	The creation options to a matrix-field are somewhat intricate, but
;;;	they allow for a range of different types of matrices to be created
;;;	(and quite easily).  The tricky part is specifying what types of
;;;	fields are to be displayed in the matrix.  The default field-type is
;;;	the synthetic gadget which is just an un-editable displayer of
;;;	an arbitrary data item (eg. string, bitmap, pixmap).  In the default
;;;	case, the col-widths and row-heights may be used to tailor the sizes
;;;	of the fields (entry-fields) in the matrix.  The entry-field is used 
;;;	unless either :row-elements or :col-elements is specified.
;;;	Following is a brief description of the non-standard creation options.
;;;	
;;;	inter-row-pad,
;;;	inter-col-pad - the space (in pixels) between each row, column 
;;;			respecively.
;;;			Default 2.
;;;	row-index -	the initial index into the data that should be added
;;;			to the row-index of the fields in the matrix to 
;;;			calculate which data items should be displayed.  See
;;;			above note on display format of a matrix.
;;;			Default 0.
;;;	col-index -	same as above but columns rather than rows.
;;;	field-table -	the initial table of fields to display in the matrix.
;;;			Default constructed at run-time.  Usually not specified
;;;	data -		the initial data to be displayed in the fields of the 
;;;			matrix.  Data may be specified in either one of two 
;;;			ways:
;;;			(1) A list of lists of row items, in other words
;;;				(<row1> <row2> <row3> . . .)
;;;			    where each row is a list of data items.
;;;			    This can also be seen as 
;;;			    (<row1>
;;;			     <row2>
;;;			     <row3>
;;;			     . . .)
;;;	
;;;			(2) An two dimensional array, in which the first
;;;			    dimension is the rows, the second is the columns
;;;
;;;			(3) A pgclos portal.
;;;			Default an array (rows cols) of nils
;;;	titles, 
;;;	col-titles -	either can be used to specify the data to display in
;;;			the column-titles (all titles are entry-fields 
;;;			currently).
;;;			Default nil.
;;;	row-titles -	""
;;;	rows -		the number of rows of fields to be created (Not all are
;;;			necessarily displayed at the same time).
;;;			Default determined dynamically based on others args.
;;;	cols -		the number of columns of fields to be created.
;;;			Default determined dynamically based on others args.
;;;	data-rows -	the number of rows to use from the data table.
;;;			Default is all rows.
;;;	data-cols -	the number of columns to use from the data table.
;;;			Default is all columns.
;;;	overflow-increment -
;;;			the increment by which to "grow" the data-table if it 
;;;			should over-flow. If nil, table can't overflow.
;;;			Default is 5.
;;;     ---------------------------------------------------------------------
;;;		NOTE: only one of row-elements or col-elements may be 
;;;		      specified (or one will be ignored).
;;;     ---------------------------------------------------------------------
;;;	row-elements -  a list of expressions which may be evaluated 
;;;			individually to actually create the fields that should
;;;			constitute the elements of each row of the matrix.
;;;			For example: '((make-meter-widget :base-height 30
;;;							 :font (serif-font))
;;;				       (:base-height 20 :editable t)
;;;				       (make-pixmap-gadget :base-height 50
;;;							   :background
;;;							   (green-color)
;;;							   :unselectable t)
;;;				       (:base-width 50 :font (default-font))
;;;				       (make-gray-button :base-height 20))
;;;			creates a matrix in which the first rows is all meter-
;;;			widgets, the second row consists of synthetic widgets,
;;;			the third row of pixmap-gadgets, the fourth of 
;;;			synthetic gadgets, and the fifth of gray-buttons.  
;;;			The :value should 
;;;			generally not be specified by any of these expressions 
;;;			as the value will only be overrided in the matrix.
;;;			Specifying either row-elements or col-elements will
;;;			override col-widths and row-heights.
;;;			A non-editable matrix row/column is typically made by
;;;			specifying one of the row/col-elements to be a gadget
;;;			of some sort (either real or synthetic).  
;;;			NOTE:  the matrix instantiation method will make all
;;;			gadgets selectable (by enclosing them in collection-
;;;			widgets).  This may be overriden if the keyword
;;;			:unselectable is specifed in the row/col-elements
;;;			eval-expression for that row or column.
;;;			For example: '((make-text-gadget :base-height 30
;;;							 :unselectable t)
;;;				       (:base-height 50 :unselectable t))
;;;			will create a matrix with one row of gadgets and one
;;;			row of synthetic gadgets in which no elements are 
;;;			selectable.
;;;			If used in association with :ROWS or :ROW-HEIGHTS, the 
;;;			table will create how ever many rows specified and 
;;;			reuse THE LAST row-element for any remaining rows not 
;;;			specified by a row-element.  This allows creation of a 
;;;			table of synthetic widgets by means of:
;;;
;;;			(make-matrix-field 
;;;		         :row-elements '((:editable t))
;;;			 :row-heights '(30 60 10 20 30 40 50 29)
;;;			 :font (serif-font))
;;;
;;;			NOTE:  synthetic widgets and gadgets can display 
;;;			(and edit) text, bitmaps, and pixmaps and are 
;;;			considerably faster than any real widget/gadget. 
;;;
;;;			USERS ARE STRONGLY URGED TO USE SYNTHETIC FIELDS
;;;			IN PLACE OF TEXT, BITMAP, AND PIXMAP WIDGETS AND GADGETS
;;;			(see above section on synthetic fields)
;;;
;;;			Default synthetic gadgets.
;;;	col-elements -	same as row-elements but determines the fields which
;;;			will constitute the elements of each column of the 
;;;			matrix.
;;;			Default entry-fields.
;;;	col-widths -	a list specifying the widths of each column in the 
;;;			matrix.  Should only be used if neither of
;;;			row-elements/col-elements are specified (in which case
;;;			col-widths and row-heights will be ignored).
;;;			Default entry-fields.
;;;	row-heights -	a list specifying heights of each row in the matrix.
;;;			Default 40.
;;;	initial-rows - 	the number of rows of fields to be displayed initially
;;;			in the matrix.
;;;			Default number of rows.
;;;	initial-cols -	the number of columns of fields to be displayed
;;;			initially in the matrix.
;;;			Default number of columns.
;;;	row-title-width - the widths of all fields in the row-titles.
;;;	col-title-height - the height of all fields in the col-titles.
;;;	default-titles - if specified t, col-titles are not specified, and data
;;;			is a portal, default column titles will be created 
;;;			consisting of the names of all the fields of the 
;;;			relation (designated by the portal).
;;;			Default t.
;;;	font -		may be used to set the font of all synthetic
;;;			widgets/gadgets in the matrix.
;;;	row-title-font - may be used to set the font of the row-titles.
;;;	col-title-font - may be used to set the font of the column-titles.
;;;	self-adjustable - the fields will automatically adjust to meet their
;;;			base-sizes.  This can be nice, but it slows things down 
;;;			somewhat and it makes the table change considerably
;;;			whenever the values displayed change considerably
;;;			(ie.  if a long string is suddenly scrolled into view).
;;;			Default nil.
;;;	unique-selection - if the passed with a value of t, ALL button clicks 
;;;			will select the data item in the selected field to be 
;;;			the current-field and deselect all other fields.
;;;			Default nil.
;;;	selection -	one of either :row :column or :entry.  Determines what
;;;			type of selection protocol to use.  If :row is used,
;;;			Clicking on any item will make that whole row current.
;;;			If :entry is used, clicking on any entry makes that
;;;			entry current.
;;;			Default :entry.
;;;


(defconstant *mf-value-init* 'mf-value-init)

;;;
;;;     Macro to determine if an object can be selectable in a matrix.
;;;

(defmethod mf-selectable-widget ((self list))
  t)

;;;
;;;	Defind matrix-field class
;;;

(defclass matrix-field (collection-widget)
  ((row-index				;; Indices into data. . .
    :initarg :row-index				 
    :initform 0
    :type integer
    :reader row-index)
   (col-index
    :initarg :col-index 
    :initform 0
    :type integer
    :reader col-index)
   (data-rows				;; Number of rows of data to use
    :initarg :data-rows				 
    :initform 0
    :type integer
    :reader data-rows)
   (data-cols				;; Number of columns of data to use
    :initarg :data-cols				 
    :initform 0
    :type integer
    :reader data-cols)
   (overflow-increment			;; Size by which to "grow" the table
    :initarg :overflow-increment			 
    :initform 5				;; if rows or colunns are inserted
    :type integer
    :accessor overflow-increment)
   (row-titles				;; Matrix of titles for rows
    :initarg :row-titles				 
    :initform nil
    :type matrix-field)
   (col-titles				;; Matrix of titles for columns
    :initarg :col-titles				 
    :initform nil
    :type matrix-field)
   (data				;; data to be displayed (may be an
    :initarg :data  
    :initform nil			;; array or portal)
    :type t
    :accessor data)
					;; info about current-fields. . .
   (current-indices			;; if using bindings, bind to this slot.
    :initarg :current-indices			 
    :initform nil
    :type t
    :reader current-indices)
   (select-func				;; executed whenever current-indices
    :initarg :select-func		;; is changed.
    :initform nil
    :type list
    :accessor select-func)
   (unselect-func			;; executed whenever current-indices
    :initarg :unselect-func		;; is changed.
    :initform nil
    :type list
    :accessor unselect-func)
   (return-func				;; executed when return pressed in
    :initarg :return-func		;; nomad-widget
    :initform '(setf (current-indices self) nil)
    :type list
    :accessor return-func)
   (current-field
    :initarg :current-field 
    :initform nil
    :type t
    :reader current-field)
   (current-fields
    :initarg :current-fields 
    :initform nil
    :type t
    :reader current-fields)
   (grid-lines				;; draw grid lines?
    :initarg :grid-lines
    :initform t
    :type atom
    :reader grid-lines)
   (nomad-creator
    :initarg :nomad-creator 
    :initform #'make-entry-widget
    :type function
    :reader nomad-creator)
   (free-nomad
    :initarg :free-nomad 
    :initform nil
    :type t
    :reader free-nomad)
   (selection
    :initarg :selection 
    :initform :entry
    :type keyword
    :accessor selection)
   (changed-indices
    :initarg :changed-indices 
    :initform nil
    :type t
    :reader changed-indices)
   (font :initform "8x13")
;;	-------------------------------------------------------------------
;;				INTERNAL USE ONLY
   (gc-spec :initform '((gc-res (:line-width 3))
			(gc-gray (:paint "gray50"))))
   (gc-gray 
    :type vector 
    :initform nil
    :reader gc-gray)
   (synths
    :initform nil
    :type list
    :accessor synths)
   (fonts
    :initform nil
    :type t
    :accessor fonts)
   (uniform-rows			;; cached info about row/col sizes. . .
    :initform nil
    :type atom
    :accessor uniform-rows)
   (uniform-cols
    :initform nil
    :type atom
    :accessor uniform-cols)
   (uni-row-types
    :initform t
    :type atom
    :accessor uni-row-types)
   (uni-col-types
    :initform t
    :type atom
    :accessor uni-col-types)
   (revert-table			;; Field table to revert to on a
    :initform nil			;; "revert" operation.
    :type t
    :accessor revert-table)
   (revert-data				;; Data table to revert to on a
    :initform nil			;; "revert" operation.
    :type t
    :accessor revert-data)
   (up-func				;; cached scrolling functions. . .
    :initform nil
    :type t
    :accessor up-func)
   (down-func
    :initform nil
    :type t
    :accessor down-func)
   (right-func
    :initform nil
    :type t
    :accessor right-func)
   (left-func
    :initform nil
    :type t
    :accessor left-func)
   (auto-prop
    :initform nil
    :type atom
    :accessor auto-prop)
   (nomad-widget
    :initform nil
    :type window
    :reader nomad-widget)
   (flag
    :initform nil
    :type atom
    :accessor flag)
   (gm :initform 'matrix-gm)
   (border-width :initform 0)
   (conform :initform :dont-conform)
   (value :initform *mf-value-init*)
   (event-mask :initform '(:exposure :button-press))
   (name :initform "A Matrix-Field")))

;;;
;;;	Field/table accessor
;;;

(defmethod field-table ((self matrix-field))
  (md-children (gm-data self)))

(defmethod (setf field-table) (val (self matrix-field))
  (setf (md-children (gm-data self)) val))

;;;
;;;	Row/col accessors
;;;

(defmethod rows ((self matrix-field))
  (array-dimension (md-children (gm-data self)) 0))

(defmethod cols ((self matrix-field))
  (array-dimension (md-children (gm-data self)) 1))

(defmethod (setf data-rows) (val (self matrix-field))
  (setf (slot-value self 'data-rows) val))

(defmethod (setf data-cols) (val (self matrix-field))
  (setf (slot-value self 'data-cols) val))

;;;
;;;	Inter-row/col accessors
;;;

(defmethod inter-row-pad ((self matrix-field))
  (md-inter-row-pad (gm-data self)))

(defmethod inter-col-pad ((self matrix-field))
  (md-inter-row-pad (gm-data self)))

(defmethod (setf inter-row-pad) (val (self matrix-field))
  (setf (md-inter-row-pad (gm-data self)) val))

(defmethod (setf inter-col-pad) (val (self matrix-field))
  (setf (md-inter-col-pad (gm-data self)) val))

;;;
;;;	Row/col-index writers
;;;

(defmethod visible-row-index ((self matrix-field))
  (if (uniform-rows self)
      (slot-value self 'row-index)
      (md-row-index (gm-data self))))

(defmethod (setf row-index) (value (self matrix-field))
  (if (uniform-rows self)
      (setf (slot-value self 'row-index) value)
      (progn
       (setf (md-row-index (gm-data self)) value)
       (force-repack self))))

(defmethod visible-col-index ((self matrix-field))
  (if (uniform-cols self)
      (slot-value self 'col-index)
      (md-col-index (gm-data self))))

(defmethod (setf col-index) (value (self matrix-field))
  (if (uniform-cols self)
      (setf (slot-value self 'col-index) value)
      (progn 
       (setf (md-col-index (gm-data self)) value)
       (force-repack self))))

;;;
;;;	Nomad setting
;;;

(defmethod (setf free-nomad) (val (self matrix-field) &aux nw)
  (when (not (eq val (slot-value self 'free-nomad)))
	(setf (slot-value self 'free-nomad) val)
	(when (consp (setq nw (nomad-widget self)))
	      (let ((gs (getf nw :geom-spec)))
		   (setq nw (aref (field-table self) (car gs) (cadr gs)))))
	(detach nw)
	(setf (slot-value nw 'parent)
	      (if val (root-window) self))
	(attach nw)))

;;;
;;;	Set current-field
;;;

(defmethod current-value ((self matrix-field) &aux cf)
  (when (and (eq (selection self) :entry) 
	     (= (length (current-indices self)) 1))
	(setq cf (car (slot-value self 'current-indices)))
	(if cf
	    (aref (data self) (car cf) (cadr cf))
	    nil)))

(defmethod current-values ((self matrix-field) &aux cf)
  (when (eq (selection self) :entry)
	(mapcar #'(lambda (ci)
			  (aref (data self) (car ci) (cadr ci)))
		(slot-value self 'current-indices))))

;;	Takes a pair of indices (x y), which should be the actual geom-spec
;;	of the field to become current.  Makes any previously current fields
;;	un-current.
(defmethod (setf current-indices) (index-list (self matrix-field) 
					      &aux cf fd ft nr nc ri ci vr
					      vc ur uc c cr cc icp irp r sel)
  (setq sel (selection self))
  (when (and (unselect-func self) sel)
	(execute 'unselect-func self index-list))
  (setq cf (current-fields self)
	c (slot-value self 'current-field)
	ft (field-table self)
	nr (data-rows self)
	nc (data-cols self)
	ri (slot-value self 'row-index)
	ci (slot-value self 'col-index)
	vr (md-visible-rows (gm-data self))
	vc (md-visible-cols (gm-data self))
	ur (uniform-rows self)
	uc (uniform-cols self)
	irp (inter-row-pad self)
	icp (inter-col-pad self))
  (when c (mf-deactivate c self))
  (dolist (cur cf)
	  (mf-uninvert cur :bw icp :bh irp :mf self))
  (when (eq sel :entry)
	(when (and index-list (atom (car index-list)))
	      (setq index-list (list index-list)))
	(when (and (= 1 (length index-list)) 
		   (>= (setq cr (- (caar index-list) ri)) 0)
		   (>= (setq cc (- (cadar index-list) ci)) 0)
		   (< cr (min nr (rows self))) 
		   (< cc (min nc (cols self))))
	      (setq c (aref ft cr cc))
	      (setf (slot-value self 'current-field) c)
	      (mf-activate c self)))
  (setq cf nil)
  (if sel
      (setf (slot-value self 'current-indices) index-list))
  (case sel
	(:entry
	 (dolist (il index-list)
		 (setq r (car il)
		       c (cadr il))
		 (cond ((and ur uc)
			(cond ((and (< r (+ ri vr)) (>= r ri) (< r nr) 
				    (< c (+ ci vc)) (>= c ci) (< c nc))
			       (setq fd (aref ft (- r ri) (- c ci)))
			       (pushnew fd cf)
			       (mf-invert fd :bw icp :bh irp :mf self))
			      (t (setq index-list (remove il index-list 
							  :test #'equal)))))
		       (ur 
			(cond ((and (< r (+ ri vr)) (>= r ri) (< r nr)) 
			       (setq fd (aref ft (- r ri) (- c ci)))
			       (pushnew fd cf)
			       (mf-invert fd :bw icp :bh irp :mf self))
			      (t (setq index-list (remove il index-list 
							  :test #'equal)))))
		       (uc
			(cond ((and (< c (+ ci vc)) (>= c ci) (< c nc))
			       (setq fd (aref ft (- r ri) (- c ci)))
			       (pushnew fd cf)
			       (mf-invert fd :bw icp :bh irp :mf self))
			      (t (setq index-list (remove il index-list 
							  :test #'equal)))))
		       (t
			(setq fd (aref ft (- r ri) (- c ci)))
			(pushnew fd cf)
			(mf-invert fd :bw icp :bh irp :mf self)))))
	(:row
	 (setq c ci)
	 (dolist (r index-list)
		 (cond ((and ur uc)
			(cond ((and (< r (+ ri vr)) (>= r ri) (< r nr) 
				    (< c (+ ci vc)) (>= c ci) (< c nc))
			       (setq fd (aref ft (- r ri) (- c ci)))
			       (pushnew fd cf)
			       (mf-invert fd :bw icp :bh irp :mf self))
			      (t (setq index-list (remove r index-list 
							  :test #'equal)))))
		       (ur 
			(cond ((and (< r (+ ri vr)) (>= r ri) (< r nr)) 
			       (setq fd (aref ft (- r ri) (- c ci)))
			       (pushnew fd cf)
			       (mf-invert fd :bw icp :bh irp :mf self))
			      (t (setq index-list (remove r index-list 
							  :test #'equal)))))
		       (uc
			(cond ((and (< c (+ ci vc)) (>= c ci) (< c nc))
			       (setq fd (aref ft (- r ri) (- c ci)))
			       (pushnew fd cf)
			       (mf-invert fd :bw icp :bh irp :mf self))
			      (t (setq index-list (remove r index-list 
							  :test #'equal)))))
		       (t
			(setq fd (aref ft (- r ri) (- c ci)))
			(pushnew fd cf)
			(mf-invert fd :bw icp :bh irp :mf self))))) 
	((:column :col)
	 (setq r ri)
	 (dolist (c index-list)
		 (cond ((and ur uc)
			(cond ((and (< r (+ ri vr)) (>= r ri) (< r nr) 
				    (< c (+ ci vc)) (>= c ci) (< c nc))
			       (setq fd (aref ft (- r ri) (- c ci)))
			       (pushnew fd cf)
			       (mf-invert fd :bw icp :bh irp :mf self))
			      (t (setq index-list (remove c index-list 
							  :test #'equal)))))
		       (ur 
			(cond ((and (< r (+ ri vr)) (>= r ri) (< r nr)) 
			       (setq fd (aref ft (- r ri) (- c ci)))
			       (pushnew fd cf)
			       (mf-invert fd :bw icp :bh irp :mf self))
			      (t (setq index-list (remove c index-list 
							  :test #'equal)))))
		       (uc
			(cond ((and (< c (+ ci vc)) (>= c ci) (< c nc))
			       (setq fd (aref ft (- r ri) (- c ci)))
			       (pushnew fd cf)
			       (mf-invert fd :bw icp :bh irp :mf self))
			      (t (setq index-list (remove c index-list 
							  :test #'equal)))))
		       (t
			(setq fd (aref ft (- r ri) (- c ci)))
			(pushnew fd cf)
			(mf-invert fd :bw icp :bh irp :mf self))))))

  (setf (slot-value self 'current-fields) cf)
  (unless index-list
	  (setf (slot-value self 'current-fields) nil
		(slot-value self 'current-field) nil))
  (when (or (and (eq sel :entry) (not index-list)) (not (eq sel :entry)))
	(setf (slot-value self 'current-field) nil))
  
  ;;  Execute select-func
  (when (and (select-func self) sel)
	(execute 'select-func self index-list)))

;;
;;	Same as above but bypasses binds
;;
(defun update-indices (self &aux index-list cf fd ft nr nc ri ci vr vc ur uc 
			    c cr cc icp irp r sel)
  (setq cf (current-fields self)
	index-list (slot-value self 'current-indices)
	sel (selection self)
	c (slot-value self 'current-field)
	ft (field-table self)
	nr (data-rows self)
	nc (data-cols self)
	ri (slot-value self 'row-index)
	ci (slot-value self 'col-index)
	vr (md-visible-rows (gm-data self))
	vc (md-visible-cols (gm-data self))
	ur (uniform-rows self)
	uc (uniform-cols self)
	irp (inter-row-pad self)
	icp (inter-col-pad self))
  (when c
	(mf-deactivate c self))
  (dolist (cur cf)
	  (mf-uninvert cur :bw icp :bh irp :mf self))
  (when (eq sel :entry)
	(when (and index-list (atom (car index-list)))
	      (setq index-list (list index-list)))
	(when (and (= 1 (length index-list)) 
		   (>= (setq cr (- (caar index-list) ri)) 0)
		   (>= (setq cc (- (cadar index-list) ci)) 0)
		   (< cr (min nr (rows self))) (< cc (min nc (cols self))))
	      (setq c (aref ft cr cc))
	      (setf (slot-value self 'current-field) c)
	      (mf-activate c self)))
  (setq cf nil)
  (case sel
	(:entry
	 (dolist (index-list index-list)
		 (setq r (car index-list)
		       c (cadr index-list))
		 (cond ((and ur uc)
			(when (and (< r (+ ri vr)) (>= r ri) (< r nr) 
				   (< c (+ ci vc)) (>= c ci) (< c nc))
			      (setq fd (aref ft (- r ri) (- c ci)))
			      (pushnew fd cf)
			      (mf-invert fd :bw icp :bh irp :mf self)))
		       (ur 
			(when (and (< r (+ ri vr)) (>= r ri) (< r nr)) 
			      (setq fd (aref ft (- r ri) (- c ci)))
			      (pushnew fd cf)
			      (mf-invert fd :bw icp :bh irp :mf self)))
		       (uc
			(when (and (< c (+ ci vc)) (>= c ci) (< c nc))
			      (setq fd (aref ft (- r ri) (- c ci)))
			      (pushnew fd cf)
			      (mf-invert fd :bw icp :bh irp :mf self)))
		       (t
			(setq fd (aref ft (- r ri) (- c ci)))
			(pushnew fd cf)
			(mf-invert fd :bw icp :bh irp :mf self)))))
	(:row
	 (setq c ci) 
	 (dolist (r index-list)
		 (cond ((and ur uc)
			(when (and (< r (+ ri vr)) (>= r ri) (< r nr) 
				   (< c (+ ci vc)) (>= c ci) (< c nc))
			      (setq fd (aref ft (- r ri) (- c ci)))
			      (pushnew fd cf)
			      (mf-invert fd :bw icp :bh irp :mf self)))
		       (ur 
			(when (and (< r (+ ri vr)) (>= r ri) (< r nr)) 
			      (setq fd (aref ft (- r ri) (- c ci)))
			      (pushnew fd cf)
			      (mf-invert fd :bw icp :bh irp :mf self)))
		       (uc
			(when (and (< c (+ ci vc)) (>= c ci) (< c nc))
			      (setq fd (aref ft (- r ri) (- c ci)))
			      (pushnew fd cf)
			      (mf-invert fd :bw icp :bh irp :mf self)))
		       (t
			(setq fd (aref ft (- r ri) (- c ci)))
			(pushnew fd cf)
			(mf-invert fd :bw icp :bh irp :mf self))))) 
	((:column :col)
	 (setq r ri) 
	 (dolist (c index-list)
		 (cond ((and ur uc)
			(when (and (< r (+ ri vr)) (>= r ri) (< r nr) 
				   (< c (+ ci vc)) (>= c ci) (< c nc))
			      (setq fd (aref ft (- r ri) (- c ci)))
			      (pushnew fd cf)
			      (mf-invert fd :bw icp :bh irp :mf self)))
		       (ur 
			(when (and (< r (+ ri vr)) (>= r ri) (< r nr)) 
			      (setq fd (aref ft (- r ri) (- c ci)))
			      (pushnew fd cf)
			      (mf-invert fd :bw icp :bh irp :mf self)))
		       (uc
			(when (and (< c (+ ci vc)) (>= c ci) (< c nc))
			      (setq fd (aref ft (- r ri) (- c ci)))
			      (pushnew fd cf)
			      (mf-invert fd :bw icp :bh irp :mf self)))
		       (t
			(setq fd (aref ft (- r ri) (- c ci)))
			(pushnew fd cf)
			(mf-invert fd :bw icp :bh irp :mf self))))))

  (setf (slot-value self 'current-fields) cf)
  (unless index-list
	  (setf (slot-value self 'current-fields) nil
		(slot-value self 'current-field) nil))
  (when (or (and (eq sel :entry) (not index-list)) (not (eq sel :entry)))
	(setf (slot-value self 'current-field) nil)))

;;	Takes a pair of indices (x y), which should be the effective data
;;	location of the field to become current.  x = car of geom-spec of
;;	field + row-index into data, y = cadr of geom-spec of field + column-
;;	index into data.  Adds this field to the list of current fields.
(defun add-current  (index-list self &aux fd ft cf nr nc ri ci vr vc ur uc r c
				icp irp sel)
  (setq ft (field-table self)
	cf (slot-value self 'current-field)
	sel (selection self)
	nr (data-rows self)
	nc (data-cols self)
	ri (slot-value self 'row-index)
	ci (slot-value self 'col-index)
	vr (md-visible-rows (gm-data self))
	vc (md-visible-cols (gm-data self))
	ur (uniform-rows self)
	uc (uniform-cols self)
	icp (inter-col-pad self)
	irp (inter-row-pad self))
  (case sel
	(:entry
	 (setq r (car index-list)
	       c (cadr index-list)))
	(:row
	 (setq r index-list
	       c ci))
	((:column :col)
	 (setq r ri
	       c index-list))
	(t
	 (return-from add-current)))
  (when cf
	(mf-deactivate cf self))
  (setf (slot-value self 'current-field) nil) 
  (setf (slot-value self 'current-indices) 
	(push index-list (slot-value self 'current-indices)))
  (setq fd (aref ft (- r ri) (- c ci))) 
  (setf (slot-value self 'current-fields) 
	(nconc (current-fields self) (list fd)))
  (cond ((and ur uc)
	 (when (and (< r (+ ri vr)) (>= r ri) (< r nr) 
		    (< c (+ ci vc)) (>= c ci) (< c nc))
	       (mf-invert fd :bw icp :bh irp :mf self :sel sel)))
	(ur 
	 (when (and (< r (+ ri vr)) (>= r ri) (< r nr)) 
	       (mf-invert fd :bw icp :bh irp :mf self :sel sel)))
	(uc
	 (when (and (< c (+ ci vc)) (>= c ci) (< c nc))
	       (mf-invert fd :bw icp :bh irp :mf self :sel sel)))
	(t
	 (mf-invert fd :bw icp :bh irp :mf self :sel sel)))
  (when (select-func self)
	(execute 'select-func self index-list))
  (do-propagate 'current-indices self))

;;	Takes a pair of indices as above and makes the corresponding field
;;	no longer current.
(defun delete-current  (index-list self &aux fd ft nr nc ri ci vr vc ur uc r c
				   icp irp sel)
  (setq ft (field-table self)
	c (slot-value self 'current-field)
	sel (selection self)
	nr (data-rows self)
	nc (data-cols self)
	ri (slot-value self 'row-index)
	ci (slot-value self 'col-index)
	vr (md-visible-rows (gm-data self))
	vc (md-visible-cols (gm-data self))
	ur (uniform-rows self)
	uc (uniform-cols self)
	icp (inter-col-pad self)
	irp (inter-row-pad self))
  (case sel
	(:entry
	 (setq r (car index-list)
	       c (cadr index-list)))
	(:row
	 (setq r index-list
	       c ci))
	((:column :col)
	 (setq r ri
	       c index-list))
	(t 
	 (return-from delete-current)))
  (setq fd (aref ft (- r ri) (- c ci)))
  (setf (slot-value self 'current-fields) 
	(delete fd (slot-value self 'current-fields)))
  (when (eq fd c)
	(mf-deactivate fd self)
	(setf (slot-value self 'current-field) nil))
  (setf (slot-value self 'current-indices) 
	(delete index-list (slot-value self 'current-indices) :test #'equal))
  (cond ((and ur uc)
	 (when (and (< r (+ ri vr)) (>= r ri) (< r nr) 
		    (< c (+ ci vc)) (>= c ci) (< c nc))
	       (mf-uninvert fd :bw icp :bh irp :mf self :sel sel)))
	(ur 
	 (when (and (< r (+ ri vr)) (>= r ri) (< r nr)) 
	       (mf-uninvert fd :bw icp :bh irp :mf self :sel sel)))
	(uc
	 (when (and (< c (+ ci vc)) (>= c ci) (< c nc))
	       (mf-uninvert fd :bw icp :bh irp :mf self :sel sel)))
	(t 
	 (mf-uninvert fd :bw icp :bh irp :mf self :sel sel)))
  (repaint-inverts self)
  (when (select-func self)
	(execute 'select-func self index-list))
  (do-propagate 'current-indices self))

;;;
;;;	Data/Value method and setf
;;;

(defmethod (setf data) (val (self matrix-field) &aux r c nr nc ft numr numc sup)
  ;;	Only process if the data has changed
  (unless (or (eq val (data self)) (eq val *mf-value-init*))
  ;;	Make into an array if necessary
  (setq r (rows self)
	c (cols self) 
	ft (field-table self))
  (when (listp val)
	(setq numr (max 
		    (setf (data-rows self) (setq nr (num-elements val)))
		    r)
	      numc (max 
		    (setf (data-cols self) (setq nc (num-cols val)))
		    c))
	(setq val (make-array (list numr numc)
			      :initial-contents (fill-2d-list val numr numc))))
  (when (> (col-index self) numc)
	(setf (col-index self) 0))
  (when (> (row-index self) numr)
	(setf (row-index self) 0))
  (when (slot-value self 'current-indices)
	(setf (current-indices self) nil))
  (setf (slot-value self 'data) val)
  ;;	Reset changed-indices
  (setf (slot-value self 'changed-indices) nil)
  (when (and (typep (setq sup (parent self)) 'table-field)
	     (eq self (matrix-field sup)))
	(ta-newdata-update sup nr nc))
  (mf-propagate self)
  (mf-repaint self)))

(defmethod value ((self matrix-field))
  (data self))

(defmethod (setf value) (val (self matrix-field))
  (setf (data self) val))

;;
;;	Grid-lines setf repaints
;;

(defmethod (setf grid-lines) (val (self matrix-field))
  (setf (slot-value self 'grid-lines) val)
  (if val
      (progn
       (mf-draw-grid self)
       (repaint-inverts self))
      (mf-repaint self)))

;;;
;;;	Gm-data access functions
;;;

(defmethod visible-rows ((self matrix-field))
  (when (gm-data self)
	(md-visible-rows (gm-data self))))

(defmethod (setf visible-rows) (val (self matrix-field))
  (when (gm-data self)
	(setf (md-visible-rows (gm-data self)) val)))

(defmethod visible-cols ((self matrix-field))
  (when (gm-data self)
	(md-visible-cols (gm-data self))))

(defmethod (setf visible-cols) (val (self matrix-field))
  (when (gm-data self)
	(setf (md-visible-cols (gm-data self)) val)))

;;;
;;; 	Accessors/Setfs for row/col-titles
;;;

(defmethod row-titles ((self matrix-field) &aux rt)
  (when (setq rt (slot-value self 'row-titles))
	(data rt)))

(defmethod col-titles ((self matrix-field) &aux ct)
  (when (setq ct (slot-value self 'col-titles))
	(data ct)))

(defmethod (setf row-titles) (val (self matrix-field) &aux rt)
  (when (setq rt (slot-value self 'row-titles))
	(when (atom (car val))
	      (setq val (mapcar #'list val)))
	(setf (data rt) val)))

(defmethod (setf col-titles) (val (self matrix-field) &aux ct)
  (when (setq ct (slot-value self 'col-titles))
	(when (atom (car val))
	      (setq val (list val)))
	(setf (data ct) val)))

(defmethod row-title-matrix ((self matrix-field))
  (slot-value self 'row-titles))

(defmethod (setf row-title-matrix) (val (self matrix-field))
  (setf (slot-value self 'row-titles) val))

(defmethod col-title-matrix ((self matrix-field))
  (slot-value self 'col-titles))

(defmethod (setf col-title-matrix) (val (self matrix-field))
  (setf (slot-value self 'col-titles) val))

;;;
;;;	Setfs for uniform-rows/cols updates caches for scrolling funcs
;;;

(defmethod (setf uniform-rows) (val (self matrix-field))
  (setf (slot-value self 'uniform-rows) val)
  (if val
      (setf (up-func self) #'mf-uni-scroll-up
	    (down-func self) #'mf-uni-scroll-down)
      (setf (up-func self) #'mf-var-scroll-up
	    (down-func self) #'mf-var-scroll-down)))

(defmethod (setf uniform-cols) (val (self matrix-field))
  (setf (slot-value self 'uniform-cols) val)
  (if val
      (setf (left-func self) #'mf-uni-scroll-left
	    (right-func self) #'mf-uni-scroll-right)
      (setf (left-func self) #'mf-var-scroll-left
	    (right-func self) #'mf-var-scroll-right)))

;;;
;;;	Reader for changed indices removes duplicates
;;;

(defmethod changed-indices ((self matrix-field))
  (setf (slot-value self 'changed-indices)
	(remove-duplicates (slot-value self 'changed-indices) 
			   :test #'pair-comp)))
