;;;
;;; Copyright (c) 1990 Regents of the University of California
;;; 
;;; $Author: picasso $
;;; $Source: RCS/list-box.cl,v $
;;; $Revision: 1.1 $
;;; $Date: 90/07/18 09:20:32 $
;;;

(in-package 'pt  :nicknames '(picasso-toolkit) :use '(lisp excl pcl))

;;;
;;;	A list-box is a table-field with only one row or one column.  A
;;;	list-box contains only synthetic widgets or gadgets (see table-field) 
;;;	and therefore creation and scrolling are pretty quick.  Other
;;;	than these restrictions, the only differences between list-boxes and
;;;	table-fields are the creation options and accessor methods.
;;;	VALUE:	pass in values in a ONE-DIMENSIONAL list.
;;;	ITEMS:	alternative name to value.
;;;	PAD:	padding (in pixels) between rows or columns
;;;	ROW-HEIGHT,
;;;	COL-WIDTH: 
;;;		may be specified instead of padding to set the height/width of 
;;;		each row or column.
;;;	ORIENTATION:
;;;		either :vertical (one column) or :horizontal (one row).
;;;	MAX-ELEMENTS:
;;;		the maximum amount of rows (if :vertical) or columns (if
;;;		:horizontal) which can be viewed at once.  This is necessary
;;;		since the table cannot "gain" rows or columns dynamically.
;;;	MAX-HEIGHT:
;;;		may be specified instead of MAX-ELEMENTS to mean the minimum
;;;		height necessary to be able to view all rows of the table at 
;;;		once.  If the table can conceivable grow to be the full 
;;;		height of the screen, this value could be specified as
;;;		(height (root-window)).  Incidentally, large values for MAX-
;;;		ELEMENTS or MAX-HEIGHT have little noticeable effect on overall 
;;;		performance of the list-box.
;;;	MAX-WIDTH:
;;;		used instead of MAX-HEIGHT if :horizontal.
;;;	TITLE:
;;;		tile of the list-box.
;;;	FONT:	font of values.
;;;	TITLE-FONT:
;;;		font of title.
;;;	
;;;	In addition to the accessor methods inherited from table-fields, Any of 
;;;	VALUE, ITEMS, PAD, ROW-HEIGHT, COL-WIDTH, FONT, TITLE, or TITLE-FONT 
;;;	may be set dynamically.
;;;

(defclass list-box (table-field)
  ((orientation 
    :initarg :orientation  
    :initform :vertical
    :type keyword
    :reader orientation)
   (pad 
    :initarg :pad  
    :initform 0
    :type integer
    :reader pad)
   (height-specified 
    :initform nil
    :type atom
    :reader height-specified)
   (font :initform "8x13")))

;;;
;;;	Accessor Methods
;;;

(defmethod title ((self list-box))
  (if (eq (orientation self) :horizontal)
      (row-titles self)
      (col-titles self)))

(defmethod (setf title) (val (self list-box))
  (if (eq (orientation self) :horizontal)
      (setf (row-titles self) val) 
      (setf (col-titles self) val)))

(defmethod (setf value) (val (self list-box))
  (when (listp val)
        (if (eq (orientation self) :horizontal)
            (setq val (list val))
            (setq val (mapcar #'list val)))
        ; (setf (slot-value self 'value) val)
        (call-next-method val self)))

(defmethod font ((self list-box))
  (font (matrix-field self)))

(defmethod (setf font) (val (self list-box))
  (setf (font (matrix-field self)) val))

(defmethod col-width ((self list-box))
  (aref (md-col-current (gm-data (matrix-field self))) 0))

(defmethod (setf col-width) (val (self list-box) &aux mf)
  (when (eq (orientation self) :horizontal) 
	(setq mf (matrix-field self))
	(dotimes (c (cols mf))
		 (setf (aref (md-col-current (gm-data mf)) c)
		       val))
	(do-repack mf)
	(mf-put mf)
	val))

(defmethod row-height ((self list-box))
  (aref (md-row-current (gm-data (matrix-field self))) 0))

(defmethod (setf row-height) (val (self list-box) &aux mf)
  (when (eq (orientation self) :vertical)
	(setq mf (matrix-field self))
	(dotimes (r (rows mf))
		 (setf (aref (md-row-current (gm-data mf)) r)
		       val))
	(do-repack mf)
	(mf-put mf)
	val))

(defmethod (setf pad) (val (self list-box)) 
  (setf (row-height self) (+ (height (font self)) val)))

;;;
;;;	If you specify :max-height, there will be as many rows as can fit in 
;;;	that height.
;;;

(defun make-list-box (&rest keys
		      &key
		      (value 		nil)
		      (data		nil)
		      (items		nil)
		      (grid-lines	nil)
		      (pad		0)
		      (col-width	nil)
		      (row-height	nil)
		      (orientation	:vertical)
		      (max-elements	nil)
		      (max-height	nil)
		      (max-width	nil)
		      (title		nil)
		      (font		"8x13")
		      (title-font	nil)
		      &allow-other-keys &aux rh cw)
  ;;	Filter out bad keywords before passing them on to table-field
  (remf keys :data)
  (remf keys :value)
  (remf keys :grid-lines)
  (when (or (getf keys :col-elements) (getf keys :row-elements)
	    (getf keys :row-title-elements)
	    (getf keys :col-title-elements))
	(warn 
	 "list-box: cannot specify the row/col/title-elements of a list-box.")
	(remf keys :col-elements)
	(remf keys :row-elements)
	(remf keys :col-title-elements)
	(remf keys :row-title-elements))

  (when data (setq value data))
  (when items (setq value items))
  (unless (font-p font)
	  (setq font (make-font :name font)))
  (unless font
	  (warn "make-list-box: illegal font \'~s\'.  Defaulting to 8x13" font))
  (if (and (null row-height) (not (attached-p font)))
      (setq row-height 13))
  (case orientation
	(:horizontal
	 (setq cw (if col-width col-width 
		      (+ (apply 
			  #'max 
			  (mapcar #'(lambda (x) (width x :font font)) 
				  value))
			 pad)))
	 (apply #'make-instance 
		'list-box
		:allow-other-keys t
		`(:data ,(list value)
		  :row-titles ,(if title (list title) nil)
		  :col-widths ,cw
		  :row-title-font ,title-font
		  :grid-lines ,grid-lines
		  :height-specified ,row-height
		  :rows 1
		  :cols ,(cond (max-elements max-elements)
			       (max-width (round (/ (- max-width 2)
						    (+ cw 2))))
			       (t 5))
		  ,@keys)))
	(t
	 (setq rh (if row-height row-height (+ (font-height font) pad)))
	 (apply #'make-instance 
		'list-box
		:allow-other-keys t
		`(:data ,(mapcar #'list value)
		  :col-titles ,(if title (list title) nil)
		  :cols 1
		  :row-heights ,rh
		  :height-specified ,row-height
		  :col-title-font ,title-font
		  :grid-lines ,grid-lines
		  :rows ,(cond (max-elements max-elements)
			       (max-height (round (/ (- max-height 2)
						     (+ rh 2))))
			       (t 30))
		  ,@keys)))))

(defmethod do-attach ((self list-box))
  (call-next-method)
  (if (not (height-specified self))
      (setf (row-height self) (+ (font-height (font self)) (pad self)))))
