;;;
;;; Copyright (c) 1990 Regents of the University of California
;;; 
;;; $Author: picasso $
;;; $Source: RCS/indicator.cl,v $
;;; $Revision: 1.1 $
;;; $Date: 90/07/18 09:17:05 $
;;;
;;; ACCESSORS AND INSTANTIATION
;;;
;;; The following accessors are provided for customization of the meter-widget:
;;; 	low	-	lower bound (number)
;;;	high	-	upper bound (number)
;;;	value	-	current position of locator relative to low and high
;;;				(number)
;;; NOTE: when value is nil, the locator is not drawn.
;;;

(in-package 'pt  :nicknames '(picasso-toolkit) :use '(lisp excl pcl))

(defclass indicator (widget)
  ((low
    :initarg :low 
    :initform 0
    :type number
    :reader low)
   (high
    :initarg :high 
    :initform 1
    :type number
    :reader high)
   (value
    :initarg :value 
    :initform nil
    :type number
    :reader value)
   (pad
    :initarg :pad 
    :initform 0
    :type integer
    :reader pad)
   (font :initform nil :type font :accessor font)
   (update-flag				;;  Like repaint-flag or repack-flag.
    :initarg :update-flag				 
    :initform t				;;  Use when changing more than one
    :type atom				;;  attribute at a time.
    :reader update-flag)
;;	-----------------------------------------------------------------
;;					;;  Internal use only
   (gc-spec :initform '((gc-res (:foreground "white" :background "black"))))
   (foreground :initform "white")
   (background :initform "black")
   (border-width :initform 0)
   (base-height :initform 25)))

(defun make-indicator (&rest args)
  (apply #'make-instance 'indicator :allow-other-keys t args))

(defmacro indicator-p (win)
  `(typep ,win 'indicator))

;;;
;;;	Accessors signal repaint
;;;

(defmethod (setf value) (val (self indicator))
  (setf (slot-value self 'value) val)
  (repaint self))

(defmethod (setf low) (val (self indicator))
  (if (null val) (setq val 0))
  (setf (slot-value self 'low) val)
  (when (update-flag self)
	(repaint self)))

(defmethod (setf high) (val (self indicator))
  (if (null val) (setq val 0))
  (setf (slot-value self 'high) val)
  (when (update-flag self)
	(repaint self)))

(defmethod (setf update-flag) (val (self indicator))
  (setf (slot-value self 'update-flag) val))

(defmethod resize-window-handler ((self indicator))
  (repaint self))

;;;
;;;	Instantiate a new indicator
;;;

(defmethod new-instance ((self indicator) &rest args
			 &key
			 (low 0)
			 (high 1)
			 (font (get-font)))
  (declare (ignore args))
  (call-next-method)
  (setf (low self) low)
  (setf (high self) high)
  (setf (font self) font)
  (make-image :name "indicator" :file "gray-triang.bitmap")
  )

;;;
;;;	Draw indicator
;;;

(defmethod do-repaint ((self indicator))
  (call-next-method)
  
  ;; There is an assumption that the width of the "indicator" bitmap
  ;; is 16 pixels.  This is probably bad, but at least now it's documented.
  (let* ((res (res self))
	 (gc (gc-res self))
	 (w (width self))

	 ;; the values to be indicated.
	 (val (value self))
	 (low (low self))
	 (high (high self))
	 
	 ;; since val is not restricted to be between low and high,
	 ;; find the real max and min.
	 (max (if val (max low high val) (max low high)))
	 (min (if val (min low high val) (min low high)))
	 
	 ;; add 8 to pad because of width of "indicator" bitmap.
	 (pad (+ (pad self) 8))
	 
	 ;; calculate scale, the pixel/value ratio.
	 (dval (- max min))
	 (dpix (- w pad pad))
	 (scale (/ (max 0 dpix) (max 1 dval)))

	 )
    (when (and (> dpix 16) low high)
	  (let* (;; pixel x-positions of values.
		 (low-pos (truncate (+ pad (* (- low min) scale))))
		 (high-pos (truncate (+ pad (* (- high min) scale))))
		 )
	    ;;	Draw horizontal line
	    (xlib:draw-line res gc
			    pad  18 (- w pad) 18)
	    ;; draw low and high.
	    (xlib:draw-rectangle res gc (- low-pos 1) 15 2 5)
	    (xlib:draw-rectangle res gc (- high-pos 1) 15 2 5)
	    ))
    (when (and (> dpix 16) low high val)
	  (let* ((val-pos (truncate (+ pad -8 (* (- val min) scale))))
		 (font (font self))
		 (val-str (format nil "~d" val))
		 (half-text (truncate (text-width val-str :font font) 2))
		 )
	    ;; draw position.
	    (xlib:put-image res gc 
			    (res (get-image "indicator"))
			    :x val-pos :y 1
			    :bitmap-p t)
	    (put val-str :font font
		 :window self :horiz-just :left :vert-just :top
		 :mask t
		 :x (min (- w half-text half-text 8)
			 (max 8
			      (- val-pos half-text -8)
			      ))
		 :y (+ 20 (height font)))
	    ))
    ))
