;;;
;;; Copyright (c) 1990 Regents of the University of California
;;;
;;; Definition for shapes.  
;;;
;;; All shapes have a list of sub-objects, and a list of viewers.
;;;
;;; The list of sub-objects (sub-objs) is a list of the sub-parts of the
;;; object.  This gives gives a hook to a hierachal scene description.
;;; Add-object and delete-object are provided to manipulate this slot,
;;; and the setf method may be used as well.  Shapes provide a method 
;;; ("flatten") to flatten out this hierarchy.
;;;
;;; Shapes have a list of viewers associated with them.  When something
;;; about a shape changes, usually changing the geomoetry, color or the
;;; addition of sub-objects, these objects receive a geom-changed,
;;; gc-changed, sub-obj-added or sub-obj-deleted messages.  The
;;; list is manipulated through the add-viewer and delete-viewer methods.
;;;
;;; One other miscellaneous property is a name for the object.
;;;
;;; A few other miscellaneous methods all shapes provide are the copy method, 
;;; to get a copy of the shape and all its sub objects, and the bbox macro,
;;; that returns a bounding box of the shape. See bbox.cl for some
;;; routines to manipulate bounding boxes.
;;;

(in-package 'pt :use '(lisp pcl))

;;;
;;; Are we a shape? Isn't that special!
;;;
(defmacro shape-p (self)
  `(typep ,self 'shape))

;;;
;;; Class definition for shape
;;;
(defclass shape ()
  ((name :initarg :name :initform "" :type string :accessor name)
   (sub-objs :initarg :sub-objs :initform nil :type list :accessor sub-objs)
   (viewers :initarg :viewers :initform nil :type list :accessor viewers)))

(defun find-shape (root pathname &aux rv)
  (cond ((null pathname) nil)
	((and (null (cdr pathname)) (equal (car pathname) (name root))) root)
	((or (equal (car pathname) (name root))
	     (equal (car pathname) "*"))
	 (do* ((so-list (sub-objs root) (cdr so-list))
	       (so (car so-list) (car so-list)))
	      ((or (null so) (setq rv (find-shape so (cdr pathname)))) rv)))
	(t nil)))

(defun make-shape (&rest keys)
  (apply #'make-instance (cons 'shape keys)))

;;;
;;; Handle construction of sub-objects
;;;
(defmethod new-instance :after ((self shape) 
				&key 
				(sub-objs nil) 
				&allow-other-keys)
  ;; Remove the make-functions from the sub-object list
  (setf (slot-value self 'sub-objs)
	(delete-if #'consp (slot-value self 'sub-objs)))
  ;; Create sub-objs using the make-functions.
  (dolist (so sub-objs) 
	  (if (consp so)
	      (add-object
	       self
	       (apply (symbol-function (car so)) (mapcar #'eval (cdr so)))))))

;;; ======================= Notify methods ====================

(defmethod notify-gc-change ((self shape))
  (mapc #'(lambda (viewer) (gc-changed viewer self))
	(viewers self)))

(defmethod notify-geom-change ((self shape))
  (mapc #'(lambda (viewer) (geom-changed viewer self))
	(viewers self)))

;;; ======================= Viewer methods ====================
;;;
;;; Add a viewer to the viewer list.
;;;
(defmethod add-viewer ((self shape) viewer)
  (pushnew viewer (viewers self)))

(defmethod add-viewer-recursively ((self shape) viewer)
  (add-viewer self viewer)
  (dolist (obj (sub-objs self))
	  (add-viewer-recursively obj viewer)))

;;;
;;; Delete a viewer from the viewer list.
;;;
(defmethod delete-viewer ((self shape) viewer)
  (setf (viewers self) (delete viewer (viewers self))))

(defmethod delete-viewer-recursively ((self shape) viewer)
  (delete-viewer self viewer)
  (dolist (obj (sub-objs self))
	  (delete-viewer-recursively obj viewer)))

;;; ================ Sub-obj hierarchy functions ==============
;;;
;;; Whenever the sub-objs of an object change, it's viewers are notified
;;; with sub-obj-added and sub-obj-deleted messages.
;;;
(defmethod (setf sub-objs) (value (self shape))
  (unless (equalp value (sub-objs self))
	  (let ((add-objs (set-difference value (sub-objs self)))
		(del-objs (set-difference (sub-objs self) value)))
	       (setf (slot-value self 'sub-objs) value)
	       (dolist (v (viewers self))
		       (dolist (obj del-objs)
			       (sub-obj-deleted v self obj))
		       (dolist (obj add-objs)
			       (sub-obj-added v self obj))))))

;;;
;;; Add an object to the sub-object list.
;;;
(defmethod add-object ((self shape) (obj shape))
  (pushnew obj (sub-objs self)))

;;;
;;; Delete an object from the sub-object list.
;;;
(defmethod delete-object ((self shape) (obj shape))
  (setf (sub-objs self) (delete obj (sub-objs self))))

;;;
;;; Return a flattened out version of the shape hierarchy implied by the
;;; sub-objects of this shape.
;;;
(defmethod flatten ((self shape))
  (let ((rv (list self)))
       (dolist (obj (sub-objs self))
	       (setq rv (nconc (flatten obj) rv)))
       rv))

(defmethod copy ((self shape))
  (let ((rv (make-instance (class-name (class-of self)) :name (name self)
					:viewers (copy-list (viewers self))))
	(sub-objs nil))
       (dolist (obj (sub-objs self))
	       (setq sub-objs (nconc sub-objs (list (copy obj)))))
       (setf (sub-objs rv) sub-objs)
       rv))
