;;;
;;; Copyright (c) 1990 Regents of the University of California
;;;
;;; simple-graphic-gadget: A very fast graphic gadget.
;;;
;;; A simple-graphic-gadget can display lines, annotations and symbols.
;;; Lines are line segments, annotations are scalable text and symbols
;;; are scalable bitmaps.
;;; 
;;; The speed is derived from the static nature of the data structures.
;;; The typical use is to load it up witha  bunch of objects, then
;;; "initialize" the gadget.  This function precomputes everything possible.
;;;
;;; $Author: bsmith $
;;; $Source: RCS/sgg-gadget.cl,v $
;;; $Revision: 1.2 $
;;; $Date: 90/08/09 14:35:51 $
;;;

(in-package 'pt :use '(lisp pcl))

;;;
;;; Class definition for simple-graphic-gadget (sgg)
;;;
(defclass simple-graphic-gadget (2d-mapper-mixin gadget) 
  (
   ; Inherited slots -- initial-values
   (background-pixmap :initform "black")
   (background :initform "black")
   (foreground :initform "white")
   (border-width :initform 0)
   
   ; The data to display in the sgg.
   (lines :initform nil :type array :accessor lines)
   (annots :initform nil :type array :accessor annots)
   (symbols :initform nil :type array :accessor symbols)
   (symbol-types :initform nil :type array :accessor symbol-types)
   (objects :initform nil :type array :accessor objects)
   
   ; Available graphic props.
   (font-list :initform nil :type list :reader font-list)
   (color-list :initform nil :type list :reader color-list)
   (colors :initform nil :type list :reader colors)
   (width-list :initform nil :type list :reader width-list)
   (sel-font-list :initform nil :type list :reader sel-font-list)
   (sel-color :initform nil :type list :reader sel-color)
   (sel-width :initform nil :type list :reader sel-width)
   (selection-highlight  :initform t :accessor selection-highlight)
   
   ;----------------------------------------------------------------
   ; Internal slots.
   ; 
   (vertex-ids :initform nil :type array :accessor vertex-ids)
   (vertices :initform nil :type array :accessor vertices)
   (mapped-vertices :initform nil :type array :accessor mapped-vertices)
   
   ; The extent (xmin, ymin, xmax ymax) of the data.
   (extent :initform nil :type list :accessor extent)
   
   ; Map from (color . font) --> gc and (color . width) -> gc
   (c&f->gc :initform (make-hash-table :test #'equal) :reader color&font->gc)
   (c&w->gc :initform (make-hash-table :test #'equal) :reader color&width->gc)
   
   ; Map from fgo-id->index
   (fgo-id->idx :initform (make-hash-table) :reader fgo-id->idx)))

;;;
;;; Constructor Function...
;;;
(defun make-simple-graphic-gadget (&rest keys)
  (apply #'make-instance 'simple-graphic-gadget :allow-other-keys t keys))

(defmacro make-sgg (&rest keys)
  `(make-simple-graphic-gadget ,@keys))

(defmethod (setf fonts) (fonts (self simple-graphic-gadget))
  ;; Initialize the fonts, heights, etc.
  (setf (slot-value self 'font-list)
	(mapcar #'(lambda (x) (make-font :name x)) fonts)))

(defmethod (setf sel-fonts) (sel-fonts (self simple-graphic-gadget))
  ;; Initialize the fonts, heights, etc.
  (setf (slot-value self 'sel-font-list)
	(mapcar #'(lambda (x) (make-font :name x)) sel-fonts)))

(defmethod (setf sel-width) (sel-width (self simple-graphic-gadget))
  (setf (slot-value self 'sel-width) sel-width))

(defmethod (setf sel-color) (sel-color (self simple-graphic-gadget))
  (setf (slot-value self 'sel-color) (make-color :name sel-color)))

(defmethod (setf widths) (widths (self simple-graphic-gadget))
  (setf (slot-value self 'width-list) widths))

(defmethod (setf colors) (colors (self simple-graphic-gadget))
  ;; Initialize the fonts, widths, colors, etc.
  (setf (slot-value self 'color-list)
	(mapcar #'(lambda (x) (make-color :name x)) colors))
  (setf (slot-value self 'colors) colors)
  )

;;
;; Set the vertices of the sgg.  Should be a list of (id x y) triplets.
;;
(defmethod (setf vertices) (vertices (self simple-graphic-gadget))
  (let* ((num-verts (length vertices)))
	;; Make of the array of id's
	(setf (slot-value self 'vertex-ids)
	      (coerce (mapcar #'car vertices) 'array))
	(setq vertices (apply #'nconc (mapcar #'cdr vertices)))
	(incf num-verts num-verts)
	(do* ((v vertices (cdr v)))
	     ((null v))
	     (setf (car v) (coerce (car v) 'single-float)))
	(setf (slot-value self 'vertices)
	      (make-array num-verts :element-type 'single-float
			  :initial-contents vertices))
	(setf (slot-value self 'mapped-vertices)
	      (make-array num-verts :element-type 'fixnum :initial-element 0))))

;;;
;;; Set the lines of the sgg.  Should be a list of (id v1 v2) triplets.
;;;
(defmethod (setf lines) (lines (self simple-graphic-gadget))
  (let* ((num-lines (length lines))
	 (array (make-array num-lines)))
	;; Make of the array of id's
	(do* ((i 0 (1+ i))
	     (l lines (cdr l))
	     (l-spec (car l) (car l)))
	    ((null l))
	    (setf (aref array i) (make-fg-line :id (first l-spec)
					       :start-vertex-id (second l-spec)
					       :end-vertex-id (third l-spec))))
	(setf (slot-value self 'lines) array)))

;;;
;;; Set the annotations of the sgg.  Should be a list of property lists:
;;; (:id 5 :text "sample" :vertex-id 'v2 :width 5 :height 5 :horiz-just :center 
;;;     :vert-just:top) 
;;;
(defmethod (setf annotations) (annots (self simple-graphic-gadget))
  (let* ((num-annots (length annots))
	 (array (make-array num-annots)))
	 ; Make of the array of id's
	(do* ((i 0 (1+ i))
	     (a annots (cdr a))
	     (a-spec (car a) (car a)))
	    ((null a))
	    ;; Verify that we have a good a-spec:
	    ;;	o Text is a string.
	    ;;	o Vertex-id is non-nil
	    ;;	o width/height are numbers > 0
	    ;;  o :horiz-just is one of :left, :center, :right
	    ;;  o :vert-just is one of :top, :center, :bottom
	    ;;  o id is non-nil, if nil, make it the text.
	    (if (and (stringp (getf a-spec :text))
		     (getf a-spec :vertex-id)
		     (numberp (getf a-spec :width))
		     (> (getf a-spec :width) 0)
		     (numberp (getf a-spec :height))
		     (> (getf a-spec :height) 0)
		     (and (member (getf a-spec :horiz-just)
				  '(nil :left :center :right)))
		     (and (member (getf a-spec :vert-just)
				  '(nil :top :center :bottom))))
		(setf (aref array i)
		      (if (getf a-spec :id)
			  (apply #'make-fg-annot a-spec)
			  (apply #'make-fg-annot :id (getf a-spec :text) a-spec)
			  ))
		(progn
		 (setf (aref array i) 
		       (make-fg-annot :text (or (getf a-spec :text) "")
				      :width 0 :height 0))
		 (warn "Illegal annotation spec ~s~%" a-spec)
		 (warn "passed to setf (annotations ~s)~%" self))))
	(setf (slot-value self 'annots) array)))

;;;
;;; Set the symbols of the sgg.  Should be a list of property lists:
;;; (:id 5 :symbol-type-id 1 :vertex-id 'v2 :horiz-just :center :vert-just :top) 
;;;
(defmethod (setf symbols) (symbols (self simple-graphic-gadget))
  (let* ((num-symbols (length symbols))
	 (array (make-array num-symbols)))
	;; Make of the array of id's
	(do* ((i 0 (1+ i))
	      (s symbols (cdr s))
	      (s-spec (car s) (car s)))
	     ((null s))
	     ;; Verify that we have a good s-spec:
	     ;;	o symbol-type-id is non-nil
	     ;;	o Vertex-id is non-nil
	     ;;  o :horiz-just is one of :left, :center, :right
	     ;;  o :vert-just is one of :top, :center, :bottom
	     ;;  o id is non-nil.
	     (unless (and (getf s-spec :symbol-type-id)
			  (getf s-spec :vertex-id)
			  (getf s-spec :id)
			  (and (member (getf s-spec :horiz-just)
				       '(nil :left :center :right)))
			  (and (member (getf s-spec :vert-just)
				       '(nil :top :center :bottom))))
		     (warn "Illegal symbol spec ~s~%" s-spec)
		     (warn "passed to setf:(symbol ~s)~%" self))
	     (setf (aref array i) (apply #'make-fg-symbol s-spec)))
	(setf (slot-value self 'symbols) array)))

;;;
;;; Set the symbol-types of the sgg.  Should be a list of property lists:
;;; (:id 5 :bitmaps <list of images> :width 5.0 :height 10.0)
;;;
(defmethod (setf symbol-types) (symbol-types (self simple-graphic-gadget))
  (let* ((num-symbols (length symbol-types))
	 (bm-list nil)
	 (array (make-array num-symbols)))
	;; Make of the array of id's
	(do* ((i 0 (1+ i))
	      (st symbol-types (cdr st))
	      (st-spec (car st) (car st)))
	     ((null st))
	     ;; Verify that we have a good st-spec:
	     ;;	o bitmaps is a non-nil list of images
	     ;;  o width and height are numbers and > 0
	     ;;  o id is non-nil.
	     (setq bm-list (getf st-spec :bitmaps))
	     (unless (and bm-list
			  (every #'(lambda (x) (typep x 'image)) bm-list)
			  (numberp (getf st-spec :width))
			  (> (getf st-spec :width) 0)
			  (numberp (getf st-spec :height))
			  (> (getf st-spec :height) 0)
			  (getf st-spec :id))
		     (warn "Illegal symbol-type spec ~s~%" st-spec)
		     (warn "passed to setf:(symbol-types ~s)~%" self))
	     (setf (aref array i) (apply #'make-fg-symbol-type st-spec)))
	(setf (slot-value self 'symbol-types) array)))

;;;
;;; Set the objects of the sgg.  Should be a list of property lists:
;;; (:id 5 :line-ids <list of line id's> :annot-ids <list of annot-id's>
;;;  :symbol-ids <list of symbol-id's> :visible t :selected t
;;;  :selectable t :line-width 5 :color "blue")
;;;
(defmethod (setf objects) (objects (self simple-graphic-gadget))
  (let* ((num-objects (length objects))
	 (array (make-array num-objects)))
	;; Make of the array of id's
	(do* ((i 0 (1+ i))
	     (o objects (cdr o))
	     (o-spec (car o) (car o)))
	    ((null o))
	    (setf (aref array i) (apply #'make-fg-object o-spec)))
	(setf (slot-value self 'objects) array)))

;;;
;;; Create all the gc's for this guy...
;;; XXX Need to get clip-mask from default gc.
;;;
(defun init-gcs (self)
  (if (attached-p self)
      (let ((c&f->gc (color&font->gc self))
	    (c&w->gc (color&width->gc self))
	    (col nil)
	    (res (res self))
	    (gc nil))
	   (let* ((sel-width (sel-width self))
		  (sel-color (sel-color self))
		  (sel-col (name sel-color)))
		 (dolist (sel-font (sel-font-list self))
			 (setq gc (xlib:create-gcontext
				   :drawable res
				   :font (res sel-font)
				   :foreground (pixel sel-color)
				   :line-width sel-width))
			 (setf (gethash (cons sel-col sel-width) c&w->gc) gc)
			 (setf (gethash (cons sel-col sel-font) c&f->gc) gc)))
	   (do* ((widths (width-list self) (cdr widths))
		 (fonts (font-list self) (cdr fonts))
		 (width (car widths) (car widths))
		 (font (car fonts) (car fonts)))
		((and (null width) (null font)))
		(dolist (color (color-list self))
			(setq col (name color))
			(cond ((and width font)
			       (setq gc (xlib:create-gcontext
					 :drawable res
					 :font (res font)
					 :foreground (pixel color)
					 :line-width width))
			       (setf (gethash (cons col width) c&w->gc) gc)
			       (setf (gethash (cons col font) c&f->gc) gc))
			      (width
			       (setq gc (xlib:create-gcontext
					 :drawable res
					 :foreground (pixel color)
					 :line-width width))
			       (setf (gethash (cons col width) c&w->gc) gc))
			      (font
			       (setq gc (xlib:create-gcontext
					 :drawable res
					 :font (res font)
					 :foreground (pixel color)))
			       (setf (gethash (cons col font) c&f->gc) gc))))
		))))

(defmethod new-instance ((self simple-graphic-gadget)
			 &key
			 (vertices nil)
			 (lines nil)
			 (annotations nil)
			 (symbols nil)
			 (symbol-types nil)
			 (objects nil)
			 (colors '("white" "yellow" "red" "orange"
					   "green" "cyan" "blue" "magenta"))
			 (widths '(0 1 2 3 4 5))
			 (fonts
			  '("*helvetica-medium-r-*--34*"
			    "*helvetica-medium-r-*--20*"
			    "*helvetica-medium-r-*--14*"
			    "*helvetica-medium-r-*--10*"
			    "*helvetica-medium-r-*--8*"
			    "nil2"))
			 (sel-fonts
			  '("*helvetica-medium-o-*--34*"
			    "*helvetica-medium-o-*--20*"
			    "*helvetica-medium-o-*--14*"
			    "*helvetica-medium-o-*--10*"
			    "*helvetica-medium-o-*--8*"
			    "nil2"))
			 (sel-color "white")
			 (sel-width 0)
			 &allow-other-keys)

  ;; Initialize the fonts, widths, colors, etc.
  (setf (fonts self) fonts)
  (setf (colors self) colors)
  (setf (widths self) widths)
  (setf (sel-width self) sel-width)
  (setf (sel-color self) sel-color)
  (setf (sel-fonts self) sel-fonts)

  (setf (vertices self) vertices)
  (setf (lines self) lines)
  (setf (annotations self) annotations)
  (setf (symbols self) symbols)
  (setf (symbol-types self) symbol-types)
  (setf (objects self) objects)

  ;; Let mapper initialize itself.
  (call-next-method)
  self)

(defmethod do-detach ((self simple-graphic-gadget))
  ;; Detach all the fonts, colors
  (mapc #'detach (color-list self))
  (detach (sel-color self))
  (let ((st nil)
	(symbol-types (symbol-types self)))
       (dotimes (i (length (symbol-types self)))
		(setq st (aref symbol-types i))
		(mapc #'detach (fgst-bitmaps st))))

  (call-next-method))

;;;
;;; Other methods that affect the world coordinate system
;;;
(defmethod zoom-extent ((self simple-graphic-gadget) &aux bb)
  (setq bb (extent self))
  (let* ((xmin (first bb))
	 (ymin (second bb))
	 (xmax (third bb))
	 (ymax (fourth bb))
	 (dx (- xmax xmin))
	 (dy (- ymax ymin)))
	(set-world self 
		   (- xmin (/ dx 20.0)) (- ymin (/ dy 20.0)) 
		   (+ xmax (/ dx 20.0)) (+ ymax (/ dy 20.0)))))

;;;
;;; Recacluate the world to device mapping function of a simple-graphic-gadget.
;;;
(defmethod recache-map ((self simple-graphic-gadget))
  (call-next-method)
  (with-slots (vertices mapped-vertices mx my bx by) self
	      (when vertices
		    (map-float vertices mapped-vertices mx my bx by)
		    (repaint self))))

(defmethod resize-window-handler ((self simple-graphic-gadget))
  (recache-map self))

;;;
;;; Initialize *all* the data structures of the browser.  Should be called
;;; only after all objects, lines, vertices, colros, fonts, etc. have
;;; been set.
;;;
(defun initialize-sgg (self)
  ;; Attach all the fonts, colors and bitmaps
  (mapc #'attach (color-list self))
  (mapc #'attach (font-list self))
  (mapc #'attach (sel-font-list self))
  (attach (sel-color self))
  (let ((st nil)
	(symbol-types (symbol-types self)))
       (dotimes (i (length (symbol-types self)))
		(setq st (aref symbol-types i))
		(mapc #'attach (fgst-bitmaps st))
		(setf (fgst-widths st) (mapcar #'width (fgst-bitmaps st)))
		(setf (fgst-heights st) (mapcar #'height (fgst-bitmaps st)))))
  
  (init-gcs self)
  (let* ((vids (vertex-ids self))
	 (xmin 0.0)
	 (ymin 0.0)
	 (xmax 1.0)
	 (ymax 1.0)
	 (lines (lines self))
	 (annots (annots self))
	 (symbols (symbols self))
	 (sym-types (symbol-types self))
	 (objects (objects self))
	 (fonts (mapcar #'res (font-list self)))
	 (sel-fonts (mapcar #'res (sel-font-list self)))
	 (vertex-ht (make-hash-table :size (length vids)))
	 (annot-ht (make-hash-table :size (length annots)))
	 (line-ht (make-hash-table :size (length lines)))
	 (symbol-ht (make-hash-table :size (length symbols)))
	 (sym-types-ht (make-hash-table :size (length sym-types)))
	 (fgo-id->idx (fgo-id->idx self))
	 (lookup-vid #'(lambda (id) (gethash id vertex-ht)))
	 (lookup-lid #'(lambda (id) (gethash id line-ht)))
	 (lookup-aid #'(lambda (id) (gethash id annot-ht)))
	 (lookup-sid #'(lambda (id) (gethash id symbol-ht)))
	 (lookup-stid #'(lambda (id) (gethash id sym-types-ht))))
	
	;; Make up all the maps so we can resolve everything...
	(dotimes (i (length vids))
		 (setf (gethash (aref vids i) vertex-ht) i))
	(dotimes (i (length lines))
		 (setf (gethash (fgl-id (aref lines i)) line-ht) i))
	(dotimes (i (length annots))
		 (setf (gethash (fga-id (aref annots i)) annot-ht) i))
	(dotimes (i (length symbols))
		 (setf (gethash (fgs-id (aref symbols i)) symbol-ht) i))
	(dotimes (i (length sym-types))
		 (setf (gethash (fgst-id (aref sym-types i)) sym-types-ht) i))
	(dotimes (i (length objects))
		 (setf (gethash (fgo-id (aref objects i)) fgo-id->idx) i))
	
	;; No go through the annots, lines, etc, caching what we can...
	(dotimes (i (length annots))
		 (let* ((annot (aref annots i))
			(text (fga-text annot)))
		       (setf (fga-vertex annot)
			     (funcall lookup-vid (fga-vertex-id annot)))
		       (let
			(widths ascents descents heights w a d)
			(dolist (f fonts)
				(multiple-value-setq
				 (w a d)
				 (xlib:text-extents f text))
				(push w widths)
				(push a ascents)
				(push d descents)
				(push (+ a d 1) heights))
			
			(setf (fga-font-widths annot) (nreverse widths))
			(setf (fga-font-heights annot) (nreverse heights))
			(setf (fga-font-ascents annot) (nreverse ascents))
			(setf (fga-font-descents annot) (nreverse descents))
			)
		       (let
			(widths ascents descents heights w a d)
			(dolist (f sel-fonts)
				(multiple-value-setq
				 (w a d)
				 (xlib:text-extents f text))
				(push w widths)
				(push a ascents)
				(push d descents)
				(push (+ a d 1) heights))
			
			(setf (fga-sel-font-widths annot) (nreverse widths))
			(setf (fga-sel-font-heights annot) (nreverse heights))
			(setf (fga-sel-font-ascents annot) (nreverse ascents))
			(setf (fga-sel-font-descents annot) (nreverse descents))
			)
		       (if (null (fga-vertex annot)) 
			   (warn "Vertex ~s in annotation ~s not found~%"
				 (fga-vertex-id annot) annot))))
	
	(dotimes (i (length lines))
		 (let* ((line (aref lines i)))
		       (setf (fgl-start-vertex line) 
			     (funcall lookup-vid (fgl-start-vertex-id line)))
		       (setf (fgl-end-vertex line) 
			     (funcall lookup-vid (fgl-end-vertex-id line)))
		       (if (null (fgl-start-vertex line)) 
			   (warn "Vertex ~s in line ~s not found~%"
				 (fgl-start-vertex-id line) line))
		       (if (null (fgl-end-vertex line)) 
			   (warn "Vertex ~s in line ~s not found~%"
				 (fgl-end-vertex-id line) line))
		       ))
	
	(dotimes (i (length symbols))
		 (let* ((symbol (aref symbols i)))
		       (setf (fgs-symbol-type symbol)
			     (funcall lookup-stid (fgs-symbol-type-id symbol)))
		       (setf (fgs-vertex symbol)
			     (funcall lookup-vid (fgs-vertex-id symbol)))
		       (if (null (fgs-vertex symbol)) 
			   (warn "Vertex ~s in symbol ~s not found~%"
				 (fgs-vertex-id symbol) symbol))
		       (if (null (fgs-symbol-type symbol)) 
			   (warn "Symbol-type ~s in symbol ~s not found~%"
				 (fgs-symbol-type-id symbol) symbol))
		       ))
	
	;; Doubly link the object to the lines, annotations and symbols.
	(dotimes (i (length objects))
		 (let* ((object (aref objects i))
			(oid (fgo-id object))
			(o-line-ids (fgo-line-ids object))
			(o-annot-ids (fgo-annot-ids object))
			(o-symbol-ids (fgo-symbol-ids object)))
		       (setf (fgo-lines object) (mapcar lookup-lid o-line-ids))
		       (setf (fgo-annots object) (mapcar lookup-aid o-annot-ids))
		       (setf (fgo-symbols object)
			     (mapcar lookup-sid o-symbol-ids))
		       (dolist (idx (fgo-lines object))
			       (pushnew oid (fgl-objects (aref lines idx))))
		       (dolist (idx (fgo-symbols object))
			       (pushnew oid (fgs-objects (aref symbols idx))))
		       (dolist (idx (fgo-annots object))
			       (pushnew oid (fga-objects (aref annots idx))))))
	
	;; Compute the extent...
	(compute-object-extent self)
	(dotimes (i (length objects))
		 (let* ((object (aref objects i))
			(extent (fgo-extent object))
			(obj-xmin (first extent))
			(obj-ymin (second extent))
			(obj-xmax (third extent))
			(obj-ymax (fourth extent)))
		       (setq xmin (min xmin obj-xmin)
			     ymin (min ymin obj-ymin)
			     xmax (max xmax obj-xmax)
			     ymax (max ymax obj-ymax))))
	(setf (slot-value self 'extent) (list xmin ymin xmax ymax))
	
	(clrhash vertex-ht)
	(clrhash annot-ht)
	(clrhash line-ht)
	(clrhash symbol-ht)
	(clrhash sym-types-ht))
  self)

;;;
;;; Define a function to repaint the simple-graphic-gadget
;;;
(defun sgg-repaint (self res x-offset y-offset)
  (declare (optimize (speed 3) (safety 0))
	   (fixnum x-offset y-offset))
  (let 
   ((annots (slot-value self 'annots))
    (c&f->gc (slot-value self 'c&f->gc))
    (c&w->gc (slot-value self 'c&w->gc))
    (font-list (slot-value self 'font-list))
    (lines (slot-value self 'lines))
    (mapped-vertices (slot-value self 'mapped-vertices))
    (mx (slot-value self 'mx))
    (my (slot-value self 'my))
    (objects (slot-value self 'objects))
    (sel-color (name (slot-value self 'sel-color)))
    (sel-font-list (slot-value self 'sel-font-list))
    (sel-width (slot-value self 'sel-width))
    (symbol-types (slot-value self 'symbol-types))
    (symbols (slot-value self 'symbols))
    (o nil)
    (res-width (xlib:drawable-width res))
    (res-height (xlib:drawable-height res)))
   (declare (type (simple-array fg-object (*)) objects)
	    (type (simple-array fixnum (*)) mapped-vertices)
	    (type (simple-array fg-symbol (*)) symbols)
	    (type (simple-array fg-symbol-type (*)) symbol-types)
	    (type (simple-array fg-line (*)) lines)
	    (type (simple-array fg-annot (*)) annots)
	    (type single-float mx my)
	    (type hash-table c&f->gc c&w->gc))
   (dotimes (i (length objects))
	    (setq o (aref objects i))
	    (if (fgo-visible o)
		(let* ((selected (and (selection-highlight self)
				      (fgo-selected o)))
		       (width (if selected sel-width (fgo-line-width o)))
		       (color (if selected sel-color (fgo-color o)))
		       (fonts (if selected sel-font-list font-list))
		       (gc nil)
		       key sv ev line annot symbol st bitmaps text
		       st-idx bmw bmh bm-idx bm hj vj w h a d fh fw fa fd
		       font-idx x1 y1 x2 y2)
		      (dolist (idx (fgo-annots o))
			      (setq annot (aref annots idx)
				    sv (fga-vertex annot)
				    w (fga-width annot)
				    h (fga-height annot))
			      (incf sv sv)
			      (setq x1 (aref mapped-vertices sv)
				    y1 (aref mapped-vertices (1+ sv))
				    w (round (* w mx))
				    h (abs (round (* h my))))
			      (setq text (fga-text annot)
				    hj (fga-horiz-just annot)
				    vj (fga-vert-just annot)
				    fw (if selected
					   (fga-font-widths annot)
					   (fga-sel-font-widths annot))
				    fh (if selected
					   (fga-font-heights annot)
					   (fga-sel-font-heights annot))
				    fa (if selected
					   (fga-font-ascents annot)
					   (fga-sel-font-ascents annot))
				    fd (if selected
					   (fga-font-descents annot)
					   (fga-sel-font-descents annot)))
			      (setq font-idx nil)
			      (do* ((fwl fw (cdr fwl))
				    (fhl fh (cdr fhl))
				    (tw (car fwl) (car fwl))
				    (th (car fhl) (car fhl))
				    (i 0 (1+ i)))
				   ((or font-idx (null fwl) (null fhl)))
				   (if (and (>= w tw) (>= h th))
				       (setq font-idx i)))
			      (setf (fga-font-idx annot) font-idx)
			      (when font-idx
				    (setq key (cons color (nth font-idx fonts))
					  gc (gethash key c&f->gc)
					  w (nth font-idx fw)
					  a (nth font-idx fa)
					  d (nth font-idx fd))
				    (case hj
					  (:center (decf x1 (round w 2)))
					  (:right (decf x1 w)))
				    (case vj 
					  (:center (incf y1 (round (- a d) 2)))
					  (:bottom (decf y1 d))
					  (:top (incf y1 a)))
				    (incf x1 x-offset)
				    (incf y1 y-offset)
				    (xlib:draw-glyphs res gc x1 y1 text)
				    ))
		      (setq gc (gethash (cons color width) c&w->gc))
		      (dolist (idx (fgo-symbols o))
			      (setq symbol (aref symbols idx)
				    sv (fgs-vertex symbol)
				    st-idx (fgs-symbol-type symbol)
				    st (aref symbol-types st-idx)
				    w (fgst-width st)
				    h (fgst-height st))
			      (incf sv sv)
			      (setq x1 (aref mapped-vertices sv)
				    y1 (aref mapped-vertices (1+ sv))
				    w (round (* w mx))
				    h (abs (round (* h my))))
			      (incf x1 x-offset)
			      (incf y1 y-offset)
			      (when (and (< (- w) x1 (+ res-width w))
					 (< (- h) y1 (+ res-height h)))
				    (setq hj (fgs-horiz-just symbol)
					  vj (fgs-vert-just symbol)
					  bitmaps (fgst-bitmaps st)
					  bmw (fgst-widths st)
					  bmh (fgst-heights st))
				    (setq bm-idx nil)
				    (do* ((bmwl bmw (cdr bmwl))
					  (bmhl bmh (cdr bmhl))
					  (bmw (car bmwl) (car bmwl))
					  (bmh (car bmhl) (car bmhl))
					  (i 0 (1+ i)))
					 ((or bm-idx (null bmwl) (null bmhl)))
					 (if (and (>= w bmw) (>= h bmh))
					     (setq bm-idx i)))
				    (setf (fgs-bitmap-idx symbol) bm-idx)
				    (when bm-idx
					  (setq w (nth bm-idx bmw)
						h (nth bm-idx bmh)
						bm (nth bm-idx bitmaps))
					  (case hj
						(:center (decf x1 (round w 2)))
						(:right (decf x1 w)))
					  (case vj
						(:center (decf y1 (round h 2)))
						(:bottom (decf y1 h)))
					  (xlib:put-image res gc (res bm)
							  :x x1 :y y1
							  :width w :height h
							  :src-x 0 :src-y 0
							  :bitmap-p t))))
		      (dolist (idx (fgo-lines o))
			      (setq line (aref lines idx)
				    sv (fgl-start-vertex line)
				    ev (fgl-end-vertex line))
			      (incf sv sv)
			      (incf ev ev)
			      (setq x1 (aref mapped-vertices sv)
				    y1 (aref mapped-vertices (1+ sv))
				    x2 (aref mapped-vertices ev)
				    y2 (aref mapped-vertices (1+ ev)))
			      (incf x1 x-offset)
			      (incf y1 y-offset)
			      (incf x2 x-offset)
			      (incf y2 y-offset)
			      (xlib:draw-line res gc x1 y1 x2 y2)
			      ))))))

(defmethod do-repaint ((self simple-graphic-gadget)
		       &key
		       (clear t)
		       &allow-other-keys)
  (call-next-method)
  (if clear (clear self))
  (sgg-repaint self (res self) (repaint-x self) (repaint-y self)))

(defun set-fgo-color (self to-change color
			   &aux (ht (fgo-id->idx self))
			   (objects (objects self)) o-idx)
  (if (not (listp to-change))
      (setq to-change (list to-change)))
  (if (color-p color) (setq color (name color)))
  (if (member color (colors self) :test #'string=)
      (dolist (object to-change)
	      (setq o-idx (gethash object ht))
	      (if o-idx
		  (setf (fgo-color (aref objects o-idx)) color)
		  (warn "Attempt to set color of unknown object ~s~%" object))
	      )))

(defun set-fgo-line-width (self to-change line-width
				&aux (ht (fgo-id->idx self))
				(objects (objects self)) o-idx)
  (if (not (listp to-change))
      (setq to-change (list to-change)))
  (if (member line-width (width-list self))
      (dolist (object to-change)
	      (setq o-idx (gethash object ht))
	      (if o-idx
		  (setf (fgo-line-width (aref objects o-idx)) line-width)
		  (warn "Attempt to set line-width of unknown object ~s~%"
			object)))))

(defun set-fgo-visible (self to-change visible
			     &aux (ht (fgo-id->idx self))
			     (objects (objects self)) o-idx)
  (declare (optimize (speed 3) (safety 0))
	   (type hash-table ht))
  (if (not (listp to-change))
      (setq to-change (list to-change)))
  (dolist (object to-change)
	  (setq o-idx (gethash object ht))
	  (if o-idx
	      (setf (fgo-visible (aref objects o-idx)) visible)
	      (warn "Attempt to set visibilty of unknown object ~s~%" object))))

(defun find-fgo (self id &aux o-idx)
  (setq o-idx (gethash id (fgo-id->idx self)))
  (if o-idx
      (aref (objects self) o-idx)
      nil))
