;;;
;;; Copyright (c) 1990 Regents of the University of California
;;;
;;; This file contains the structure definitions for graphic objects
;;; that are used by simple-graphic-gadgets and simple-graphic-browsers.
;;;
;;; $Author: bsmith $
;;; $Source: RCS/sgg-data-utils.cl,v $
;;; $Revision: 1.2 $
;;; $Date: 90/08/14 17:33:03 $
;;;

(in-package 'pt :use '(lisp pcl))

(defstruct (fg-annot (:conc-name fga-)) 
  id			;; Unique id of annotation.
  text			;; Text to display inside.
  vertex-id		;; Id of vertex which is justification point.
  (horiz-just :center)	;; One of :left :center :right
  (vert-just :center)	;; One of :top :center :bottom
  width			;; Width, in world coords, of annotation
  height		;; Height, in world coords, of annotation
  ;; -------------------  CACHED VALUES ----------------------------
  objects		;; A list of objects that contain this annotation.
  font-widths		;; List with width of string in the various fonts.
  font-heights		;; List with height of string in the various fonts.
  font-ascents		;; List with ascent of string in the various fonts.
  font-descents		;; List with descents of string in the various fonts.
  sel-font-widths	;; List with width of string in the selected fonts.
  sel-font-heights	;; List with height of string in the selected fonts.
  sel-font-ascents	;; List with ascent of string in the selected fonts.
  sel-font-descents	;; List with descents of string in the selected fonts.
  font-idx		;; Last font index used to draw.
  vertex		;; Index of vertex in vertex-table.
  )

(defstruct (fg-symbol (:conc-name fgs-)) 
  id			;; Unique id of symbol.
  symbol-type-id	;; Id of symbol type.
  vertex-id		;; Id of vertex which is justification point.
  (horiz-just :center)	;; One of :left :center :right
  (vert-just :center)	;; One of :top :center :bottom
  ;; -------------------  CACHED VALUES ----------------------------
  objects		;; A list of objects that contain this symbol.
  vertex		;; Index of vertex in vertex-table.
  symbol-type		;; Index of symbol type in symbol-type-table.
  bitmap-idx		;; Last bitmap index used to draw.
  )

(defstruct (fg-symbol-type (:conc-name fgst-))
  id			;; Unique id of symbol type.
  bitmaps		;; List of bitmaps to try, best first.
  widths		;; List of the widths of the bitmaps
  heights		;; List of the heights of the bitmaps
  width			;; Width, in world coords, of symbol
  height		;; Height, in world coords, of symbol
  )

(defstruct (fg-line (:conc-name fgl-)) 
  id			;; Unique id of line.
  start-vertex-id	;; Id's of start and end vertices of line.
  end-vertex-id
  ;; -------------------  CACHED VALUES ----------------------------
  objects		;; A list of objects that contain this line.
  start-vertex		;; Indices of start and end of line in vertex-table.
  end-vertex
  )

(defstruct (fg-object (:conc-name fgo-)) 
  id			;; Unique id of object.
  line-ids		;; List of ids of lines
  annot-ids		;; List of ids of annotations
  symbol-ids		;; List of ids of symbols
  (visible t)		;; Is this object visible?
  selected		;; Is this object selected?
  (selectable t)	;; Is this object selectable?
  (line-width 0)	;; Line width to use on object lines.
  (color "white")	;; Color use on object lines/annotations/symbols.
  ;; -------------------  CACHED VALUES ----------------------------
  lines			;; List of the indexes of the lines
  annots		;; List of the indexes of the annotations
  symbols		;; List of the indexes of the symbols
  extent		;; List of xmin ymin xmax ymax of object
  )

(defun compute-object-extent (sgg &aux sgg-objects sgg-lines sgg-annots
				  verts sgg-symbols sgg-symbol-types)
  (setq verts (vertices sgg)
	sgg-objects (objects sgg)
	sgg-lines (lines sgg)
	sgg-annots (annots sgg)
	sgg-symbols (symbols sgg)
	sgg-symbol-types (symbol-types sgg))
  ;; Compute the extent of the objects...
  (dotimes (i (length sgg-objects))
	   (let ((xmin most-positive-fixnum)
		 (ymin most-positive-fixnum)
		 (xmax most-negative-fixnum)
		 (ymax most-negative-fixnum)
		 (object (aref sgg-objects i)))
		(dolist (j (fgo-lines object))
			(let* ((l (aref sgg-lines j))
			       x1 x2 y1 y2
			       (sv (fgl-start-vertex l))
			       (ev (fgl-end-vertex l)))
			      (setq sv (+ sv sv)
				    ev (+ ev ev)
				    x1 (aref verts sv)
				    y1 (aref verts (1+ sv))
				    x2 (aref verts ev)
				    y2 (aref verts (1+ ev))
				    xmin (min xmin x1 x2)
				    ymin (min ymin y1 y2)
				    xmax (max xmax x1 y2)
				    ymax (max ymax y1 y2))))
		(dolist (j (fgo-annots object))
			(let* ((a (aref sgg-annots j))
			       x1 x2 y1 y2
			       (w (fga-width a))
			       (h (fga-height a))
			       (hj (fga-horiz-just a))
			       (vj (fga-vert-just a))
			       (v (fga-vertex a)))
			      (setq v (+ v v)
				    x1 (aref verts v)
				    y1 (aref verts (1+ v))
				    x2 x1
				    y2 y1)
			      (case hj
				    (:right (decf x1 w))
				    (:left (incf x2 w))
				    (:center (decf x1 (/ w 2))
					     (incf x2 (/ w 2))))
			      (case vj
				    (:top (decf y1 h))
				    (:bottom (incf y2 h))
				    (:center (decf y1 (/ h 2))
					     (incf y2 (/ h 2))))
			      (setq xmin (min xmin x1)
				    ymin (min ymin y1)
				    xmax (max xmax x2)
				    ymax (max ymax y2))))
		(dolist (j (fgo-symbols object))
			(let* ((s (aref sgg-symbols j))
			       (st (aref sgg-symbol-types (fgs-symbol-type s)))
			       x1 x2 y1 y2
			       (w (fgst-width st))
			       (h (fgst-height st))
			       (hj (fgs-horiz-just s))
			       (vj (fgs-vert-just s))
			       (v (fgs-vertex s)))
			      (setq v (+ v v)
				    x1 (aref verts v)
				    y1 (aref verts (1+ v))
				    x2 x1
				    y2 y1)
			      (case hj
				    (:right (decf x1 w))
				    (:left (incf x2 w))
				    (:center (decf x1 (/ w 2))
					     (incf x2 (/ w 2))))
			      (case vj
				    (:top (decf y1 h))
				    (:bottom (incf y2 h))
				    (:center (decf y1 (/ h 2))
					     (incf y2 (/ h 2))))
			      (setq xmin (min xmin x1)
				    ymin (min ymin y1)
				    xmax (max xmax x2)
				    ymax (max ymax y2))))
		(setf (fgo-extent object) (list xmin ymin xmax ymax)))))

(defun print-vertices (vertex-ids vertices mapped-vertices)
  (format t "Vertex Table~%")
  (dotimes (i (length vertex-ids))
	   (format t "~s	(~f,~f)	(~d,~d)~%" (aref vertex-ids i)
		   (aref vertices (+ i i))
		   (aref vertices (+ i i 1))
		   (aref mapped-vertices (+ i i))
		   (aref mapped-vertices (+ i i 1))))
  (format t "~%"))

(defun print-lines (lines)
  (format t "Line Table~%")
  (map nil #'(lambda (x) (format t "~s~%" x)) lines)
  (format t "~%"))

(defun print-annots (annots)
  (format t "Annotation Table~%")
  (map nil #'(lambda (x) (format t "~s~%" x)) annots)
  (format t "~%"))

(defun print-symbols (symbols)
  (format t "Symbol Table~%")
  (map nil #'(lambda (x) (format t "~s~%" x)) symbols)
  (format t "~%"))

(defun print-symbol-types (symbol-types)
  (format t "Symbol-Type Table~%~%")
  (map nil #'(lambda (x) (format t "~s~%" x)) symbol-types)
  (format t "~%"))

(defun print-objects (objects)
  (format t "Object Table~%")
  (pprint objects)
  (format t "~%"))

(defun print-sgg (self)
  (print-vertices (vertex-ids self) (vertices self) (mapped-vertices self))
  (pprint (lines self))
  (terpri) (terpri)
  (pprint (annots self))
  (terpri) (terpri)
  (pprint (symbols self))
  (terpri) (terpri)
  (pprint (symbol-types self))
  (terpri) (terpri)
  (pprint (objects self))
  (terpri) (terpri)
  
  (format t "Fonts:~%")
  (format t "	~s~%" (font-list self))
  (format t "~%Selected Fonts:~%")
  (format t "	~s~%" (sel-font-list self))
  (format t "~%Colors:~%")
  (format t "	~s~%" (color-list self))
  (format t "~%Widths:~%")
  (format t "	~s~%" (width-list self))
  (format t "~%Selected Color:~%")
  (format t "	~s~%" (sel-color self))
  (format t "~%Selected Width:~%")
  (format t "	~s~%" (sel-width self))
  (format t "~%Extent~%")
  (format t "	~s~%" (extent self)))

(defun dist-to-annot (self a x y)
  (let* ((verts (mapped-vertices self))
	 (fw (fga-font-widths a))
	 (fh (fga-font-heights a))
	 (font-idx (fga-font-idx a))
	 (hj (fga-horiz-just a))
	 (vj (fga-vert-just a))
	 (v (fga-vertex a))
	 x1 x2 y1 y2 w h)
	(if (null font-idx)
	    most-positive-fixnum
	    (progn
	     (setq v (+ v v)
		   x1 (aref verts v)
		   y1 (aref verts (1+ v))
		   x2 x1
		   y2 y1
		   w (nth font-idx fw)
		   h (nth font-idx fh))
	     (case hj
		   (:right (decf x1 w))
		   (:left (incf x2 w))
		   (:center (decf x1 (round w 2))
			    (incf x2 (round w 2))))
	     (case vj
		   (:bottom (decf y1 h))
		   (:top (incf y2 h))
		   (:center (decf y1 (round h 2))
			    (incf y2 (round h 2))))
	     (dist-to-bb (list x1 y1 x2 y2) x y)))))

(defun dist-to-symbol (self s x y)
  (let* ((verts (mapped-vertices self))
	 (symbol-types (symbol-types self))
	 (st (aref symbol-types (fgs-symbol-type s)))
	 (bmw (fgst-widths st))
	 (bmh (fgst-heights st))
	 (bitmap-idx (fgs-bitmap-idx s))
	 (hj (fgs-horiz-just s))
	 (vj (fgs-vert-just s))
	 (v (fgs-vertex s))
	 x1 x2 y1 y2 w h)
	(if (null bitmap-idx)
	    most-positive-fixnum
	    (progn
	     (setq v (+ v v)
		   x1 (aref verts v)
		   y1 (aref verts (1+ v))
		   x2 x1
		   y2 y1
		   w (nth bitmap-idx bmw)
		   h (nth bitmap-idx bmh))
	     (case hj
		   (:right (decf x1 w))
		   (:left (incf x2 w))
		   (:center (decf x1 (round w 2))
			    (incf x2 (round w 2))))
	     (case vj
		   (:bottom (decf y1 h))
		   (:top (incf y2 h))
		   (:center (decf y1 (round h 2))
			    (incf y2 (round h 2))))
	     (dist-to-bb (list x1 y1 x2 y2) x y)))))

(defun dist-to-line (self l x y)
  (let* ((verts (mapped-vertices self))
	 x1 x2 y1 y2
	 (sv (fgl-start-vertex l))
	 (ev (fgl-end-vertex l)))
	(setq sv (+ sv sv)
	      ev (+ ev ev)
	      x1 (aref verts sv)
	      y1 (aref verts (1+ sv))
	      x2 (aref verts ev)
	      y2 (aref verts (1+ ev)))
	(int-pt-line-dist x1 y1 x2 y2 x y)))

(defun dist-to-object (self object x y)
  (let ((d most-positive-fixnum)
	(sgg-lines (lines self))
	(sgg-symbols (symbols self))
	(sgg-annots (annots self)))
       (dolist (l (fgo-lines object))
	       (setq d (min d (dist-to-line self (aref sgg-lines l) x y))))
       (dolist (l (fgo-annots object))
	       (setq d (min d (dist-to-annot self (aref sgg-annots l) x y))))
       (dolist (l (fgo-symbols object))
	       (setq d (min d (dist-to-symbol self (aref sgg-symbols l) x y))))
       d))

(defun pt-in-bb (x y xmin ymin xmax ymax)
  (and (>= xmax x xmin) (>= ymax y ymin)))

(defun annot-in-bb (self a xmin ymin xmax ymax)
  (let* ((verts (mapped-vertices self))
	 (fw (fga-font-widths a))
	 (fh (fga-font-heights a))
	 (font-idx (fga-font-idx a))
	 (hj (fga-horiz-just a))
	 (vj (fga-vert-just a))
	 (v (fga-vertex a))
	 x1 x2 y1 y2 w h)
	(if (null font-idx)
	    most-positive-fixnum
	    (progn
	     (setq v (+ v v)
		   x1 (aref verts v)
		   y1 (aref verts (1+ v))
		   x2 x1
		   y2 y1
		   w (nth fw font-idx)
		   h (nth fh font-idx))
	     (case hj
		   (:right (decf x1 w))
		   (:left (incf x2 w))
		   (:center (decf x1 (round w 2))
			    (incf x2 (round w 2))))
	     (case vj
		   (:bottom (decf y1 h))
		   (:top (incf y2 h))
		   (:center (decf y1 (round h 2))
			    (incf y2 (round h 2))))
	     (and (pt-in-bb x1 y1 xmin ymin xmax ymax)
		  (pt-in-bb x2 y2 xmin ymin xmax ymax))))))

(defun symbol-in-bb (self s xmin ymin xmax ymax)
  (let* ((verts (mapped-vertices self))
	 (symbol-types (symbol-types self))
	 (st (aref symbol-types (fgs-symbol-type s)))
	 (bmw (fgst-widths st))
	 (bmh (fgst-heights st))
	 (bitmap-idx (fgs-bitmap-idx s))
	 (hj (fgs-horiz-just s))
	 (vj (fgs-vert-just s))
	 (v (fgs-vertex s))
	 x1 x2 y1 y2 w h)
	(if (null bitmap-idx)
	    most-positive-fixnum
	    (progn
	     (setq v (+ v v)
		   x1 (aref verts v)
		   y1 (aref verts (1+ v))
		   x2 x1
		   y2 y1
		   w (nth bmw bitmap-idx)
		   h (nth bmh bitmap-idx))
	     (case hj
		   (:right (decf x1 w))
		   (:left (incf x2 w))
		   (:center (decf x1 (round w 2))
			    (incf x2 (round w 2))))
	     (case vj
		   (:bottom (decf y1 h))
		   (:top (incf y2 h))
		   (:center (decf y1 (round h 2))
			    (incf y2 (round h 2))))
	     (and (pt-in-bb x1 y1 xmin ymin xmax ymax)
		  (pt-in-bb x2 y2 xmin ymin xmax ymax))))))

(defun line-in-bb (self l xmin ymin xmax ymax)
  (let* ((verts (mapped-vertices self))
	 x1 x2 y1 y2
	 (sv (fgl-start-vertex l))
	 (ev (fgl-end-vertex l)))
	(setq sv (+ sv sv)
	      ev (+ ev ev)
	      x1 (aref verts sv)
	      y1 (aref verts (1+ sv))
	      x2 (aref verts ev)
	      y2 (aref verts (1+ ev)))
	(and (pt-in-bb x1 y1 xmin ymin xmax ymax)
	     (pt-in-bb x2 y2 xmin ymin xmax ymax))))

(defun object-in-bb (self object bb)
  (let ((x1 (first bb))
	(y1 (second bb))
	(x2 (third bb))
	(y2 (fourth bb))
	(sgg-lines (lines self))
	(sgg-symbols (symbols self))
	(sgg-annots (annots self))
	(in t))
       (do* ((ll (fgo-lines object) (cdr ll))
	     (l (car ll) (car ll)))
	    ((or (not in) (null ll)))
	    (setq in (line-in-bb self (aref sgg-lines l) x1 y1 x2 y2)))
       (do* ((al (fgo-annots object) (cdr al))
	     (a (car al) (car al)))
	    ((or (not in) (null al)))
	    (setq in (annot-in-bb self (aref sgg-annots a) x1 y1 x2 y2)))
       (do* ((sl (fgo-symbols object) (cdr sl))
	     (s (car sl) (car sl)))
	    ((or (not in) (null sl)))
	    (setq in (symbol-in-bb self (aref sgg-symbols s) x1 y1 x2 y2)))
       in))

(defun dist-to-bb (bb x y)
  (let ((xmin (first bb))
	(ymin (second bb))
	(xmax (third bb))
	(ymax (fourth bb))
	x-dist y-dist)
       (setq x-dist
	     (if (>= xmax x xmin) 0
		 (if (< x xmin) (- xmin x) (- x xmax))))
       (setq y-dist (if (>= ymax y ymin) 0
			(if (< y ymin) (- ymin y) (- y ymax))))
       (+ (* x-dist x-dist) (* y-dist y-dist))))

