;;;
;;; Copyright (c) 1990 Regents of the University of California
;;;
;;; Definition for segment-dl, a class for mapping
;;; from linestrings to lists that can be easily drawn in an X window.
;;;
;;; Todo:  Recycle arrays in mov-abs.
;;;

(in-package 'pt :use '(lisp pcl))

;;;
;;; Are we a segment-dl? Isn't that special!
;;;
(defmacro segment-dl-p (self)
  `(typep ,self 'segment-dl))

;;;
;;; Class definition for segment-dl
;;;
(defclass segment-dl (display-list)
  ((fp-polyline-list :initform nil :type list :accessor fp-polyline-list)
   (int-polyline-list :initform nil :type list :accessor int-polyline-list)
   (color :initform nil :accessor color)
   (line-style :initform nil :accessor line-style)
   (line-width :initform nil :accessor line-width)))

(defmethod new-instance ((self segment-dl)
			 &key
			 (ignore nil)
			 &allow-other-keys)
  (declare (ignore ignore))
  (setf (slot-value self 'prop-list) 
	(list :background "black"
	      :foreground "white"
	      :line-style :solid
	      :line-width 0)))

(defun make-segment-dl (&rest args) 
  (apply #'make-instance (cons 'segment-dl args)))

(defun clear-segment-dl (gs)
  (setf (fp-polyline-list gs) nil
	(int-polyline-list gs) nil))

(defun mov-abs (gs x y)
  (let ((fp-pl (make-array '(4) :fill-pointer t :adjustable t)))
       (setf (aref fp-pl 0) x)
       (setf (aref fp-pl 1) y)
       (setf (fill-pointer fp-pl) 2)
       (push fp-pl (fp-polyline-list gs))))

(defun lin-abs (gs x y)
  (let ((fp-pl (car (fp-polyline-list gs))))
       (vector-push-extend x fp-pl)
       (vector-push-extend y fp-pl)))

(defmethod draw ((self segment-dl) viewer)
  (validate self viewer)
  (let ((res (res viewer))
	(gc (draw-gc self)))
       (if gc
	   (mapc #'(lambda (pl) (xlib:draw-lines res gc pl))
		 (int-polyline-list self)))))

(defmethod erase ((self segment-dl) viewer)
  (validate self viewer)
  (let ((res (res viewer))
	(gc (erase-gc self)))
       (if gc
	   (mapc #'(lambda (pl) (xlib:draw-lines res gc pl))
		 (int-polyline-list self)))))

(defmethod dl-bbox ((self segment-dl))
  (let ((xmin most-positive-fixnum)
	(ymin most-positive-fixnum)
	(xmax most-negative-fixnum)
	(ymax most-negative-fixnum))
       (dolist (int-pl (int-polyline-list self))
	       (dotimes (i (/ (length int-pl) 2))
			(let ((x (aref int-pl (* 2 i)))
			      (y (aref int-pl (1+ (* 2 i)))))
			     (setq xmin (min xmin x)
				   ymin (min ymin y)
				   xmax (max xmax x)
				   ymax (max ymax y)))))
       (list xmin ymin xmax ymax)))

(defun int-pt-line-dist (xa ya xb yb xp yp)
  (let* ((xab (- xb xa))
	 (yab (- yb ya))
	 (xap (- xp xa))
	 (yap (- yp ya))
	 (len (+ (* xab xab) (* yab yab)))
	 (s 0))
	(if (not (zerop len))
	    (setq s (/ (+ (* xab xap) (* yab yap)) len)))
	(if (< s 0)
	    (setq s 0)
	    (if (> s 1)
		(setq s 1)))
	(setq xab (* xab s)
	      yab (* yab s))
	(setq xab (- xap xab)
	      yab (- yap yab))
	(+ (* xab xab) (* yab yab))))

(defun dist-to-int-pl (int-pl rad dx dy)
  (if (< (length int-pl) 4) 
      (return-from dist-to-int-pl most-positive-fixnum))
  (let ((min most-positive-fixnum)
	(pd 0)
	(len (length int-pl))
	(x1 0)
	(y1 0)
	(x2 0)
	(y2 0))
       ;; Check against bbox of int-pl
       (setq y1 (setq x1 (reduce #'min int-pl)))
       (setq y2 (setq x2 (reduce #'max int-pl)))
       (when (and (> (+ dx rad) x1) (> (+ dy rad) x1)
		  (< (- dx rad) x2) (< (- dy rad) x2))
	     (setq x1 (aref int-pl 0))
	     (setq y1 (aref int-pl 1))
	     (dotimes (i (- (/ len 2) 2))
		      (setq x2 (aref int-pl (+ (* 2 i) 2)))
		      (setq y2 (aref int-pl (+ (* 2 i) 3)))
		      ;; Check against bbox of line
		      (when (> rad (min (abs (- x1 dx)) (abs (- y1 dy))
					(abs (- x2 dx)) (abs (- y2 dy)))
			       (setq pd (int-pt-line-dist x1 y1 x2 y2 dx dy))
			       (setq min (min min pd))))
		      (setq x1 x2)
		      (setq y1 y2)))
       min))

(defmethod dist-to-dl ((dl segment-dl) rad dx dy)
  (if (or (not (visible dl)) (null (int-polyline-list dl)))
      (return-from dist-to-dl most-positive-fixnum))
  (let ((min most-positive-fixnum))
       (dolist (int-pl (int-polyline-list dl))
	       (setq min (min min (dist-to-int-pl int-pl rad dx dy))))
       min))

(defmethod do-validation ((self segment-dl) viewer
			  &aux 
			  (mx (mx viewer))
			  (my (my viewer))
			  (bx (bx viewer))
			  (by (by viewer))
			  (prop-list (prop-list self))
			  int-pl-list)
  (do* ((fp-pl-l (fp-polyline-list self) (cdr fp-pl-l))
	(int-pl-l (int-polyline-list self) (cdr int-pl-l))
	(fp-pl (car fp-pl-l) (car fp-pl-l))
	(int-pl (car int-pl-l) (car int-pl-l)))
       ((null fp-pl-l))
       (let* ((len (length fp-pl)))
	     (if (/= (length int-pl) len)
		 (setq int-pl (make-array (list len)
					  :element-type 'fixnum
					  :initial-element 0)))
	     (map-float fp-pl int-pl mx my bx by)
	     (push int-pl int-pl-list)))
  (setf (int-polyline-list self) int-pl-list)
  (let* ((shape (shape self))
	 (bg (or (background viewer) "black"))
	 (fg (or (color self) (color shape) (foreground viewer)))
	 (ls (or (line-style self) (line-style shape) :solid))
	 (lw (or (line-width self) (line-width shape) 0)))
	(setf (getf prop-list :background) bg)
	(setf (getf prop-list :line-style) ls)
	(setf (getf prop-list :line-width) lw)
	(setf (getf prop-list :foreground) bg)
	(setf (erase-gc self) (make-shared-gc viewer prop-list))
	(setf (getf prop-list :foreground) fg)
	(setf (draw-gc self) (make-shared-gc viewer prop-list))))
