;;;
;;; Copyright (c) 1990 Regents of the University of California
;;;
;;; Definition for polygons, a subclass of shapes.
;;;
;;; Polygons are types of shapes whose control points are the vertices
;;; of the polygon.  Polygons can be open or closed, as specified by
;;; the closed flag.
;;;

(in-package 'pt :use '(lisp pcl))

;;;
;;; Are we a polygon? Isn't that special!
;;;
(defmacro polygon-p (self)
  `(typep ,self 'polygon))

;;;
;;; Class definition for polygon
;;;
(defclass polygon (segment)
  ((closed :initarg :closed :initform nil :type t :accessor closed)
   (hook-pt :initarg :hook-pt :initform (make-2d-point :x 0 :y 0)
	    :type 2d-point :accessor hook-pt)))

(defun make-polygon (&rest keys)
  (apply #'make-instance (cons 'polygon keys)))

(defmethod (setf closed) (closed (self polygon))
  (setq closed (not (null closed)))
  (unless (eq closed (slot-value self 'closed))
	  (setf (slot-value self 'closed) closed)
	  (notify-geom-change self)))

(defmethod (setf hook-pt) (hook-pt (self polygon))
  (2dv-copy (slot-value self 'hook-pt) hook-pt)
  (notify-geom-change self))

(defmethod new-instance ((self polygon)
			 &key
			 (ignore nil)
			 &allow-other-keys)
  (declare (ignore ignore))
  (call-next-method)
  (setf (slot-value self 'closed) (not (null (closed self))))
  (setf (slot-value self 'coord-list)
	(make-array (list (length (ctrl-pts self)))
		    :adjustable t :fill-pointer 0))
  self)

;;;
;;; Copy a polygon.
;;;
(defmethod copy ((self polygon))
  (let ((rv (call-next-method)))
       (setf (slot-value rv 'closed) (closed self))
       (setf (slot-value rv 'hook-pt) (duplicate-2d (hook-pt self)))))

;;;
;;; Bounding box of a polygon.
;;;
(defmethod find-bbox ((self polygon))
  (linestr-bbox (ctrl-pts self)))

;;;
;;; Return the gravity points for a polygon.  This is a list of param-vals.
;;;
(defmethod gravity-pts ((self polygon) do-midpt)
  (linestr-gravity-pt (ctrl-pts self) (closed self) do-midpt))

;;;
;;; Return the point on a polygon at a given parametric value.
;;; The parametric value for a polygon is [0-n), where n is the number
;;; of vertices in the polygon.
;;;
(defmethod point ((self polygon) value)
  (if (eq value :hook-pt)
      (duplicate-2d (hook-pt self))
      (linestr-point (ctrl-pts self) (closed self) value)))
 
;;;
;;; Return the normal to a polygon at a given parametric value.
;;; The parametric value for a polygon is [0-n), where n is the number
;;; of vertices in the polygon.
;;;
(defmethod normal ((self polygon) value)
  (if (eq value :hook-pt)
      (alloc-2d 0 -1)
      (linestr-normal (ctrl-pts self) (closed self) value)))
 
;;;
;;; Return the point and normal to a polygon at a given parametric value.
;;; The parametric value for a polygon is [0-n), where n is the number
;;; of vertices in the polygon.
;;;
(defmethod pt-normal ((self polygon) value)
  (if (eq value :hook-pt)
      (list (duplicate-2d (hook-pt self)) (alloc-2d 0 -1))
      (linestr-pt-normal (ctrl-pts self) (closed self) value)))
 
;;;
;;; Translate a polygon.  Just moves the hook-pt and adjust ctrl-pts.
;;;
(defmethod 2d-translate ((self polygon) tx ty)
  (call-next-method)
  (let* ((hook-pt (hook-pt self)))
	(incf (2d-point-x hook-pt) tx)
	(incf (2d-point-y hook-pt) ty)
	(dolist (cp (ctrl-pts self))
		(incf (2d-point-x cp) tx)
		(incf (2d-point-y cp) ty))
	(notify-geom-change self)))

;;;
;;; Rotate a polygon about (ox, oy).  Just rotates the hook-pt 
;;; and ctrl-pts about (ox, oy).
;;;
(defmethod 2d-rotate ((self polygon) theta ox oy)
  (call-next-method)
  (let* ((hook-pt (hook-pt self))
	 (x (2d-point-x hook-pt))
	 (y (2d-point-y hook-pt))
	 (c (cos theta))
	 (s (sin theta)))
	(declare (float c s x y)
		 (number ox oy))
	(decf x ox)
	(decf y oy)
	(copy-2d hook-pt (+ (- (* x c) (* y s)) ox) (+ (* x s) (* y c) oy))
	(dolist (cp (ctrl-pts self))
		(setq x (2d-point-x cp) y (2d-point-y cp))
		(decf x ox)
		(decf y oy)
		(copy-2d cp (+ (- (* x c) (* y s)) ox) (+ (* x s) (* y c) oy)))
	(notify-geom-change self)))

;;;
;;; Scale a polygon about (ox, oy).  Changes hook-pt, ctrl-pts
;;;
(defmethod 2d-scale ((self polygon) sf ox oy)
  (call-next-method)
  (let* ((hook-pt (hook-pt self))
	 (x (2d-point-x hook-pt))
	 (y (2d-point-y hook-pt)))
	(declare (number sf x y ox oy))
	(decf x ox)
	(decf y oy)
	(copy-2d hook-pt (+ ox (* x sf)) (+ (* y sf) oy))
	(dolist (cp (ctrl-pts self))
		(setq x (2d-point-x cp) y (2d-point-y cp))
		(decf x ox)
		(decf y oy)
		(copy-2d cp (+ ox (* x sf)) (+ (* y sf) oy)))
	(notify-geom-change self)))

;;;
;;; Return the coord-list of a polygon;  this is just a copy of
;;; the ctrl-pts of the sub-objs.  If the polygon is less than
;;; two pixels, a single point is returned.
;;;
(defmethod 2d-coord-list ((self polygon) xleft xright ybottom ytop w h)
  (if (null (ctrl-pts self))
      ;; No poly.
      (return-from 2d-coord-list nil))
  
  (let ((rv (slot-value self 'coord-list)))
       (setf (fill-pointer rv) 0)
       (do* ((i 0 (1+ i))
	     (iter (ctrl-pts self) (cdr iter))
	     (pt (car iter) (car iter)))
	    ((null iter) nil)
	    (vector-push-extend (duplicate-2d pt) rv))
       (if (closed self) 
	   (let ((start (car (ctrl-pts self))))
		(vector-push-extend (duplicate-2d start) rv)))
       (return-from 2d-coord-list rv)))

;;;
;;; Find the nearest point on the polygon to the point pt.
;;; Return a list of (parametric-value dist).
;;;
(defmethod nearest-pt ((self polygon) pt)
  (if (null (ctrl-pts self))
      (list 0 most-positive-single-float)
      (nearest-pt-to-linestr (ctrl-pts self) (closed self) pt)))
