;;;
;;; Copyright (c) 1990 Regents of the University of California
;;; 
;;; $Author seitz $
;;; $Source $
;;; $Revision $
;;; $Date $
;;;

(in-package 'pt  :nicknames '(picasso-toolkit) :use '(lisp excl pcl))

;;;
;;;	A plot-widget may be used for displaying arrays of points in a two-
;;;	dimensional graph.  The plot-widget has zoom and panning facilities
;;;	as well.  
;;;	NOTE:  Currently the spline feature does not work properly
;;;
;;;	Sample usage:
;;;	(setq pts 
;;;	      (list (make-array 5 :initial-contents
;;;				'((10 . 10) (20 . 30) (50 . 15)
;;;				  (60 . 70) (90 . 40)))
;;;		    (make-array 8 :initial-contents
;;;				'((5 . 60) (15 . 30) (30 . 15) 
;;;				  (35 . 12) (38 . 10) (45 . 8) 
;;;				  (65 . 40) (90 . 80)))))
;;;	(make-plot-widget :base-size '(200 200) 
;;;			  :domain (cons 10 30)
;;;			  :x-label "Hello" :y-label "There" 
;;;			  :value pts 
;;;			  :paints (list "green" "red")))
;;;

(defclass pw-pane (collection-widget)
  ((value
    :initarg :value 
    :initform nil
    :type list
    :reader value)
   (paints
    :initarg :paints 
    :initform nil
    :type list
    :reader paints)
   (range
    :initarg :range 
    :initform nil
    :type cons
    :reader range)
   (domain
    :initarg :domain 
    :initform nil
    :type cons
    :reader domain)
   (x-increment
    :initarg :x-increment 
    :initform 5
    :type number
    :reader x-increment)
   (y-increment
    :initarg :y-increment 
    :initform 5
    :type number
    :reader y-increment)
   (spline
    :initarg :spline 
    :initform nil
    :type atom
    :reader spline)
   (axes
    :initarg :axes 
    :initform t
    :type atom
    :reader axes)
   (mark-points
    :initarg :mark-points
    :initform t
    :type atom
    :reader mark-points)
   (update-flag				;; Like repaint-flag or repack-flag.
    :initarg :update-flag				 
    :initform t				;; Use when changing more than one
    :type atom				;; attribute at a time.
    :reader update-flag)
   (curve-labels             ;; a list of strings, at least one for each curve
    :initform nil
    :type list
    :accessor curve-labels)
;; ----------------------------------------------------------------------------
;;					For internal use. . .
   (invalidate-flag
    :initform nil
    :type atom
    :reader invalidate-flag)
   (x-scale
    :initform nil
    :type number
    :accessor x-scale)
   (y-scale
    :initform nil
    :type number
    :accessor y-scale)
   (intern-origin
    :initform nil
    :type cons
    :accessor intern-origin)
   (diff-points
    :initform nil
    :type list
    :accessor diff-points)
   (limits
    :initform nil
    :type list
    :accessor limits)
   (x-axis
    :initform t
    :type t 
    :accessor x-axis)
   (y-axis
    :initform t
    :type t
    :accessor y-axis)
   (vlists
    :initform nil
    :type t
    :accessor vlists)
   (box
    :initform nil
    :type t
    :accessor box)
   (event-mask :initform '(:exposure :button-press :button-release
				     :button-1-motion))
   (gc-spec :initform '((gc-res "default")
			(gc-graph "default")
			(gc-invert (:function 10))))
   (gc-graph
    :initform nil
    :type vector
    :reader gc-graph)
   (gc-invert
    :initform nil
    :type vector
    :reader gc-invert)
   (gm :initform 'anchor-gm)))

;;;
;;;	Plot-widget class
;;;

(defclass plot-widget (collection-widget)
  ((pane
    :initarg :pane 
    :initform nil
    :type pw-pane
    :reader pane)
   (font 
    :initform nil
    :type font
    :reader font)
   (mark-font
    :initarg :mark-font 
    :initform nil
    :type font
    :accessor mark-font)
   (x-label
    :initarg :x-label 
    :initform nil
    :type string
    :reader x-label)
   (y-label
    :initarg :y-label 
    :initform nil
    :type string
    :reader y-label)
   (x-pad
    :initarg :x-pad 
    :initform 5
    :type integer
    :reader x-pad)
   (y-pad
    :initarg :y-pad 
    :initform 5
    :type integer
    :reader y-pad)
   (gc-spec :initform '(gc-res "default"))
   (name :initform "A Plot-Widget" :type string)
   (gm :initform 'linear-gm)))

;;;
;;;	Accessor macros
;;;

(defmacro pw-make-point (x y)
  `(cons ,x ,y))

(defmacro x-crd (point)
  `(car ,point))

(defmacro y-crd (point)
  `(cdr ,point))

(defmacro set-x-crd (point val)
  `(setf (car ,point) ,val))

(defmacro set-y-crd (point val)
  `(setf (cdr ,point) ,val))

;;;
;;;	Accessors update cache
;;;

(defmethod (setf invalidate-flag) (val (self pw-pane))
  (setf (slot-value self 'invalidate-flag) 
	(logior val (slot-value self 'invalidate-flag))))

(defmethod compute-limits ((self pw-pane) &aux val
			   (x-low nil) (x-high nil)
			   (y-low nil) (y-high nil))
  (setq val (slot-value self 'value))
  (unless (listp val) (setq val (list val)))
  (dolist (v val)
	  (dotimes (i (length v))
		   (let* ((p (aref v i))
			  (x (x-crd p))
			  (y (y-crd p))
			  )
		     (when (or (null x-low) (< x x-low))
			   (setq x-low x))
		     (when (or (null x-high) (> x x-high))
			   (setq x-high x))
		     (when (or (null y-low) (< y y-low))
			   (setq y-low y))
		     (when (or (null y-high) (> y y-high))
			   (setq y-high y)))))
  (setf (limits self) (list (cons x-low x-high) (cons y-low y-high))))

(defmethod (setf value) (val (self pw-pane) &aux sup cdiff 
			     x-low x-high y-low y-high)
  (when (invalidate-flag self)
	(unless (listp val) (setq val (list val)))
	(setf (slot-value self 'value) val)
	(dolist (v val)
		(dotimes (i (length v))
			 (let* ((p (aref v i))
				(x (x-crd p))
				(y (y-crd p)))
			   (when (or (null x-low) (< x x-low))
				 (setq x-low x))
			   (when (or (null x-high) (> x x-high))
				 (setq x-high x))
			   (when (or (null y-low) (< y y-low))
				 (setq y-low y))
			   (when (or (null y-high) (> y y-high))
				 (setq y-high y)))))
	(if (null x-low) (setq x-low 0))
	(if (null x-high) (setq x-high 0))
	(if (null y-low) (setq y-low 0))
	(if (null y-high) (setq y-high 0))
	(setf (limits self) (list (cons x-low x-high) (cons y-low y-high)))
	(setq cdiff (- (length val) (length (paints self))))
	(when (plusp cdiff)
	      (setf (slot-value self 'paints)
		    (nconc (paints self) 
			   (make-list cdiff :initial-element 
				      (foreground self)))))
	(setf (slot-value self 'domain) (cons x-low x-high))
	(setf (slot-value self 'range) (cons y-low y-high))
	(do-propagate 'area self)
	(do-propagate 'domain self)
	(do-propagate 'range self)
	(if (update-flag self)
	    (progn
	     (pw-update-cache self :invalidate t)
	     (if (typep (setq sup (parent self)) 'plot-widget)
		 (pw-repaint sup)
		 (repaint self)))
	    (setf (invalidate-flag self) 3))))

(defmethod (setf spline) (val (self pw-pane) &aux sup)
  (setf (slot-value self 'spline) val)
  (if (update-flag self)
      (progn
       (pw-update-graph self :invalidate t)
       (if (typep (setq sup (parent self)) 'plot-widget)
	   (pw-repaint sup)
	   (repaint self)))
      (setf (invalidate-flag self) 3)))

(defmethod (setf domain) (val (self pw-pane) &aux sup)
  (setf (slot-value self 'domain) val)
  (if (update-flag self)
      (progn
       (pw-update-domain self :invalidate t)
       (if (typep (setq sup (parent self)) 'plot-widget)
	   (pw-repaint sup)
	   (repaint self)))
      (setf (invalidate-flag self) 1)))

(defmethod (setf range) (val (self pw-pane) &aux sup)
  (setf (slot-value self 'range) val)
  (if (update-flag self)
      (progn
       (pw-update-range self :invalidate t)
       (if (typep (setq sup (parent self)) 'plot-widget)
	   (pw-repaint sup)
	   (repaint self)))
      (setf (invalidate-flag self) 2)))

(defmethod (setf axes) (val (self pw-pane))
  (setf (slot-value self 'axes) val)
  (repaint self))

(defmethod x-low ((self pw-pane))
  (car (domain self)))

(defmethod y-low ((self pw-pane))
  (car (range self)))

(defmethod x-high ((self pw-pane))
  (cdr (domain self)))

(defmethod y-high ((self pw-pane))
  (cdr (range self)))

(defmethod (setf x-low) (val (self pw-pane))
  (setf (car (domain self)) val)
  (setf (domain self) (domain self)))

(defmethod (setf y-low) (val (self pw-pane))
  (setf (car (range self)) val)
  (setf (range self) (range self)))

(defmethod (setf x-high) (val (self pw-pane))
  (setf (cdr (domain self)) val)
  (setf (domain self) (domain self)))

(defmethod (setf y-high) (val (self pw-pane))
  (setf (cdr (range self)) val)
  (setf (range self) (range self)))

(defmethod (setf x-increment) (val (self pw-pane) &aux sup)
  (setf (slot-value self 'x-increment) val)
  (if (update-flag self)
      (progn
       (pw-update-domain self :invalidate t)
       (if (typep (setq sup (parent self)) 'plot-widget)
	   (pw-repaint sup)
	   (repaint self)))
      (setf (invalidate-flag self) 1)))

(defmethod (setf y-increment) (val (self pw-pane) &aux sup)
  (setf (slot-value self 'y-increment) val)
  (if (update-flag self)
      (progn
       (pw-update-range self :invalidate t)
       (if (typep (setq sup (parent self)) 'plot-widget)
	   (pw-repaint sup)
	   (repaint self)))
      (setf (invalidate-flag self) 2)))

(defmethod (setf mark-points) (val (self pw-pane) &aux sup)
  (unless (eq val (mark-points self))
	  (setf (slot-value self 'mark-points) val)
	  (if (typep (setq sup (parent self)) 'plot-widget)
	      (pw-repaint sup)
	      (repaint self))))

(defmethod (setf x-label) (val (self pw-pane) &aux sup)
  (when (typep (setq sup (parent self)) 'plot-widget)
	(setf (slot-value sup 'x-label) val)
	(pw-repaint sup)))

(defmethod (setf y-label) (val (self pw-pane) &aux sup)
  (when (typep (setq sup (parent self)) 'plot-widget)
	(setf (slot-value sup 'y-label) val)
	(pw-repaint sup)))

(defmethod (setf paints) (val (self pw-pane) &aux sup cdiff)
  (unless (listp val) (setq val (list val)))
  (setq cdiff (- (length (value self)) (length val)))
  (when (plusp cdiff)
	(setq val
	      (nconc val (make-list cdiff 
				    :initial-element (foreground self)))))
  (setf (slot-value self 'paints) val)
  (if (typep (setq sup (parent self)) 'plot-widget)
      (pw-repaint sup)
      (repaint self)))

(defmethod (setf update-flag) (val (self pw-pane) &aux sup)
  (setf (slot-value self 'update-flag) val)
  (when val
	(pw-update-cache 
	 self 
	 :invalidate (case (invalidate-flag self) 
			   (0 nil)
			   (1 'x)
			   (2 'y)
			   (3 t))))
  (setf (slot-value self 'invalidate-flag) 0)
  (if (typep (setq sup (parent self)) 'plot-widget)
      (pw-repaint sup)
      (repaint self)))

(defmethod area ((self pw-pane))
  (list (domain self) (range self)))

(defmethod (setf area) (val (self pw-pane))
  (setf (domain self) (car val)
	(range self) (cadr val))
  (setf (update-flag self) t))

(defmethod (setf rover-update) (val (self pw-pane) &aux dom ran dfact rfact)
  (when (and (invalidate-flag self) 
	     val (setq dom (domain self)) (setq ran (range self)))
	(setq dfact (round (* (- (cdr dom) (car dom)) (car val))))
	(setf (update-flag self) nil)
	(setf (domain self) (cons (+ (car dom) dfact) (+ (cdr dom) dfact)))
	(setq rfact (round (* (- (cdr ran) (car ran)) (cdr val))))
	(setf (range self) (cons (+ (car ran) rfact) (+ (cdr ran) rfact)))
	(setf (update-flag self) t)
	(do-propagate 'area self)))

;;;
;;;	Function to calculate and cache vertex and scaling info
;;;

(defun pw-update-cache (self &key (invalidate nil)) 
  (pw-update-x-axis self :invalidate (or (eq invalidate t) (eq invalidate 'x)))
  (pw-update-y-axis self :invalidate (or (eq invalidate t) (eq invalidate 'y)))
  (pw-update-graph self :invalidate invalidate))

(defun pw-update-domain (self &key (invalidate nil))
  (pw-update-x-axis self :invalidate invalidate)
  (pw-update-x-graph self))

(defun pw-update-range (self &key (invalidate nil))
  (pw-update-y-axis self :invalidate invalidate)
  (pw-update-y-graph self))

;;	Updates x-axis vertex cache
(defun pw-update-x-axis (self &key (invalidate nil) 
			      &aux scale vlist len inc incr diff domain)
  (unless (x-axis self)
	  (return-from pw-update-x-axis))
  ;;	First calculate and store scale
  (setq domain (domain self)
	inc (x-increment self))
  (setq scale (/ (max 0 (- (width self) 3))
		 (max 1 (- (cdr domain) (car domain)))))
  
  (cond (invalidate 
	 ;;	Create vertex-list
	 (setq incr (round (* inc scale))
	       diff (max 0 (- (cdr domain) (car domain))))
	 (when (> 2 incr)
	       (setq inc (max 1 diff))
	       (setq incr (round (* inc scale))))
	 (setq len (max 8 (round (* (1+ (/ diff inc)) 8)))) 
	 (when (> (* incr (/ len 8)) (width self))
	       (decf incr 1))
	 (do ((x 0 (+ x inc)))
	     ((> x diff))
	     (setq vlist (nconc vlist (list incr 0 0 -3 0 6 0 -3))))
	 (setf (car vlist) 3
	       (cadr vlist) (- (height self) 3))
	 (setq inc incr))
	(t
	 ;;	Reposition vertices
	 (setq vlist (x-axis self)
	       inc (round (* inc scale)))
	 (setf (cadr vlist) (- (height self) 3))
	 (setq len (length vlist))
	 (when (> (* inc (/ len 8)) (width self)) 
	       (setq inc (1- inc)))
	 (do ((x 8 (+ x 8)))
	     ((>= x len))
	     (setf (nth x vlist) inc)))) 
  (setf (x-axis self) vlist)
  (setf (slot-value self 'x-scale)
	(/ (max 0 (* inc (1- (/ len 8))))
	   (max 1 (- (cdr domain) (car domain))))))

;;	Updates x-axis vertex cache and returns scale
(defun pw-update-y-axis (self &key (invalidate nil) 
			      &aux scale vlist len inc incr diff range)
  (unless (y-axis self)
	  (return-from pw-update-y-axis))
  ;;	First calculate and store scale
  (setq range (range self)
	inc (y-increment self))
  (setq scale (/ (max 0 (- (height self) 6))
		 (max 1 (- (cdr range) (car range)))))
  
  (cond (invalidate
	 ;;	Create vertex-list 
	 (setq incr (round (* inc scale))
	       diff (max 0 (- (cdr range) (car range))))
	 (when (> 2 incr)
	       (setq inc (max 1 diff))
	       (setq incr (round (* inc scale))))
	 (setq len (max 8 (round (* (1+ (/ diff inc)) 8))))
	 (when (> (* incr (/ len 8)) (height self))
	       (setq incr (1- incr)))
	 (setq incr (- incr))
	 (do ((y 0 (+ y inc)))
	     ((> y diff))
	     (setq vlist (nconc vlist (list 0 incr -3 0 6 0 -3 0))))
	 (setf (cadr vlist) (- (height self) 3)
	       (car vlist) 3)
	 (setq inc incr))
	(t
	 ;;	Reposition vertices
	 (setq vlist (y-axis self)
	       inc (round (* inc scale)))
	 (setf (cadr vlist) (- (height self) 3))
	 (setq len (length vlist))
	 (when (> (* inc (/ len 8)) (width self)) 
	       (setq inc (1- inc)))
	 (setq inc (- inc))
	 (do ((i 9 (+ i 8)))
	     ((>= i len))
	     (setf (nth i vlist) inc))))
  (setf (y-axis self) vlist)
  (setf (slot-value self 'y-scale)
	(/ (max 0 (- (* inc (1- (/ len 8)))))
	   (max 1 (- (cdr range) (car range))))))

;;	Updates graph vertices
(defun pw-update-graph (self &key (invalidate nil) 
			     &aux x-scale y-scale vlist vlists len prev
			     x-cur y-cur opt (diff-points nil) dp pt
			     x-diff y-diff spline io nx ny)
  ;;	Get misc info
  (setq x-scale (x-scale self)
	y-scale (y-scale self)
	vlists (vlists self)
	spline (spline self))
  (setq io (intern-origin self)
	nx (+ 3 (- (round (* x-scale (car (domain self))))))
	ny (+ (- (height self) 3) 
	      (round (* y-scale (car (range self))))))
  (setf (intern-origin self) (cons nx ny))
  (when invalidate
	(when (setq vlists (vlists self))
	      (setq vlists nil))
	(dolist (points (value self))
		(setq len (length points))
		(setq diff-points
		      (cons
		       (setq dp (make-array (list len)))
		       diff-points))
		(setq opt (aref points 0))
		(setq x-cur (x-crd opt) y-cur (y-crd opt))
		(setq vlist (list x-cur y-cur))
		(setf (aref dp 0) (pw-make-point x-cur y-cur))
		(dotimes (i (1- len))
			 (setq opt (aref points (1+ i)))
			 (setq prev x-cur)
			 (setq x-cur (x-crd opt)) 
			 (setq x-diff (- x-cur prev))
			 (setq prev y-cur)
			 (setq y-cur (y-crd opt)) 
			 (setq y-diff (- prev y-cur))
			 (setf (aref dp (1+ i)) (pw-make-point x-diff y-diff))
			 (setq vlist (nconc vlist 
					    (list (round (* x-diff x-scale)) 
						  (round (* y-diff y-scale)))))) 
		(setf (car vlist) (+ (round (* x-scale (car vlist))) nx) 
		      (cadr vlist) (- ny (round (* y-scale (cadr vlist)))))
		(setq vlists (cons vlist vlists)))
	(setf (diff-points self) (reverse diff-points)
	      (vlists self) (reverse vlists))
;;	(pw-move-graph self)
	(return-from pw-update-graph))
  
  ;;	Else don't invalidate
  (dolist (pts (diff-points self))
	  (setq vlist (car vlists))
	  (setq len (length vlist))
	  (do ((i 1 (1+ i))
	       (j 2 (+ j 2)))
	      ((>= j len))
	      (setq pt (aref pts i))
	      (setf (nth (1+ j) vlist)
		    (round (* (y-crd pt) y-scale))
		    (nth j vlist)
		    (round (* (x-crd pt) x-scale))))
	  (setq pt (aref pts 0))
	  (setf (car vlist) (+ (round (* x-scale (x-crd pt))) nx) 
		(cadr vlist) (- ny (round (* y-scale (y-crd pt))))) 
	  (setq vlists (cdr vlists))))

;;	Updates graph vertices x-offsets
(defun pw-update-x-graph (self &aux scale vlist vlists len io ox nx)
  ;;	Get misc info
  (setq scale (x-scale self)
	vlists (vlists self)) 
  (setq io (intern-origin self))
  (setq ox (if io (car io) 0))
  (setq nx (+ 3 (- (round (* (x-scale self) (car (domain self)))))))
  (setf (intern-origin self) (cons nx (cdr io)))
  (dolist (pts (diff-points self))
	  (setq vlist (car vlists))
	  (setq len (length vlist))
	  (do ((i 1 (1+ i)) 
	       (j 2 (+ j 2)))
	      ((>= j len))
	      (setf (nth j vlist) 
		    (round (* (x-crd (aref pts (1- i))) scale)))) 
	  (setf (car vlist) (- (car vlist) ox (- nx)))
	  (setq vlists (cdr vlists))))

;;	Updates graph vertices y-offsets
(defun pw-update-y-graph (self &aux scale vlist vlists len io oy ny)
  ;;	Get misc info
  (setq scale (y-scale self)
	vlists (vlists self))
  (setq io (intern-origin self))
  (setq oy (if io (cdr io) 0))
  (setq ny (+ (- (height self) 3) 
	      (round (* (y-scale self) (car (range self))))))
  (setf (intern-origin self) (cons (car io) ny))
  (dolist (pts (diff-points self))
	  (setq vlist (car vlists))
	  (setq len (length vlist))
	  (do ((i 1 (1+ i))
	       (j 2 (+ j 2)))
	      ((>= j len))
	      (setf (nth (1+ j) vlist)
		    (round (* (y-crd (aref pts (1- i))) scale))))
	  (setf (cadr vlist) (- (cadr vlist) oy (- ny)))
	  (setq vlists (cdr vlists))))

(defun pw-move-axes (self &aux vlist)
  (setq vlist (x-axis self))
  (setf (car vlist) 3
	(cadr vlist) (- (height self) 3))
  (setq vlist (y-axis self))
  (setf (car vlist) 3
	(cadr vlist) (- (height self) 3)))

(defun pw-move-graph (self &aux ox oy nx ny)
  (setq ox (intern-origin self))
  (psetq ox (car ox) oy (cdr ox))
  (setq nx (+ 3 (- (round (* (x-scale self) (car (domain self))))))
	ny (+ (- (height self) 3) 
		 (round (* (y-scale self) (car (range self))))))
  (setf (intern-origin self) (cons nx ny))
  (dolist (vlist (vlists self))
	  (setf (car vlist) (- (car vlist) ox (- nx))
		(cadr vlist) (- (cadr vlist) oy (- ny)))))

(defhandler begin-zoom-box ((self pw-pane) &key x y &allow-other-keys 
			    &aux res gc vlist w h
			    &default :button-press)
  ;;	erase old box
  (setq res (res self)
	gc (gc-invert self))
  (when (setq vlist (box self))
	(xlib:draw-lines res gc vlist :relative-p t))

  ;;	create vertex-list
  (setq w 1 h 1)
  (setf (box self)
	(setq vlist (list x y w 0 0 h (- w) 0 0 (- h))))
  
  ;;	Draw first border
  (xlib:draw-lines res gc vlist :relative-p t))

(defhandler end-zoom-box ((self pw-pane) &rest event &aux vlist
			  &default :button-release)
  (declare (ignore event))
  (when (and (setq vlist (box self))
	     (or (>= 1 (nth 2 vlist))
		 (>= 1 (nth 5 vlist))))
	(setf (box self) nil)))

(defhandler stretch-zoom-box ((self pw-pane) &key x y &allow-other-keys 
			      &aux res gc vlist w h
			      &default :pointer-motion)
  (unless (setq vlist (box self))
	  (return-from pw-pane-stretch-zoom-box))
  
  ;;	erase old box
  (setq res (res self)
	gc (gc-invert self))
  (xlib:draw-lines res gc vlist :relative-p t)
  
  ;;	update vertex-list
  (setf (nth 2 vlist) (setq w (max 1 (- x (car vlist))))
	(nth 5 vlist) (setq h (max 1 (- y (cadr vlist)))))
  (setf (nth 6 vlist) (- w)
	(nth 9 vlist) (- h))
  
  ;;	draw new vertex-list
  (xlib:draw-lines res gc vlist :relative-p t))

;;;	Resizing just updates cache
(defmethod resize-window-handler ((self pw-pane))
  (when (invalidate-flag self)
	(pw-update-cache self :invalidate t)
	(call-next-method)))

;;;
;;;	Instantiate a new pw-pane
;;;

(defmethod new-instance ((self pw-pane)
			 &key 
			 (domain	nil)
			 (range		nil)
			 (axes		t)
			 (x-low		nil)
			 (x-high	nil)
			 (y-low		nil)
			 (y-high	nil)
			 (paints	nil)
			 (no-pan	nil)
			 (no-zoom	nil)
			 &allow-other-keys &aux cdiff limits)

  ;;	Initialize domain & range
  (unless (listp paints) (setq paints (list paints)))
  (setq cdiff (- (length (value self)) (length paints)))
  (when (plusp cdiff)
	(setf (slot-value self 'paints)
	      (nconc paints (make-list cdiff 
				       :initial-element (foreground self)))))

  ;;	Set up normal window stuff
  (call-next-method)

  ;;	Make zoom and rover widgets
  (unless (or no-pan no-zoom) 
	  (make-button 
	   :geom-spec '(:anchor (:top 0 :right 0)) 
	   :parent self
	   :border-width 1
	   :base-size '(20 20)
	   :value (make-image :file "zoom-in.bitmap")
	   :press-func
	   `(when (and (box ',self) (value ',self))
		  (let* ((vlist (box ',self)) 
			 (x (round 
			     (+ (/ (- (car vlist) 3) 
				   (x-scale ',self))
				(car (domain ',self)))))
			 (y (round 
			     (+ (/ (- (height ',self)
				      (cadr vlist) 3)
				   (y-scale ',self))
				(car (range ',self)))))
			 (w (round (/ (nth 2 vlist)
				      (x-scale ',self))))
			 (h (round (/ (nth 5 vlist)
				      (y-scale ',self)))))
		    (setf (box ',self) nil)
		    (setf (update-flag ',self) nil)
		    (setf (domain ',self) (cons x (+ x w))
			  (range ',self) (cons (- y h) y))
		    (do-propagate 'area ',self)
		    (setf (update-flag ',self) t)))) 
	  (make-button 
	   :geom-spec '(:anchor (:top 24 :right 0)) 
	   :parent self
	   :border-width 1
	   :base-size '(20 20)
	   :value (make-image :file "zoom-out.bitmap")
	   :press-func `(let ((limits (limits ',self)))
			  (setf (update-flag ',self) nil)
			  (setf (domain ',self) (car limits))
			  (setf (range ',self) (cadr limits))
			  (do-propagate 'area ',self)
			  (setf (update-flag ',self) t))) 
	  (bind-slot 'rover-update self 
		     `(var value ,(make-rover-widget 
				   :parent self
				   :geom-spec '(:anchor (:top 0 :right 24))
				   :base-size '(44 44)
				   :base-height 45 :border-width 1))))

  ;;	Set limits
  (setq limits (compute-limits self))
  (unless domain
	  (setq domain
		(cond ((not (or domain x-low x-high))
		       (car limits))
		      ((and x-low x-high)
		       (cons x-low x-high))
		      (x-low
		       (cons x-low (caar limits)))
		      (x-high
		       (cons (cdar limits) x-high)))))
  (unless range
	  (setq range
		(cond ((not (or range y-low y-high))
		       (cadr limits))
		      ((and y-low y-high)
		       (cons y-low y-high))
		      (y-low
		       (cons y-low (caadr limits)))
		      (y-high
		       (cons (cdadr limits) y-high)))))
  (setf (slot-value self 'domain) domain
	(slot-value self 'range) range
	(slot-value self 'invalidate-flag) 0)

  ;;	Initialize axes
  #|(case axes
	(nil nil)
	('x (pw-update-x-axis self :invalidate t))
	('y (pw-update-y-axis self :invalidate t))
	(t (pw-update-x-axis self :invalidate t)
	   (pw-update-y-axis self :invalidate t)))|#
  (setf (value self) (slot-value self 'value)))

;;;
;;;	Draw axes, graphs
;;;

(defmethod do-repaint ((self pw-pane)
		       &aux
		       res gc gc-graph
		       x y paint
		       clrs box
		       labels
		       label
		       label-index
		       font
		       mark-points
		       )
  (setq res (res self)
	gc (gc-res self)
	gc-graph (gc-graph self)
	clrs (paints self)
	labels (curve-labels self)
	label-index 0
	font (font self)
	mark-points (mark-points self)
	)
  
  ;;	Draw axes
  (when (setq x (x-axis self))
	(xlib:draw-lines res gc x :relative-p t))
  (when (setq y (y-axis self))
	(xlib:draw-lines res gc y :relative-p t))
  
  ;;	Draw graphs
  (dolist (vlist (vlists self))
	  (setq paint (car clrs))
	  (setq clrs (cdr clrs))
	  (if (color-p paint) 
	      (setf (xlib:gcontext-foreground gc-graph) (pixel paint)
		    (xlib:gcontext-fill-style gc-graph) :solid)
	      (setf (xlib:gcontext-tile gc-graph) (res paint)
		    (xlib:gcontext-fill-style gc-graph) :tiled))
	  (xlib:draw-lines res gc-graph vlist :relative-p t)

	  (when labels
		;; label on the middle point
		(setq label-index (* 2
				     (truncate (length vlist) 4)))
		(setq label (car labels))
		(setq labels (cdr labels)))
	  
	  (setq x (car vlist)
		y (cadr vlist)) 
	  (if mark-points (xlib:draw-rectangle res gc (1- x) (1- y) 3 3 t))
	  (do ((i 2 (+ i 2)))
	      ((>= i (length vlist))) 
	      (setq x (+ x (nth i vlist)) 
		    y (+ y (nth (1+ i) vlist))) 
	      (if mark-points (xlib:draw-rectangle res gc (1- x) (1- y) 3 3 t))
	      (when (= i label-index)
		    (put label :window self :font font
			 :horiz-just :left :vert-just :top
			 :x x
			 :y y)
		    )
	      ))
  
  (when (setq box (box self))
	(xlib:draw-lines res (gc-invert self) box :relative-p t)))

;;;
;;;	Plot-widget constructor
;;;

(defun make-plot-widget (&rest args)
  (apply #'make-instance 'plot-widget :allow-other-keys t args))

(defmacro plot-widget-p (pb)
  `(typep ,pb 'plot-widget))

;;;
;;;	Plot-widget accessors
;;;

(defmethod value ((self plot-widget))
  (value (pane self)))

(defmethod (setf value) (val (self plot-widget) &aux pane)
  (when (setq pane (pane self))
	(setf (value pane) val)))

(defmethod spline ((self plot-widget))
  (spline (pane self)))

(defmethod (setf spline) (val (self plot-widget) &aux pane)
  (when (setq pane (pane self))
	(setf (spline pane) val)))

(defmethod paints ((self plot-widget))
  (paints (pane self)))

(defmethod (setf paints) (val (self plot-widget) &aux pane)
  (when (setq pane (pane self))
	(setf (paints pane) val)))

(defmethod range ((self plot-widget))
  (range (pane self)))

(defmethod (setf range) (val (self plot-widget))
  (setf (range (pane self)) val))

(defmethod domain ((self plot-widget))
  (domain (pane self)))

(defmethod (setf domain) (val (self plot-widget))
  (setf (domain (pane self)) val))

(defmethod x-increment ((self plot-widget))
  (x-increment (pane self)))

(defmethod (setf x-increment) (val (self plot-widget))
  (setf (x-increment (pane self)) val))

(defmethod y-increment ((self plot-widget))
  (y-increment (pane self)))

(defmethod (setf y-increment) (val (self plot-widget))
  (setf (y-increment (pane self)) val))

(defmethod x-low ((self plot-widget))
  (x-low (pane self)))

(defmethod (setf x-low) (val (self plot-widget))
  (setf (x-low (pane self)) val))

(defmethod x-high ((self plot-widget))
  (x-high (pane self)))

(defmethod (setf x-high) (val (self plot-widget))
  (setf (x-high (pane self)) val))

(defmethod y-low ((self plot-widget))
  (y-low (pane self)))

(defmethod (setf y-low) (val (self plot-widget))
  (setf (y-low (pane self)) val))

(defmethod y-high ((self plot-widget))
  (y-high (pane self)))

(defmethod (setf y-high) (val (self plot-widget))
  (setf (y-high (pane self)) val))

(defmethod update-flag ((self plot-widget))
  (update-flag (pane self)))

(defmethod (setf update-flag) (val (self plot-widget))
  (setf (update-flag (pane self)) val))

(defmethod (setf font) (val (self plot-widget))
  (attach val)
  (setf (slot-value self 'font) val)
  (repaint self))

(defmethod (setf mark-font) (val (self plot-widget))
  (attach val)
  (setf (slot-value self 'mark-font) val)
  (repaint self))

(defmethod (setf x-label) (val (self plot-widget))
  (setf (x-label (pane self)) val))

(defmethod (setf y-label) (val (self plot-widget))
  (setf (y-label (pane self)) val))

(defmethod curve-labels ((self plot-widget))
  (curve-labels (pane self)))

(defmethod (setf curve-labels) (val (self plot-widget))
  (setf (curve-labels (pane self)) val))

;;;
;;;	Instantiate a new plot-widget
;;;

(defmethod new-instance ((self plot-widget)
			 &rest args
			 &key 
			 (x-label	"")
			 (y-label	"")
			 (x-pad		nil)
			 (y-pad		nil)
			 (font		(get-font))
			 (mark-font	(get-font))
			 &allow-other-keys)
  (call-next-method)
  (setf (font self) font)
  (setf (mark-font self) mark-font)
  (unless x-pad (setf (slot-value self 'x-pad) 
		      (setq x-pad
			    (+ (max (text-width y-label :font font)
				    (text-width "0000.0" :font font))
			       5))))
  (unless y-pad (setf (slot-value self 'y-pad)
		      (setq y-pad (+ (font-height font) 5))))
  (remf args :geom-spec)
  (remf args :x-offset)
  (remf args :y-offset)
  (remf args :base-width)
  (remf args :base-height)
  (remf args :location)
  (remf args :region)
  (remf args :border-width)
  (remf args :parent)
  (setf (slot-value self 'pane)
	(apply #'make-instance 'pw-pane :allow-other-keys t
	       `(:parent ,self :x-offset ,x-pad 
			 :geom-spec ,(list 0 y-pad) 
			 :border-width 0 
			 ,@args 
			 :parent ,self))))

;;;
;;;	Redraw
;;;

(defmethod do-repaint ((self plot-widget)
		       &aux font mark-font x-pad y-pad x-diff y-diff x-low 
		       x-high y-low y-high pane xw yh temp)
  (setq x-pad (x-pad self)
	y-pad (y-pad self)
	pane (pane self))
  
  ;;	Draw graph-panel
  (call-next-method)
  
  ;;	Draw labels
  (setq font (font self)
	mark-font (mark-font self))
  (when (setq temp (y-label self))
	(put temp :window self :font font :height (height self) 
	     :width x-pad :horiz-just :center :vert-just :center))
  (when (setq temp (x-label self))
	(put temp :window self :font font :y (- (height self) y-pad)
	     :width (width self) :height y-pad
	     :horiz-just :center :vert-just :center))
  
  ;;
  ;;	Draw markers
  ;;
  (setq x-low (princ-to-string (x-low self))
	x-high (princ-to-string (x-high self))
	y-low (princ-to-string (y-low self))
	y-high (princ-to-string (y-high self)))
  (setq xw (text-width x-low :font mark-font)
	yh (font-height mark-font))
  (setq x-diff (max 0 (- x-pad xw))
	y-diff (max 0 (- (height self) y-pad)))
  (when (x-axis pane)
	(put x-low :window self :font mark-font 
	     :x (x-offset pane) :y (+ y-diff 3)
	     :width (width y-low :font mark-font) :height yh)
	(put x-high :window self :font mark-font
	     :x (max 0 (+ x-pad (width pane) 
			  (- (width x-high :font mark-font)))) 
	     :y (+ y-diff 3)
	     :width (width x-high :font mark-font) :height yh))
  (when (y-axis pane)
	(put y-low :window self :font mark-font 
	     :x (max 0 (- x-pad (width y-high :font mark-font)))
	     :y (max 0 (- y-diff yh -3)) 
	     :width xw :height yh)
	(put y-high :window self :font mark-font
	     :x (max 0 (- x-pad (width y-high :font mark-font)))
	     :y (max 0 (+ (y-offset pane) yh))
	     :width (width y-high :font font) :height yh)))

(defun pw-repaint (pw &aux pane)
  (repaint pw)
  (when (setq pane (pane pw))
	(repaint pane)))

(defmethod do-attach ((self plot-widget) &aux temp)
  ;;	Attach self
  (call-next-method)
  ;;	Attach paints
  (when (setq temp (paints self))
	(setf (repaint-flag (pane self)) nil)
	(setf (paints self)
	      (mapcar #'(lambda (x) 
				(when (stringp x)
				      (if (get-paint x)
					  (setq x (get-paint x))
					  (setq x (make-color :name x))))
				(if (not (paint-p x))
				    (setq x (foreground self)))
				(when (stringp x)
				      (if (get-paint x)
					  (setq x (get-paint x))
					  (setq x (make-color :name x))))
				(if (not (paint-p x))
				    (setq x (get-paint "black")))
				(if (color-p x) 
				    (color-attach x)
				    (setf (window x) (pane self))) 
				x)
		      temp))
	(setf (repaint-flag (pane self)) t)))

(defmethod do-detach ((self plot-widget))
  ;;	First detach self
  (call-next-method)
  ;;	Detach paints
  (dolist (c (paints self))
	  (setq c (get-paint c))
	  (if (paint-p c)
	      (do-detach c))))
