;;;
;;; Copyright (c) 1990 Regents of the University of California
;;; 
;;; $Author: picasso $
;;; $Source: RCS/line-string.cl,v $
;;; $Revision: 1.1 $
;;; $Date: 90/07/17 21:11:44 $
;;;
;;;
;;; Some functions to manipulate line-strings, which are lists
;;; of points that describe a polygon.  If the closed parameter to the
;;; functions is true, the line-string is interpreted as closed; ie,
;;; that its first and last vertices are connected.
;;;
;;;

(in-package 'pt :use '(lisp pcl))

;;;
;;; dopoints is a macro to iterate through all the points of a line string.
;;; p1 and p2 are assigned succesive pairs of the line segments for the line
;;; string passed.
;;;
(defmacro dopoints ((p1 p2 linestr closed) &rest body)
  `(let* ((,p1 (car ,linestr))
	  (,p2 nil))
	 (dolist (,p2 (cdr ,linestr))
		 ,@body
		 (setq ,p1 ,p2))
	 ;; Do wrap-around
	 (when ,closed
	       (setq ,p2 (car ,linestr))
	       ,@body)))

;;;
;;; linestr-gravity-pt finds the gravity points for the line string passed.
;;; It returns a list of param-vals that describe the gravity points of the
;;; line string.  If do-midpt is t, the midpoints are included as gravity
;;; points.
;;;
(defun linestr-gravity-pt (linestr closed do-midpt)
  (if (null linestr)
      nil
      (let ((rv nil)
	    (len (length linestr)))
	   (if (not closed)
	       (decf len))
	   (dotimes (l-num len)
		    (push l-num rv)
		    (if do-midpt
			(push (+ l-num 0.5) rv)))
	   (if (not closed)
	       (push (- len 0.0001) rv))
	   rv)))
  
;;;
;;; Return the point on a line string at a given parametric value.
;;; The parametric value for a line string is [0-n), where n is the number
;;; of vertices in the line string.
;;;
(defun linestr-point (linestr closed value)
  (let* ((n (floor value))
	 (nv (length linestr))
	 (temp nil)
	 (pt1 nil)
	 (pt2 nil)
	 (l nil))
	(if (not closed)
	    (decf nv))
	(when (or (> value nv) (< value 0))
	      (warn "Invalid reference to sub-part of line string")
	      (return-from linestr-point nil))
        (if (= value nv)
	    (setq value 0.0 n 0)
	    (setq value (- value n)))
	(setq temp (nthcdr n linestr))
	(setq pt1 (car temp))
	(if (null (second temp))
	    (setq pt2 (car linestr))
	    (setq pt2 (second temp)))
	(setq l (2dv-scale! (2dv- pt2 pt1) value))
	(2dv+! l pt1)
	l))
 
;;;
;;; Return the normal to a line string at a given parametric value.
;;; The parametric value for a line string is [0-n), where n is the number
;;; of vertices in the line string.
;;;
(defun linestr-normal (linestr closed value)
  (let* ((n (floor value))
	 (nv (length linestr))
	 (temp nil)
	 (pt1 nil)
	 (pt2 nil)
	 (rv nil)
	 (l nil))
	(if (not closed)
	    (decf nv))
	(when (or (>= value nv) (< value 0))
	      (warn "Invalid reference to sub-part of line string")
	      (return-from linestr-normal nil))
	(setq value (- value n))
	(setq temp (nthcdr n linestr))
	(setq pt1 (car temp))
	(if (null (second temp))
	    (setq pt2 (car linestr))
	    (setq pt2 (second temp)))
	(setq l (2dv-normalize! (2dv- pt2 pt1)))
	(setq rv (alloc-2d (2d-point-y l) (- (2d-point-x l))))
	(free-2d l)
	rv))
 
;;;
;;; Return the point and normal to a line string at a given parametric value.
;;; The parametric value for a line string is [0-n), where n is the number
;;; of vertices in the line string.
;;;
(defun linestr-pt-normal (linestr closed value)
  (list (linestr-point linestr closed value)
	(linestr-normal linestr closed value)))
 
;;;
;;; Find the nearest point on the line string to the point pt.
;;; Return a list of (parametric-value dist).
;;;
(defun nearest-pt-to-linestr (linestr closed pt)
  (let* ((best (list 0 0))
	 (temp nil)
	 (l-num 0)
	 (min-dist most-positive-single-float))
	(dopoints (pt1 pt2 linestr closed)
		  (setq temp (pt-line-dist pt1 pt2 pt))
		  (when (< (2d-point-y temp) min-dist)
			(setf (car best) (2d-point-x temp)
			      (cadr best) (2d-point-y temp))
			(incf (first best) l-num)
			(setq min-dist (second best)))
		  (free-2d temp)
		  (incf l-num))
	best))

;;;
;;; Return the bounding box of the line-string passed.
;;;
(defun linestr-bbox (line-string)
  (if (null line-string)
      nil
      (let* ((min-x most-positive-single-float)
	     (min-y most-positive-single-float)
	     (max-x most-negative-single-float)
	     (max-y most-negative-single-float)
	     (x 0)
	     (y 0))
	    (dolist (p1 line-string)
		    (setq x (2d-point-x p1)
			  y (2d-point-y p1)
			  min-x (min min-x x)
			  min-y (min min-y y)
			  max-x (max max-x x)
			  max-y (max max-y y)))
	    (make-2d-bbox min-x min-y max-x max-y))))
