;;;
;;; Copyright (c) 1990 Regents of the University of California
;;;
;;; Graphic gadget: A gadget to display the contents of a graphic data
;;; structure.  Unless otherwise noted, all structures are 2D.
;;;
;;; A graphic gadget inherits mapping abilities from the 2d-mapper-mixin
;;; which gives it the ability to map from a floating point world coordinate
;;; system into the device coordinate system of it's window.  See the file
;;; mapper.cl for more details.
;;;
;;; The value displayed by a graphic-gadget is a 2d-shape, which is actually
;;; a tree of 2d-shapes.  See the file shape.cl for details about the
;;; construction and manipulation of these objects.
;;;

(in-package 'pt :use '(lisp pcl))

;;;
;;; Class definition for graphic-gadget
;;;
(defclass graphic-gadget (2d-mapper-mixin gadget) 
  (
   ; Inherited slots -- initial-values
   (background-pixmap :initform "black")
   (background :initform "black")
   (foreground :initform "white")

   ; The "value" of a graphic-gadget is a shape to display. These slots
   ; hold sorted, flattened version of the value slot, and are 
   ; incrementally maintained.
   (segments :initform nil :type list :accessor segments)
   (annotations :initform nil :type list :accessor annotations)
   (other-objs :initform nil :type list :accessor other-objs)

   ;----------------------------------------------------------------
   ; Internal slots.
   (dl-tab :initform (make-hash-table) :type hash-table :accessor dl-tab)
   ))

;;;
;;; Constructor Function...
;;;
(defun make-graphic-gadget (&rest keys)
  (apply #'make-instance (cons 'graphic-gadget keys)))

;;;
;;; Define the setf method on value that's an shape...
;;;
(defmethod (setf value) ((value shape) (self graphic-gadget))
  (let* ((obj-list (flatten value))
	 (new-annotations (remove-if-not #'annotation-p obj-list))
	 (new-segments (remove-if-not #'segment-p obj-list)))
	(setq obj-list (delete-if #'annotation-p obj-list))
	(setq obj-list (delete-if #'segment-p obj-list))
	(update-segment-list new-segments self)
	(update-annotation-list new-annotations self)
	(setf (slot-value self 'other-objs) obj-list)
	(setf (slot-value self 'value) value)
	(repaint self :clear t)))

(defmethod new-instance ((self graphic-gadget)
			 &key
			 (value nil)
			 &allow-other-keys)
  ;; Let mapper initialize itself.
  (call-next-method)

  ;; Set the value slot using the setf method...
  (setf (slot-value self 'value) nil)
  (setf (value self) value)
  (zoom-extent self)
  self)

;;;
;;; Other methods that affect the world coordinate system
;;;
(defmethod zoom-extent ((self graphic-gadget) &aux bb)
  (setq bb (2d-bb-union (bbox (value self)) (sub-objs-bbox (value self))))
  (if bb
      (let* ((xmin (2d-bb-xmin bb))
	     (ymin (2d-bb-ymin bb))
	     (xmax (2d-bb-xmax bb))
	     (ymax (2d-bb-ymax bb))
	     (dx (- xmax xmin))
	     (dy (- ymax ymin)))
	    (set-world self 
		       (- xmin (/ dx 20.0)) (- ymin (/ dy 20.0)) 
		       (+ xmax (/ dx 20.0)) (+ ymax (/ dy 20.0))))))

;;;
;;; Recacluate the world to device mapping function of a graphic-gadget.
;;;
(defmethod recache-map ((self graphic-gadget))
  (call-next-method)
  (update-all-segs self))

(defmethod resize-window-handler ((self graphic-gadget))
  (recache-map self))

;;;
;;; Incrementally maintain the segments slot
;;;
(defun update-segment-list (new-segments self)
  (let* ((old-segments (segments self))
	 (to-add (set-difference new-segments old-segments))
	 (to-del (set-difference old-segments new-segments)))
	(dolist (obj to-del)
		(delete-seg self obj)
		(delete-viewer obj self))
	(dolist (obj to-add)
		(add-seg self obj)
		(add-viewer obj self))
	(setf (slot-value self 'segments) new-segments)))

;;;
;;; Incrementally maintain the annotations slot
;;;
(defun update-annotation-list (new-annotations self)
  (let* ((old-annotations (annotations self))
	 (to-add (set-difference new-annotations old-annotations))
	 (to-del (set-difference old-annotations new-annotations)))
	(dolist (obj to-del)
		(delete-seg self obj)
		(delete-viewer obj self))
	(dolist (obj to-add)
		(add-seg self obj)
		(add-viewer obj self))
	(setf (slot-value self 'annotations) new-annotations)))

;;; ====================================================================
;;;
;;; Convert a line-string to a segment-dl for drawing.  This code should
;;; be moved into the segment-dl code.
;;;
(defun coord-list-to-dl (cl dl &aux (move-only t) pt x y)
  (clear-segment-dl dl)
  (dotimes (i (fill-pointer cl))
	   (setq pt (aref cl i))
	   (if pt
	       (progn
		(setq x (float (2d-point-x pt))
		      y (float (2d-point-y pt)))
		(free-2d pt)
		(if move-only
		    (mov-abs dl x y)
		    (lin-abs dl x y))
		(setq move-only nil))
	       ;; pt is nil -- indicates break in line.
	       (setq move-only t)))
  dl)

(defun draw-objects (self objlist)
  (let ((dl nil)
	(dl-tab (dl-tab self)))
       (dolist (obj objlist)
	       (setq dl (gethash obj dl-tab))
	       (if (visible dl) (draw dl self)))))

(defun erase-objects (self objlist)
  (let ((dl nil)
	(dl-tab (dl-tab self)))
       (dolist (obj objlist)
	       (setq dl (gethash obj dl-tab))
	       (if (visible dl) (erase dl self)))))

;;;
;;; Make sure all graphic properties and display-lists associated with
;;; the annotation are set up.
;;;
(defun update-annotation (self shape &aux (dl-tab (dl-tab self)))
  (let ((dl (gethash shape dl-tab)))
       ;; Make sure display-list is set up
       (if (not (annot-dl-p dl))
	   (setf (gethash shape dl-tab)
		 (setq dl (make-annot-dl :shape shape))))
       (invalidate dl)))

;;;
;;; Make sure all graphic properties and display-lists associated with
;;; the segment are set up.
;;;
(defmethod update-seg ((self graphic-gadget) shape
		       &aux
		       (ht (dl-tab self))
		       (xleft (xleft self))
		       (xright (xright self))
		       (ybottom (ybottom self))
		       (ytop (ytop self))
		       (w (width self))
		       (h (height self)))
  (let* ((dl (gethash shape ht))
	 (cl (2d-coord-list shape xleft xright ybottom ytop w h)))
       (if (not (segment-dl-p dl))
	   (setf (gethash shape ht)
		 (setq dl (make-segment-dl :shape shape))))
       (invalidate dl)
       (if cl (coord-list-to-dl cl dl))))

(defmethod update-all-segs ((self graphic-gadget) &aux ht count dl)
  (if (not (and (attached-p self) (exposed-p self)))
      (return-from update-all-segs))
  (clear self)
  (setq ht (dl-tab self))
  (setq count 0)
  (dolist (obj (segments self))
	  (setq dl (gethash obj ht))
	  (cond ((null dl)
		 (update-seg self obj)
		 (setq dl (gethash obj ht)))
		((visible dl)
		 (update-seg self obj))
		(t
		 (invalidate dl)))
	  (when (visible dl)
		(validate dl self)
		(draw dl self)
		(incf count)
		(when (> count 20)
		      (flush-display)
		      (setq count 0))))
  (dolist (obj (annotations self))
	  (setq dl (gethash obj ht))
	  (cond ((null dl)
		 (update-annotation self obj)
		 (setq dl (gethash obj ht)))
		((visible dl)
		 (update-annotation self obj))
		(t
		 (invalidate dl)))
	  (when (visible dl)
		(validate dl self)
		(draw dl self)
		(incf count)
		(when (> count 20)
		      (flush-display)
		      (setq count 0)))))

;;;==================================================================
;;;
;;; Add a seg for a shape to the hash table. 
;;; Returns new seg if it was added, nil if already in hash table.
;;;
(defmethod add-seg ((self graphic-gadget) shape)
  (if (null (gethash shape (dl-tab self)))
      (cond ((annotation-p shape)
	     (update-annotation self shape))
	    ((segment-p shape)
	     (update-seg self shape)))))

;;;
;;; Delete the seg of a shape from the hash table. 
;;; Returns t if it was deleted, nil if not in hash table.
;;;
(defmethod delete-seg ((self graphic-gadget) shape 
			 &aux (ht (dl-tab self)))
  (let* ((old-seg (gethash shape ht)))
	(if old-seg (remhash shape ht))))

(defmethod sub-obj-added ((self graphic-gadget) shape obj)
  (declare (ignore shape))
  (let* ((to-add (flatten obj))
	 (new-segments nil)
	 (new-annotations nil)
	 (new-others nil))

	;; Go through all the sub objects, registering ourselves as a
	;; viewer, adding a seg and drawing the new shape.
	(dolist (obj to-add)
		(add-seg self obj)
		(add-viewer obj self))
	(draw-objects self to-add)

	;; Put the annotations in one list and the segments in another.
	(setf new-annotations (remove-if-not #'annotation-p to-add)
	      new-segments (remove-if-not #'segment-p to-add)
	      new-others (delete-if #'segment-p to-add)
	      new-others (delete-if #'annotation-p to-add))
	(setf (slot-value self 'other-objs)
	      (nconc (other-objs self) new-others))
	(setf (slot-value self 'segments)
	      (nconc (segments self) new-segments))
	(setf (slot-value self 'annotations)
	      (nconc (annotations self) new-annotations))))

(defmethod sub-obj-deleted ((self graphic-gadget) shape obj)
  (declare (ignore shape))
  (let ((to-del (flatten obj)))
       
       ;; Go through all the sub objects, unregistering ourselves as a 
       ;; viewer, deleting the seg and erasing the shape.
       (erase-objects self to-del)
       (dolist (obj to-del)
	       (delete-seg self obj)
	       (delete-viewer obj self))
       
       ;; Update the annotations and segment lists.
       (setf (slot-value self 'other-objs) 
	     (nset-difference (other-objs self) to-del))
       (setf (slot-value self 'segments) 
	     (nset-difference (segments self) to-del))
       (setf (slot-value self 'annotations) 
	     (nset-difference (annotations self) to-del))))

(defun set-visibility-recursively (self shape visible 
					&optional (ht (dl-tab self)))
  ;; Convert to t/nil value
  (setq visible (not (null visible)))
  (let ((dl (gethash shape ht)))
       (unless (display-list-p dl)
	       (if (segment-p shape) (update-seg self shape))
	       (if (annotation-p shape) (update-annotation self shape))
	       (setq dl (gethash shape ht)))
       (when (and (display-list-p dl) (not (eq visible (visible dl))))
	     (setf (visible dl) visible)
	     ;; Update display...
	     (if visible (draw dl self) (erase dl self)))
       ;; Do recursive setf
       (mapc #'(lambda (x) (set-visibility-recursively self x visible ht))
	     (sub-objs shape))))

(defun shape-visible (self shape)
  (let ((dl (gethash shape (dl-tab self))))
       (and (display-list-p dl) (visible dl))))

(defun shape-color (self shape)
  (let ((dl (gethash shape (dl-tab self))))
       (if (slot-exists-p 'color dl)
	   (slot-value dl 'color))))

(defun shape-line-width (self shape)
  (let ((dl (gethash shape (dl-tab self))))
       (if (slot-exists-p 'line-width dl)
	   (slot-value dl 'line-width))))

(defun translate-paint (win color)
  (if (stringp color)
      (setq color (get-paint color)))
  (cond ((image-p color)
	 (make-tile :window win :image color))
	((color-p color) color)
	(t nil)))

(defun set-line-width-recursively (self shape line-width 
					&optional (ht (dl-tab self)))
  ;; Convert to t/nil value
  (setq color (translate-paint self color))
  (let ((dl (gethash shape ht)))
       (unless (display-list-p dl)
	       (if (segment-p shape) (update-seg self shape))
	       (if (annotation-p shape) (update-annotation self shape))
	       (setq dl (gethash shape ht)))
       (when (and (display-list-p dl) 
		  (slot-exists-p dl 'line-width)
		  (not (eq line-width (line-width dl))))
	     (if (visible dl) (erase dl self))
	     (setf (line-width dl) line-width)
	     ;; Update display...
	     (if (visible dl) (draw dl self)))
       ;; Do recursive setf
       (mapc #'(lambda (x) (set-line-width-recursively self x line-width ht))
	     (sub-objs shape))))

(defun set-color-recursively (self shape color 
					&optional (ht (dl-tab self)))
  ;; Convert to t/nil value
  (setq color (translate-paint self color))
  (let ((dl (gethash shape ht)))
       (unless (display-list-p dl)
	       (if (segment-p shape) (update-seg self shape))
	       (if (annotation-p shape) (update-annotation self shape))
	       (setq dl (gethash shape ht)))
       (when (and (display-list-p dl) 
		  (slot-exists-p dl 'color)
		  (not (eq color (color dl))))
	     (setf (color dl) color)
	     ;; Update display...
	     (if (visible dl) (draw dl self) (erase dl self)))
       ;; Do recursive setf
       (mapc #'(lambda (x) (set-color-recursively self x color ht))
	     (sub-objs shape))))

(defmethod gc-changed ((self graphic-gadget) shape &aux (ht (dl-tab self)))
  (when (attached-p self)
	(let ((dl (gethash shape ht))
	      (sl (list shape)))
	     ;; Find the old seg and erase it from the screen.
	     (erase-objects self (list shape))
	     (invalidate dl)
	     (draw-objects self (list shape)))))

(defmethod geom-changed ((self graphic-gadget) shape)
  (when (attached-p self)
	;; Find the old seg and erase it from the screen.
	(erase-objects self (list shape))
	;; Make new ones and redraw
	(update-seg self shape)
	(draw-objects self (list shape))))

;;;
;;; Define a function to repaint the graphic-gadget
;;;
(defmethod do-repaint ((self graphic-gadget)
		       &key
		       (clear t)
		       &allow-other-keys)
  (call-next-method)
  (if clear (clear self))
  (draw-objects self (segments self))
  (draw-objects self (annotations self)))
