;;;
;;; Copyright (c) 1990 Regents of the University of California
;;;
;;; Definition for box, a subclass of shapes.
;;;
;;; Boxes are types of shapes whose control points are the corners
;;; of the box and are constrined to be rectilinear.
;;;

(in-package 'pt :use '(lisp pcl))

;;;
;;; Are we a box? Isn't that special!
;;;
(defmacro box-p (self)
  `(typep ,self 'box))

;;;
;;; Class definition for box
;;;
(defclass box (polygon)
  ((width :initarg :width :initform 1.0 :type float :reader width)
   (height :initarg :height :initform 1.0 :type float :reader height)
   (closed :initform t)))

(defun update-box-ctrl-pts (self)
  (let* ((ctrl-pts (ctrl-pts self))
	 (hook-pt (hook-pt self))
	 (x (2d-point-x hook-pt))
	 (y (2d-point-y hook-pt))
	 (w (width self))
	 (h (height self)))
	(if (null ctrl-pts)
	    (setq ctrl-pts (list
			    (make-2d-point :x 0.0 :y 0.0)
			    (make-2d-point :x 0.0 :y 0.0)
			    (make-2d-point :x 0.0 :y 0.0)
			    (make-2d-point :x 0.0 :y 0.0))))
	(copy-2d (first ctrl-pts) x y)
	(copy-2d (second ctrl-pts) (+ x w) y)
	(copy-2d (third ctrl-pts) (+ x w) (+ y h))
	(copy-2d (fourth ctrl-pts) x (+ y h))
	(setf (slot-value self 'ctrl-pts) ctrl-pts)))

(defmethod (setf hook-pt) (hook-pt (self box))
  (2dv-copy (slot-value self 'hook-pt) hook-pt)
  (update-box-ctrl-pts self)
  (call-next-method))

(defmethod (setf width) (w (self box))
  (setf (slot-value self 'width) (float w))
  (update-box-ctrl-pts self)
  (notify-geom-change self))

(defmethod (setf height) (h (self box))
  (setf (slot-value self 'height) (float h))
  (update-box-ctrl-pts self)
  (notify-geom-change self))

(defmethod new-instance ((self box)
			 &key
			 (width 1.0)
			 &allow-other-keys)
  (declare (ignore width))
  (call-next-method)
  (update-box-ctrl-pts self)
  self)

(defun make-box (&rest keys)
  (apply #'make-instance (cons 'box keys)))

;;;
;;; Rotate a box about (ox, oy).  Just rotates the hook-pt about (ox, oy).
;;;
(defmethod 2d-rotate ((self box) theta ox oy)
  (declare (ignore theta ox oy))
  (let ((viewers (viewers self)))
       (setf (slot-value self 'bbox) nil)
       (call-next-method)
       (update-box-ctrl-pts self)
       (notify-geom-change self)))

(defmethod 2d-scale ((self box) sf ox oy)
  (declare (ignore ox oy))
  (setf (slot-value self 'width) (* (slot-value self 'width) sf))
  (setf (slot-value self 'height) (* (slot-value self 'height) sf))
  (call-next-method))
