;;;
;;; Copyright (c) 1990 Regents of the University of California
;;;
;;; Definition for 2d-shapes.  
;;;
;;; All 2d-shapes have a list of control points.
;;;
;;; The control point (ctrl-pts) are points that are active and, in some
;;; sense, describe the shape.  For a rectangle, these are the four sides of
;;; the rectangle.  For a circle, they are the center and four points that are
;;; the radius away along each of the orthogonal axes.  For a bezier curve,
;;; they are the points of the control polygon.
;;;
;;; All 2d-shapes provide a method to return the coord-list of a shape. The
;;; coord-list is an extendable vector of (x y) pairs that
;;; describe, in a world coordinate system, how to draw the 2d-shape.  A nil
;;; value in this array indicates a break in the line.  The method is
;;; passed all the information needed to create a this list of points.
;;;
;;; All 2d-shapes provide a paramterization that describes a sub-part of the
;;; 2d-shape to calling routines.  For example, the parameterization of a
;;; circle is 0 to 2*pi, and the parameterization of a polygon is [0-n),
;;; [0-1) is the first edge, [1-2) is the second edge, etc.  Each 2d-shape
;;; provides methods, to return the point and normal on the 2d-shape at the
;;; given parametric value.
;;;
;;; The bbox method is provided to obtained a bounding box of the 2d-shape.
;;;


(in-package 'pt :use '(lisp pcl))

;;;
;;; Are we a 2d-shape? Isn't that special!
;;;
(defmacro 2d-shape-p (self)
  `(typep ,self '2d-shape))

;;;
;;; Class definition for 2d-shape
;;;
(defclass 2d-shape (shape) 
  ((coord-list :initform nil :type array)
   (ctrl-pts :initform nil :type list :reader ctrl-pts)
   (bbox :initform nil :type list)))

(defun make-2d-shape (&rest keys)
  (apply #'make-instance (cons '2d-shape keys)))

(defmethod (setf ctrl-pts) (ctrl-pts (self 2d-shape))
  ;; Convert control points...
  (do* ((cp ctrl-pts (cdr cp)))
       ((null cp))
       (if (consp (car cp))
	   (setf (car cp) (apply #'alloc-2d (car cp)))))
  (setf (slot-value self 'ctrl-pts) ctrl-pts))

(defmethod new-instance ((self 2d-shape)
			 &key
			 (ctrl-pts nil)
			 &allow-other-keys)
  (setf (ctrl-pts self) ctrl-pts))

;;; ======================= Notify methods ====================

(defmethod notify-geom-change ((self 2d-shape))
  (setf (slot-value self 'bbox) nil)
  (call-next-method))

;;; ======================= Methods that affect geometry ======
;;;
;;; Translate a 2d-shape.  Just affects the sub-objs.
;;;
(defmethod 2d-translate ((self 2d-shape) tx ty)
  (dolist (obj (sub-objs self))
	  (2d-translate obj tx ty)))

;;;
;;; Rotate a 2d-shape about the axis and origin.  Just affects the sub-objs
;;;
(defmethod 2d-rotate ((self 2d-shape) theta ox oy)
  (dolist (obj (sub-objs self))
	  (2d-rotate obj theta ox oy)))

;;;
;;; Scale a 2d-shape about the origin.  Just affects the sub-objs
;;;
(defmethod 2d-scale ((self 2d-shape) sf ox oy)
  (dolist (obj (sub-objs self))
	  (2d-scale obj sf ox oy)))


;;; ======================= Multimap Methods ======================
;;;
;;; The gravity-points of the 2d-shape.  Returns a list of param-vals.
;;; This is just a stub for the function, which is filled in by subclasses.
;;;
(defmethod gravity-pts ((self 2d-shape) do-midpt)
  (declare (ignore do-midpt))
  nil)

;;;
;;; The point on the 2d-shape at the given parametric value.  This is just a
;;; stub for the function, which is filled in by subclasses.
;;;
(defmethod point ((self 2d-shape) value)
  (declare (ignore value))
  (alloc-2d 0 0))

;;;
;;; The normal of the 2d-shape at the given parametric value.  This is just a
;;; stub for the function, which is filled in by subclasses.
;;;
(defmethod normal ((self 2d-shape) value)
  (declare (ignore value))
  (alloc-2d 0 -1))

;;;
;;; The point/normal of the 2d-shape.  This is just a stub for the function,
;;; which is filled in by subclasses.
;;;
(defmethod pt-normal ((self 2d-shape) value)
  (list (point self value) (normal self value)))

;;;
;;; Return a list of (param-value dist) triplets.
;;;
(defmethod nearest-pt ((self 2d-shape) pt)
  (declare (ignore pt))
  (list 0.0 most-positive-single-float))

;;; ===================== Misc functions ======================
;;;
;;; Bounding box of a 2d-shape.  Since we don't know anything about 
;;; the 2d-shape, we'll just return a null bbox.
;;;
(defmethod find-bbox ((self 2d-shape))
  nil)

;;;
;;; bbox function uses the cache.
;;;
(defun bbox (self)
  (cond ((slot-value self 'bbox))
	(t (setf (slot-value self 'bbox) (find-bbox self)))))

;;;
;;; Copy a 2d-shape.
;;;
(defmethod copy ((self 2d-shape))
  (let ((rv (call-next-method)))
       (setf (ctrl-pts rv) (copy-tree (ctrl-pts self)))
       rv))

;;;
;;; Bounding box of a sub-objs of a 2d-shape.
;;;
(defmethod sub-objs-bbox ((self 2d-shape))
  (let ((rv nil))
       (dolist (obj (sub-objs self))
	       (setq rv (2d-bb-union rv (bbox obj)))
	       (setq rv (2d-bb-union rv (sub-objs-bbox obj))))
       rv))

