;;;
;;; Copyright (c) 1990 Regents of the University of California
;;;
;;; Definition for 2d bounding-boxes.  
;;;
;;; A bounding box is either a list of four numbers, giving the
;;; (xmin, xmax, ymin, ymax) coordinates, or nil, indicating a zero size
;;; bounding box.  Operations are 2d-bb-union, 2d-bb-intersection, 
;;; 2d-pt-in-bb and 2d-pt-near-bb.
;;;

(in-package 'pt :use '(lisp pcl))

(defmacro nil-bbox-p (bb) `(null ,bb))
(defmacro 2d-bbox-p (bb) `(= 4 (length ,bb)))

(defmacro make-2d-bbox (xmin ymin xmax ymax) `(list ,xmin ,ymin ,xmax ,ymax))

(defmacro 2d-bb-xmin (bb) `(first ,bb))
(defmacro 2d-bb-ymin (bb) `(second ,bb))
(defmacro 2d-bb-xmax (bb) `(third ,bb))
(defmacro 2d-bb-ymax (bb) `(fourth ,bb))

;;;
;;; Return union (a bounding box) of two bounding boxes.
;;;
(defun 2d-bb-union (b1 b2)
  (cond ((null b1) b2)
	((null b2) b1)
	(t
	 (make-2d-bbox (min (2d-bb-xmin b1) (2d-bb-xmin b2))
		       (min (2d-bb-ymin b1) (2d-bb-ymin b2))
		       (max (2d-bb-xmax b1) (2d-bb-xmax b2))
		       (max (2d-bb-ymax b1) (2d-bb-ymax b2))))))

;;;
;;; Return intersection (a bounding box) of two bounding boxes.
;;;
(defun 2d-bb-intersection (b1 b2)
  (cond ((null b1) nil)
	((null b2) nil)
	(t
	 (setq b1 (make-2d-bbox (max (2d-bb-xmin b1) (2d-bb-xmin b2))
				(max (2d-bb-ymin b1) (2d-bb-ymin b2))
				(min (2d-bb-xmax b1) (2d-bb-xmax b2))
				(min (2d-bb-ymax b1) (2d-bb-ymax b2))))
	 (if (or (>= (2d-bb-xmin b1) (2d-bb-xmax b1))
		 (>= (2d-bb-ymin b1) (2d-bb-ymax b1)))
	     nil
	     b1))))

;;;
;;; Return whether the two bounding boxes intersect.
;;;
(defun 2d-bb-intersect-p (b1 b2)
  (cond ((null b1) nil)
	((null b2) nil)
	(t
	 (and (<= (2d-bb-xmin b1) (2d-bb-xmax b2))
	      (<= (2d-bb-xmin b2) (2d-bb-xmax b1))
	      (<= (2d-bb-ymin b1) (2d-bb-ymax b2))
	      (<= (2d-bb-ymin b2) (2d-bb-ymax b1))))))

;;;
;;; Return whether a point is within a bounding box.
;;;
(defun 2d-pt-in-bb (pt bb)
  (if (null bb)
      nil
      (and bb
	   (<= (2d-bb-xmin bb) (2d-point-x pt) (2d-bb-xmax bb)) 
	   (<= (2d-bb-ymin bb) (2d-point-y pt) (2d-bb-ymax bb)))))

;;;
;;; Return whether a point is within d-tol of a bounding box.
;;;
(defun 2d-pt-near-bb (pt bb d-tol)
  (if (null bb)
      nil
      (and bb 
	   (<= (- (2d-bb-xmin bb) d-tol) 
	       (2d-point-x pt)
	       (+ (2d-bb-xmax bb)  d-tol))
	   (<= (- (2d-bb-ymin bb) d-tol) 
	       (2d-point-y pt)
	       (+ (2d-bb-ymax bb) d-tol)))))
